// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Creates a coarsened version of the input
		\author Romain Behar (romainbehar@yahoo.com)
*/

#include <k3dsdk/basic_math.h>
#include <k3dsdk/object.h>
#include <k3dsdk/persistence.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/mesh_filter.h>
#include <k3dsdk/module.h>
#include <k3dsdk/plugins.h>
#include <k3dsdk/utility.h>

#include "gts_interface.h"

namespace libk3dgts
{

static gdouble cost_angle(GtsEdge* e)
{
	if(e->triangles && e->triangles->next)
		{
			GtsTriangle* t1 = static_cast<GtsTriangle*>(e->triangles->data);
			GtsTriangle* t2 = static_cast<GtsTriangle*>(e->triangles->next->data);
			return std::fabs(gts_triangles_angle(t1, t2));
		}

	return G_MAXDOUBLE;
}

static GtsVolumeOptimizedParams volume_params = { 0.5, 0.5, 0. };

/////////////////////////////////////////////////////////////////////////////
// coarsen_polyhedra_implementation

class coarsen_polyhedra_implementation :
	public k3d::mesh_filter<k3d::persistent<k3d::object> >
{
	typedef k3d::mesh_filter<k3d::persistent<k3d::object> > base;

public:
	coarsen_polyhedra_implementation(k3d::idocument& Document) :
		base(Document),
		m_cost_function(k3d::init_name("cost_function") + k3d::init_description("Cost function [enumeration]") + k3d::init_value(LENGTH) + k3d::init_enumeration(cost_values()) + k3d::init_document(Document)),
		m_midvertex_function(k3d::init_name("midvertex_function") + k3d::init_description("Mid-vertex function [enumeration]") + k3d::init_value(VOLUMEOPTIMIZED) + k3d::init_enumeration(midvertex_values()) + k3d::init_document(Document)),
		m_stop_function(k3d::init_name("stop_function") + k3d::init_description("Stop function [enumeration]") + k3d::init_value(EDGENUMBER) + k3d::init_enumeration(stop_values()) + k3d::init_document(Document)),
		m_max_fold_angle(k3d::init_name("max_fold_angle") + k3d::init_description("Maximum fold angle [number]") + k3d::init_value(k3d::radians<double>(1.0)) + k3d::init_precision(2) + k3d::init_step_increment(0.01) + k3d::init_units(typeid(k3d::measurement::angle)) + k3d::init_document(Document)),
		m_edge_number(k3d::init_name("edge_number") + k3d::init_description("Final edge number [integer]") + k3d::init_value(100) + k3d::init_constraint(k3d::constraint::minimum(1UL)) + k3d::init_precision(0) + k3d::init_step_increment(1) + k3d::init_units(typeid(k3d::measurement::scalar)) + k3d::init_document(Document)),
		m_max_edge_cost(k3d::init_name("max_edge_cost") + k3d::init_description("Maximum edge cost [number]") + k3d::init_value(1.0) + k3d::init_precision(2) + k3d::init_step_increment(0.01) + k3d::init_units(typeid(k3d::measurement::scalar)) + k3d::init_document(Document)),
		m_volume_weight(k3d::init_name("volume_weight") + k3d::init_description("Weight used for volume optimization [number]") + k3d::init_value(0.5) + k3d::init_precision(2) + k3d::init_step_increment(0.1) + k3d::init_units(typeid(k3d::measurement::scalar)) + k3d::init_document(Document)),
		m_boundary_weight(k3d::init_name("boundary_weight") + k3d::init_description("Weight used for boundary optimization [number]") + k3d::init_value(0.5) + k3d::init_precision(2) + k3d::init_step_increment(0.1) + k3d::init_units(typeid(k3d::measurement::scalar)) + k3d::init_document(Document)),
		m_shape_weight(k3d::init_name("shape_weight") + k3d::init_description("Weight used for shape optimization [number]") + k3d::init_value(0.0) + k3d::init_precision(2) + k3d::init_step_increment(0.1) + k3d::init_units(typeid(k3d::measurement::scalar)) + k3d::init_document(Document))
	{
		enable_serialization(k3d::persistence::proxy(m_cost_function));
		enable_serialization(k3d::persistence::proxy(m_midvertex_function));
		enable_serialization(k3d::persistence::proxy(m_stop_function));
		enable_serialization(k3d::persistence::proxy(m_max_fold_angle));
		enable_serialization(k3d::persistence::proxy(m_edge_number));
		enable_serialization(k3d::persistence::proxy(m_max_edge_cost));
		enable_serialization(k3d::persistence::proxy(m_volume_weight));
		enable_serialization(k3d::persistence::proxy(m_boundary_weight));
		enable_serialization(k3d::persistence::proxy(m_shape_weight));

		register_property(m_cost_function);
		register_property(m_midvertex_function);
		register_property(m_stop_function);
		register_property(m_max_fold_angle);
		register_property(m_edge_number);
		register_property(m_max_edge_cost);
		register_property(m_volume_weight);
		register_property(m_boundary_weight);
		register_property(m_shape_weight);

		m_input_mesh.changed_signal().connect(SigC::slot(*this, &coarsen_polyhedra_implementation::on_reset_geometry));
		m_cost_function.changed_signal().connect(SigC::slot(*this, &coarsen_polyhedra_implementation::on_reset_geometry));
		m_midvertex_function.changed_signal().connect(SigC::slot(*this, &coarsen_polyhedra_implementation::on_reset_geometry));
		m_stop_function.changed_signal().connect(SigC::slot(*this, &coarsen_polyhedra_implementation::on_reset_geometry));
		m_max_fold_angle.changed_signal().connect(SigC::slot(*this, &coarsen_polyhedra_implementation::on_reset_geometry));
		m_edge_number.changed_signal().connect(SigC::slot(*this, &coarsen_polyhedra_implementation::on_reset_geometry));
		m_max_edge_cost.changed_signal().connect(SigC::slot(*this, &coarsen_polyhedra_implementation::on_reset_geometry));
		m_volume_weight.changed_signal().connect(SigC::slot(*this, &coarsen_polyhedra_implementation::on_reset_geometry));
		m_boundary_weight.changed_signal().connect(SigC::slot(*this, &coarsen_polyhedra_implementation::on_reset_geometry));
		m_shape_weight.changed_signal().connect(SigC::slot(*this, &coarsen_polyhedra_implementation::on_reset_geometry));
		m_output_mesh.need_data_signal().connect(SigC::slot(*this, &coarsen_polyhedra_implementation::on_create_geometry));
	}

	void on_reset_geometry()
	{
		m_output_mesh.reset();
	}

	k3d::mesh* on_create_geometry()
	{
		// TODO : process each polyhedron (+ commit_mesh_changes)

		// Get the input geometry ...
		k3d::mesh* const input = m_input_mesh.property_value();
		if(!input)
			return 0;

		const double fold = k3d::radians<double>(m_max_fold_angle.property_value());

		GtsSurface* surface = gts_surface(*input);

		// Select the right coarsening process
		GtsKeyFunc cost_func = 0;
		gpointer cost_data = 0;
		switch(m_cost_function.property_value())
			{
				case OPTIMIZED:
					cost_func = (GtsKeyFunc) gts_volume_optimized_cost;
					cost_data = &volume_params;

					volume_params.volume_weight = static_cast<gdouble>(m_volume_weight.property_value());
					volume_params.boundary_weight = static_cast<gdouble>(m_boundary_weight.property_value());
					volume_params.shape_weight = static_cast<gdouble>(m_shape_weight.property_value());
				break;
				case LENGTH:
					cost_func = 0;
				break;
				case ANGLE:
					cost_func = (GtsKeyFunc) cost_angle;
				break;
				default:
					assert_not_reached();
			}

		GtsCoarsenFunc coarsen_func = 0;
		gpointer coarsen_data = 0;
		GtsStopFunc stop_func = 0;
		gpointer stop_data = 0;
		switch(m_midvertex_function.property_value())
			{
				case MIDVERTEX:
					coarsen_func = 0;
				break;
				case VOLUMEOPTIMIZED:
					coarsen_func = (GtsCoarsenFunc) gts_volume_optimized_vertex;
					coarsen_data = &volume_params;

					volume_params.volume_weight = static_cast<gdouble>(m_volume_weight.property_value());
					volume_params.boundary_weight = static_cast<gdouble>(m_boundary_weight.property_value());
					volume_params.shape_weight = static_cast<gdouble>(m_shape_weight.property_value());
				break;
				default:
					assert_not_reached();
			}

		guint edge_number = static_cast<guint>(m_edge_number.property_value());
		gdouble max_edge_cost = static_cast<gdouble>(m_max_edge_cost.property_value());
		switch(m_stop_function.property_value())
			{
				case EDGENUMBER:
					stop_func = (GtsStopFunc) gts_coarsen_stop_number;
					stop_data = &edge_number;
				break;
				case COST:
					stop_func = (GtsStopFunc) gts_coarsen_stop_cost;
					stop_data = &max_edge_cost;
					break;
				default:
					assert_not_reached ();
			}

		gts_surface_coarsen(surface, cost_func, cost_data, coarsen_func, coarsen_data, stop_func, stop_data, fold);

		// Create output geometry ...
		k3d::mesh* const output = new k3d::mesh();
		copy_surface(surface, *output);

		// Copy materials ...
		if(input->polyhedra.size() && output->polyhedra.size())
			output->polyhedra.front()->material = input->polyhedra.front()->material;

		return output;
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<coarsen_polyhedra_implementation>,
				k3d::interface_list<k3d::imesh_source,
				k3d::interface_list<k3d::imesh_sink > > > factory(
				k3d::uuid(0xdaae61bd, 0xd5b94f9b, 0x90a54f79, 0xf3f78729),
				"CoarsenPolyhedra",
				"Coarsens polygonal surfaces",
				"Objects",
				k3d::iplugin_factory::EXPERIMENTAL);

		return factory;
	}

private:
	/// Enumerates cost function types
	typedef enum
	{
		OPTIMIZED,
		LENGTH,
		ANGLE
	} cost_t;

	friend std::ostream& operator << (std::ostream& Stream, const cost_t& Value)
	{
		switch(Value)
			{
				case OPTIMIZED:
					Stream << "optimized";
					break;
				case LENGTH:
					Stream << "length";
					break;
				case ANGLE:
					Stream << "angle";
					break;
			}

		return Stream;
	}

	friend std::istream& operator >> (std::istream& Stream, cost_t& Value)
	{
		std::string text;
		Stream >> text;

		if(text == "optimized")
			Value = OPTIMIZED;
		else if(text == "length")
			Value = LENGTH;
		else if(text == "angle")
			Value = ANGLE;
		else
			std::cerr << __PRETTY_FUNCTION__ << ": unknown enumeration [" << text << "]" << std::endl;

		return Stream;
	}

	const k3d::ienumeration_property::values_t& cost_values()
	{
		static k3d::ienumeration_property::values_t values;
		if(values.empty())
			{
				values.push_back(k3d::ienumeration_property::value_t("Optimized", "optimized", "Use optimized point cost"));
				values.push_back(k3d::ienumeration_property::value_t("Length", "length", "Use length^2 as cost function"));
				values.push_back(k3d::ienumeration_property::value_t("Angle", "angle", "Use angle as cost function"));
			}

		return values;
	}

	/// Enumerates mid-vertex functions
	typedef enum
	{
		MIDVERTEX,
		VOLUMEOPTIMIZED
	} midvertex_t;

	friend std::ostream& operator << (std::ostream& Stream, const midvertex_t& Value)
	{
		switch(Value)
			{
				case MIDVERTEX:
					Stream << "midvertex";
				break;
				case VOLUMEOPTIMIZED:
					Stream << "volumeoptimized";
				break;
			}

		return Stream;
	}

	friend std::istream& operator >> (std::istream& Stream, midvertex_t& Value)
	{
		std::string text;
		Stream >> text;

		if(text == "midvertex")
			Value = MIDVERTEX;
		else if(text == "volumeoptimized")
			Value = VOLUMEOPTIMIZED;
		else
			std::cerr << __PRETTY_FUNCTION__ << ": unknown enumeration [" << text << "]" << std::endl;

		return Stream;
	}

	const k3d::ienumeration_property::values_t& midvertex_values()
	{
		static k3d::ienumeration_property::values_t values;
		if(values.empty())
			{
				values.push_back(k3d::ienumeration_property::value_t("Mid-vertex", "midvertex", "Use mid-vertex as replacement vertex"));
				values.push_back(k3d::ienumeration_property::value_t("Volume optimized", "volumeoptimized", "Use volume optimized point"));
			}

		return values;
	}

	/// Enumerates stop functions
	typedef enum
	{
		EDGENUMBER,
		COST
	} stop_t;

	friend std::ostream& operator << (std::ostream& Stream, const stop_t& Value)
	{
		switch(Value)
			{
				case EDGENUMBER:
					Stream << "number";
				break;
				case COST:
					Stream << "cost";
				break;
			}

		return Stream;
	}

	friend std::istream& operator >> (std::istream& Stream, stop_t& Value)
	{
		std::string text;
		Stream >> text;

		if(text == "number")
			Value = EDGENUMBER;
		else if(text == "cost")
			Value = COST;
		else
			std::cerr << __PRETTY_FUNCTION__ << ": unknown enumeration [" << text << "]" << std::endl;

		return Stream;
	}

	const k3d::ienumeration_property::values_t& stop_values()
	{
		static k3d::ienumeration_property::values_t values;
		if(values.empty())
			{
				values.push_back(k3d::ienumeration_property::value_t("Number", "number", "Stop he coarsening process if the number of edges was to fall below"));
				values.push_back(k3d::ienumeration_property::value_t("Cost", "cost", "Stop the coarsening process if the cost of collapsing an edge is larger"));
			}

		return values;
	}

	k3d_enumeration_property(cost_t, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_cost_function;
	k3d_enumeration_property(midvertex_t, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_midvertex_function;
	k3d_enumeration_property(stop_t, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_stop_function;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_max_fold_angle;
	k3d_measurement_property(unsigned long, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::with_constraint) m_edge_number;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_max_edge_cost;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_volume_weight;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_boundary_weight;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_shape_weight;
};

/////////////////////////////////////////////////////////////////////////////
// coarsen_polyhedra_factory

k3d::iplugin_factory& coarsen_polyhedra_factory()
{
	return coarsen_polyhedra_implementation::get_factory();
}

} // namespace libk3dgts


