// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Anders Dahnielson (anders@dahnielson.com)
*/

#include <k3dsdk/module.h>
#include <k3dsdk/plugins.h>

#include "bitmap_filter_simple.h"

namespace libk3dbitmap
{

/////////////////////////////////////////////////////////////////////////////
// bitmap_add_implementation

class bitmap_add_implementation :
	public bitmap_filter_simple
{
	typedef bitmap_filter_simple base;

public:
	bitmap_add_implementation(k3d::idocument& Document) :
		base(Document),
		m_value(k3d::init_name("value") + k3d::init_description("Value [double]") + k3d::init_value(0.0) + k3d::init_document(Document))
	{
		enable_serialization(k3d::persistence::proxy(m_value));
		register_property(m_value);
		m_value.changed_signal().connect(SigC::slot(*this, &bitmap_add_implementation::on_value_change));
	}

	void on_value_change()
	{
		m_value_cache = m_value.property_value();
		on_reset_bitmap();
	}
	
        void on_filter(const k3d::pixel& I, k3d::pixel& O)
	{
		double I_red   = k3d::color_traits<double>::convert(I.red);
		double I_green = k3d::color_traits<double>::convert(I.green);
		double I_blue  = k3d::color_traits<double>::convert(I.blue);
		double I_alpha = k3d::color_traits<double>::convert(I.alpha);

		// Matte divide ...
		if (I_alpha != 0)
		{
			I_red   = I_red   / I_alpha;
			I_green = I_green / I_alpha;
			I_blue  = I_blue  / I_alpha;
		}

		// Color operation ...
		const double O_red   = I_red   + m_value_cache;
		const double O_green = I_green + m_value_cache;
		const double O_blue  = I_blue  + m_value_cache;

		// Matte multiply ...
		O.red   = k3d::bitmap::pixel_type::sample_traits::convert(O_red   * I_alpha);
		O.green = k3d::bitmap::pixel_type::sample_traits::convert(O_green * I_alpha);
		O.blue  = k3d::bitmap::pixel_type::sample_traits::convert(O_blue  * I_alpha);
		O.alpha = I.alpha;
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<bitmap_add_implementation>,
			k3d::interface_list<k3d::ibitmap_source,
			k3d::interface_list<k3d::ibitmap_sink> > > factory(
				k3d::uuid(0x30d648f9, 0x74844196, 0x801a34f9, 0x6ff15920),
				"BitmapAdd",
				"Add value to each pixel",
				"Objects",
				k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d_data_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_value;
	double m_value_cache;
};

/////////////////////////////////////////////////////////////////////////////
// bitmap_add_factory

k3d::iplugin_factory& bitmap_add_factory()
{
	return bitmap_add_implementation::get_factory();
}

} // namespace libk3dbitmap

