// K3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Implements plugin functions
		\author Tim Shead (tshead@k-3d.com)
*/

#include "data.h"
#include "iapplication_plugin_factory.h"
#include "ideletable.h"
#include "idocument.h"
#include "idocument_plugin_factory.h"
#include "iobject_collection.h"
#include "iobject.h"
#include "plugins.h"
#include "result.h"
#include "utility.h"

namespace k3d
{

iplugin_factory* plugin(const uuid& ClassID)
{
	for(k3d::iplugin_factory_collection::factories_t::const_iterator factory = k3d::application().plugins().begin(); factory != k3d::application().plugins().end(); ++factory)
		if((*factory)->class_id() == ClassID)
			return *factory;

	return 0;
}

const iplugin_factory_collection::factories_t plugins(const std::string PluginName)
{
	iplugin_factory_collection::factories_t factories;

	for(k3d::iplugin_factory_collection::factories_t::const_iterator factory = k3d::application().plugins().begin(); factory != k3d::application().plugins().end(); ++factory)
		if((*factory)->name() == PluginName)
			factories.insert(*factory);

	return factories;
}

iunknown* create_application_plugin(iplugin_factory& Factory)
{
	iapplication_plugin_factory* const factory = dynamic_cast<iapplication_plugin_factory*>(&Factory);
	return_val_if_fail(factory, 0);

	return factory->create_plugin();
}

iunknown* create_application_plugin(const uuid& ClassID)
{
	iplugin_factory* const factory = plugin(ClassID);
	return_val_if_fail(factory, 0);

	return create_application_plugin(*factory);
}

iunknown* create_application_plugin(const std::string& PluginName)
{
	const factories_t factories(plugins(PluginName));
	return_val_if_fail(1 == factories.size(), 0);

	return create_application_plugin(**factories.begin());
}

iobject* create_document_plugin(iplugin_factory& Factory, idocument& Document, const std::string& Name)
{
	// Make sure this is a document object factory ...
	idocument_plugin_factory* const factory = dynamic_cast<idocument_plugin_factory*>(&Factory);
	return_val_if_fail(factory, 0);

	// Create the object ...
	iobject* object = factory->create_plugin(Document);
	return_val_if_fail(object, 0);

	// Give the object a unique ID ...
	object->set_id(Document.objects().next_available_id());
	
	// Set the object's name ...
	object->set_name(Name);

	// Make sure the object gets cleaned-up properly after an undo ...
	k3d::undoable_new(dynamic_cast<k3d::ideletable*>(object), Document);

	// Add it to the document object collection ...
	Document.objects().add_objects(make_collection<iobject_collection::objects_t>(object));

	return object;
}

iobject* create_document_plugin(const uuid& ClassID, idocument& Document, const std::string& Name)
{
	iplugin_factory* const factory = plugin(ClassID);
	return_val_if_fail(factory, 0);

	return create_document_plugin(*factory, Document, Name);
}

iobject* create_document_plugin(const std::string& PluginName, idocument& Document, const std::string& Name)
{
	const factories_t factories(plugins(PluginName));
	return_val_if_fail(1 == factories.size(), 0);

	return create_document_plugin(**factories.begin(), Document, Name);
}

} // namespace k3d


