
/*
 * vfs.C -- written for Juice
 *	Copyright (C) 1999, 2000, 2001 Abraham vd Merwe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifndef DIALOGS_VFS_C
#define DIALOGS_VFS_C

#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <unistd.h>
#include <time.h>

#include "typedefs.h"
#include "vfs.h"
#include "regexp.h"

FifoStack::FifoStack ()
{
   fifostack = NULL;
}

FifoStack::~FifoStack ()
{
   FifoStackList *tmp;
   while (fifostack != NULL)
	 {
		if (fifostack->next == NULL)
		  {
			 free (fifostack->buffer);
			 free (fifostack);
			 fifostack = NULL;
		  }
		else
		  {
			 tmp = fifostack;
			 while (tmp->next->next != NULL) tmp = tmp->next;
			 free (tmp->next->buffer);
			 free (tmp->next);
			 tmp->next = NULL;
		  }
	 }
}

void FifoStack::Push (const char *buffer)
{
   FifoStackList *nw,*tmp;
   nw = (FifoStackList *) malloc (sizeof (FifoStackList));
   nw->buffer = (char *) malloc (strlen (buffer) + 1);
   strcpy (nw->buffer,buffer);
   nw->next = NULL;
   if (fifostack == NULL)
     fifostack = nw;
   else
     {
		tmp = fifostack;
		while (tmp->next != NULL) tmp = tmp->next;
		tmp->next = nw;
	 }
}

char *FifoStack::Pull ()
{
   FifoStackList *tmp;
   char *buffer = NULL;
   if (fifostack != NULL)
	 {
		tmp = fifostack;
		fifostack = fifostack->next;
		buffer = tmp->buffer;
		free (tmp);
	 }
   return buffer;
}

char *FifoStack::Pop ()
{
   FifoStackList *tmp;
   char *buffer = NULL;
   if (fifostack != NULL)
	 {
		if (fifostack->next == NULL)
		  {
			 buffer = fifostack->buffer;
			 free (fifostack);
			 fifostack = NULL;
		  }
		else
		  {
			 tmp = fifostack;
			 while (tmp->next->next != NULL) tmp = tmp->next;
			 buffer = tmp->next->buffer;
			 free (tmp->next);
			 tmp->next = NULL;
		  }
	 }
   return buffer;
}

bool FifoStack::IsEmpty ()
{
   return (fifostack == NULL);
}

VirtualFS::VirtualFS ()
{
   nentries = 0;
   shortenmethod = METHOD_SPLIT;
   entries = NULL;
}

VirtualFS::~VirtualFS ()
{
   if (nentries)
	 {
		free (entries);
		nentries = 0;
	 }
}

bool VirtualFS::cutfield (char *field,const char *pathname,int fieldno)
{
   int i = 0,j,curfield = 0;
   while ((curfield < fieldno) && (i < (int) strlen (pathname)))
	 {
		while ((pathname[i] == '/') && (i < (int) strlen (pathname))) i++;
		curfield++;
		j = 0;
		while ((pathname[i] != '/') && (i < (int) strlen (pathname))) field[j++] = pathname[i++];
		field[j] = '\0';
	 }
   return (curfield == fieldno ? TRUE : FALSE);
}

void VirtualFS::shortname (char **newname,const char *fullname,int maxlen)
{
   int len = (maxlen - 1) >> 1;
   if ((int) strlen (fullname) <= maxlen)
     {
		*newname = (char *) malloc (strlen (fullname) + 1);
		strcpy (*newname,fullname);
	 }
   else
     {
		if (shortenmethod == METHOD_SPLIT)
		  {
			 *newname = (char *) malloc (maxlen + 1);
			 strncpy (*newname,fullname,len);
			 (*newname)[len] = '~';
			 strncpy (*newname + len + 1,fullname + strlen (fullname) - len,len);
			 (*newname)[maxlen] = '\0';
		  }
		else /*if (shortenmethod == METHOD_TRUNCATE)*/
		  {
			 *newname = (char *) malloc (maxlen + 1);
			 strncpy (*newname,fullname,maxlen);
			 (*newname)[maxlen] = '\0';
		  }
	 }
   /* we don't want TAB's in the shortened name */
   for (int i = 0; i < (int) strlen (*newname); i++) if ((*newname)[i] == '\t') (*newname)[i] = ' ';
}

bool VirtualFS::isdir (const char *dirname,const char *filename)
{
   struct stat filestat;
   char *pathname;
   pathname = (char *) malloc (strlen (dirname) + strlen (filename) + 2);
   strcpy (pathname,dirname);
   strcat (pathname,"/");
   strcat (pathname,filename);
   if (stat (pathname,&filestat) != 0)
	 {
		free (pathname);
		return FALSE;
	 }
   free (pathname);
   return (S_ISDIR (filestat.st_mode));
}

bool VirtualFS::isdir2 (const char *name)
{
   struct stat filestat;
   if (stat (name,&filestat) != 0) return FALSE;
   return (S_ISDIR (filestat.st_mode));
}

void VirtualFS::CreateDirListing (const char *dirname,int maxdirlen)
{
   DIR *directory = opendir (dirname);
   struct dirent *entry;
   char *tmp;
   int i = 0;
   while ((entry = readdir (directory)) != NULL) if (isdir (dirname,entry->d_name) && (strcmp (entry->d_name,".") != 0)) i++;
   closedir (directory);
   nentries = i;
   entries = (EntryRec *) malloc (nentries * sizeof (EntryRec));
   directory = opendir (dirname);
   i = 0;
   while ((entry = readdir (directory)) != NULL) if (isdir (dirname,entry->d_name) && (strcmp (entry->d_name,".") != 0))
     {
		entries[i].filename = (char *) malloc (strlen (entry->d_name) + strlen (dirname) + 2);
		strcpy (entries[i].filename,dirname);
		strcat (entries[i].filename,"/");
		strcat (entries[i].filename,entry->d_name);
		shortname (&tmp,entry->d_name,maxdirlen);
		entries[i].name = (char *) malloc (strlen (tmp) + 1);
		strcpy (entries[i].name,tmp);
		free (tmp);
		entries[i].type = isdir (dirname,entry->d_name) ? FT_DIR : FT_FILE;
		i++;
	 }
   closedir (directory);
}

void VirtualFS::CreateFileListing (const char *dirname,const char *filespec,int maxfilelen)
{
   DIR *directory = opendir (dirname);
   struct dirent *entry;
   char *tmp;
   int i = 0;
   Match mask;
   while ((entry = readdir (directory)) != NULL)
	 if ((strcmp (entry->d_name,".") != 0) && (isdir (dirname,entry->d_name) || mask.MatchPattern (entry->d_name,filespec) || (strlen (filespec) == 0))) i++;
   closedir (directory);
   nentries = i;
   entries = (EntryRec *) malloc (nentries * sizeof (EntryRec));
   directory = opendir (dirname);
   i = 0;
   while ((entry = readdir (directory)) != NULL)
	 if ((strcmp (entry->d_name,".") != 0) && (isdir (dirname,entry->d_name) || mask.MatchPattern (entry->d_name,filespec) || (strlen (filespec) == 0)))
	 {
		entries[i].filename = (char *) malloc (strlen (entry->d_name) + strlen (dirname) + 2);
		strcpy (entries[i].filename,dirname);
		strcat (entries[i].filename,"/");
		strcat (entries[i].filename,entry->d_name);
		shortname (&tmp,entry->d_name,maxfilelen);
		entries[i].name = (char *) malloc (strlen (tmp) + 1);
		strcpy (entries[i].name,tmp);
		free (tmp);
		entries[i].type = isdir (dirname,entry->d_name) ? FT_DIR : FT_FILE;
		i++;
	 }
   closedir (directory);
}

void VirtualFS::AddEntry (const EntryRec *entry)
{
   if (nentries == 0) entries = NULL;
   entries = (EntryRec *) realloc (entries,(nentries + 1) * sizeof (EntryRec));
   entries[nentries].filename = (char *) malloc (strlen (entry->filename) + 1);
   entries[nentries].name = (char *) malloc (strlen (entry->name) + 1);
   entries[nentries].type = entry->type;
   strcpy (entries[nentries].filename,entry->filename);
   strcpy (entries[nentries].name,entry->name);
   nentries++;
}

void VirtualFS::ChangeEntry (int entryno,const EntryRec *entry)
{
   entries[entryno].filename = (char *) realloc (entries[entryno].filename,strlen (entry->filename) + 1);
   strcpy (entries[entryno].filename,entry->filename);
   entries[entryno].name = (char *) realloc (entries[entryno].name,strlen (entry->name) + 1);
   strcpy (entries[entryno].name,entry->name);
   entries[entryno].type = entry->type;
}

void VirtualFS::RemoveEntry (int entryno)
{
   nentries--;
   for (int i = entryno; i < nentries; i++)
     {
		free (entries[i].filename);
		free (entries[i].name);
		entries[i].filename = (char *) malloc (strlen (entries[i + 1].filename) + 1);
		entries[i].name = (char *) malloc (strlen (entries[i + 1].name) + 1);
		entries[i].type = entries[i + 1].type;
		strcpy (entries[i].filename,entries[i + 1].filename);
		strcpy (entries[i].name,entries[i + 1].name);
	 }
   entries = (EntryRec *) realloc (entries,nentries * sizeof (EntryRec));
}

void VirtualFS::SwapEntries (int entry1,int entry2)
{
   char *tmp;
   int i;
   tmp = entries[entry1].filename;
   entries[entry1].filename = entries[entry2].filename;
   entries[entry2].filename = tmp;
   tmp = entries[entry1].name;
   entries[entry1].name = entries[entry2].name;
   entries[entry2].name = tmp;
   i = entries[entry1].type;
   entries[entry1].type = entries[entry2].type;
   entries[entry2].type = i;
}

int VirtualFS::NumEntries ()
{
   return nentries;
}

void VirtualFS::RemoveEntries ()
{
   for (int i = 0; i < nentries; i++)
     {
		free (entries[i].filename);
		free (entries[i].name);
	 }
   free (entries);
   entries = NULL;
   nentries = 0;
}

void VirtualFS::ShuffleEntries ()
{
   int i,*shufflelist = (int *) malloc (nentries * sizeof (int));
   srand (time (NULL));
   for (i = 0; i < nentries; i++) shufflelist[i] = -1;
   for (i = 0; i < nentries; i++)
	 {
		int seed = (int) ((float) nentries * rand () / (RAND_MAX + 1.0));
		while (shufflelist[seed] != -1) seed = seed + 1 < nentries ? seed + 1 : 0;
		shufflelist[seed] = i;
	 }
   for (i = 0; i < nentries; i++) SwapEntries (i,shufflelist[i]);
   free (shufflelist);
}

const char *VirtualFS::getext (const char *str)
{
   const char *d;
   if (!strlen (str)) return "";
   for (d = str + strlen (str) - 1; d >= str; d--)
	 {
		if (*d == '.') return d + 1;
	 }
   return "";
}

int VirtualFS::compare (const char *s1,const char *s2)
{
   int result;
   bool ad,bd;
   ad = isdir2 (s1);
   bd = isdir2 (s2);
   if (sort_mode == SORT_NAME)
	 {
		if (ad == bd) result = strcmp (s1,s2); else result = bd - ad;
	 }
   else	/* sort_mode == SORT_EXT */
	 {
		if (ad == bd)
		  {
			 result = strcmp (getext (s1),getext (s2));
			 if (!result) result = strcmp (s1,s2);
		  }
		else result = bd - ad;
	 }
   return result;
}

void VirtualFS::quicksort (int l,int r)
{
   int i,j;
   char *m;
   i = l;
   j = r;
   m = EntryFilename ((l + r) >> 1);
   do
	 {
		while (compare (EntryFilename (i),m) < 0) i++;
		while (compare (m,EntryFilename (j)) < 0) j--;
		if (i <= j)
		  {
			 SwapEntries (i,j);
			 i++;
			 j--;
		  }
	 }
   while (i <= j);
   if (l < j) quicksort (l,j);
   if (i < r) quicksort (i,r);
}

void VirtualFS::SortEntries (int mode)
{
   sort_mode = mode;
   quicksort (0,NumEntries () - 1);
}

char *VirtualFS::EntryName (int entryno)
{
   return entries[entryno].name;
}

int VirtualFS::EntryType (int entryno)
{
   return entries[entryno].type;
}

char *VirtualFS::EntryFilename (int entryno)
{
   return entries[entryno].filename;
}

#endif
