/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

package org.apache.jmeter.protocol.http.gui;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.FlowLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.util.Iterator;

import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.JButton;
import javax.swing.JFileChooser;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.table.TableCellEditor;
import javax.swing.table.TableColumn;

import org.apache.jmeter.gui.util.FileDialoger;
import org.apache.jmeter.gui.util.HeaderAsPropertyRenderer;
import org.apache.jmeter.protocol.http.sampler.HTTPSamplerBase;
import org.apache.jmeter.protocol.http.util.HTTPFileArg;
import org.apache.jmeter.testelement.TestElement;
import org.apache.jmeter.util.JMeterUtils;
import org.apache.jorphan.gui.ObjectTableModel;
import org.apache.jorphan.reflect.Functor;

/*
 * Note: this class is currently only suitable for use with HTTSamplerBase.
 * If it is required for other classes, then the appropriate configure() and modifyTestElement()
 * method code needs to be written.
 */
/**
 * A GUI panel allowing the user to enter file information for http upload.
 * Used by MultipartUrlConfigGui for use in HTTP Samplers.
 */
public class HTTPFileArgsPanel extends JPanel implements ActionListener {

    /** The title label for this component. */
    private JLabel tableLabel;

    /** The table containing the list of files. */
    private transient JTable table;

    /** The model for the files table. */
    private transient ObjectTableModel tableModel;

    /** A button for adding new files to the table. */
    private JButton add;

    /** A button for browsing file system to set path of selected row in table. */
    private JButton browse;

    /** A button for removing files from the table. */
    private JButton delete;

    /** Command for adding a row to the table. */
    private static final String ADD = "add"; // $NON-NLS-1$

    /** Command for browsing filesystem to set path of selected row in table. */
    private static final String BROWSE = "browse"; // $NON-NLS-1$

    /** Command for removing a row from the table. */
    private static final String DELETE = "delete"; // $NON-NLS-1$

    private static final String FILEPATH = "send_file_filename_label"; // $NON-NLS-1$

    /** The parameter name column title of file table. */
    private static final String PARAMNAME = "send_file_param_name_label"; //$NON-NLS-1$

    /** The mime type column title of file table. */
    private static final String MIMETYPE = "send_file_mime_label"; //$NON-NLS-1$

    public HTTPFileArgsPanel() {
        this(""); // required for unit tests
    }

    /**
     * Create a new HTTPFileArgsPanel as an embedded component, using the
     * specified title.
     *
     * @param label
     *  the title for the component.
     */
    public HTTPFileArgsPanel(String label) {
        tableLabel = new JLabel(label);
        init();
    }

    /**
     * Initialize the table model used for the http files table.
     */
    private void initializeTableModel() {
        tableModel = new ObjectTableModel(new String[] {
                FILEPATH, PARAMNAME, MIMETYPE},
            HTTPFileArg.class,
            new Functor[] {
                new Functor("getPath"), //$NON-NLS-1$
                new Functor("getParamName"), //$NON-NLS-1$
                new Functor("getMimeType")}, //$NON-NLS-1$
            new Functor[] {
                new Functor("setPath"), //$NON-NLS-1$
                new Functor("setParamName"), //$NON-NLS-1$
                new Functor("setMimeType")}, //$NON-NLS-1$
            new Class[] {String.class, String.class, String.class});
    }

    public static boolean testFunctors(){
        HTTPFileArgsPanel instance = new HTTPFileArgsPanel(""); //$NON-NLS-1$
        instance.initializeTableModel();
        return instance.tableModel.checkFunctors(null,instance.getClass());
    }

    /**
     * Resize the table columns to appropriate widths.
     *
     * @param table
     *  the table to resize columns for
     */
    private void sizeColumns(JTable table) {
        int resizeMode = table.getAutoResizeMode();
        table.setAutoResizeMode(JTable.AUTO_RESIZE_OFF);
        fixSize(table.getColumn(PARAMNAME));
        fixSize(table.getColumn(MIMETYPE));
        table.setAutoResizeMode(resizeMode);
    }

    /**
     * Resize the table column to a fixed size.
     *
     * @param column
     *  the column whose size will be fixed
     */
    private void fixSize(TableColumn column) {
        column.sizeWidthToFit();
        column.setMaxWidth(column.getWidth() * 2);
        column.setWidth(column.getMaxWidth());
        column.setResizable(false);
    }

    /**
     * Save the GUI data in the HTTPSamplerBase element.
     * 
     * @param testElement
     */
    public void modifyTestElement(TestElement testElement) {
        stopTableEditing();
        if (testElement instanceof HTTPSamplerBase) {
            HTTPSamplerBase base = (HTTPSamplerBase) testElement;
            int rows = tableModel.getRowCount();
            Iterator modelData = tableModel.iterator();
            HTTPFileArg[] files = new HTTPFileArg[rows];
            int row=0;
            while (modelData.hasNext()) {
                HTTPFileArg file = (HTTPFileArg) modelData.next();
                files[row++]=file;
            }
            base.setHTTPFiles(files);
        }
    }

    /**
     * A newly created component can be initialized with the contents of a
     * HTTPSamplerBase object by calling this method. The component is responsible for
     * querying the Test Element object for the relevant information to display
     * in its GUI.
     *
     * @param testElement the HTTPSamplerBase to be used to configure the GUI
     */
    public void configure(TestElement testElement) {
        if (testElement instanceof HTTPSamplerBase) {
            HTTPSamplerBase base = (HTTPSamplerBase) testElement;
            tableModel.clearData();
            HTTPFileArg[] files = base.getHTTPFiles();
            for(int i=0; i< files.length; i++){
                tableModel.addRow(files[i]);            
            }
            checkDeleteAndBrowseStatus();
        }
    }


    /**
     * Enable or disable the delete button depending on whether or not there is
     * a row to be deleted.
     */
    private void checkDeleteAndBrowseStatus() {
        // Disable DELETE and BROWSE buttons if there are no rows in
        // the table to delete.
        if (tableModel.getRowCount() == 0) {
            browse.setEnabled(false);
            delete.setEnabled(false);
        } else {
            browse.setEnabled(true);
            delete.setEnabled(true);
        }
    }

    /**
     * Clear all rows from the table.
     */
    public void clear() {
        stopTableEditing();
        tableModel.clearData();
    }

    /**
     * Invoked when an action occurs. This implementation supports the add and
     * delete buttons.
     *
     * @param e
     *  the event that has occurred
     */
    public void actionPerformed(ActionEvent e) {
        String action = e.getActionCommand();
        if (action.equals(ADD)) {
            addFile(""); //$NON-NLS-1$
        }
        runCommandOnSelectedFile(action);
    }

    /**
     * runs specified command on currently selected file.
     *
     * @param command specifies which process will be done on selected
     * file. it's coming from action command currently catched by
     * action listener.
     *
     * @see runCommandOnRow
     */
    private void runCommandOnSelectedFile(String command) {
        // If a table cell is being edited, we must cancel the editing before
        // deleting the row
        if (table.isEditing()) {
            TableCellEditor cellEditor = table.getCellEditor(table.getEditingRow(), table.getEditingColumn());
            cellEditor.cancelCellEditing();
        }
        int rowSelected = table.getSelectedRow();
        if (rowSelected >= 0) {
            runCommandOnRow(command, rowSelected);
            tableModel.fireTableDataChanged();
            // Disable DELETE and BROWSE if there are no rows in the table to delete.
            checkDeleteAndBrowseStatus();
            // Table still contains one or more rows, so highlight (select)
            // the appropriate one.
            if (tableModel.getRowCount() != 0) {
                int rowToSelect = rowSelected;
                if (rowSelected >= tableModel.getRowCount()) {
                    rowToSelect = rowSelected - 1;
                }
                table.setRowSelectionInterval(rowToSelect, rowToSelect);
            }
        }
    }

    /**
     * runs specified command on currently selected table row.
     *
     * @param command specifies which process will be done on selected
     * file. it's coming from action command currently catched by
     * action listener.
     *
     * @param rowSelected index of selected row.
     */
    private void runCommandOnRow(String command, int rowSelected) {
        if (DELETE.equals(command)) {
            tableModel.removeRow(rowSelected);
        } else if (BROWSE.equals(command)) {
            String path = browseAndGetFilePath();
            tableModel.setValueAt(path, rowSelected, 0);
        }
    }

    /**
     * Add a new file row to the table.
     */
    private void addFile(String path) {
        // If a table cell is being edited, we should accept the current value
        // and stop the editing before adding a new row.
        stopTableEditing();

        tableModel.addRow(new HTTPFileArg(path));

        // Enable DELETE (which may already be enabled, but it won't hurt)
        delete.setEnabled(true);
        browse.setEnabled(true);

        // Highlight (select) the appropriate row.
        int rowToSelect = tableModel.getRowCount() - 1;
        table.setRowSelectionInterval(rowToSelect, rowToSelect);
    }

    /**
     * opens a dialog box to choose a file and returns selected file's
     * path.
     *
     * @return a new File object
     */
    private String browseAndGetFilePath() {
        String path = ""; //$NON-NLS-1$
        JFileChooser chooser = FileDialoger.promptToOpenFile();
        if (chooser != null) {
            File file = chooser.getSelectedFile();
            if (file != null) {
                path = file.getPath();
            }
        }
        return path;
    }

    /**
     * Stop any editing that is currently being done on the table. This will
     * save any changes that have already been made.
     */
    protected void stopTableEditing() {
        if (table.isEditing()) {
            TableCellEditor cellEditor = table.getCellEditor(table.getEditingRow(), table.getEditingColumn());
            cellEditor.stopCellEditing();
        }
    }

    /**
     * Create the main GUI panel which contains the file table.
     *
     * @return the main GUI panel
     */
    private Component makeMainPanel() {
        initializeTableModel();
        table = new JTable(tableModel);
        table.getTableHeader().setDefaultRenderer(new HeaderAsPropertyRenderer());
        table.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
        return makeScrollPane(table);
    }

    /**
     * Create a panel containing the title label for the table.
     *
     * @return a panel containing the title label
     */
    private Component makeLabelPanel() {
        JPanel labelPanel = new JPanel(new FlowLayout(FlowLayout.CENTER));
        labelPanel.add(tableLabel);
        return labelPanel;
    }

    /**
     * Create a panel containing the add and delete buttons.
     *
     * @return a GUI panel containing the buttons
     */
    private JPanel makeButtonPanel() {
        add = new JButton(JMeterUtils.getResString("add")); // $NON-NLS-1$
        add.setActionCommand(ADD);
        add.setEnabled(true);

        browse = new JButton(JMeterUtils.getResString("browse")); // $NON-NLS-1$
        browse.setActionCommand(BROWSE);

        delete = new JButton(JMeterUtils.getResString("delete")); // $NON-NLS-1$
        delete.setActionCommand(DELETE);

        checkDeleteAndBrowseStatus();

        JPanel buttonPanel = new JPanel();
        buttonPanel.setBorder(BorderFactory.createEmptyBorder(0, 10, 0, 10));
        add.addActionListener(this);
        browse.addActionListener(this);
        delete.addActionListener(this);
        buttonPanel.add(add);
        buttonPanel.add(browse);
        buttonPanel.add(delete);
        return buttonPanel;
    }

    /**
     * Initialize the components and layout of this component.
     */
    private void init() {
        JPanel p = this;

        p.setLayout(new BorderLayout());

        p.add(makeLabelPanel(), BorderLayout.NORTH);
        p.add(makeMainPanel(), BorderLayout.CENTER);
        // Force a minimum table height of 70 pixels
        p.add(Box.createVerticalStrut(70), BorderLayout.WEST);
        p.add(makeButtonPanel(), BorderLayout.SOUTH);

        table.revalidate();
        sizeColumns(table);
    }

    private JScrollPane makeScrollPane(Component comp) {
        JScrollPane pane = new JScrollPane(comp);
        pane.setPreferredSize(pane.getMinimumSize());
        return pane;
    }
}
