
/****************************************************************************
 *
 * All portions copyright their respective authors.  All rights reserved.
 *
 * This file is part of IVMan (ivm).
 *
 * This file may be distributed under the terms of the Q Public License
 * as defined by Troll Tech AS of Norway and appearing in the file
 * LICENSE.QPL included in the packaging of this file.
 * 
 * See http://www.troll.no/qpl for QPL licensing information.
 *
 * $Id: hal_interface.c,v 1.33 2006/01/19 08:53:47 ro_han Exp $
 *****************************************************************************/

#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>
#include <libhal.h>
#include <stdlib.h>

#include "common.h"
#include "manager.h"
#include "hal_interface.h"

#include <string.h>

#include "IvmConfig/IvmConfigProperties.h"
#include "IvmConfig/IvmConfigConditions.h"


void dump_devices();
void print_device(gpointer key, gpointer val, gpointer data);

// Code borrowed from Gnome-Volume-Manager by RML

/** Invoked when a device is added to the Global Device List. 
*
*  @param  ctx                 LibHal context
*  @param  udi                 Universal Device Id
*/
void
hal_device_added(LibHalContext * ctx __attribute__ ((__unused__)),
                 const char *udi)
{
    DEBUG(_("New Device: %s"), udi);
    
    DBusError dbus_error;
    dbus_error_init( &dbus_error );

    if ( libhal_device_property_exists (
           hal_ctx,
           udi,
           "info.locked",
           &dbus_error
         ) && libhal_device_get_property_bool (
           hal_ctx,
           udi,
           "info.locked",
           &dbus_error
         )
    ) {
        DEBUG(_("Device %s is ignored, as it is currently locked by another process."), udi );
        ivm_check_dbus_error( &dbus_error );
        return;
    }

    ivm_check_dbus_error( &dbus_error );
    ivm_media_changed(udi);

}

/** Invoked when a device is removed from the Global Device List. 
*
*  @param  ctx                 LibHal context
*  @param  udi                 Universal Device Id
*/
void
hal_device_removed(LibHalContext * ctx __attribute__ ((__unused__)),
                   const char *udi)
{
    DEBUG(_("Device removed: %s"), udi);
    char *device = g_hash_table_lookup(devices, (gpointer) udi);

    if (NULL != device)
    {
        ivm_umount_if_needed(udi);
        g_hash_table_remove(devices, (gpointer) udi);
    }
    // g_hash_table_foreach(execuns, print_device, NULL);

    // if there's any execun to be done, do it here
    char **execun = NULL;

    if ((execun = (char **) g_hash_table_lookup(execuns, (gpointer) udi)))
    {
        ivm_run_commands(execun, udi, TRUE);
        g_hash_table_remove(execuns, (gpointer) udi);
    }

}

/** Invoked when device in the Global Device List acquires a new capability.
*
*  @param  ctx                 LibHal context
*  @param  udi                 Universal Device Id
*  @param  capability          Name of capability
*/
void
hal_device_new_capability(LibHalContext * ctx __attribute__ ((__unused__)),
                          const char *udi __attribute__ ((__unused__)),
                          const char *capability
                          __attribute__ ((__unused__)))
{
}

/** Invoked when device in the Global Device List loses a capability.
*
*  @param  ctx                 LibHal context
*  @param  udi                 Universal Device Id
*  @param  capability          Name of capability
*/
void
hal_device_lost_capability(LibHalContext * ctx
                           __attribute__ ((__unused__)), const char *udi
                           __attribute__ ((__unused__)),
                           const char *capability
                           __attribute__ ((__unused__)))
{
}

/** Invoked when a property of a device in the Global Device List is
*  changed, and we have we have subscribed to changes for that device.
*
*  @param  ctx                 LibHal context
*  @param  udi                 Univerisal Device Id
*  @param  key                 Key of property
*/
void
hal_device_property_modified(LibHalContext * ctx
                             __attribute__ ((__unused__)), const char *udi,
                             const char *key, dbus_bool_t is_removed
                             __attribute__ ((__unused__)),
                             dbus_bool_t is_added
                             __attribute__ ((__unused__)))
{
    DBusError dbus_error;

    dbus_error_init(&dbus_error);

    // DEBUG("%s:%s modified",udi,key);

    if (!g_strcasecmp(key, "volume.is_mounted"))
    {
        dbus_bool_t val;

        val =
            libhal_device_get_property_bool(hal_ctx, udi, key,
                                            &dbus_error);
        if (val == TRUE)
        {
            DEBUG(_("Mounted: %s"), udi);
            // Add to device list
            char *str =
                (char *) libhal_device_get_property_string(hal_ctx, udi,
                                                           "block.device",
                                                           &dbus_error);


            g_hash_table_insert(devices,
                                (gpointer) g_strdup(udi),
                                (gpointer) g_strdup(str));
            libhal_free_string(str);

            DEBUG(_("Devices table has %d entries"),
                  g_hash_table_size(devices));
            dump_devices();

        }
        else
        {
            DEBUG(_("Unmounted: %s"), udi);
            g_hash_table_remove(devices, udi);
            DEBUG(_("Devices table has %d entries"),
                  g_hash_table_size(devices));
            dump_devices();
        }
    }

    char *file = ivm_get_config_file("IvmConfigProperties.xml");
    IvmConfigProperties *cfg =
        parseIvmConfigProperties(file, (char *) udi, (char *) key);
    free(file);

    if (!cfg)
    {
        xmlReadFailed("IvmConfigProperties.xml");
        return;
    }

    ivm_run_commands(cfg->exec, udi, FALSE);
    ivm_free_str_array(cfg->exec);
    free(cfg);

    ivm_check_dbus_error(&dbus_error);
}


/** Invoked when a device in the GDL emits a condition that cannot be
*  expressed in a property (like when the processor is overheating)
*
*  @param  ctx                 LibHal context
*  @param  udi                 Univerisal Device Id
*  @param  condition_name      Name of condition
*  @param  message             D-BUS message with parameters
*/
#ifdef HAL_0_4
void
hal_device_condition(LibHalContext * ctx __attribute__ ((__unused__)),
                     const char *udi,
                     const char *condition_name, DBusMessage * message
                     __attribute__ ((__unused__)))
#else
void
hal_device_condition(LibHalContext * ctx __attribute__ ((__unused__)),
                     const char *udi,
                     const char *condition_name,
                     const char *condition_detail
                     __attribute__ ((__unused__)))
#endif
{
    // VolumeUnmountForced occurs when USB disk is yanked or similar
    // We have to ensure that device is removed from hash table
    if (!strcmp(condition_name, "VolumeUnmountForced"))
    {
        DEBUG(_("Forced unmount: %s"), udi);

        g_hash_table_remove(devices, udi);
        DEBUG(_("Devices table has %d entries"),
              g_hash_table_size(devices));
        dump_devices();
    }

    char *file = ivm_get_config_file("IvmConfigConditions.xml");
    IvmConfigConditions *cfg =
        parseIvmConfigConditions(file, udi, condition_name);
    free(file);

    if (!cfg)
    {
        xmlReadFailed("IvmConfigConditions.xml");
        return;
    }

    ivm_run_commands(cfg->exec, udi, FALSE);
    ivm_free_str_array(cfg->exec);
    free(cfg);
}


/** Invoked by libhal for integration with our mainloop. 
*
*  @param  ctx                 LibHal context
*  @param  dbus_connection     D-BUS connection to integrate
*/
#ifdef HAL_0_4
void
hal_mainloop_integration(LibHalContext * ctx __attribute__ ((__unused__)),
                         DBusConnection * dbus_connection)
{
    dbus_connection_setup_with_g_main(dbus_connection, NULL);
}
#else
dbus_bool_t
hal_mainloop_integration(LibHalContext * ctx, DBusError * error)
{
    DBusConnection *dbus_connection =
        dbus_bus_get(DBUS_BUS_SYSTEM, error);

    if ( !dbus_connection ) {
        ivm_check_dbus_error(error);
        return FALSE;
    }

    dbus_connection_setup_with_g_main(dbus_connection, NULL);

    return libhal_ctx_set_dbus_connection(ctx, dbus_connection);

}
#endif

void dump_devices()
{
    g_hash_table_foreach(devices, print_device, NULL);
}

void
print_device(gpointer key, gpointer val,
             gpointer data __attribute__ ((__unused__)))
{
    DEBUG("\t\"%s\" \t \"%s\"", (gchar *) key, (gchar *) val);
}
