/****************************************************************************
 *                         TreeWidgetNode.cc
 *
 * Author: Matthew Ballance
 * Desc:   Describes a node in the tree-widget tree
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 ****************************************************************************/
#include "TreeWidgetNode.h"
#include "TreeWidget.h"

/********************************************************************
 * TreeWidgetNode()
 ********************************************************************/
TreeWidgetNode::TreeWidgetNode() : d_nodeName(16), d_nodeData(16), 
    d_nodeText(16)
{
    base_init();
    d_nodeData = "";
    d_nodeText = "";
    d_nodeName = "";
}

/********************************************************************
 * TreeWidgetNode()
 ********************************************************************/
TreeWidgetNode::TreeWidgetNode(const char *text) : 
    d_nodeName(text), d_nodeData(16), d_nodeText(text)
{
    base_init();
    d_nodeData = "";
}

/********************************************************************
 * TreeWidgetNode()
 ********************************************************************/
TreeWidgetNode::TreeWidgetNode(const char *text, const char *data) : 
    d_nodeName(text), d_nodeData(data), d_nodeText(text)
{
    base_init();
}

/********************************************************************
 * base_init()
 ********************************************************************/
void TreeWidgetNode::base_init()
{
    d_userData      = 0;
    d_parent        = 0;
    d_parentNode    = 0;
    d_DimValid      = 0;
    d_ItemSetHeight = 0;
    d_expanded      = 0;
    d_indent        = 0;
    d_cross         = CP_Auto;
    d_selected      = 0;
    d_drawBranch    = true;
}

/********************************************************************
 * ~TreeWidgetNode()
 ********************************************************************/
TreeWidgetNode::~TreeWidgetNode()
{
    if (d_parent) {
        d_parent->removeNode(this);
    }

    for (Uint32 i=0; i<d_children.length(); i++) {
/*        d_children.idx(i)->d_parentNode = 0; */
        delete d_children.idx(i);
    }
}

/********************************************************************
 * SetParent()
 ********************************************************************/
void TreeWidgetNode::SetParent(TreeWidget *parent)
{
    d_parent = parent;
    d_drawBranch = parent->d_config.drawbranch;
}

/********************************************************************
 * SetParentNode()
 ********************************************************************/
void TreeWidgetNode::SetParentNode(TreeWidgetNode *parent)
{
    d_parentNode = parent;
    d_indent     = parent->d_indent + 1;
}

/********************************************************************
 * SetText()
 ********************************************************************/
void TreeWidgetNode::SetText(const char *text)
{
    d_nodeText = text;
    d_DimValid = 0;
}

/********************************************************************
 * GetDimensions()
 *
 * - height always set to default_height on entry.
 ********************************************************************/
void TreeWidgetNode::GetDimensions(Uint32 &width, Uint32 &height)
{
    Int32 text_width;
    GCObj       *gcObjs = d_parent->getGCs();

    if (!d_parent) {
        fprintf(stderr, "TreeWidget ERROR :: no parent\n");
        return;
    }

    if (!d_DimValid) {
        GCObj          &fg = gcObjs[TreeWidget::C_Foreground];
        Tk_TextLayout   layout;
        layout = fg.ComputeTextLayout(d_nodeText.value(), d_nodeText.length(),
            TK_JUSTIFY_CENTER, (Int32 *)&text_width, (Int32 *)&d_ItemHeight);
        fg.FreeTextLayout(layout);

        d_ItemHeight = d_parent->GetLineSpace()+2*d_parent->GetPadY();

        if (d_ItemHeight & 1) {
            d_ItemHeight++;
        }

        /**** width = col_indent*(d_indent-1)+deltax+text_width ****/
        d_ItemWidth = (d_parent->GetColPixIndent()*(d_indent-1))+
            d_parent->GetDeltaX()+text_width;
    }

    width  = d_ItemWidth;
    height = d_ItemHeight;
}

/********************************************************************
 * DrawNode()
 ********************************************************************/
void TreeWidgetNode::DrawNode(PixmapObj &pixmap, Uint32 x_off, Uint32 y_off)
{
    Tk_TextLayout   layout;
    Int32           width, height;
    Uint32          y, x, deltax, col_indent;
    GCObj          *gcObjs = d_parent->getGCs();

    if (!d_parent) {
        fprintf(stderr, "TreeWidget ERROR :: no parent\n");
        return;
    }

    Uint32          lines = d_parent->GetLineSpace()+2*d_parent->GetPadY();
    deltax = d_parent->GetDeltaX();
    col_indent = d_parent->GetColPixIndent();
    GCObj          &fg = (d_selected)?
        gcObjs[TreeWidget::C_HighlightFg]:
        gcObjs[TreeWidget::C_Foreground];

    if (lines & 1) {
        lines++;
    }

    layout = fg.ComputeTextLayout(d_nodeText.value(), d_nodeText.length(),
            TK_JUSTIFY_CENTER, &width, &height);

    /**** If the item's height is invalid and not preset, then update
     **** dimensions here...
     ****/
    if (!d_ItemSetHeight && !d_DimValid) {
        d_ItemWidth  = width;
        d_ItemHeight = lines;
    }

    /**** Text goes in the middle of ItemHeight, offset by y_off ****/
    y = y_off;

    /**** Find the baseline for this node ****/
    x = col_indent*(d_indent-1);

    /**** Now, draw the lines/boxes to the left of this node... 
     **** 
     **** - Start with the (optional) open/close box
     ****/
    d_chkBoxX = x+4;
    d_chkBoxY = y+(lines/2)-5;
    d_chkBoxW = 8;
    d_chkBoxH = 8;

    if (DrawCheck()) {
        GCObj  &cfg = gcObjs[TreeWidget::C_CheckFg];
        GCObj  &cbg = gcObjs[TreeWidget::C_CheckBg];


        cbg.fill_rect(d_chkBoxX, d_chkBoxY, d_chkBoxW, d_chkBoxH);
        cfg.rect(d_chkBoxX, d_chkBoxY, d_chkBoxW, d_chkBoxH);

#if defined(__MINGW32__) || defined(MAC_OS_X)
        /*** Windows and OSX display these check-boxes differently... ***/
        cfg.line(d_chkBoxX+2, d_chkBoxY+(d_chkBoxW/2),
                 d_chkBoxX+(d_chkBoxW-1), d_chkBoxY+(d_chkBoxW/2));
        if (!d_expanded) {
            cfg.line(d_chkBoxX+(d_chkBoxW/2), d_chkBoxY+2,
                    d_chkBoxX+(d_chkBoxW/2),  d_chkBoxY+(d_chkBoxH-1));
        }
#else 
        cfg.line(d_chkBoxX+2, d_chkBoxY+(d_chkBoxW/2),
                 d_chkBoxX+(d_chkBoxW-2), d_chkBoxY+(d_chkBoxW/2));
        if (!d_expanded) {
            cfg.line(d_chkBoxX+(d_chkBoxW/2), d_chkBoxY+2,
                    d_chkBoxX+(d_chkBoxW/2),  d_chkBoxY+(d_chkBoxH-2));
        }
#endif
    }

    /**** If the node has a predecessor, draw a line up to that node...
     ****/
    if (DrawBranch()) {
        if (d_idx > 0) {
            GCObj    &lfg = gcObjs[TreeWidget::C_Foreground];
            TreeWidgetNode *p = d_parentNode->d_children.idx(d_idx-1);

            if (DrawCheck()) {
                lfg.line(d_chkBoxX+(d_chkBoxW)+1, d_chkBoxY+(d_chkBoxH/2), 
                    x+deltax-2, d_chkBoxY+(d_chkBoxH/2));
                if (p->DrawCheck()) {
                    lfg.line(d_chkBoxX+(d_chkBoxW/2), d_chkBoxY-1,
                        d_chkBoxX+(d_chkBoxW/2), 
                        (p->d_chkBoxY)+(p->d_chkBoxH)+1);
                } else {
                    lfg.line(d_chkBoxX+(d_chkBoxW/2), d_chkBoxY,
                        d_chkBoxX+(d_chkBoxW/2), 
                        (p->d_chkBoxY)+(p->d_chkBoxH/2));
                }
            } else {
                lfg.line(d_chkBoxX+(d_chkBoxW/2), d_chkBoxY+(d_chkBoxH/2), 
                    x+deltax-2, d_chkBoxY+(d_chkBoxH/2));
                if (p->DrawCheck()) {
                    lfg.line(d_chkBoxX+(d_chkBoxW/2), d_chkBoxY+(d_chkBoxH/2),
                        d_chkBoxX+(d_chkBoxW/2), (p->d_chkBoxY)+(p->d_chkBoxH));
                } else {
                    lfg.line(d_chkBoxX+(d_chkBoxW/2), d_chkBoxY+(d_chkBoxH/2),
                        d_chkBoxX+(d_chkBoxW/2), 
                        (p->d_chkBoxY)+(p->d_chkBoxH/2));
                }
            }
        } else if (d_parentNode) {
            /**** This node is the head node... Since we're here, the parent
             **** is expanded...
             ****/
            GCObj    &lfg = gcObjs[TreeWidget::C_Foreground];
            TreeWidgetNode *p = d_parentNode; // ->d_children.idx(d_idx-1);

            if (DrawCheck()) {
                lfg.line(d_chkBoxX+(d_chkBoxW), d_chkBoxY+(d_chkBoxH/2), 
                    x+deltax-2, d_chkBoxY+(d_chkBoxH/2));
                if (p->DrawCheck()) {
                    lfg.line(d_chkBoxX+(d_chkBoxW/2), d_chkBoxY,
                        d_chkBoxX+(d_chkBoxW/2), 
                        (p->d_chkBoxY)+(p->d_chkBoxH)+2);
                } else {
                    lfg.line(d_chkBoxX+(d_chkBoxW/2), d_chkBoxY,
                        d_chkBoxX+(d_chkBoxW/2), 
                        (p->d_chkBoxY)+(p->d_chkBoxH/2));
                }
            } else {
                lfg.line(d_chkBoxX+(d_chkBoxW/2), d_chkBoxY+(d_chkBoxH/2), 
                    x+deltax-2, d_chkBoxY+(d_chkBoxH/2));
                if (p->DrawCheck()) {
                    lfg.line(d_chkBoxX+(d_chkBoxW/2), d_chkBoxY+(d_chkBoxH/2),
                        d_chkBoxX+(d_chkBoxW/2), 
                        (p->d_chkBoxY)+(p->d_chkBoxH)+2);
                } else {
                    lfg.line(d_chkBoxX+(d_chkBoxW/2), d_chkBoxY+(d_chkBoxH/2),
                        d_chkBoxX+(d_chkBoxW/2), 
                        (p->d_chkBoxY)+(p->d_chkBoxH/2));
                }
            }
        }
    }

    fg.DrawTextLayout(layout, x+deltax, y+(lines/2)-(height/2));
    fg.FreeTextLayout(layout);
}

/********************************************************************
 * Configure()
 ********************************************************************/
int TreeWidgetNode::Configure(Tcl_Interp *interp,
        Uint32 argc, Tcl_Obj *const objv[])
{
    for (Uint32 i=0; i<argc; i++) {
        char *str = Tcl_GetString(objv[i]);

        if (String::equal(str, "-open")) {
            i++;
            if (Tcl_GetBooleanFromObj(interp, objv[i], &d_expanded) != TCL_OK) {
                return TCL_ERROR;
            }
        } else
        if (String::equal(str, "-text")) {
            i++;
            d_nodeText = Tcl_GetString(objv[i]);
        } else
        if (String::equal(str, "-data")) {
            i++;
            d_nodeData = Tcl_GetString(objv[i]);
        } else 
        if (String::equal(str, "-drawcross")) {
            char *val = Tcl_GetString(objv[++i]);
            if (String::equal(val, "always")) {
                d_cross = CP_Always;
            } else if (String::equal(val, "never")) {
                d_cross = CP_Never;
            } else if (String::equal(val, "auto")) {
                d_cross = CP_Auto;
            } else {
                Tcl_AppendResult(interp, "unknown -drawcross option ",
                        val, 0);
                return TCL_ERROR;
            }
        } else 
        if (String::equal(str, "-drawbranch")) {
            i++;
            if (Tcl_GetBooleanFromObj(interp, objv[i], &d_drawBranch)
                    != TCL_OK) {
                return TCL_ERROR;
            }
        } else
        if (String::equal(str, "-selected")) {
            i++;
            if (Tcl_GetBooleanFromObj(interp, objv[i], &d_selected)
                    != TCL_OK) {
                return TCL_ERROR;
            }
        } else
        {
            /* For now, ignore 
            Tcl_AppendResult(interp, "unknown option ", str, 0);
            return TCL_ERROR;
             */
        }
    }
    return TCL_OK;
}

/********************************************************************
 * Cget()
 ********************************************************************/
int TreeWidgetNode::Cget(Tcl_Interp *interp, char *option)
{
    if (String::equal(option, "-open")) {
        Tcl_SetObjResult(interp, Tcl_NewBooleanObj(d_expanded));
    } else if (String::equal(option, "-text")) {
        Tcl_SetObjResult(interp, Tcl_NewStringObj(
                    d_nodeText.value(), d_nodeText.length()));
    } else if (String::equal(option, "-data")) {
        Tcl_SetObjResult(interp, Tcl_NewStringObj(
                    d_nodeData.value(), d_nodeData.length()));
    } else if (String::equal(option, "-drawcross")) {
        Tcl_SetObjResult(interp, Tcl_NewStringObj(
            (d_cross==CP_Always)?"always":
            (d_cross==CP_Never)?"never":
            (d_cross==CP_Auto)?"auto":"unknown", -1));
    } else if (String::equal(option, "-drawbranch")) {
        Tcl_SetObjResult(interp, Tcl_NewBooleanObj(d_drawBranch));
    } else {
        Tcl_AppendResult(interp, "no tree item option ", option, 0);
        return TCL_ERROR;
    }

    return TCL_OK;
}

/********************************************************************
 * IsOverCheck()
 ********************************************************************/
bool TreeWidgetNode::IsOverCheck(Uint32 x, Uint32 y)
{
    if (!d_children.length() && (d_cross != CP_Always)) {
        return false;
    }

    /**** Okay, where does this item live? ****/
    if ((x >= d_chkBoxX && x <= (d_chkBoxX+d_chkBoxW)) &&
            (y >= d_chkBoxY && y <= (d_chkBoxY+d_chkBoxH))) {
        return true;
    }

    return false;
}

/********************************************************************
 * Open()
 ********************************************************************/
void TreeWidgetNode::Open(bool is_open)
{
    d_expanded = (is_open)?1:0;
}

/********************************************************************
 *
 ********************************************************************/
bool TreeWidgetNode::DrawBranch()
{
    return d_drawBranch;
}

