/****************************************************************************
 *                             TreeWidget.h
 *
 * Author: Matthew Ballance
 * Desc:   Data structures for a tree
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 ****************************************************************************/
#ifndef INCLUDED_TREE_WIDGET_H
#define INCLUDED_TREE_WIDGET_H
#include "XObjWidgetBase.h"
#include "XPixmapObj.h"
#include "LogMgr.h"

class TreeWidgetNode;
class TreeWidget : public XObjWidgetBase {

    friend class TreeWidgetNode;

    /****************************************************************
     * Public Methods
     ****************************************************************/
    public:

        /************************************************************
         * TreeWidget()
         ************************************************************/
        TreeWidget(
                Tcl_Interp        *interp,
                Uint32             objc,
                Tcl_Obj           *const objv[]);

        /************************************************************
         * ~TreeWidget()
         ************************************************************/
        virtual ~TreeWidget();

        /************************************************************
         * getOptionSpec()
         ************************************************************/
        static Tk_OptionSpec *getOptionSpec();

        /************************************************************
         * CreateInst()
         ************************************************************/
        static int CreateInst(
                ClientData        clientData,
                Tcl_Interp       *interp,
                int               objc,
                Tcl_Obj          *const objv[]);

        /************************************************************
         * IdleProc()
         ************************************************************/
        virtual void IdleProc();

        /************************************************************
         * EventProc()
         ************************************************************/
        virtual void EventProc(XEvent *eventPtr);

        /************************************************************
         * InstCmd()
         ************************************************************/
         virtual int InstCmd(
                 Uint32        objc,
                 Tcl_Obj      *const objv[]);

        /************************************************************
         * getCmdStructs()
         *
         * Gets the list of sub-commands. The derived object 
         * overrides this method to provide its own list of commands
         ************************************************************/
        virtual ObjWidgetBase::CmdStruct *getCmdStructs();

        /************************************************************
         * GetColIndent()
         ************************************************************/
        double GetColIndent() { return d_config.col_indent; }

        /************************************************************
         * GetColPixIndent()
         ************************************************************/
        Uint32 GetColPixIndent() { 
            return (Uint32)(d_config.col_indent * d_fontMetrics.linespace); }

        /************************************************************
         * GetDeltaX()
         ************************************************************/
        inline Uint32 GetDeltaX() { 
            return (Uint32)(d_config.deltax * d_fontMetrics.linespace); }

        /************************************************************
         * GetLineSpace()
         ************************************************************/
        inline Uint32 GetLineSpace() const { return d_fontMetrics.linespace; }

        /************************************************************
         * GetPadY()
         ************************************************************/
        inline Uint32 GetPadY() const { return d_config.pady; }

        /************************************************************
         * RedrawTree()
         ************************************************************/
        virtual void RedrawTree(
            PixmapObj     &pixmap,
            Uint32         repaintYStart,
            Uint32        *repaintYOffset,
            Uint32         repaintYSize);

        /************************************************************
         * RedrawTree()
         ************************************************************/
        void RedrawTree(
            PixmapObj     &pixmap,
            Uint32         repaintYStart,
            Uint32        *repaintYOffset,
            Uint32         repaintYSize,
            bool           exceedSize);

        /************************************************************
         * setUseXGC()
         ************************************************************/
        virtual void setUseXGC(bool useX) { d_useXGC = useX; }

        /************************************************************
         * getTreeDim()
         ************************************************************/
        virtual void getTreeDim(Uint32 &max_width, Uint32 &max_height);

        /************************************************************
         * findNode()
         ************************************************************/
        TreeWidgetNode *findNode(char *nodeName);

    /****************************************************************
     * Public Data
     ****************************************************************/
    public:
        static Tk_OptionTable    d_OptTab;

    /****************************************************************
     * Private Methods
     ****************************************************************/
    private:

        struct SelRange {

            SelRange(Uint32 y) { start_y = y; end_y = y; }

            Uint32     start_y;
            Uint32     end_y;
        };

        enum RepaintFlags {
            RepaintExpose          = RedrawNumFlags,
            RepaintB1Down          = (RedrawNumFlags << 1),
            RepaintB2Down          = (RedrawNumFlags << 2),
            RepaintB3Down          = (RedrawNumFlags << 3),
            RepaintScrollX         = (RedrawNumFlags << 4),
            RepaintScrollY         = (RedrawNumFlags << 5),
            RepaintTreeUpdate      = (RedrawNumFlags << 6)
        };

        /************************************************************
         * Configure()
         ************************************************************/
        int Configure(Uint32 argc, Tcl_Obj *const objv[]);

        /************************************************************
         * SetupGraphics()
         ************************************************************/
        void SetupGraphics();

        /************************************************************
         * SetupGeometry()
         ************************************************************/
        void SetupGeometry();


        /************************************************************
         * RedrawTreeSub()
         ************************************************************/
        void RedrawTreeSub(
            PixmapObj        &pixmap,
            TreeWidgetNode   *node,
            Uint32           &currY,
            Uint32            endY);

        /************************************************************
         * FindStartNode()
         ************************************************************/
        TreeWidgetNode *FindStartNode(
            TreeWidgetNode *node, 
            Uint32          repaintYStart, 
            Uint32         &curr_y);

        /************************************************************
         * ComputeTreeHeight()
         ************************************************************/
        void ComputeTreeHeight(
                TreeWidgetNode *node, Uint32 *height,
                Uint32 *max_width, Uint32 indent);

        /************************************************************
         * UpdatePixmaps()
         ************************************************************/
        void UpdatePixmaps();
        
        /************************************************************
         * insertNode()
         ************************************************************/
        int insertNode(Uint32 objc, Tcl_Obj *const objv[]);

        /************************************************************
         * Info()
         ************************************************************/
        int Info(int objc, Tcl_Obj *const objv[]);

        /************************************************************
         * ItemConfig()
         ************************************************************/
        int ItemConfig(int objc, Tcl_Obj *const objv[]);

        /************************************************************
         * createNode()
         ************************************************************/
        TreeWidgetNode *createNode(Uint32 objc, Tcl_Obj *const objv[]);

        /************************************************************
         * deleteNode()
         ************************************************************/
        int deleteNode(Uint32 objc, Tcl_Obj *const objv[]);

        /************************************************************
         * SelectionSetRange()
         ************************************************************/
        int SelectionSetRange(char *start, char *end);

        /************************************************************
         * Selection()
         ************************************************************/
        int Selection(Uint32 objc, Tcl_Obj *const objv[]);

        /************************************************************
         * selectionAdd()
         ************************************************************/
        void selectionAdd(TreeWidgetNode *node);

        /************************************************************
         * selectionClr()
         ************************************************************/
        void selectionClr();

        /************************************************************
         * findNodeAtY()
         *
         * Find a node located at the specified coordinates
         ************************************************************/
        TreeWidgetNode *findNodeAtY(
                TreeWidgetNode        *startNode,
                Uint32                &nodeY,
                Uint32                 targetY);

        /************************************************************
         * findMoveBarPos()
         ************************************************************/
        Int32 findMoveBarPos(Int32 y);

        /************************************************************
         * drawMoveBar()
         ************************************************************/
        void drawMoveBar();

        /************************************************************
         * buttonAction()
         ************************************************************/
        static int buttonAction(
                ClientData        clientData,
                Tcl_Interp       *interp,
                int               objc,
                Tcl_Obj          *const objv[]);

        /************************************************************
         * motionAction()
         ************************************************************/
        static int motionAction(
                ClientData        clientData,
                Tcl_Interp       *interp,
                int               objc,
                Tcl_Obj          *const objv[]);

        /************************************************************
         * motionAction()
         ************************************************************/
        void motionAction(Int32 x, Int32 y);

        /************************************************************
         * updateRangeSel()
         ************************************************************/
        void updateRangeSel();

        /************************************************************
         * selNodesFromRange()
         ************************************************************/
        TreeWidgetNode *selNodesFromRange(
                TreeWidgetNode         *rootNode,
                Uint32                  startY,
                Uint32                 &nodeY,
                Uint32                  endY);

        /************************************************************
         * buttonRelease()
         ************************************************************/
        static int buttonRelease(
                ClientData        clientData,
                Tcl_Interp       *interp,
                int               objc,
                Tcl_Obj          *const objv[]);

        /************************************************************
         * b1Action()
         *
         * NOTE: x/y are in
         ************************************************************/
        void b1Action(Uint32 x, Uint32 y, Uint32 mask);

        /************************************************************
         * UpdateScrollbars()
         ************************************************************/
        void UpdateScrollbars();

        /************************************************************
         * Yview()
         ************************************************************/
        int Yview(Uint32 objc, Tcl_Obj *const objv[]);

        /************************************************************
         * Xview()
         ************************************************************/
        int Xview(Uint32 objc, Tcl_Obj *const objv[]);

        /************************************************************
         * removeNode()
         ************************************************************/
        void removeNode(TreeWidgetNode *node);

        /************************************************************
         * moveButtonAction()
         *
         * called during a move operation...
         ************************************************************/
        void moveButtonAction(Uint32 x, Uint32 y);

        /************************************************************
         * getGCs()
         ************************************************************/
        GCObj *getGCs() { return (d_useXGC)?d_gcObjs:d_psGcObjs; }


    /****************************************************************
     * Private Data
     ****************************************************************/
    private:
        enum {
            C_Foreground,
            C_HighlightFg,
            C_HighlightBg,
            C_CheckFg,
            C_CheckBg,
            C_MoveBarFg,
            C_NumColors
        };

        enum {
            Opt_Graphics     = 0x0001,
            Opt_Geometry     = 0x0002,
            Opt_Scrollbars   = 0x0004,
            Opt_SelectMode   = 0x0008,
            Opt_Redraw       = 0x0010,
            Opt_NumOpts
        };

        enum SelectMode {
            SelMode_Disabled,
            SelMode_Single,
            SelMode_Multiple,
            SelMode_Extended
        };

        GCObj          d_gcObjs[C_NumColors];
        GCObj          d_psGcObjs[C_NumColors];
        bool           d_useXGC;

        /************************************************************
         * TreeConfig
         ************************************************************/
        struct TreeConfig {
            Tk_3DBorder        background;
            Tk_3DBorder        check_bg;
            Tk_Font            tkfont;
            XColor            *colors[C_NumColors];
            Tcl_Obj           *width;
            Tcl_Obj           *height;

            Tcl_Obj           *borderwidth;

            double             col_indent;
            Int32              redraw;
            char              *yscrollcommand;
            char              *xscrollcommand;

            char              *opencmd;
            char              *closecmd;
            char              *browsecmd;

            double             deltax;
            Int32              pady;

            Int32              drawbranch;
            char              *selModeStr;
            SelectMode         selMode;
            Int32              lineheight;

            char              *movedropcmd;
        };

        TreeConfig             d_config;
        Tk_FontMetrics         d_fontMetrics;

        XPixmapObj             d_backingPixmap;
        Uint32                 d_backingPixmapValid;
        Uint32                 d_widgetWidth;
        Uint32                 d_widgetHeight;

        Uint32                 d_displayStartXPos;
        Uint32                 d_displayStartYPos;

        Uint32                 d_pixmapStartYPos;

        XPixmapObj             d_displayPixmap;

        ClientData             d_ButtonPressCb;
        ClientData             d_ButtonReleaseCb;
        ClientData             d_MotionCb;
        Int32                  d_buttonDown;
        Vector<TreeWidgetNode> d_SelectedNodes;
        Vector<SelRange>       d_selNodes;

        TreeWidgetNode        *d_rootNode;

        Tcl_HashTable          d_nodeHash;

        LogInstance           *log_inst;
        LogInstLevel           log_high;
        LogInstLevel           log_med;
        LogInstLevel           log_low;

        Tcl_Obj               *d_CbArgs[16];
        CallbackList          *d_openCbList;
        CallbackList          *d_closeCbList;

        bool                   d_moveMode;
        Int32                  d_moveCurrY;
};

#endif /* INCLUDED_TREE_WIDGET_H */


