/****************************************************************************
 *                              TreeWidget.cc
 * Author: Matthew Ballance
 * Desc:   Implements a tree widget for general use throughout IVI
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 ****************************************************************************/
#include "TreeWidget.h"
#include "TreeWidgetNode.h"
#include "Stack.h"
#include "CallbackMgr.h"

/********************************************************************
 * TreeWidget()
 ********************************************************************/
TreeWidget::TreeWidget(
        Tcl_Interp        *interp,
        Uint32             objc,
        Tcl_Obj           *const objv[]) : 
    XObjWidgetBase(interp, objc, objv)
{
    if (!ok) {
        return;
    }
    ok = 0;

    memset(&d_config, 0, sizeof(TreeConfig));

    d_config.selMode = SelMode_Single;
    d_backingPixmapValid = 0;
    d_displayStartXPos   = 0;
    d_displayStartYPos   = 0;
    d_pixmapStartYPos    = 0;
    d_buttonDown         = 0;
    d_moveMode           = false;
    d_useXGC             = true;

    Tcl_InitHashTable(&d_nodeHash, TCL_STRING_KEYS);

    d_rootNode                  = new TreeWidgetNode();

    for (Uint32 i=0; i<16; i++) {
        d_CbArgs[i] = Tcl_NewObj();
    }

    Tk_SetClass(getWin(), "TreeWidget");

    d_backingPixmap.setWindow(getWin());
    d_displayPixmap.setWindow(getWin());

    if (Tk_InitOptions(d_interp, (char *)&d_config, d_OptTab, getWin()) 
            != TCL_OK) {
        return;
    }

    if (Configure(objc-2, &objv[2]) != TCL_OK) {
        return;
    }

    SetupGraphics();
    SetupGeometry();

    d_WinPixmap.blank();

    Bind("<Button-1>", "1 %x %y 0", &TreeWidget::buttonAction, this);
    Bind("<<Mult-Select>>", "1 %x %y 1", &TreeWidget::buttonAction, this);
    Bind("<Motion>", "%x %y", &TreeWidget::motionAction, this);
    Bind("<ButtonRelease>", "1 %x %y", &TreeWidget::buttonRelease, this);

//    Bind("<Button-3>", "3 %x %y 0", &TreeWidget::buttonAction, this);

    LogRegionType    *lrt = 0;
    LogMgr::FindRegionType("TreeWidget", &lrt);

    if (!lrt) {
        fprintf(stderr, "ERROR :: Null lrt\n");
    }

    log_inst = new LogInstance(lrt, Tcl_GetString(objv[1]));

    log_low(log_inst, LogMgr::DebugLevel_Low);
    log_med(log_inst, LogMgr::DebugLevel_Med);
    log_high(log_inst, LogMgr::DebugLevel_High);

    LogDestination  *ld = new LogDestination();
    
    log_inst->AddDestination(ld);
    ld->setLogLevel(LogMgr::DebugLevel_Off);

    WidgetMgr_AddInst(d_interp, "tree_widget",
            Tcl_GetString(objv[1]), this);

    Tcl_SetObjResult(d_interp, Tcl_DuplicateObj(objv[1]));

    ok = 1;
}

/********************************************************************
 * ~TreeWidget()
 ********************************************************************/
TreeWidget::~TreeWidget()
{
    Tcl_DeleteHashTable(&d_nodeHash);

    Tk_FreeConfigOptions((char *)&d_config, d_OptTab, getWin());

#if 0
    CallbackMgr_DeleteCb(d_ButtonPressCb); 
    CallbackMgr_DeleteCb(d_ButtonReleaseCb); 
    CallbackMgr_DeleteCb(d_MotionCb); 
#endif
}

/********************************************************************
 * FindStartNode()
 *
 * Parameters:
 * - curr_y
 *   Returns the offset of the first row from the requested 
 *   repaintYStart (repaintYStart+curr_y)=actualY.
 *   NOTE: actualY >= repaintYStart
 ********************************************************************/
TreeWidgetNode *TreeWidget::FindStartNode(
        TreeWidgetNode *node, 
        Uint32          repaintYStart, 
        Uint32         &curr_y)
{
    TreeWidgetNode *ret = 0;
    Uint32          w, h;

    if (!node) {
        node   = d_rootNode;
        curr_y = 0;
    }

    /**** Add size of each node... 
     ****/
    for (Uint32 i=0; i<node->d_children.length(); i++) {
        ret = node->d_children.idx(i);

        if (curr_y >= repaintYStart) {
            /**** Done ****/
            curr_y -= repaintYStart;
            break;
        }

        ret->GetDimensions(w, h);

        /**** If adding this entry's height would advance us to/beyond the
         **** specified repaintYStart, then we're done...
         ****/
        curr_y += h;

        if (ret->d_expanded) {
            /**** If the sub-search returned a positive match, then
             **** we're done - bail
             ****/
            if ((ret = FindStartNode(ret, repaintYStart, curr_y))) {
                break;
            }
        }

        ret = 0;
    }

    return ret;
}

/********************************************************************
 * RedrawTreeSub()
 *
 * This function is invoked to draw an entire subtree.
 * Parameters:
 * - pixmap
 *   Draw stuff here...
 *
 * - node
 *   Current node that we're operating on
 ********************************************************************/
void TreeWidget::RedrawTreeSub(
        PixmapObj        &pixmap,
        TreeWidgetNode   *node,
        Uint32           &currY,
        Uint32            endY)
{
    Uint32      w, h;
    GCObj      *gcObjs = getGCs();

    /**** draw all nodes in this node's level...
     ****/
    for (Uint32 i=0; i<node->d_children.length(); i++) {
        TreeWidgetNode *n = node->d_children.idx(i);

        n->GetDimensions(w, h);

        if (n->d_selected) {
            GCObj &hl = gcObjs[TreeWidget::C_HighlightBg];
            hl.fill_rect(0, currY, pixmap.width(), h);
        }

        n->DrawNode(pixmap, 0, currY);

        /**** Accumulate the distance we've travelled ****/
        currY += h;

        if (currY >= endY) {
            break;
        }

        /**** If the node is expanded, recurse now... ****/
        if (n->d_expanded) {
            RedrawTreeSub(pixmap, n, currY, endY);
            if (currY >= endY) {
                break;
            }
        }

    }
}

/********************************************************************
 * RedrawTree()
 ********************************************************************/
void TreeWidget::RedrawTree(
        PixmapObj     &pixmap,
        Uint32         repaintYStart,
        Uint32        *repaintYOffset,
        Uint32         repaintYSize)
{
    RedrawTree(pixmap, repaintYStart, 
            repaintYOffset, repaintYSize, true);
}

/********************************************************************
 * RedrawTree()
 ********************************************************************/
void TreeWidget::RedrawTree(
        PixmapObj     &pixmap,
        Uint32         repaintYStart,
        Uint32        *repaintYOffset,
        Uint32         repaintYSize,
        bool           exceedSize)
{
    Uint32             i, totalY, y_off = 0, idx, currY, endY, w, h;
    TreeWidgetNode    *startNode, *parent;
    GCObj             *gcObjs = getGCs();

    endY  = repaintYStart + repaintYSize;
    currY = 0;

    pixmap.addGCs(gcObjs, C_NumColors);

    startNode = FindStartNode(0, repaintYStart, y_off);
    *repaintYOffset = y_off;

    repaintYStart += y_off;

    /**** Nothing to draw - exit ****/
    if (!startNode) {
        return;
    }

    /**** We have been given the startNode. We want to draw this node and
     **** its siblings until we hit an expanded node or we exceed the 
     **** drawing limit...
     ****/

    /**** First, find the parent node for this node. Find what the
     **** index of this entry is in the parent's child-list
     ****/
    parent = startNode->d_parentNode;
    for (idx=0; idx<parent->d_children.length(); idx++) {
        if (startNode == parent->d_children.idx(idx)) {
            break;
        }
    }

    if (idx == parent->d_children.length()) {
        fprintf(stderr, "Tree Widget ERROR :: startNode not "
                "on parentNode list\n");
        return;
    }

    while (parent && (currY < repaintYSize)) {
        /**** Now, draw each node at this level... ****/
        for (i=idx; i<parent->d_children.length(); i++) {
            TreeWidgetNode *n = parent->d_children.idx(i);

            /**** TODO :: Calculate the X-offset ****/

            n->GetDimensions(w, h);

            if (n->d_selected) {
                GCObj &hl = gcObjs[TreeWidget::C_HighlightBg];
                hl.fill_rect(0, currY, pixmap.width(), h);
            }

            n->DrawNode(pixmap, 0, currY);

            currY += h;

            if (currY >= repaintYSize) {
                break;
            }

            if (n->d_expanded) {
                RedrawTreeSub(pixmap, n, currY, repaintYSize);
                if (currY >= repaintYSize) {
                    break;
                }
            }
        }

        /**** Now, move out to the previous level of hierarchy ****/
        if (currY < repaintYSize) {

            /**** Locate the position of this node within its parent's
             **** list
             ****/
            if (parent) {
                TreeWidgetNode *p = parent->d_parentNode;

                if (p) {
                    for (idx=0; idx<p->d_children.length(); idx++) {
                        if (p->d_children.idx(idx) == parent) {
                            break;
                        }
                    }

                    if (idx == p->d_children.length()) {
                        /**** Hmmm... Didn't find it ****/
                        fprintf(stderr, "ERROR :: Couldn't find node\n");
                        break;
                    } else if (idx < p->d_children.length()) {
                        idx++;
                    } else {
                        break;
                    }
                }
                parent = parent->d_parentNode;
            }
        }
    }

    d_backingPixmapValid = 1;

    if (!d_useXGC) {
        pixmap.removeGCs(gcObjs, C_NumColors);
    }
}

/********************************************************************
 * getTreeDim()
 ********************************************************************/
void TreeWidget::getTreeDim(Uint32 &max_width, Uint32 &max_height)
{
    Uint32 w, h;

    ComputeTreeHeight(0, &h, &w, 0);

    max_width = w;
    max_height = h;
}

/********************************************************************
 * ComputeTreeHeight()
 ********************************************************************/
void TreeWidget::ComputeTreeHeight(
        TreeWidgetNode *node, Uint32 *height, Uint32 *max_width, Uint32 indent)
{
    Uint32             w, h;

    if (!node) {
        *height = 0;
        *max_width = 0;
        node = d_rootNode;
    }

    for (Uint32 i=0; i<node->d_children.length(); i++) {
        TreeWidgetNode    *n = node->d_children.idx(i);

        n->GetDimensions(w, h);
        *height += h;

        if (w > *max_width) {
            *max_width = w;
        }

        if (n->d_expanded) {
            ComputeTreeHeight(n, height, max_width, 
                    (Uint32)(indent+d_config.col_indent));
        }
    }
}

/********************************************************************
 * UpdatePixmaps()
 *
 * - Want dimensions to be:
 *   - 2x widgetHeight
 *   - >= max(maxWidth, widgetWidth)
 ********************************************************************/
void TreeWidget::UpdatePixmaps()
{
    Uint32 pm_width, t_height = 0, t_width = 0;

    log_high.enter("UpdatePixmaps()\n");
    log_high("width=%d height=%d\n", width(), height());


    d_widgetWidth = width();
    d_widgetHeight = height();

    ComputeTreeHeight(0, &t_height, &t_width, 0);
    /**** If the tree changed, must check to see what the dimensions
     **** of the tree are...
     ****/
    if (IsFlagSet(RepaintTreeUpdate)) {

    }

    pm_width = (d_widgetWidth>t_width)?d_widgetWidth:t_width;

    d_backingPixmap.width_height(2*pm_width, 2*d_widgetHeight);
    d_displayPixmap.width_height(pm_width, d_widgetHeight);

    log_high.leave("UpdatePixmaps()\n");
}

/********************************************************************
 * insertNode()
 *
 * <insert> index parent node ?args?
 ********************************************************************/
int TreeWidget::insertNode(Uint32 objc, Tcl_Obj *const objv[])
{
    TreeWidgetNode   *parent, *node;

    if (objc < 3) {
        Tcl_AppendResult(d_interp, "too few args - expecting 3", 0);
        return TCL_ERROR;
    }

    if (!strcmp(Tcl_GetString(objv[1]), "root")) {
        parent = d_rootNode;
    } else if (!(parent = findNode(Tcl_GetString(objv[1])))) {
        Tcl_AppendResult(d_interp, "Cannot find parent-node ",
                Tcl_GetString(objv[1]), 0);
        return TCL_ERROR;
    }

    if ((node = findNode(Tcl_GetString(objv[2])))) {
        Tcl_AppendResult(d_interp, "node ", Tcl_GetString(objv[2]), 
                " already exists", 0);
        return TCL_ERROR;
    }

    node = new TreeWidgetNode(Tcl_GetString(objv[2]));
    node->SetParent(this);
    node->Configure(d_interp, objc-3, &objv[3]);
    node->SetParentNode(parent);

    int itmp;
    Tcl_HashEntry *entry = Tcl_CreateHashEntry(&d_nodeHash, 
            Tcl_GetString(objv[2]), &itmp);
    Tcl_SetHashValue(entry, node);

    node->d_idx = parent->d_children.length();
    parent->d_children.append(node);

    if (d_config.redraw) {
        SetupRedraw(RepaintTreeUpdate|RedrawForce);
    }


    return TCL_OK;
}

/********************************************************************
 * findNode()
 ********************************************************************/
TreeWidgetNode *TreeWidget::findNode(char *nodeName)
{
    Tcl_HashEntry *entry;

    if (!(entry = Tcl_FindHashEntry(&d_nodeHash, nodeName))) {
        return 0;
    }

    return (TreeWidgetNode *)Tcl_GetHashValue(entry);
}

enum {
    IC_Children,
    IC_NumCmds
};

static ObjWidgetBase::CmdStruct  info_cmds[] = {
    {"children",            IC_Children        },
    {"",                    0                  }
};

/********************************************************************
 * Info()
 *
 * <info> <subcmd> args...
 ********************************************************************/
int TreeWidget::Info(int objc, Tcl_Obj *const objv[])
{
    Int32              cmd, i;
    Tcl_Obj           *ret = 0;
    TreeWidgetNode    *node, *p;

    if (objc < 2) {
        Tcl_AppendResult(d_interp, "too few args to tree_widget info", 0);
        return TCL_ERROR;
    }

    cmd = ObjWidgetBase::CmdSwitch(info_cmds, Tcl_GetString(objv[1]));
    switch (cmd) {
        case IC_Children:
            if (objc < 3) {
                node = d_rootNode;
            } else {
                if (!(node = findNode(Tcl_GetString(objv[2])))) {
                    Tcl_AppendResult(d_interp, "no tree node named ",
                            Tcl_GetString(objv[2]), 0);
                    return TCL_ERROR;
                }
            }

            ret = Tcl_NewListObj(0, 0);
            for (i=0; i<node->d_children.length(); i++) {
                p = node->d_children.idx(i);
                Tcl_ListObjAppendElement(d_interp, ret, 
                        Tcl_NewStringObj(p->d_nodeName.value(), 
                            p->d_nodeName.length()));
            }
            Tcl_SetObjResult(d_interp, ret);
            break;

        default:
            Tcl_AppendResult(d_interp, "unknown sub-cmd ", 
                    Tcl_GetString(objv[1]), 0);
            return TCL_ERROR;
            break;
    }

    return TCL_OK;
}

/********************************************************************
 * ItemConfig()
 ********************************************************************/
int TreeWidget::ItemConfig(int objc, Tcl_Obj *const objv[])
{
    TreeWidgetNode *node;
    Int32           ret;

    if (objc < 1) {
        Tcl_AppendResult(d_interp, "too few arguments", 0);
        return TCL_ERROR;
    }

    if (!(node = findNode(Tcl_GetString(objv[0])))) {
        Tcl_AppendResult(d_interp, "no node named ", Tcl_GetString(objv[0]), 0);
        return TCL_ERROR;
    }

    ret = node->Configure(d_interp, objc-1, &objv[1]);

    if (d_config.redraw) {
        SetupRedraw(RedrawForce);
    }

    return ret;
}

/********************************************************************
 * createNode()
 *
 * nodeName ?args?
 ********************************************************************/
TreeWidgetNode *TreeWidget::createNode(Uint32 objc, Tcl_Obj *const objv[])
{
    char    *name = Tcl_GetString(objv[0]);
    TreeWidgetNode *node;
   
    node = new TreeWidgetNode(name);

    node->Configure(d_interp, objc-1, &objv[1]);

    return node;
}

/********************************************************************
 * deleteNode()
 *
 * <delete> ?node?
 ********************************************************************/
int TreeWidget::deleteNode(Uint32 objc, Tcl_Obj *const objv[])
{
    TreeWidgetNode *node;
    char           *node_name;

    selectionClr();

    node_name = Tcl_GetString(objv[0]);

    if (!strcmp(node_name, "root")) {
        node = d_rootNode;
    } else {
        Tcl_AppendResult(d_interp, "Deleting node not root", 0);
        return TCL_ERROR;
    }

    for (Uint32 i=0; i<node->d_children.length(); i++) {
        TreeWidgetNode *n = node->d_children.idx(i);
        delete n;
    }
    node->d_children.setLength(0);

    if (d_config.redraw) {
        SetupRedraw(RepaintTreeUpdate|RedrawForce);
    }

    return TCL_OK;
}

/********************************************************************
 * removeNode()
 ********************************************************************/
void TreeWidget::removeNode(TreeWidgetNode *node)
{
    Tcl_HashEntry *entry;

    if (!(entry = Tcl_FindHashEntry(&d_nodeHash, node->d_nodeName.value()))) {
        fprintf(stderr, "ERROR :: treeNode \"%s\" isn't in hash table\n",
                node->d_nodeName.value());
        return;
    }

    Tcl_DeleteHashEntry(entry);
}

/********************************************************************
 * SelectionSetRange()
 ********************************************************************/
int TreeWidget::SelectionSetRange(char *start, char *end)
{
    TreeWidgetNode *startNode, *endNode;
    TreeWidgetNode *startParent;
    Tcl_HashEntry  *entry;
    Uint32          start_idx;

    if (!(entry = Tcl_FindHashEntry(&d_nodeHash, start))) {
        Tcl_AppendResult(d_interp, "no node named ", start, 0);
        return TCL_ERROR;
    } else {
        startNode = (TreeWidgetNode *)Tcl_GetHashValue(entry);
    }

    if (!(entry = Tcl_FindHashEntry(&d_nodeHash, end))) {
        Tcl_AppendResult(d_interp, "no node named ", end, 0);
        return TCL_ERROR;
    } else {
        endNode = (TreeWidgetNode *)Tcl_GetHashValue(entry);
    }

    startParent = startNode->d_parentNode;

    for (Uint32 i=0; i<startParent->d_children.length(); i++) {
        if (startParent->d_children.idx(i) == startNode) {
            start_idx = i;
            break;
        }
    }

    /**** Okay, now we have the parent of the start-node and the 
     **** index of the startNode within the parent's node vector
     ****/
    for (Uint32 i=start_idx; i<startParent->d_children.length(); i++) {
        TreeWidgetNode *node = startParent->d_children.idx(i);

        selectionAdd(node);

        if (node == endNode) {
            break;
        }
    }

    SetupRedraw(RedrawForce);

    return TCL_OK;
}

/********************************************************************
 * Selection()
 *
 * selection [get|set]
 ********************************************************************/
int TreeWidget::Selection(Uint32 objc, Tcl_Obj *const objv[])
{
    Tcl_HashEntry    *entry;

    if (objc < 2) {
        Tcl_AppendResult(d_interp, "too few args", 0);
        return TCL_ERROR;
    }

    char *cmd = Tcl_GetString(objv[1]);

    if (String::equal(cmd, "get")) {
        Tcl_Obj *ret = Tcl_NewListObj(0, 0);
        for (Uint32 i=0; i<d_SelectedNodes.length(); i++) {
            TreeWidgetNode *n = d_SelectedNodes.idx(i);

            Tcl_ListObjAppendElement(d_interp, ret,
                    Tcl_NewStringObj(n->d_nodeName.value(), 
                        n->d_nodeName.length()));
        }
        Tcl_SetObjResult(d_interp, ret);
    } else if (String::equal(cmd, "clear")) {
        selectionClr();
        SetupRedraw(RedrawForce);
    } else if (String::equal(cmd, "set")) {
        if (objc < 4) {
            Tcl_AppendResult(d_interp, "too few args - expecting 4", 0);
            return TCL_ERROR;
        }

        return SelectionSetRange(Tcl_GetString(objv[2]), 
                Tcl_GetString(objv[3]));
    } else if (String::equal(cmd, "add")) {
        if (!(entry=Tcl_FindHashEntry(&d_nodeHash, Tcl_GetString(objv[2])))) {
            Tcl_AppendResult(d_interp, "no entry ", Tcl_GetString(objv[2]));
            return TCL_ERROR;
        }

        selectionAdd((TreeWidgetNode *)Tcl_GetHashValue(entry));
    } else {
        Tcl_AppendResult(d_interp, "unknown selection sub-cmd ", cmd, 0);
    }

    return TCL_OK;
}

/********************************************************************
 * selectionAdd()
 ********************************************************************/
void TreeWidget::selectionAdd(TreeWidgetNode *node)
{
    if (!node->d_selected) {
        d_SelectedNodes.append(node);
        node->d_selected = 1;
    }
}

/********************************************************************
 * selectionClr()
 ********************************************************************/
void TreeWidget::selectionClr()
{
    for (Uint32 i=0; i<d_SelectedNodes.length(); i++) {
        d_SelectedNodes.idx(i)->d_selected = 0;
    }

    for (Uint32 i=0; i<d_selNodes.length(); i++) {
        delete d_selNodes.idx(i);
    }

    d_selNodes.empty();
    d_SelectedNodes.empty();
}

/********************************************************************
 * UpdateScrollbars()
 *
 * - displayStartY
 * - height
 * - widgetHeight;
 ********************************************************************/
void TreeWidget::UpdateScrollbars()
{
    Uint32 height, max_width;

    ComputeTreeHeight(0, &height, &max_width, GetColPixIndent());

    UpdateScrollbar(d_config.yscrollcommand, d_displayStartYPos,
            d_widgetHeight, height);

    UpdateScrollbar(d_config.xscrollcommand, d_displayStartXPos,
            d_widgetWidth, max_width);
}

/********************************************************************
 * Yview()
 ********************************************************************/
int TreeWidget::Yview(Uint32 objc, Tcl_Obj *const objv[])
{
    Uint32 t_height = 0, max_width;
    Uint32 new_displayStart = d_displayStartYPos;
    Int32  ret;

#if 0
    fprintf(stderr, "TreeWidget::Yview( ");
    for (int i=2; i<objc; i++) {
        fprintf(stderr, "%s ", Tcl_GetString(objv[i]));
    }
    fprintf(stderr, ")\n");
#endif

    ComputeTreeHeight(0, &t_height, &max_width, GetColPixIndent());

    ret = ProcessViewCmd(objc, objv, t_height, d_widgetHeight, 
            d_displayStartYPos);

    if ((d_displayStartYPos+d_widgetHeight) >= 
            (d_pixmapStartYPos+d_backingPixmap.height()) ||
        (d_displayStartYPos < d_pixmapStartYPos)) {
        /**** re-center the display window ****/
        Center(d_displayStartYPos, height(), d_pixmapStartYPos, 
                d_backingPixmap.height());
        SetupRedraw(RedrawForce);
    }

    SetupRedraw(RepaintScrollY);

    return ret;
}

/********************************************************************
 * Xview()
 ********************************************************************/
int TreeWidget::Xview(Uint32 objc, Tcl_Obj *const objv[])
{
    Uint32 t_height = 0, max_width = 0;
    Int32  ret;

    ComputeTreeHeight(0, &t_height, &max_width, GetColPixIndent());

    ret = ProcessViewCmd(objc, objv, ((max_width*3)/2), d_widgetWidth,
            d_displayStartXPos);

    /*
    if ((d_displayStartXPos+d_widgetWidth) > ((max_width*3)/2)) {
        d_displayStartXPos = ((max_width*3)/2) - d_widgetWidth;
    }
     */

    SetupRedraw(RepaintScrollY);
    return TCL_OK;
}

/********************************************************************
 * buttonAction()
 *
 *
 ********************************************************************/
int TreeWidget::buttonAction(
        ClientData        clientData,
        Tcl_Interp       *interp,
        int               objc,
        Tcl_Obj          *const objv[])
{
    TreeWidget      *tree = (TreeWidget *)clientData;
    TreeWidgetNode  *node;
    Uint32           nodeY = 0;
    Int32            x, y, mask;

    Tcl_GetIntFromObj(interp, objv[1], &tree->d_buttonDown);

    if (!tree->d_moveMode && (tree->d_buttonDown != 1)) {
        goto button_exit;
    }

    Tcl_GetIntFromObj(interp, objv[2], &x);
    Tcl_GetIntFromObj(interp, objv[3], &y);
    Tcl_GetIntFromObj(interp, objv[4], &mask);

    if (tree->d_moveMode) {
        tree->moveButtonAction(x, y);
    } else {
        tree->b1Action(x, y, mask);
    }

button_exit:
    return TCL_OK;
}

/********************************************************************
 * moveButtonAction()
 ********************************************************************/
void TreeWidget::moveButtonAction(Uint32 x, Uint32 y)
{
    int     ret;
    Uint32  i, idx, nodeY = 0;
    char   *nodeName = "end";
    TreeWidgetNode   *node;

    if (d_buttonDown != 1 && d_buttonDown != 3) {
        return;
    }

    /**** Cancel grab... ****/
    Tk_Ungrab(getWin());
    d_moveMode = false;
    SetupRedraw(RedrawForce);

    if (d_buttonDown != 1) {
        return;
    }

    /**** Must find the name of the node that is just after the drop-bar
     ****/

    if ((node = findNodeAtY(d_rootNode, nodeY,
                    (d_moveCurrY+d_displayStartYPos)))) {
        Uint32    w, h;
        TreeWidgetNode *parent = node->d_parentNode;
        TreeWidgetNode *tnode;

        for (idx=0; idx<parent->d_children.length(); idx++) {
            if (parent->d_children.idx(idx) == node) {
                break;
            }
        }

        node->GetDimensions(w, h);
        if ((d_moveCurrY+d_displayStartYPos) > (nodeY+(h/2))) {
            if (idx+1 < parent->d_children.length()) {
                node = parent->d_children.idx(idx+1);
            } 
            idx++;
        }

        for (i=idx; i<parent->d_children.length(); i++) {
            tnode = parent->d_children.idx(i);
            if (!tnode->d_selected) {
                node = tnode;
                break;
            }
        }

        if (i == parent->d_children.length()) {
            nodeName = "end";
        } else {
            nodeName = node->d_nodeName.value();
        }
    }

    if (d_config.movedropcmd) {
        ret = Tcl_VarEval(d_interp, d_config.movedropcmd, " ", nodeName, 0);
        if (ret != TCL_OK) {
            fprintf(stderr, "dropcmd failed: %s\n", 
                    Tcl_GetStringResult(d_interp));
        }
    }
}

/********************************************************************
 * buttonRelease()
 ********************************************************************/
int TreeWidget::buttonRelease(
        ClientData         clientData,
        Tcl_Interp        *interp,
        int                objc,
        Tcl_Obj           *const objv[])
{
    ((TreeWidget *)clientData)->d_buttonDown = 0;

    return TCL_OK;
}

/********************************************************************
 * motionAction()
 ********************************************************************/
int TreeWidget::motionAction(
        ClientData         clientData,
        Tcl_Interp        *interp,
        int                objc,
        Tcl_Obj           *const objv[])
{
    TreeWidget     *tree = (TreeWidget *)clientData;
    Int32           x, y, nodeY;

    if (tree->d_moveMode) {
        Tcl_GetIntFromObj(interp, objv[2], &y);
        nodeY = tree->findMoveBarPos(y);
        if (nodeY != tree->d_moveCurrY) { 
            tree->d_moveCurrY = nodeY;
            tree->SetupRedraw(RedrawForce);
        }
    } else if ((tree->d_buttonDown == 1) &&
            (tree->d_config.selMode == SelMode_Extended)) {
        Tcl_GetIntFromObj(interp, objv[1], &x);
        Tcl_GetIntFromObj(interp, objv[2], &y);
        tree->motionAction(x, y);
    }
    
    return TCL_OK;
}

/********************************************************************
 * motionAction()
 ********************************************************************/
void TreeWidget::motionAction(Int32 x, Int32 y)
{
    TreeWidgetNode  *node;
    Uint32           nodeY = 0;
    
    if ((node = findNodeAtY(d_rootNode, nodeY, (y+d_displayStartYPos)))) {
        SelRange *r = d_selNodes.idx(d_selNodes.length()-1);

        if (r) {
            r->end_y = y+d_displayStartYPos;
            updateRangeSel();
            SetupRedraw(RedrawForce);
        }
    }
}

/********************************************************************
 * selNodesFromRange()
 ********************************************************************/
TreeWidgetNode *TreeWidget::selNodesFromRange(
        TreeWidgetNode         *rootNode,
        Uint32                  startY,
        Uint32                 &nodeY,
        Uint32                  endY)
{
    TreeWidgetNode *n=0;
    Uint32          w, h;

    for (Uint32 i=0; i<rootNode->d_children.length(); i++) {
        n = rootNode->d_children.idx(i);

        n->GetDimensions(w, h);

        /**** See if we should be flagging nodes as selected... ****/
        if (nodeY+h >= startY) {
            selectionAdd(n);
        }

        /**** See if we've reached the end of the selected range... ****/
        if ((nodeY >= endY) || (nodeY+h >= endY)) {
            break;
        }

        nodeY += h;

        if (n->d_expanded) {
            if ((n = selNodesFromRange(n, startY, nodeY, endY))) {
                break;
            }
        }
        n = 0;
    }

    return n;
}

/********************************************************************
 * updateRangeSel()
 ********************************************************************/
void TreeWidget::updateRangeSel()
{
    Uint32 startY, nodeY, endY;
    SelRange *r;

    for (Uint32 i=0; i<d_SelectedNodes.length(); i++) {
        d_SelectedNodes.idx(i)->d_selected = 0;
    }
    d_SelectedNodes.empty();


    /**** Now, step through each range-node ****/
    for (Uint32 i=0; i<d_selNodes.length(); i++) {
        r = d_selNodes.idx(i);

        nodeY = 0;
        if (r->end_y > r->start_y) {
            endY = r->end_y;
            startY = r->start_y;
        } else {
            endY = r->start_y;
            startY = r->end_y;
        }

        selNodesFromRange(d_rootNode, startY, nodeY, endY);
    }
}

/********************************************************************
 * b1Action()
 *
 * Handles actions performed when a button is pressed...
 *
 * NOTE :: x/y are in uniform coordinates - 0=tree-top
 ********************************************************************/
#define MULT_SEL_MASK    1
#define RANGE_SEL_MASK   2
void TreeWidget::b1Action(Uint32 x, Uint32 y, Uint32 mask)
{
    TreeWidgetNode     *node;
    Uint32              nodeY = 0;
    SelRange           *selRange;

    /**** First, see if this button-press targets any item
     ****/
    if (!(node = findNodeAtY(d_rootNode, nodeY, (y+d_displayStartYPos)))) {
        selectionClr();
        SetupRedraw(RedrawForce);
        return;
    }

    /**** Okay, we are over a node...
     **** - May be opening a closed node
     **** - May be selecting a node
     ****/
    if (node->IsOverCheck(x, ((y+d_displayStartYPos)-d_pixmapStartYPos))) {
        SetFlag(RepaintScrollY);
        if (node->IsOpen()) {
            if (d_config.closecmd) {
                if (Tcl_VarEval(d_interp, d_config.closecmd, " ",
                        node->d_nodeName.value(), 0) != TCL_OK) {
                    Tcl_BackgroundError(d_interp);
                }
            }
            node->Open(0);
        } else {
            if (d_config.opencmd) {
                if (Tcl_VarEval(d_interp, d_config.opencmd, " ",
                        node->d_nodeName.value(), 0) != TCL_OK) {
                    Tcl_BackgroundError(d_interp);
                }
            }
            node->Open(1);
        }
    } else {
        switch (d_config.selMode) {
            case SelMode_Disabled:
                break;

            case SelMode_Single:
                selectionClr();
                selectionAdd(node);
                break;

            case SelMode_Multiple:
                if (mask != MULT_SEL_MASK) {
                    selectionClr();
                }
                selectionAdd(node);
                break;

            /********************************************************
             * Extended selection mode
             *
             * - If control down, then we're starting a new sequence.
             * - Otherwise, clear everything...
             ********************************************************/
            case SelMode_Extended:
                if (mask != MULT_SEL_MASK) {
                    selectionClr();
                }

                selRange = new SelRange(y+d_displayStartYPos);
                d_selNodes.append(selRange);
                updateRangeSel();
                break;

            default:
                break;
        }
    }

    if (d_config.browsecmd) {
        if (Tcl_VarEval(d_interp, d_config.browsecmd, " ", 
                    getInstName(), " ",
                    node->d_nodeName.value(), 0) != TCL_OK) {
            Tcl_BackgroundError(d_interp);
        }
    }

    SetupRedraw(RedrawForce);
}

/********************************************************************
 * findNodeAtY()
 ********************************************************************/
TreeWidgetNode *TreeWidget::findNodeAtY(
        TreeWidgetNode     *startNode,
        Uint32             &nodeY,
        Uint32              targetY)
{
    TreeWidgetNode  *n = 0;
    Uint32           w, h;

    for (Uint32 i=0; i<startNode->d_children.length(); i++) {
        n = startNode->d_children.idx(i);

        n->GetDimensions(w, h);
       
        if ((targetY >= nodeY) && (targetY <= (nodeY+h))) {
            /**** Done... ****/
            break;
        }

        nodeY += h;

        /**** If the node is expanded, then recurse... ****/
        if (n->d_expanded) {
            /**** If the recursion found the node, then return ****/
            if ((n = findNodeAtY(n, nodeY, targetY))) {
                break;
            }
        }

        n = 0;
    }

    return n;
}

/********************************************************************
 * findMoveBarPos()
 ********************************************************************/
Int32 TreeWidget::findMoveBarPos(Int32 y)
{
    TreeWidgetNode    *node;
    Uint32             nodeY = 0;
    if (y < 0) {
        y = 0;
    }

    if (y > height()) {
        y = height();
    }

    if ((node = findNodeAtY(d_rootNode, nodeY, (y+d_displayStartYPos)))) {
        Uint32     w, h;
        Uint32     yoff = d_pixmapStartYPos;

        node->GetDimensions(w, h);

        if ((y+d_displayStartYPos) > (nodeY+(h/2))) {
            nodeY += h;
        }
    } 

    return nodeY;
}

/********************************************************************
 * drawMoveBar()
 ********************************************************************/
void TreeWidget::drawMoveBar()
{
    TreeWidgetNode *node;
    Uint32          nodeY = 0;

    if (!d_moveMode) {
        return;
    }

    GCObj    *gcObjs = getGCs();
    GCObj    &g = gcObjs[TreeWidget::C_MoveBarFg];
    Uint32    yoff = d_pixmapStartYPos;

    g.line(0, d_moveCurrY-yoff, width(), d_moveCurrY-yoff);
}

/********************************************************************
 * IdleProc()
 ********************************************************************/
void TreeWidget::IdleProc()
{
    GCObj    *gcObjs = getGCs();
    log_high.enter("IdleProc()\n");

    if (!winValid()) {
        return;
    }

    d_backingPixmap.addGCs(gcObjs, C_NumColors);

    if (IsFlagSet(RedrawForce|RepaintTreeUpdate|RepaintExpose) 
            || !d_backingPixmapValid) {
        UpdatePixmaps();
    }

    if (IsFlagSet(RedrawForce|RepaintScrollY|RepaintScrollX|RepaintExpose)) {
        UpdateScrollbars();
    }

    if (IsFlagSet(RedrawForce|RepaintExpose) || 
            !d_backingPixmapValid) {
        Uint32 dummy = 0;

        log_high("blank backing pixmap\n");
        d_backingPixmap.blank();

        RedrawTree(d_backingPixmap, d_pixmapStartYPos, &dummy, 
                d_backingPixmap.height());

        d_pixmapStartYPos += dummy;
    }


    log_high("d_WinPixmap = d_backingPixmap(%d,%d %d,%d)\n",
            d_displayStartXPos, d_displayStartYPos, d_widgetWidth, 
            d_widgetHeight);

    if (d_moveMode) {
        drawMoveBar();
    }

    d_WinPixmap = d_backingPixmap(d_displayStartXPos, 
            (d_displayStartYPos-d_pixmapStartYPos),
            d_widgetWidth, d_widgetHeight);


    ClrFlag(RepaintTreeUpdate|RepaintExpose|RepaintScrollX|RepaintScrollY);

    log_high.leave("IdleProc()\n");
}

/********************************************************************
 * EventProc()
 ********************************************************************/
void TreeWidget::EventProc(XEvent *eventPtr)
{
    log_high.enter("EventProc()\n");

    switch (eventPtr->type) {
        /************************************************************
         * Expose
         ************************************************************/
        case Expose:
            log_high("Expose: (x=%d,y=%d) (w=%d,h=%d)\n",
                    eventPtr->xexpose.x, eventPtr->xexpose.y,
                    eventPtr->xexpose.width, eventPtr->xexpose.height);

            if (d_backingPixmapValid && (d_widgetWidth == width()) &&
                    (d_widgetHeight == height())) {
                d_WinPixmap(eventPtr->xexpose.x, eventPtr->xexpose.y) =
                    d_backingPixmap(eventPtr->xexpose.x, eventPtr->xexpose.y,
                            eventPtr->xexpose.width, eventPtr->xexpose.height);
            } else {
                SetupRedraw(RedrawForce);
            }
            break;

        case DestroyNotify:
            delete this;
            break;

        case NoExpose:
            break;

        case ConfigureNotify:
            SetupRedraw(RedrawForce);
            break;
    }

    log_high.leave("EventProc()\n");
}

enum {
    TWC_Config,
    TWC_Cget,
    TWC_Insert,
    TWC_Delete,
    TWC_Xview,
    TWC_Yview,
    TWC_Selection,
    TWC_Exists,
    TWC_Info,
    TWC_ItemConfig,
    TWC_ItemCget,
    TWC_Nearest,
    TWC_StartMove,
    TWC_NumCmds
};

/********************************************************************
 * getCmdStructs()
 ********************************************************************/
ObjWidgetBase::CmdStruct *TreeWidget::getCmdStructs()
{
    static ObjWidgetBase::CmdStruct   cmds[] = {
        {"configure",       TWC_Config       },
        {"config",          TWC_Config       },
        {"cget",            TWC_Cget         },
        {"insert",          TWC_Insert       },
        {"delete",          TWC_Delete       },
        {"xview",           TWC_Xview        },
        {"yview",           TWC_Yview        },
        {"selection",       TWC_Selection    },
        {"exists",          TWC_Exists       },
        {"info",            TWC_Info         },
        {"itemconfigure",   TWC_ItemConfig   },
        {"itemconfig",      TWC_ItemConfig   },
        {"itemcget",        TWC_ItemCget     },
        {"nearest",         TWC_Nearest      },
        {"start_move",      TWC_StartMove    },
        {"",                0                }
    };

    return cmds;
}

/********************************************************************
 * InstCmd()
 ********************************************************************/
int TreeWidget::InstCmd(
        Uint32        objc,
        Tcl_Obj      *const objv[])
{
    Int32              cmd, ret, targY;
    Tcl_Obj           *cget_res;
    TreeWidgetNode    *node;
    Uint32             nodeY;

    switch (CmdSwitch(Tcl_GetString(objv[1]))) {

        case TWC_Config:
            return Configure(objc-2, &objv[2]);
            break;

        case TWC_Cget:
            cget_res = Tk_GetOptionValue(
                    d_interp, (char *)&d_config, d_OptTab, objv[2],
                    getWin());
            if (!cget_res) {
                return TCL_ERROR;
            } else {
                Tcl_SetObjResult(d_interp, cget_res);
            }
            break;

        /************************************************************
         * ItemConfig
         ************************************************************/
        case TWC_ItemConfig:
            return ItemConfig(objc-2, &objv[2]);
            break;

        /************************************************************
         * ItemCget
         *
         * <inst> <itemcget> <item> <option>
         ************************************************************/
        case TWC_ItemCget:
            if (objc < 4) {
                Tcl_AppendResult(d_interp, "too few args - need 4", 0);
                return TCL_ERROR;
            }

            if (!(node = findNode(Tcl_GetString(objv[2])))) {
                Tcl_AppendResult(d_interp, "no sub-element named ",
                        Tcl_GetString(objv[2]));
                return TCL_ERROR;
            }

            return node->Cget(d_interp, Tcl_GetString(objv[3]));
            break;

        /************************************************************
         * Insert
         *
         * insert index parent node <args>
         ************************************************************/
        case TWC_Insert:
            return insertNode(objc-2, &objv[2]);
            break;

        /************************************************************
         * Delete
         ************************************************************/
        case TWC_Delete:
            return deleteNode(objc-2, &objv[2]);
            break;

        /************************************************************
         * Xview
         ************************************************************/
        case TWC_Xview:
            Xview(objc, objv);
            break;

        /************************************************************
         * Yview
         ************************************************************/
        case TWC_Yview:
            return Yview(objc, objv);
            break;

        /************************************************************
         * Selection
         ************************************************************/
        case TWC_Selection:
            return Selection(objc-1, &objv[1]);
            break;

        /************************************************************
         * Exists
         ************************************************************/
        case TWC_Exists:
            if (findNode(Tcl_GetString(objv[2]))) {
                Tcl_SetObjResult(d_interp, Tcl_NewIntObj(1));
            } else {
                Tcl_SetObjResult(d_interp, Tcl_NewIntObj(0));
            }
            break;

        /************************************************************
         * Info
         ************************************************************/
        case TWC_Info:
            return Info(objc-1, &objv[1]);

        /************************************************************
         * Nearest
         ************************************************************/
        case TWC_Nearest:
            if (objc < 3) {
                Tcl_AppendResult(d_interp, "too few args - expect 3", 0);
                return TCL_ERROR;
            }

            if (Tcl_GetIntFromObj(d_interp, objv[2], &targY) != TCL_OK) {
                return TCL_ERROR;
            }

            nodeY = 0;
            if ((node = findNodeAtY(d_rootNode, nodeY, 
                            targY+d_displayStartYPos))) {
                Tcl_SetObjResult(d_interp, 
                    Tcl_NewStringObj(node->d_nodeName.value(), 
                        node->d_nodeName.length()));
            }
            break;

        /************************************************************
         * TWC_StartMove
         ************************************************************/
        case TWC_StartMove:
            Tk_Grab(d_interp, getWin(), false);
            d_moveMode = true;
            d_moveCurrY = -1;
            break;

        default:
            fprintf(stderr, "Unknown cmd %s\n", Tcl_GetString(objv[1]));
            return TCL_ERROR;
    }

    return TCL_OK;
}

/********************************************************************
 * SetupGraphics()
 ********************************************************************/
void TreeWidget::SetupGraphics()
{
    GCObj    *gcObjs = d_gcObjs;
    log_med.enter("SetupGraphics()\n");

    GCObj::invalidate(gcObjs, C_NumColors);

    for (Uint32 i=0; i<C_NumColors; i++) {
        gcObjs[i].setFont(d_config.tkfont);
        gcObjs[i].setBgColor(d_config.background);
        gcObjs[i].setFgColor(d_config.colors[i]);
    }

    d_WinPixmap.setBgColor(d_config.background);
    d_backingPixmap.setBgColor(d_config.background);
    d_displayPixmap.setBgColor(d_config.background);

    Tk_GetFontMetrics(d_config.tkfont, &d_fontMetrics);

    d_config.lineheight = GetLineSpace();

    if ((d_config.lineheight & 1)) {
        d_config.lineheight++;
    }

    /**** Now, we modify some fontMetrics fields to ensure that
     **** certain base-features can be drawn appropriately
     ****/

    for (Uint32 i=0; i<C_NumColors; i++) {
        d_psGcObjs[i].setFont(d_config.tkfont);
        d_psGcObjs[i].setFgColor((Uint32)0);
    }
    d_psGcObjs[C_CheckBg].setFgColor((Uint32)0xFFFFFFFF);

    log_med.leave("SetupGraphics()\n");
}

/********************************************************************
 * SetupGeometry()
 ********************************************************************/
void TreeWidget::SetupGeometry()
{
    int    width, height, bd_width;

    Tcl_GetIntFromObj(d_interp, d_config.width, &width);
    Tcl_GetIntFromObj(d_interp, d_config.height, &height);
    Tcl_GetIntFromObj(d_interp, d_config.borderwidth, &bd_width);

    Tk_GeometryRequest(getWin(), width, height);
    Tk_SetInternalBorder(getWin(), bd_width);
}

/********************************************************************
 * Configure()
 ********************************************************************/
int TreeWidget::Configure(Uint32 objc, Tcl_Obj *const objv[])
{
    GCObj    *gcObjs = getGCs();
    Tk_SavedOptions     savedOptions;
    int                 mask;

    if (Tk_SetOptions(d_interp, (char *)&d_config, d_OptTab,
                objc, objv, getWin(), &savedOptions, &mask) != TCL_OK) {
        Tk_RestoreSavedOptions(&savedOptions);
        fprintf(stderr, "ERROR: %s\n", Tcl_GetStringResult(d_interp));
        return TCL_ERROR;
    }

    if (mask & Opt_Graphics) {
        SetupGraphics();
        GCObj::invalidate(gcObjs, C_NumColors);
        SetupRedraw(RedrawForce);
    }

    if (mask & Opt_Geometry) {
        SetupGeometry();

        if (d_config.redraw) {
            SetupRedraw(RedrawForce);
        }
    }

    if (mask & Opt_Scrollbars) {
        SetupRedraw(RedrawForce|RepaintScrollX|RepaintScrollY);
    }

    if (mask & Opt_SelectMode) {
        if (!strcmp(d_config.selModeStr, "disabled")) {
            d_config.selMode = SelMode_Disabled;
            selectionClr();
        } else 
        if (!strcmp(d_config.selModeStr, "single")) {
            d_config.selMode = SelMode_Single;
        } else 
        if (!strcmp(d_config.selModeStr, "multiple")) {
            d_config.selMode = SelMode_Multiple;
        } else
        if (!strcmp(d_config.selModeStr, "extended")) {
            d_config.selMode = SelMode_Extended;
        } else {
            d_config.selMode = SelMode_Single;
            Tcl_AppendResult(d_interp, "unknown select-mode ",
                    d_config.selModeStr, 0);
            return TCL_ERROR;
        }

        SetupRedraw(RedrawForce);
    }

    if ((mask & Opt_Redraw) && d_config.redraw) {
        SetupRedraw(RedrawForce);
    }

    Tk_FreeSavedOptions(&savedOptions);

    return TCL_OK;
}

/********************************************************************
 * getOptionSpec()
 ********************************************************************/
Tk_OptionSpec *TreeWidget::getOptionSpec()
{
    static Tk_OptionSpec optSpecs[] = {
        {TK_OPTION_BORDER, "-background", "background", "Background",
            "white", -1, Tk_Offset(TreeConfig, background),
            0, (ClientData)"white", Opt_Graphics},
        {TK_OPTION_FONT, "-font", "font", "Font", "Fixed 18",
            -1, Tk_Offset(TreeConfig, tkfont), 0, 0, Opt_Graphics},

        {TK_OPTION_INT, "-width", "width", "Width",
            "100", Tk_Offset(TreeConfig, width), -1, 0, 0, Opt_Geometry},
        {TK_OPTION_INT, "-height", "height", "Height",
            "100", Tk_Offset(TreeConfig, height), -1, 0, 0, Opt_Geometry},

        {TK_OPTION_DOUBLE, "-indent", "indent", "Indent",
            "1.25", -1, Tk_Offset(TreeConfig, col_indent), 0, 0, Opt_Geometry},
        {TK_OPTION_DOUBLE, "-deltax", "deltax", "Deltax",
            "1.25", -1, Tk_Offset(TreeConfig, deltax), 0, 0, Opt_Geometry},
        {TK_OPTION_INT, "-pady", "pady", "Pady",
            "1", -1, Tk_Offset(TreeConfig, pady), 0, 0, Opt_Geometry},
        {TK_OPTION_INT, "-lineheight", "lineheight", "LineHeight",
            "1", -1, Tk_Offset(TreeConfig, lineheight), 0, 0, 0},

        {TK_OPTION_INT, "-borderwidth", "borderwidth", "BorderWidth",
            "1", Tk_Offset(TreeConfig, borderwidth), -1, 0, 0, Opt_Geometry},

        {TK_OPTION_BOOLEAN, "-redraw", "redraw", "Redraw",
            "1", Tk_Offset(TreeConfig, redraw), -1, 0, 0, Opt_Geometry},

        /**** Colors
         ****/
        {TK_OPTION_COLOR, "-foreground", "foreground", "Foreground",
            "black", -1, Tk_Offset(TreeConfig, colors[C_Foreground]),
            0, (ClientData)"black", Opt_Graphics},
        {TK_OPTION_COLOR, "-selectbackground", "selectbackground", 
            "SelectBackground",
            "#000080", -1, Tk_Offset(TreeConfig, colors[C_HighlightBg]),
            0, (ClientData)"black", Opt_Graphics},
        {TK_OPTION_COLOR, "-selectforeground", "selectforeground", 
            "SelectForeground",
            "white", -1, Tk_Offset(TreeConfig, colors[C_HighlightFg]),
            0, (ClientData)"white", Opt_Graphics},
        {TK_OPTION_COLOR, "-checkforeground", "checkforeground", 
            "CheckForeground",
            "black", -1, Tk_Offset(TreeConfig, colors[C_CheckFg]),
            0, (ClientData)"black", Opt_Graphics},
        {TK_OPTION_COLOR, "-checkbackground", "checkbackground", 
            "CheckBackground",
            "white", -1, Tk_Offset(TreeConfig, colors[C_CheckBg]),
            0, (ClientData)"white", Opt_Graphics},
        {TK_OPTION_COLOR, "-movebarfg", "movebarfg", "MoveBarFg",
            "white", -1, Tk_Offset(TreeConfig, colors[C_MoveBarFg]),
            0, (ClientData)"white", Opt_Graphics},

        {TK_OPTION_STRING, "-yscrollcommand", "yscrollcommand", 
            "YscrollCommand", (char *)0, -1, 
            Tk_Offset(TreeConfig, yscrollcommand), 0, 0, Opt_Scrollbars},
        {TK_OPTION_STRING, "-xscrollcommand", "xscrollcommand", 
            "XscrollCommand", (char *)0, -1, 
            Tk_Offset(TreeConfig, xscrollcommand), 0, 0, Opt_Scrollbars},

        {TK_OPTION_STRING, "-movedropcmd", "movedropcmd", 
            "MoveDropCmd", (char *)0, -1,
            Tk_Offset(TreeConfig, movedropcmd), 0, 0, 0},

        {TK_OPTION_STRING, "-opencmd", "opencmd",
            "OpenCmd", (char *)0, -1, Tk_Offset(TreeConfig, opencmd), 0, 0, 0},
        {TK_OPTION_STRING, "-closecmd", "closecmd",
            "CloseCmd", (char *)0, -1, 
            Tk_Offset(TreeConfig, closecmd), 0, 0, 0},
        {TK_OPTION_STRING, "-browsecmd", "browsecmd",
            "BrowseCmd", (char *)0, -1, 
            Tk_Offset(TreeConfig, browsecmd), 0, 0, 0},
        {TK_OPTION_BOOLEAN, "-drawbranch", "drawbranch", "DrawBranch",
            "true", -1, Tk_Offset(TreeConfig, drawbranch), 0, 0, 0},
        {TK_OPTION_STRING, "-selectmode", "selectmode",
            "SelectMode", "single", -1, 
            Tk_Offset(TreeConfig, selModeStr), 0, 0, Opt_SelectMode},

        {TK_OPTION_END, (char *)NULL, (char *)NULL,
            (char *)NULL, (char *)NULL, -1, 0, 0, 0, 0}
    };

    return optSpecs;
}

/********************************************************************
 * CreateInst()
 ********************************************************************/
int TreeWidget::CreateInst(
        ClientData        clientData,
        Tcl_Interp       *interp,
        int               objc,
        Tcl_Obj          *const objv[])
{
    TreeWidget *tree = new TreeWidget(interp, objc, objv);

    if (!tree->ok) {
        fprintf(stderr, "Construct error\n");
    }
    return (tree->ok)?TCL_OK:TCL_ERROR;
}

/********************************************************************
 * Tree_widget_Init()
 ********************************************************************/
extern "C" int Tree_widget_Init(Tcl_Interp *interp)
{
    LogRegionType    *lr;

    Tcl_PkgProvide(interp, "tree_widget", "8.1");

    lr = new LogRegionType("TreeWidget",
            "Tree",
            "",
            "",
            "");

    LogMgr::AddRegionType(lr);

    TreeWidget::d_OptTab = Tk_CreateOptionTable(interp,
            TreeWidget::getOptionSpec());

    int ret = ObjWidgetBase::WidgetInit(
            interp,
            "tree_widget",
            "1.0",
            "tree_widget",
            &TreeWidget::CreateInst);

    if (Tk_InitStubs(interp, "8.1", 0) == NULL) {
        Tcl_AppendResult(interp, "tk stubs init failed...", 0);
        return TCL_ERROR;
    }

    return ret;
}

extern "C" int Tree_widget_SafeInit(Tcl_Interp *interp)
{
    return Tree_widget_SafeInit(interp);
}

Tk_OptionTable TreeWidget::d_OptTab = 0;

