/****************************************************************************
 *                                ShmDFIO.h
 *
 * Author: Matthew Ballance
 * Desc:   Base class functionality for shared memory access. Used by both
 *         the ShmDFIOWriter and ShmDFIOReader classes
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 *
 ****************************************************************************/
#ifndef INCLUDED_SHM_DFIO_H
#define INCLUDED_SHM_DFIO_H
#include "DFIO.h"
#include "ShmDFIOFile.h"
#include "ShmDFIOHeader.h"

/********************************************************************/
/** \class ShmDFIO
 *  \brief Shared-memory implementation of the DFIO interface
 ********************************************************************/
class ShmDFIO : public DFIO {

    /****************************************************************
     * Public Methods
     ****************************************************************/
    public:

        /************************************************************/
        /** ShmDFIO
         ************************************************************/
        ShmDFIO(
                Tcl_Interp       *interp,
                Uint32            argc,
                Char            **argv);

        /************************************************************/
        /** ~ShmDFIO
         ************************************************************/
        virtual ~ShmDFIO();

        /************************************************************/
        /** Malloc()
         *
         * Allocates a block of shared memory. Returns the in-memory
         * pointer to the block of data. 
         * The returned pointer is 32-bit aligned
         *
         * \param size Number of bytes to allocate. 
         * 
         * \sa Ptr2ShmPtr ShmPtr2Ptr
         ************************************************************/
        virtual char *Malloc(Uint32 size);

        /************************************************************/
        /** Ptr2ShmPtr()
         *
         * Converts a memory pointer obtained from Malloc() to a
         * shm pointer, which is an offset into the block of 
         * shared memory
         *
         * \param ptr Memory pointer to convert to Shared-memory 
         *            offset
         *
         * \sa ShmPtr2Ptr
         ************************************************************/
        ShmDFIOPtr Ptr2ShmPtr(char *ptr);

        /************************************************************/
        /** ShmPtr2Ptr()
         *
         * Converts a shared-memory pointer to an in-memory pointer
         *
         * \param ptr Shared-memory offset to convert to a memory
         *            pointer
         *
         * \sa Ptr2ShmPtr
         ************************************************************/
        char *ShmPtr2Ptr(ShmDFIOPtr ptr);

        /************************************************************/
        /** UpdateMapping()
         *
         * Updates the mapping of the shared memory block. This 
         * function ensures that, as the file grows, its full 
         * length is mapped into memory
         ************************************************************/
        virtual void UpdateMapping() { ; }

        /************************************************************
         * getTimeUnit()
         ************************************************************/
        virtual Int32 getTimeUnit();

        /************************************************************
         * setTimeUnit()
         ************************************************************/
        virtual void setTimeUnit(Int32 unit);

        bool     ok;

    /****************************************************************
     * Protected Methods
     ****************************************************************/
    protected:

        /************************************************************/
        /** Remap()
         *
         * Re-maps the shared-memory file to ensure that 'size' 
         * bytes are mapped. This is called by both reader and 
         * writer as the file size changes
         ************************************************************/
        int Remap(Uint32 size);


    /****************************************************************
     * Protected Data
     ****************************************************************/
    protected:

        ShmDFIOFile          *d_file;
        ShmDFIOHeader        *d_header;
        Uint32                d_pagesize;
        const char           *d_filename;
};

#endif /* INCLUDED_SHM_DFIO_H */

