#*****************************************************************************
#*                            GuiConsole.tcl
#*
#* Author: Matthew Ballance
#* Desc:   Implements a GUI console
#*
#*
#* <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
#*
#*    This source code is free software; you can redistribute it
#*    and/or modify it in source code form under the terms of the GNU
#*    General Public License as published by the Free Software
#*    Foundation; either version 2 of the License, or (at your option)
#*    any later version.
#*
#*    This program is distributed in the hope that it will be useful,
#*    but WITHOUT ANY WARRANTY; without even the implied warranty of
#*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#*    GNU General Public License for more details.
#*
#*    You should have received a copy of the GNU General Public License
#*    along with this program; if not, write to the Free Software
#*    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
#*
#* </Copyright>
#*
#*
#*****************************************************************************
namespace eval GuiConsole {

    variable configspec {
            {-intr_handler ""}
            {-font         ""}
            {-log_fp       ""}
    }
}

#*********************************************************
#* GuiConsole
#*********************************************************
proc GuiConsole::GuiConsole {path args} {
    variable configspec

    set path [frame $path]

    rename $path ::$path:cmd
    proc ::$path {cmd args} "return \[eval GuiConsole::cmd $path \$cmd \$args\]"

    array set $path {_dummy _dummy}
    upvar #0 $path data

    foreach spec $configspec {
        set data([lindex $spec 0]) [lindex $spec 1]
    }
    eval GuiConsole::Configure $path $args

    set data(his_count) 0
    set data(cmd_running) 0

    set data(w:vscroll) [scrollbar $path.s -command "$path.t yview"]

    set data(w:text)    [text $path.t -relief sunken -bd 2 \
                         -yscrollcommand "$path.s set" \
                         -width 60 -height 13 -setgrid true \
                         -bg white]
    pack $data(w:vscroll) -side right -fill y
    pack $data(w:text)    -side right -expand yes -fill both

    bind $data(w:text) <Return> "GuiConsole::ReturnKey $path ; break"

    bind $data(w:text) <Delete> {
        catch {[%W tag remove sel sel.first promptEnd}
        if {[%W tag nextrange sel 1.0 end] == ""} {
            if {[%W compare insert < promptEnd]} {
                break
            }
        }
    }

    bind $data(w:text) <BackSpace> {
        catch {[%W tag remove sel sel.first promptEnd}
        if {[%W compare insert <= promptEnd]} {
            break
        }
        if {[%W tag nextrange sel 1.0 end] == ""} {
            if {[%W compare insert < promptEnd]} {
                break
            }
        }
    }

    bind $data(w:text) <Up> "GuiConsole::Up $path -1; break"
    bind $data(w:text) <Down> "GuiConsole::Up $path 1; break"
    bind $data(w:text) <Control-c> "GuiConsole::CtrlC $path; break"
    bind $data(w:text) <Control-C> "GuiConsole::CtrlC $path; break"
    bind $data(w:text) <Home> {%W mark set insert promptEnd ; break}

    $data(w:text) configure -foreground blue
    if {0} {
    if {$data(-font) != ""} {
        $data(w:text) tag configure t_bold -font $data(-font) \
                -foreground black
    }
    }
    $data(w:text) tag configure t_bold -foreground black

    $data(w:text) tag configure t_error -foreground    red
    $data(w:text) tag configure t_answer -foreground   black

    set data(cmd) ""

    GuiConsole::prompt $path 
    focus $data(w:text)

    #*** Now, register ourselves with the console manager
    ConsoleMgr::AddConsole $path

    return $path
}

#*********************************************************
#* cmd
#*********************************************************
proc GuiConsole::cmd {path cmd args} {
    upvar #0 $path data

    switch $cmd {

        save {
            eval GuiConsole::save $path $args
        }

        puts {
            eval GuiConsole::c_puts $path $args
        }

        configure -
        config {
            eval GuiConsole::Configure $path $args
        }

        cget {
            error "GuiConsole doesn't have \"cget\""
        }

        print_line {
            c_puts $path [lindex $args 0]
        }

        enter_command {
            prompt $path
            $data(w:text) delete promptEnd end
            $data(w:text) insert insert [string trimleft [lindex $args 0]]
            ReturnKey $path
        }

        default {
            error "no GuiConsole sub-cmd $cmd"
        }
    }
}

#*********************************************************
#* Configure
#*********************************************************
proc GuiConsole::Configure {path args} {
    upvar #0 $path data

    while {[llength $args] > 0} {
        set arg [lindex $args 0]

        set val [array get data $arg]
        if {$val == ""} {
            error "no option \"$arg\""
        }

        if {[info commands GuiConsole::config$arg] != ""} {
            set data($arg) [GuiConsole::config$arg $path [lindex $args 1]]
        } else {
            set data($arg) [lindex $args 1]
        }

        set args [lrange $args 2 end]
    }
}

#*********************************************************
#* save
#*********************************************************
proc GuiConsole::save {w filename} {
    upvar #0 $w data

    set fp [open $filename "w"]

    puts $fp [$data(w:text) get 1.0 end]

    close $fp
}

#*********************************************************
#* InsertMarkedData
#*********************************************************
proc GuiConsole::InsertMarkedData {w str} {
    upvar #0 $w data

    set s_start 0
    set tmp_str $str

    while {1} {

    }
}

#*********************************************************
#* puts
#*********************************************************
proc GuiConsole::c_puts {w str} {
    upvar #0 $w data

    if {[$data(w:text) compare insert != "insert linestart"]} {
        $data(w:text) insert insert \n
    }

    if {$data(cmd_running) != 1} {
        $data(w:text) tag remove t_bold {promptEnd linestart} promptEnd
        $data(w:text) delete {promptEnd linestart} insert
        set data(cmd) ""
    }

#    set str [lindex $str 0]

    $data(w:text) insert insert "# $str"

    if {$data(-log_fp) != ""} {
        puts $data(-log_fp) "# $str"
        flush $data(-log_fp)
    }

    if {$data(cmd_running) != 1} {
        $data(w:text) insert insert \n
        GuiConsole::prompt $w
    }

    $data(w:text) yview -pickplace insert
}

#*********************************************************
#* CtrlC Handler
#*********************************************************
proc GuiConsole::CtrlC {w} {
    upvar #0 $w data

    if {$data(cmd_running) == 1 && $data(-intr_handler) != ""} {
        $data(-intr_handler)
    } else {
        $data(w:text) insert insert \n
        GuiConsole::prompt $w
        set data(cmd) ""
        $data(w:text) yview -pickplace insert
    }
}

#*********************************************************
#* relev
#*********************************************************
proc GuiConsole::relev {w a} {
    upvar #0 $w data

    set cmd ""

    set curr_id [ConsoleMgr::GetNextCmdId]

    if {$a < 0} {
        set next_id [expr $data(his_count) - 1]
        if {$next_id >= 0} {
            set cmd [ConsoleMgr::GetHistoryCmd $next_id]
            set data(his_count) $next_id
        }
    } else {
        set next_id [expr $data(his_count) + 1]
        if {$next_id <= $curr_id} {
            set cmd [ConsoleMgr::GetHistoryCmd $next_id]
            set data(his_count) $next_id
        }
    }

    return $cmd
}

#*********************************************************
#* Up
#*********************************************************
proc GuiConsole::Up {w dir} {
    upvar #0 $w data

    if {[$data(w:text) compare insert < promptEnd]} {
    } else {
        $data(w:text) delete promptEnd end
        $data(w:text) insert insert \
            [string trimleft [GuiConsole::relev $w $dir]] 
    }

    $data(w:text) yview -pickplace insert
}

#*********************************************************
#* prompt
#*********************************************************
proc GuiConsole::prompt {w} {
    upvar #0 $w data

    $data(w:text) mark set promptBegin {insert}
    set id [expr [ConsoleMgr::GetNextCmdId] + 1]
    $data(w:text) insert insert "ivi ($id) "

    if {$data(-log_fp) != ""} {
        puts -nonewline $data(-log_fp) "ivi ($id) "
        flush $data(-log_fp)
    }

    $data(w:text) mark set promptEnd {insert}
    $data(w:text) mark gravity promptEnd left
    $data(w:text) tag add t_bold {promptEnd linestart} promptEnd
}

#*********************************************************
#* prompt2
#*********************************************************
proc GuiConsole::prompt2 {w} {
    upvar #0 $w data
    $data(w:text) insert insert "=> "

    if {$data(-log_fp) != ""} {
        puts -nonewline $data(-log_fp) "=> "
        flush $data(-log_fp)
    }
    $data(w:text) mark set promptEnd {insert}
    $data(w:text) mark gravity promptEnd left
    $data(w:text) tag add t_bold {promptEnd linestart} promptEnd
}

#*********************************************************
#* ReturnKey
#*********************************************************
proc GuiConsole::ReturnKey {w} {
    upvar #0 $w data

    $data(w:text) mark set insert {end - 1c}
    set data(cmd) "$data(cmd) [$data(w:text) get promptEnd insert]"
    $data(w:text) insert insert \n

    $data(w:text) mark set promptBegin {insert}

    if {$data(-log_fp) != ""} {
        puts -nonewline $data(-log_fp) "$data(cmd)\n"
        flush $data(-log_fp)
    }
 
    if { $data(cmd) != " "} {
        if [info complete $data(cmd)] {
            set cmd $data(cmd)
            set data(cmd) ""

            run_cmd $w $cmd
        } else {
            set data(cmd) "$data(cmd)\n"
            GuiConsole::prompt2 $w
        }
        set data(his_count) [ConsoleMgr::GetNextCmdId]
    } else {
        set data(cmd) ""
        GuiConsole::prompt $w
    }

    $data(w:text) yview -pickplace insert
}

#*********************************************************
#* run_cmd
#*********************************************************
proc GuiConsole::run_cmd {w cmd} {
    upvar #0 $w data

    set data(cmd_running) 1

    if {[catch {ConsoleMgr::RunConsoleCmd $w $cmd} res]} {
#        puts "Error: $res"
#        $data(w:text) insert insert "ERROR: "
#        $data(w:text) tag add t_error {insert linestart} insert
#        $data(w:text) insert insert "$res\n"
#        if {$data(-log_fp) != ""} {
#            puts $data(-log_fp) "ERROR: $res"
#            flush $data(-log_fp)
#        }
    } elseif {[string compare "" "$res"]} {
#        $data(w:text) insert insert "$res"
#        $data(w:text) tag add t_answer {insert linestart} insert
#        $data(w:text) insert insert "\n"
#        if {$data(-log_fp) != ""} {
#            puts $data(-log_fp) "$res"
#            flush $data(-log_fp)
#        }
    }

    set data(cmd_running) 0

#    set data(hist_count) [history nextid]
    set data(hist_count) [ConsoleMgr::GetNextCmdId]

    if {[$data(w:text) compare insert != "insert linestart"]} {
        $data(w:text) insert insert \n
    }

    GuiConsole::prompt $w

    $data(w:text) yview -pickplace insert
}
 
#*********************************************************
#* insert_cmd
#*********************************************************
proc GuiConsole::insert_cmd {w cmd} {
    upvar #0 $w data

    c_puts $w $cmd

    ReturnKey $w 
}


