/*****************************************************************************
 *                                CverDesignDB.cc
 * Author: Matthew Ballance
 * Desc:   Design database. This class allows the ui to access the design
 *         being simulated as a hierarchical tree...
 *
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 *
 *****************************************************************************/
#include "CverDesignDB.h"
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "CmdSwitcher.h"
#include "WidgetManager.h"
#include "CallbackMgr.h"

#undef  MODULE_DEBUG
#include "debug.h"

#undef ALL_PATHEL_REGEXP

static int TclCmd_DdbInstCmd(ClientData cdata, Tcl_Interp *interp,
                      int argc, char **argv);

static int CverDesignDB_ScanDesign(
        ClientData        clientData,
        Tcl_Interp       *interp,
        int               argc,
        Tcl_Obj          *const objv[]);

/*********************************************************
 * CverDesignDB()
 *********************************************************/
CverDesignDB::CverDesignDB(
        Tcl_Interp    *interp,
        int            argc,
        char         **argv) : DesignDB(interp, argc, argv)
{
}

/*********************************************************
 * ~CverDesignDB()
 *********************************************************/
CverDesignDB::~CverDesignDB()
{
}

/*********************************************************
 * scanDesign()
 * Use the simulator's VPI interface to obtain info on 
 * the design. This builds the design tree...
 *********************************************************/
void CverDesignDB::scanDesign()
{
    vpiHandle    modIt, modHndl;
    TreeNode    *thisLev;
    TreeNode    *module;
    TreeNode    *modules;

    if (!sim) {
        return;
    }

    if (designTree) {
        delete designTree;
    }

    designTree = new TreeNode(sepChar);
    thisLev = designTree;
    populateTree(designTree);

    modules     = designTree->findNode("@modules@");

    if (modIt = sim->vpi_iterate(vpiModule, 0)) {
        while (modHndl = sim->vpi_scan(modIt)) {
            module = cverScanModule(thisLev, modHndl);
            designTree->addChild(module);
            modules->addChild(module);
        }
    }

    CallbackMgr_Invoke(CBTYPE_DDB_UPDATE, instName.value(), 0, 0);

    DEBUG_1("    Top-level = %s\n", module->nodeName->value());
}

/*********************************************************
 * cverScanModule()
 *********************************************************/
TreeNode *CverDesignDB::cverScanModule(TreeNode *parent, vpiHandle modHndl)
{
    TreeNode    *module, *tmpNode;
    TreeNode    *signals, *tasks, *functions, *modules;
    vpiHandle    modIt, modIn;

    module = parent->newNodeOfType("ModuleType", 
             sim->vpi_get_str(vpiName, modHndl));
    module->findField("vpiHandle")->setVoidVal(modHndl);
    module->findField("fullName")->setStringVal(
        sim->vpi_get_str(vpiFullName, modHndl));

    signals     = module->findNode("@signals@");
    tasks       = module->findNode("@tasks@");
    functions   = module->findNode("@functions@");

   /**** modules holds the module nodes that are sub-modules of 
    **** this module
    ****/
    modules     = module->findNode("@modules@");

    DEBUG_1("Scanning module \"%s\"\n", sim->vpi_get_str(vpiName, modHndl));

    if (modIt = sim->vpi_iterate(vpiModule, modHndl)) {
        while ((modIn = sim->vpi_scan(modIt))) {

            if (sim->vpi_get(vpiType, modIn) != vpiModule) {
                fprintf(stderr, "ERROR :: type is NOT Module\n");
            }

            DEBUG_0("    vpiModule\n");
            tmpNode = cverScanModule(parent, modIn);
            modules->addChild(tmpNode);
            module->addChild(tmpNode);
        }
    }

    if (modIt = sim->vpi_iterate(vpiReg, modHndl)) {
        while ((modIn = sim->vpi_scan(modIt))) {
            if (sim->vpi_get(vpiType, modIn) != vpiReg &&
                    sim->vpi_get(vpiType, modIn) != vpiIntegerVar) {
                fprintf(stderr, "ERROR :: type is NOT Reg\n");
                fprintf(stderr, "\ttype is \"%s\"\n", 
                        IviSim::GetVpiObjString(sim->vpi_get(vpiType, modIn)));
            }

            DEBUG_0("    vpiReg\n");
            tmpNode = cverScanSignal(module, modIn, (Uint32)vpiReg); 
            signals->addChild(tmpNode);
            module->addChild(tmpNode);
        }
    }

    if (modIt = sim->vpi_iterate(vpiNet, modHndl)) {
        while ((modIn = sim->vpi_scan(modIt))) {
            if (sim->vpi_get(vpiType, modIn) != vpiNet) {
                fprintf(stderr, "ERROR :: type is NOT Net\n");
            }

            DEBUG_0("    vpiNet\n");
            tmpNode = cverScanSignal(module, modIn, (Uint32)vpiNet); 
            signals->addChild(tmpNode);
            module->addChild(tmpNode);
        }
    }

    if (modIt = sim->vpi_iterate(vpiInternalScope, modHndl)) {
        while (modIn = sim->vpi_scan(modIt)) {
            if (sim->vpi_get(vpiType, modIn) != vpiModule) {
                /*
                fprintf(stderr, "NOTE :: InternalScope :: %s\n",
                        sim->vpi_get_str(vpiName, modIn));
                 */
            }
        }
    }

    return module;     
}

/*********************************************************
 * cverScanSignal()
 *********************************************************/
TreeNode *CverDesignDB::cverScanSignal(TreeNode *parent, vpiHandle sigHndl,
                               Uint32    sigType)
{
    TreeNode *newSig;
    vpiHandle    sigBounds;
    s_vpi_value  v_val;
    s_vpi_value  r_val, l_val;

    DEBUG_1("    cverScanSignal(%s)\n", sim->vpi_get_str(vpiName, sigHndl));

    newSig = parent->newNodeOfType("SignalType", 
                                     sim->vpi_get_str(vpiName, sigHndl));

    newSig->findField("vpiHandle")->setVoidVal(sigHndl);
    newSig->findField("sigType")->setVoidVal((void *)sigType);
    newSig->findField("fullName")->setStringVal(
        sim->vpi_get_str(vpiFullName, sigHndl));

    r_val.value.integer = 0;
    l_val.value.integer = 0;

    if (sim->vpi_get(vpiVector, sigHndl)) { 
        r_val.format = vpiIntVal;
        sigBounds = sim->vpi_handle(vpiRightRange, sigHndl);
        sim->vpi_get_value(sigBounds, &r_val);

        l_val.format = vpiIntVal;
        l_val.value.integer = 0;
        sigBounds = sim->vpi_handle(vpiLeftRange, sigHndl);
        sim->vpi_get_value(sigBounds, &l_val);
    }

    DEBUG_1("        lsb = 0x%08x\n", r_val.value.integer);
    newSig->findField("sigLsb")->setVoidVal((void *)r_val.value.integer);

    DEBUG_1("        msb = 0x%08x\n", l_val.value.integer);
    newSig->findField("sigMsb")->setVoidVal((void *)l_val.value.integer);



    return newSig;
}

