/****************************************************************************
 *                              tree_node.cc  
 * Author: Matthew Ballance
 * Desc:   Implementation of a generic tree...
 *
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 *
 ****************************************************************************/
#include "tree_node.h"
#include <string.h>
#include <stdio.h>

/**********************************************************
 * TreeNode(TreeNode *)
 **********************************************************/
TreeNode::TreeNode(TreeNode *templ)
{
    Uint32    i;
    Vector<TreeNode>     *treeV;
    Vector<TreeField>    *treeF;

    isType  = false;
    nodeName = new String("");
    treeTop = templ->getTreeTop();
    parent  = templ->getParent();
    if (templ->nodeName != 0) {
        nodeName = new String(templ->nodeName);
    } else nodeName = 0;

    sepChar = templ->sepChar;
    if (templ->getTreeType() != 0) {
        treeType = new String(templ->getTreeType());
    } else treeType = new String("");

    treeV = templ->getChildren();
    children = new Vector<TreeNode>;
    for (i=0; i<treeV->length(); i++) {
        children->append(new TreeNode(treeV->idx(i)));
    }

    treeV = templ->getTypes();
    if (treeV != 0) {
        types = new Vector<TreeNode>;
        for (i=0; i<treeV->length(); i++) {
            types->append(new TreeNode(treeV->idx(i)));
        }
    } else {
        types = 0;
    }

    treeF = templ->getFields();
    fields = new Vector<TreeField>;
    for (i=0; i<treeF->length(); i++) {
        fields->append(new TreeField(treeF->idx(i)));
    }
}

/**********************************************************
 * TreeNode(sepChar)
 * - Used to create new tree root...
 **********************************************************/
TreeNode::TreeNode(char         sepChar)
{
    this->treeTop = this;
    this->sepChar = sepChar;
    this->parent  = 0;
    nodeName = new String("");
    children      = new Vector<TreeNode>;
    fields        = new Vector<TreeField>;
    types         = new Vector<TreeNode>;
    isType        = false;
    treeType      = new String("");
}

/**********************************************************
 * TreeNode()
 * NOTE: Used for TreeNode types.
 **********************************************************/
TreeNode::TreeNode(char *nodeName)
{
    children      = new Vector<TreeNode>;
    fields        = new Vector<TreeField>;
    parent        = 0;
    sepChar       = 0;
    treeTop       = 0;
    types         = 0;
    this->nodeName = new String(nodeName);
    isType        = false;
    treeType      = new String("");
}

/**********************************************************
 * newNodeOfType()
 **********************************************************/
TreeNode *TreeNode::newNodeOfType(char    *nodeTypeName,
                                  char    *nodeName)
{
    TreeNode *nodeType, *newNode;

    nodeType = findNodeType(nodeTypeName);
    if (nodeType == 0) { return 0; }

    newNode = new TreeNode(nodeType);
    *newNode->nodeName = nodeName;
    newNode->parent  = this;
    newNode->treeTop = treeTop;

    return newNode;
}

/**********************************************************
 * addField()
 **********************************************************/
void TreeNode::addField(TreeField *newField)
{
    if (!fields) {
        fields = new Vector<TreeField>();
    }
    fields->append(newField);
}

/**********************************************************
 * ~TreeNode()
 **********************************************************/
TreeNode::~TreeNode(void)
{
  
}

/**********************************************************
 * findNodeType()
 **********************************************************/
TreeNode *TreeNode::findNodeType(char       *nodeType)
{
   Uint32        i;

   if (parent != 0) {
       return treeTop->findNodeType(nodeType);
   }

   for (i=0; i<types->length(); i++)
   {
       if (types->idx(i)->nodeName->equal(nodeType)) {
           return types->idx(i);
       }
   }

   return (TreeNode *)0;
}

/********************************************************** 
 * findNode()
 * 
 * Find a node based on the supplied path...
 **********************************************************/
TreeNode *TreeNode::findNode(char *pathToNode)
{
    TreeNode *startPoint;
    char     *startCh = 0, *sep;
    Uint32    i, elemlen;

    /*** Absolutely-rooted path... Start with tree top
     ***/
    if (pathToNode[0] == sepChar) {
        if (pathToNode[1] == 0) {
            if (parent == 0) {
                return this;
            } else {
                return treeTop;
            } 
        } else if (parent != 0) {
            return treeTop->findNode(&pathToNode[1]);
        } else {
            startCh = &pathToNode[1];
        }
    } else {
        startCh = pathToNode;
    }

    /*** Otherwise, we've located the place to begin the search...
     ***/
    if (!startCh) {
        fprintf(stderr, "ERROR :: startCh == NULL\n");
    }

    sep = strchr(startCh, sepChar);

    if (sep != 0) {
        elemlen = sep - startCh;
    } else {
        elemlen = strlen(startCh);
    }

    for (i=0; i<children->length(); i++) {
        if (children->idx(i)->nodeName->equal(startCh, elemlen)) {
            if ((sep != 0) && (sep[1] != 0) && (sep[1] != ' ')) {
                return children->idx(i)->findNode((sep+1));
            } else {
                return children->idx(i);
            }
        } 
    }

   return (TreeNode *)0;
}

/**********************************************************
 * findField
 **********************************************************/
TreeField *TreeNode::findField(char    *fieldName)
{
    Uint32 i, len;
    len = strlen(fieldName);

    for (i=0; i<fields->length(); i++) {
        if (fields->idx(i)->fieldName->equal(fieldName)) {
            return fields->idx(i);
        }
    }

    return (TreeField *)0;
}

/**********************************************************
 * newNodeType
 * Creates a new node type...
 **********************************************************/
TreeNode *TreeNode::newNodeType(char        *nodeType)
{
    Uint32         i;
    TreeNode      *newType;

    if (parent != 0) {
        return treeTop->newNodeType(nodeType);
    }

   /**** If we _ARE_ the tree top, then go ahead and try to create this
    **** type...
    ****/

    newType = findNodeType(nodeType);
    if (newType != 0) return newType;
 
    newType = new TreeNode(nodeType);
    newType->isType = true;
    newType->sepChar = sepChar;
    newType->treeType = new String(nodeType);
    
    types->append(newType);
  
    return newType;
}

/***************************************************
 * addChild()
 ***************************************************/
TreeNode *TreeNode::addChild(char           *nodeName)
{
    TreeNode *newChild = 0, *tmp;
    Uint32    i, len = children->length();

    for (i=0; i<len; i++) {
        tmp = children->idx(i);
        if (tmp->nodeName->equal(nodeName)) {
            newChild = tmp;
            return newChild;
        }
    }

    if (!newChild) {
        newChild = new TreeNode(sepChar);
        newChild->treeTop = treeTop;
        newChild->parent = this;
       *newChild->nodeName = nodeName;
        children->append(newChild);
    }

    return newChild;
}

/***************************************************
 * addChild()
 ***************************************************/
TreeNode *TreeNode::addChild(TreeNode       *newNode)
{
    TreeNode *newChild = 0, *tmp;
    Uint32    i, len;

    if (children == 0) { children = new Vector<TreeNode>; }
    len = children->length();

    for (i=0; i<len; i++) {
        tmp = children->idx(i);
        if (tmp->nodeName == newNode->nodeName) {
            fprintf(stderr, "ERROR :: Adding duplicate node %s\n",
                    newNode->nodeName->value());
            return tmp;
        }
    }

    newNode->treeTop = treeTop;
    newNode->parent  = this;
    newNode->sepChar = sepChar; 
    children->append(newNode);

    return newNode;
}

/***************************************************
 * addChildOfType() 
 * Add a child of 
 ***************************************************/
TreeNode *TreeNode::addChildOfType(char         *nodeType,
                                   char         *nodeName
                                  )
{
    TreeNode *newNode = newNodeOfType(nodeType, nodeName);

    if (newNode == 0) { return 0; }

    return addChild(newNode);

} 


