/****************************************************************************
 *                             StackTrace.cc
 *
 * Author: Matthew Ballance
 * Desc:    
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 ****************************************************************************/
#include "StackTrace.h"
#include "config.h"

#ifdef ENABLE_STACK_TRACE 
#include <unistd.h>
#include <sys/select.h>
#include <sys/wait.h>
#include <signal.h>
#include <stdio.h>
#include <time.h>
#include <string.h>

static bool stack_trace_done = false;

/********************************************************************
 * prv_stack_trace_sigchild()
 ********************************************************************/
static void prv_stack_trace_sigchild(int signum)
{
    stack_trace_done = true;
}

/********************************************************************
 * prv_stack_trace()
 ********************************************************************/
static void prv_stack_trace(char **args)
{
    int pid;
    int in_fd[2];
    int out_fd[2];
    fd_set  fdset, readset;
    struct timeval tv;
    int sel, index, state;
    char buffer[512];
    char c;
    FILE   *fp;

    fprintf(stderr, "IVI detected a fatal error\n");

    strcpy(buffer, "ivi_stacktrace.");
    strcat(buffer, args[2]);

    if ((fp = fopen(buffer, "w"))) {
        fprintf(stderr, "A stack-trace will be written to ivi_stacktrace.%s\n",
            args[2]);
    } else {
        fprintf(stderr, "Cannot open ivi_stacktrace.%s - stacktrace goes to "
                "stdout\n");
        fp = stdout;
    }

    stack_trace_done = false;
    signal(SIGCHLD, prv_stack_trace_sigchild);

    if ((pipe(in_fd) == -1) || (pipe(out_fd) == -1)) {
        perror("unable to open pipe");
        _exit(0);
    }

    if (!(pid = fork())) {
        /**** Child ****/
        close(0); dup(in_fd[0]);    /* set stdin to the in pipe   */
        close(1); dup(out_fd[1]);   /* set stdout to the out pipe */
        close(2); dup(out_fd[1]);   /* set stderr to the out pipe */

        execvp(args[0], args);
        perror("exec of gdb failed");
        _exit(0);
    } else if (pid < 0) {
        perror("unable to fork");
        _exit(0);
    }

    FD_ZERO(&fdset);
    FD_SET(out_fd[0], &fdset);

    write(in_fd[1], "backtrace\n", 10);
    write(in_fd[1], "p x = 0\n", 8);
    write(in_fd[1], "quit\n", 5);

    index = 0;
    state = 0;

    while (1) {
        readset = fdset;
        tv.tv_sec = 1;
        tv.tv_usec = 0;

        sel = select(out_fd[0]+1, &readset, NULL, NULL, &tv);
        if (sel == -1) {
            break;
        }

        if ((sel > 0) && (FD_ISSET(out_fd[0], &readset))) {

            if (read(out_fd[0], &c, 1)) {
                switch (state) {
                    case 0:
                        if (c == '#') {
                            state = 1;
                            index = 0;
                            buffer[index++] = c;
                        }
                        break;

                    case 1:
                        buffer[index++] = c;
                        if ((c == '\n') || (c == '\r')) {
                            buffer[index] = 0;
                            fprintf(fp, "%s", buffer);
                            state = 0;
                            index = 0;
                        }
                        break;

                    default:
                        break;
                }
            }
        } else if (stack_trace_done) {
            break;
        }
    }

    if (fp != stdout) {
        fclose(fp);
    }

    close(in_fd[0]);
    close(in_fd[1]);
    close(out_fd[0]);
    close(out_fd[1]);
    _exit(0);
}

#endif /* ifdef ENABLE_STACK_TRACE */


/********************************************************************
 * StackTrace()
 ********************************************************************/
void StackTrace(const char *prog_name)
{
#ifdef ENABLE_STACK_TRACE
    int   pid, status;
    char *args[4] = {"gdb", NULL, NULL, NULL };
    char  pbuf[16];

    if (!prog_name) {
        return;
    }

    sprintf(pbuf, "%u", (int)getpid());

    args[1] = (char *)prog_name;
    args[2] = pbuf;

    if (!(pid = fork())) {
        /**** Child ****/
        prv_stack_trace(args);
    } else if (pid < 0) {
        perror("Unable to start gdb");
    }

    waitpid(pid, &status, 0);
    
    _exit(0);
#endif /* ifdef ENABLE_STACK_TRACE */
}

