/*****************************************************************************
 *                              SdbSignal.cc
 * 
 * Author: Matthew Ballance
 * Desc:   
 *
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 *
 *****************************************************************************/
#include "SdbSignal.h"
#include "SdbReader.h"
#include "tclOldConfig.h"
#include "TclBitVector.h"
#include "DFIOTrace.h"
#include "DFIO.h"
#include "BitVector.h"
#include "BitVectorABVal.h"
#include <stdlib.h>

static int SdbSignal_InstCmd(
        ClientData        clientData,
        Tcl_Interp       *interp,
        int               argc,
        char            **argv);

static void SdbSignal_Delete(char *ptr);

/****************************************************************
 * SdbSignal()
 ****************************************************************/
SdbSignal::SdbSignal(
    SdbReader   *sdbr,
    Tcl_Interp  *interp,
    Char        *id,
    Char        *fullName, 
    Uint32       msb,
    Uint32       lsb,
    DFIOTrace  *dfioTrace)
{
    this->sdbr     = sdbr;
    this->interp   = interp;
    this->msb      = msb;
    this->lsb      = lsb;
    this->radix    = 0;
    this->sdbName  = 0;

    d_EnumHash     = 0;

    if (lsb < msb) {
        this->len = msb - lsb + 1;
    } else {
        this->len = lsb - msb + 1;
    }
    
    this->fullName = Tcl_Alloc(strlen(fullName)+1);
    strcpy(this->fullName, fullName);

    this->dispName = Tcl_Alloc(strlen(fullName)+1);
    strcpy(this->dispName, fullName);

    this->id = new Char[strlen(id)+1];
    strcpy(this->id, id);

    this->dfioTrace = dfioTrace;

    valueFormat = ValFmt_BitStr;

    separator = 0;
    expanded = 0;

    Configure(0, 0, 0);

    Tcl_CreateCommand(interp, id, 
            (Tcl_CmdProc *)SdbSignal_InstCmd, this, NULL);
}

/****************************************************************
 * ~SdbSignal
 ****************************************************************/
SdbSignal::~SdbSignal(void)
{
    /**** Remove this signal from its parent SDBR ****/
    sdbr->delSignal(this);
    if (dfioTrace) {
        dfioTrace->RemoveClient(this);
    }

    Tcl_DeleteCommand(interp, id);
    delete[] id;
    Tcl_Free(fullName);
    Tcl_Free(dispName); 
}

/****************************************************************
 * FormatValue()
 ****************************************************************/
void SdbSignal::FormatValue(
    DFIOValChg           *value,
    String               &fmtVal)
{
    Uint32 flags;
    FormatValue(value, fmtVal, flags);
}

/****************************************************************
 * FormatValue()
 ****************************************************************/
void SdbSignal::FormatValue(
    DFIOValChg           *value,
    String               &fmtVal,
    Uint32               &flags)
{
    flags = 0;

    switch (valueFormat) {
        case ValFmt_BitStr:
            BitVectorABVal::fmtBin(value->bitVal, len, fmtVal, 0, flags);
            break;

        case ValFmt_Hex:
            BitVectorABVal::fmtHex(value->bitVal, len, fmtVal, 0, flags);
            break;

        case ValFmt_Dec:
            BitVectorABVal::fmtDec(value->bitVal, len, fmtVal, 0, flags);
            break;

        case ValFmt_Oct:
            BitVectorABVal::fmtOct(value->bitVal, len, fmtVal, 0, flags);
            break;

        case ValFmt_String:
            BitVectorABVal::fmtString(value->bitVal, len, fmtVal, 0, flags);
            break;

        case ValFmt_Enum:
            fmtEnum(value->bitVal, fmtVal, flags);
            break;

        default:
            fprintf(stderr, "ERROR :: Unknown value format!!\n");
            break;
    }
}

/******************************************************************
 * fmtEnum()
 ******************************************************************/
void SdbSignal::fmtEnum(const DFIOVal *value, String &fmtVal, Uint32 &flags)
{
    IviUintHashTabEntry<String *>   *e;
    Uint32       key=0;
    Uint32       nDigits = (len/8)+((len%8)?1:0);
    Uint32       nz=0, nx=0, n01=0;

    for (Uint32 i=0; i<nDigits; i++) {
        const DFIOVal *db = &value[i];

        key |= (db->aval << (8*i));

        if (db->bval) {
            nx++;
        }
    }

    if (!d_EnumHash ||
            !(e = d_EnumHash->find(key))) {
        fmtVal = "Unknown";
    } else {
        fmtVal = *e->getValue();
    }
}

/******************************************************************
 * findEdge()
 ******************************************************************/
void SdbSignal::findEdge(Uint32 time, Uint32 fwd, Uint32 *nextTime)
{
    Vector<DFIOValChg>   *vals;

    vals = dfioTrace->getValue(time, time, 
            (fwd)?DFIOTrace::GVF_OnePlus:DFIOTrace::GVF_OneMinus);

    if (vals->length() > 1) {
        if (fwd) {
            *nextTime = vals->idx(1)->changeTime;
        } else {
            *nextTime = vals->idx(0)->changeTime;
        }
    } else {
        *nextTime = time;
    }

    delete vals;
}

/****************************************************************
 * getValue()
 ****************************************************************/
Int32 SdbSignal::getValue(Uint32 argc, Char **argv)
{
    Vector<DFIOValChg>     *val;
    Char                    buf[1024];
    DFIOValChg             *ptr;
    Uint32                  i, time, valid;
    Tcl_Obj                *list, *r_val, *tmp;

    if (isSeparator() || !dfioTrace) {
        Tcl_AppendResult(interp, " ", 0);
        return TCL_OK;
    }

    time = strtoul(argv[0], 0, 0);

    val  = dfioTrace->getValue(time, time, 0);
    list = Tcl_NewListObj(0, 0);

    valid = (val->length())?1:0;

    Tcl_ListObjAppendElement(interp, list,
                    Tcl_NewStringObj("-- No Data --", -1));
#if 0
    if (valid && val->idx(0)) {
        ptr = val->idx(0);
        tmp = Tcl_NewBitVector_DFIOVal(ptr->bitVal, len);

        fprintf(stderr, "FIXME: Tcl_NewBitVector_BinStr(ptr, len);\n");
//        tmp = Tcl_NewBitVector_BinStr(ptr, len);

        Tcl_ListObjAppendElement(interp, list, tmp);
    } else {
        Tcl_ListObjAppendElement(interp, list,
                    Tcl_NewStringObj("-- No Data --", -1));
    }

    if (expanded) {
        for (i=0; i<len; i++) {
            if (valid) {
                Tcl_ListObjAppendElement(interp, list,
                        Tcl_NewBitVector_DFIOVal(&ptr->bitVal[i], 1));
            } else {
                Tcl_ListObjAppendElement(interp, list,
                        Tcl_NewStringObj("-- No Data --", -1));
            }
        }
    }
#endif

    Tcl_SetObjResult(interp, list);

    if (val) {
        delete val;
    }
    
    return TCL_OK;
}

/****************************************************************
 * Configure()
 ****************************************************************/
Int32 SdbSignal::Configure(Uint32 argc, Char **argv, Uint32 flags)
{
    Int32 ret, i;
    Char  tmp[2048], *ptr;

    ret = Tcl_ConfigureWidget(interp, 0, getConfigSpec(), argc, argv,
            (char *)this, flags);

    /**** Check to see if information has been changed...
     ****/

    /**** Update the number of visable elements
     ****
     **** Do the calculation to compute the display-name when:
     **** - Visel changes (and DispName not specified)
     **** - Dispname change (and dispName is NULL)
     ****/
    if ((optionSpecified(Opt_Visel) && !optionSpecified(Opt_Dispname)) ||
        (optionSpecified(Opt_Dispname) && (!dispName || !dispName[0]))) {

        if (dispName) {
            Tcl_Free(dispName);
        }
        if (!visEl) {
            dispName = Tcl_Alloc(strlen(fullName)+1);
            strcpy(dispName, fullName);
        } else {
            ptr = &fullName[strlen(fullName)];
            for (i=0; i<visEl; i++) {
                while ((ptr >= fullName) && *ptr != '.') {
                    ptr--;
                }
                if (ptr < fullName) {
                    ptr++;
                    break;
                } else {
                    ptr--;
                }
            }

            if (ptr[1] == '.') {
                ptr += 2;
            }

            dispName = Tcl_Alloc(strlen(ptr)+1);
            strcpy(dispName, ptr);
        }
    }

    if (optionSpecified(Opt_Radix)) {
        if (String::equal(radix, "bin")) {
            valueFormat = ValFmt_BitStr;
        } else 
            if (String::equal(radix, "hex")) {
                valueFormat = ValFmt_Hex;
        } else
            if (String::equal(radix, "dec")) {
                valueFormat = ValFmt_Dec;
        } else 
            if (String::equal(radix, "oct")) {
                valueFormat = ValFmt_Oct;
        } else 
            if (String::equal(radix, "str")) {
                valueFormat = ValFmt_String;
        } else
            if (String::equal(radix, "enum")) {
                valueFormat = ValFmt_Enum;
        } else {
            valueFormat = ValFmt_BitStr;
        }
    }

    sdbr->update();

    return ret;
}

/****************************************************************
 * InstCmd
 ****************************************************************/
Int32 SdbSignal::InstCmd(Uint32 argc, Char **argv)
{
    Uint32   next;
    Uint32   cmd = CmdSwitch(sdbSignalCmds, argv[1]);

    switch (cmd) {
        case SDS_Config:
            return Configure(argc-2, &argv[2], TK_CONFIG_ARGV_ONLY);
            break;

        case SDS_Cget:
            if (argc != 3) {
                Tcl_AppendResult(interp, "wrong # args", 0);
                return TCL_ERROR;
            }
            return Tcl_ConfigureValue(interp, 0, getConfigSpec(),
                    (char *)this, argv[2], 0);
            break;

        case SDS_Delete:
//            sdbr->delSignal(this);
            Tcl_EventuallyFree(this, SdbSignal_Delete);
            break;

        case SDS_Index:
            Tcl_SetObjResult(interp, Tcl_NewIntObj(sdbr->sigIndex(this)));
            break;

        case SDS_GetValue:
            return getValue(argc-2, &argv[2]);
            break;

        case SDS_NextEdge:
        case SDS_PrevEdge:
            findEdge(strtoul(argv[2], 0, 0), (cmd == SDS_NextEdge)?1:0,
                    &next);
            Tcl_SetObjResult(interp, Tcl_NewIntObj(next));
            break;

        case SDS_SetEnum:
            return setEnum(argc, argv);
            break;

        case SDS_ClrEnum:
            return clrEnum(argc, argv);
            break;

        case SDS_GetEnum:
            return getEnum(argc, argv);
            break;

        default:
            break;
    }

    return TCL_OK;
}

/********************************************************************
 * setEnum()
 *
 * argv[0] = <obj>
 * argv[1] = set_enum
 ********************************************************************/
int SdbSignal::setEnum(int argc, char **argv)
{
    int  num_set = 0;

    if ((argc-2) % 2) {
        Tcl_AppendResult(interp, "enum list must have an even length", 0);
        return TCL_ERROR;
    }
   
    if (d_EnumHash) {
        delete d_EnumHash;
    }

    d_EnumHash = new IviUintHashTab<String *>();

    for (int i=2; i<argc; i+=2) {
        IviUintHashTabEntry<String *>  *e;
        char *intVal = argv[i];
        char *txtVal = argv[i+1];

        if (!*intVal || !*txtVal) {
            continue;
        }

        num_set++;

        String *val = new String(txtVal);
        e = new IviUintHashTabEntry<String *>(strtoul(intVal, 0, 10), val);
        d_EnumHash->add(e);
    }

    if (num_set) {
        hasEnum = 1;
    } else {
        hasEnum = 0;
        delete d_EnumHash;
        d_EnumHash = 0;
    }

    sdbr->update();

    return TCL_OK;
}

/********************************************************************
 * clrEnum()
 ********************************************************************/
int SdbSignal::clrEnum(int argc, char **argv)
{
    if (d_EnumHash) {
        delete d_EnumHash;
    }
    d_EnumHash = 0;

    hasEnum = 0;

    return TCL_OK;
}

/********************************************************************
 * getEnum()
 *
 * argv[0] = <obj>
 * argv[1] = set_enum
 ********************************************************************/
int SdbSignal::getEnum(int argc, char **argv)
{
    Tcl_Obj    *list = Tcl_NewListObj(0, 0);

    if (d_EnumHash) {
        Vector< IviUintHashTabEntry<String *> >  vect; 

        d_EnumHash->getList(vect);

        for (Uint32 i=0; i<vect.length(); i++) {
            IviUintHashTabEntry<String *>    *e = vect.idx(i);

            Tcl_ListObjAppendElement(interp, list, Tcl_NewIntObj(e->getKey()));
            Tcl_ListObjAppendElement(interp, list, 
                    Tcl_NewStringObj(e->getValue()->value(), -1));
        }
    }

    Tcl_SetObjResult(interp, list);
    return TCL_OK;
}

/****************************************************************
 * getConfigSpec()
 ****************************************************************/
Tk_ConfigSpec *SdbSignal::getConfigSpec()
{
    static Tk_ConfigSpec   sdbSigSpec[] = {
    {TK_CONFIG_STRING, "-radix", "radix", "Radix",
        "bin", Tk_Offset(SdbSignal, radix), 0},
    {TK_CONFIG_INT, "-separator", "separator", "Separator",
        "0",   Tk_Offset(SdbSignal, separator), TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_INT, "-visel", "visel", "VisEl",
        "0",   Tk_Offset(SdbSignal, visEl), 0},
    {TK_CONFIG_INT, "-msb", "msb", "Msb",
        "0",   Tk_Offset(SdbSignal, msb), TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_INT, "-lsb", "lsb", "Lsb",
        "0",   Tk_Offset(SdbSignal, lsb), TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_STRING, "-sig_fullname", "sig_fullname", "SigFullname",
        (char *)NULL, Tk_Offset(SdbSignal, fullName), 
        TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_STRING, "-sig_dispname", "sig_dispname", "SigDispName",
        (char *)NULL, Tk_Offset(SdbSignal, dispName),
        TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_INT, "-expanded", "expanded", "Expanded",
        "0", Tk_Offset(SdbSignal, expanded), 0},
    {TK_CONFIG_STRING, "-sdb", "sdb", "Sdb",
        (char *)"", Tk_Offset(SdbSignal, sdbName), 0},
    {TK_CONFIG_INT, "-has_enum", "hasEnum", "HasEnum",
        "0", Tk_Offset(SdbSignal, hasEnum), 0},
    {TK_CONFIG_END, (char *)NULL, (char *)NULL, (char *)NULL,
        (char *)NULL, 0, 0}
    };
    return sdbSigSpec;
}

/****************************************************************
 * SdbSignal_Delete()
 ****************************************************************/
static void SdbSignal_Delete(char *ptr)
{
    delete ((SdbSignal *)ptr);
}

/****************************************************************
 * SdbSignal_InstCmd()
 ****************************************************************/
static int SdbSignal_InstCmd(
        ClientData        clientData,
        Tcl_Interp       *interp,
        int               argc,
        char            **argv)
{
    SdbSignal    *sdbSignal = (SdbSignal *)clientData;
    int           ret;

    Tcl_Preserve(clientData);
    ret = sdbSignal->InstCmd(argc, argv);
    Tcl_Release(clientData);
    return ret;
}


