/****************************************************************************
 *                            LogInstance.cc
 *
 * Author: Matthew Ballance
 * Desc:   Describes an instance of a logger... The instance may have
 *         sub-regions.
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 ****************************************************************************/
#include "LogInstance.h"
#include "LogMgr.h"
#include "LogDestination.h"
#include <string.h>

/********************************************************************
 * LogInstance()
 ********************************************************************/
LogInstance::LogInstance(
        LogRegionType *regionType, 
        const char *instNameStr) : LogRegionType(regionType),
    instName(instNameStr), d_regionType(regionType), prvBuffer(4096)
{
    destList   = new Vector<LogDestination>();
    destRefCnt = 0;
}

/********************************************************************
 * ~LogInstance()
 ********************************************************************/
LogInstance::~LogInstance()
{
    delete destList;
}

/********************************************************************
 * AddDestination()
 *
 * Adds an output to this instance... This may cause the enable 
 * state of this instance to change...
 ********************************************************************/
int LogInstance::AddDestination(LogDestination  *dest)
{
    destList->append(dest);

    dest->addInstance(this);
    /**** See if the state of this dest changes the overall state
     **** of the object...
     ****/
    destRefCnt += (Uint32)dest->getDebugLevel();

    return 0;
}

/********************************************************************
 * DestStateUpdate()
 *
 * Called when the state of a destination changes. This will be 
 * called when:
 * - debug level changes on the destination
 * 
 ********************************************************************/
int LogInstance::DestStateUpdate(
        LogDestination       *dest, 
        LogMgr::DebugLevel    oldState)
{
    if (destRefCnt) {
        destRefCnt -= oldState;
    }

    destRefCnt += dest->getDebugLevel();
    return 0;
}

/********************************************************************
 * print()
 ********************************************************************/
int LogInstance::vprint(LogMgr::DebugLevel dbgLevel, const char *fmt, 
        va_list ap)
{
    int                ret;
    Uint32             len, i;
    LogDestination    *dest;

    if (!destRefCnt) {
        return 0;
    }

    /**** Create the message in the buffer, then distribute it to 
     **** interested log destinations
     ****/
    ret = vsnprintf(prvBuffer.value(), prvBuffer.max_len(), fmt, ap);

    if (ret < 0) {
        fprintf(stderr, "ERROR :: Buffer overflow...\n");
    }

    /**** Now, loop through each destination that is attached to this instance
     **** - If the destination's debug level is >= the level of this message,
     ****   then we can send the message...
     ****/

    len = destList->length();
    for (i=0; i<len; i++) {
        dest = destList->idx(i);

        if (((Uint32)dest->getDebugLevel()) >= ((Uint32)dbgLevel)) {
            dest->write(prvBuffer.value());
        }
    }

    return 0;
}

/********************************************************************
 * venter()
 ********************************************************************/
void LogInstance::venter(LogMgr::DebugLevel dbgLevel, 
        const char *fmt, va_list ap)
{
    Uint32          len, i;
    int             ret;
    LogDestination *dest;

    if (!destRefCnt) {
        return;
    }

    strcpy(prvBuffer.value(), "----> ");
    ret = vsnprintf(&prvBuffer.value()[6], prvBuffer.max_len(), fmt, ap);

    if (ret < 0) {
        fprintf(stderr, "ERROR :: Buffer overflow...\n");
    }

    len = destList->length();
    for (i=0; i<len; i++) {
        dest = destList->idx(i);

        if (((Uint32)dest->getDebugLevel()) >= ((Uint32)dbgLevel)) {
            dest->write(prvBuffer.value());
            dest->enter();
        }
    }
}

/********************************************************************
 *
 ********************************************************************/
void LogInstance::vleave(LogMgr::DebugLevel dbgLevel, 
        const char *fmt, va_list ap)
{
    Uint32          len, i;
    int             ret;
    LogDestination *dest;

    if (!destRefCnt) {
        return;
    }

    strcpy(prvBuffer.value(), "<---- ");
    ret = vsnprintf(&prvBuffer.value()[6], prvBuffer.max_len(), fmt, ap);

    if (ret < 0) {
        fprintf(stderr, "ERROR :: Buffer overflow...\n");
    }

    len = destList->length();
    for (i=0; i<len; i++) {
        dest = destList->idx(i);

        if (((Uint32)dest->getDebugLevel()) >= ((Uint32)dbgLevel)) {
            dest->leave();
            dest->write(prvBuffer.value());
        }
    }
}


