/****************************************************************************
 *                             IviRemoteProc.h
 *
 * Author: Matthew Ballance
 * Desc:   Describes a remote process. 
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 ****************************************************************************/
#ifndef INCLUDED_IVI_REMOTE_PROC_H
#define INCLUDED_IVI_REMOTE_PROC_H
#include "types.h"
#include <tcl.h>
#include "RemoteAppConnection.h"
#include "vector.h"

#ifdef __MINGW32__
#include <windows.h>
#endif

class TclChannelObj;
class IviRemoteProc {
    /****************************************************************
     * Public Methods
     ****************************************************************/
    public:

        /************************************************************
         * IviRemoteProc()
         ************************************************************/
        IviRemoteProc(Tcl_Interp *interp, int argc, char **argv);

        /************************************************************
         * ~IviRemoteProc()
         ************************************************************/
        ~IviRemoteProc();

        /************************************************************
         * setInvokeString()
         *
         * Sets the command to execute
         ************************************************************/
        void setInvokeString(const char *invokeStr);

        /************************************************************
         * setInvoke()
         ************************************************************/
        void setInvoke(int argc, char **argv);

        /************************************************************
         * setInvoke()
         ************************************************************/
        void setInvoke(Tcl_Obj *obj);

        /************************************************************
         * invoke()
         *
         * Invokes the command specified by this object
         ************************************************************/
        int invoke();

        /************************************************************
         * wait()
         ************************************************************/
        int wait(int *status);

        /************************************************************
         * kill()
         *
         * Kills the process represented by this object
         ************************************************************/
        int kill();

        /************************************************************
         * isRunning()
         ************************************************************/
        bool isRunning();

        /************************************************************
         * isRunning()
         ************************************************************/
        bool isRunning(int ms_timeout);

        /************************************************************
         * findProc()
         ************************************************************/
        static IviRemoteProc *findProc(Uint32 pid);

        /************************************************************
         * deleteProc()
         ************************************************************/
        static void deleteProc(IviRemoteProc *proc);

        /************************************************************
         * getCommConnect()
         ************************************************************/
        RemoteAppConnection *getCommConnect() { return d_conn; }

        /************************************************************
         * setCommConnect()
         ************************************************************/
        void setCommConnect(RemoteAppConnection *conn) { d_conn = conn; }

        /************************************************************
         * getStdoutChannelName()
         ************************************************************/
        const char *getStdoutChannelName();

        /************************************************************
         * getProcRunning()
         ************************************************************/
        bool getProcRunning() { return d_running; }

    /****************************************************************
     * Private Methods
     ****************************************************************/
    private:

        /************************************************************
         * InputProc()
         ************************************************************/
        static void InputProc(ClientData cdata, int mask);

        /************************************************************ 
         * WaitHandler()
         ************************************************************/
        static void WaitHandler(ClientData cdata);

        /************************************************************
         * InstCmd()
         ************************************************************/
        static int InstCmd(
                ClientData        clientData,
                Tcl_Interp       *interp,
                int               objc,
                Tcl_Obj          *const objv[]) {
            int ret;
            Tcl_Preserve(clientData);
            ret = ((IviRemoteProc *)clientData)->InstCmd(objc, objv);
            Tcl_Release(clientData);
            return ret;
        }

        /************************************************************
         * InstCmd()
         ************************************************************/
        int InstCmd(int objc, Tcl_Obj *const objv[]);

        /************************************************************
         * Destroy()
         ************************************************************/
        static void Destroy(char *data) {
            delete ((IviRemoteProc *)data);
        }
        void Destroy() {
            Tcl_EventuallyFree(this, &IviRemoteProc::Destroy);
        }

    /****************************************************************
     * Private Data
     ****************************************************************/
    private:
        char                    *d_invokeStr;

        int                      d_procArgc;
        char                   **d_procArgv;
        Vector<char>             d_procArgs;


        Uint32                   d_pid;
        bool                     d_running;
        bool                     d_mergeStd;
        bool                     d_channelActive;
        Tcl_Interp              *d_interp;
        int                      d_outFd[2];
        Tcl_TimerToken           d_waitTimer;
        RemoteAppConnection     *d_conn;

        TclChannelObj           *d_stdout;

        Tcl_Command              d_cmdToken;

        int                      d_procExitCode;

#ifdef __MINGW32__
        PROCESS_INFORMATION      d_procInfo;
#endif
   
        Tcl_Channel              d_procOutput;

        Uint32                   d_procIdx;

        static IviRemoteProc    *d_procList[128];
        static Uint32            d_numProcs;
};

#endif /* INCLUDED_IVI_REMOTE_PROC_H */

