/****************************************************************************
 *                               ConfigDB.cc
 *
 * Author: Matthew Ballance
 * Desc:   
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 ****************************************************************************/
#ifndef INCLUDED_CONFIG_DB_H
#define INCLUDED_CONFIG_DB_H
#include "types.h"
#include "ivi_String.h"
#include "Stack.h"
#include <tcl.h>
#include "vector.h"

class ConfigDBItem;
class ConfigDBElem;

/********************************************************************/
/** \class ConfigDBCallback
 *  \brief Represents a callback for a config-database element
 ********************************************************************/ 
class ConfigDBCallback {

    /****************************************************************
     * Public Methods
     ****************************************************************/
    public:

        /************************************************************
         * ConfigDBCallback()
         ************************************************************/
        ConfigDBCallback(
                ConfigDBElem         *elem, 
                Tcl_ObjCmdProc        func,
                ClientData            clientData);

        /************************************************************
         * ConfigDBCallback()
         ************************************************************/
        ConfigDBCallback(
                ConfigDBElem         *elem,
                Tcl_Interp           *interp,
                Tcl_Obj              *cmd);

        /************************************************************
         * ~ConfigDBCallback()
         ************************************************************/
        ~ConfigDBCallback();

        /************************************************************
         * Invoke()
         ************************************************************/
        void Invoke(const char *elem_name);

    /****************************************************************
     * Private Methods
     ****************************************************************/
    private:

        /************************************************************
         * TclCmdFunc
         ************************************************************/
        static int TclCmdFunc(
                ClientData               clientData,
                Tcl_Interp              *interp,
                int                      objc,
                Tcl_Obj                 *const objv[]);

    /****************************************************************
     * Private Data
     ****************************************************************/
    private:
        Tcl_Obj                          *d_tclCmd;
        Tcl_ObjCmdProc                   *d_func;
        ClientData                        d_userData;
        Tcl_Interp                       *d_interp;
        ConfigDBElem                     *d_parent;
};

/********************************************************************/
/** \class ConfigDBElem
 *  \brief Holds properties of all config elements
 ********************************************************************/
class ConfigDBElem {

    /****************************************************************
     * Public Methods
     ****************************************************************/
    public:

        typedef enum {
            Type_Item,
            Type_Section
        } Type;

        /************************************************************
         * ConfigDBElem()
         ************************************************************/
        ConfigDBElem(
                ConfigDBElem        *parent,
                const char          *name,
                ConfigDBElem::Type   type);

        /************************************************************
         * ~ConfigDBElem()
         ************************************************************/
        virtual ~ConfigDBElem();

        /************************************************************
         * getElemName()
         ************************************************************/
        const char *getElemName() { return d_elemName; }

        /************************************************************
         * getFullElemName()
         ************************************************************/
        const char *getFullElemName() { return d_fullElemName; }

        /************************************************************
         * getType()
         ************************************************************/
        ConfigDBElem::Type getType() { return d_type; }

        /************************************************************
         * addCallback()
         ************************************************************/
        void addCallback(ConfigDBCallback *cb) { d_callbacks.append(cb); }

        /************************************************************
         * removeCallback()
         ************************************************************/
        void removeCallback(ConfigDBCallback *cb) { d_callbacks.remove(cb); }

        /************************************************************
         * callCallbacks()
         ************************************************************/
        void callCallbacks(const char *name);

        /************************************************************
         * write()
         ************************************************************/
        virtual bool write(FILE *fp, Uint32 levelMask) = 0;

    /****************************************************************
     * Protected Data
     ****************************************************************/
    protected:
        ConfigDBElem               *d_parent;
        char                       *d_elemName;
        char                       *d_fullElemName;
        Type                        d_type;
        Vector<ConfigDBCallback>    d_callbacks;
};

/********************************************************************/
/** \class  ConfigDBSection
 *  \brief  Describes a section within the configuration
 ********************************************************************/
class ConfigDBSection : public ConfigDBElem {

    /****************************************************************
     * Public Methods
     ****************************************************************/
    public:

        /************************************************************
         * ConfigDBSection()
         ************************************************************/
        ConfigDBSection(
                ConfigDBSection        *parent,
                const char             *name);

        /************************************************************
         * ~ConfigDBSection()
         ************************************************************/
        ~ConfigDBSection();

        /************************************************************
         * addChild()
         ************************************************************/
        void addChild(ConfigDBElem *child);

        /************************************************************
         * removeChild()
         ************************************************************/
        void removeChild(ConfigDBElem *child);

        /************************************************************
         * findOrAddItem()
         */
        /** Finds the named node. If it doesn't exist, it is created
         ************************************************************/
        ConfigDBItem *findOrAddItem(const char *name);

        /************************************************************
         * findOrAddSection()
         */
        /** Finds the named node. If it doesn't exist, it is created
         ************************************************************/
        ConfigDBSection *findOrAddSection(const char *name);

        /************************************************************
         * findItem()
         ************************************************************/
        ConfigDBItem *findItem(const char *name);

        /************************************************************
         * findSection()
         ************************************************************/
        ConfigDBSection *findSection(const char *name);

        /************************************************************
         * findElem()
         ************************************************************/
        ConfigDBElem *findElem(const char *name);

        /************************************************************
         * getChildren()
         ************************************************************/
        Vector<ConfigDBElem> *getChildren() { return &d_children; }

        /************************************************************
         * write()
         ************************************************************/
        virtual bool write(FILE *fp, Uint32 levelMask);

    /****************************************************************
     * Private Data
     ****************************************************************/
    private:
        Vector<ConfigDBElem>                  d_children;
};

/********************************************************************/
/** \class ConfigDBItem
 *  \brief 
 ********************************************************************/
class ConfigDBItem : public ConfigDBElem {

    /****************************************************************
     * Public Methods
     ****************************************************************/
    public:

        /************************************************************
         * ConfigDBItem()
         ************************************************************/
        ConfigDBItem(
                ConfigDBSection        *parent,
                const char             *name);

        /************************************************************
         * ~ConfigDBItem()
         ************************************************************/
        ~ConfigDBItem();

        /************************************************************
         * setValue()
         ************************************************************/
        void setValue(
                Uint32              level,
                Tcl_Obj            *value);

        /************************************************************
         * getValue()
         ************************************************************/
        const Tcl_Obj *getValue();

        /************************************************************
         * write()
         ************************************************************/
        virtual bool write(FILE *fp, Uint32 levelMask);

    /****************************************************************
     * Private Data
     ****************************************************************/
    private:

        char                          *d_typeName;
        Tcl_Obj                      **d_values;
        Uint32                         d_numValues;
};

/********************************************************************/
/** \class ConfigDB
 *  \brief Class implements a 
 ********************************************************************/
class ConfigDB {

    /****************************************************************
     * Public Methods
     ****************************************************************/
    public:

        /************************************************************
         * ConfigDB()
         ************************************************************/
        ConfigDB(
                Tcl_Interp *interp,
                Char        *inst_name);

        /************************************************************
         * ~ConfigDB()
         ************************************************************/
        ~ConfigDB(void);

        /************************************************************
         * InstCmd()
         ************************************************************/
        int InstCmd(
                Uint32      objc,
                Tcl_Obj    *const objv[]);


        /************************************************************
         * GetCurrent()
         ************************************************************/
        Tcl_Obj *GetCurrent(const char *itemName);

        /************************************************************
         * SetCurrent()
         ************************************************************/
        void  SetCurrent(const char *itemName, Uint32 level, Tcl_Obj *value);

        /************************************************************
         * SetCurrent()
         ************************************************************/
        void  SetCurrent(const char *itemName, Tcl_Obj *value);

        /************************************************************
         * WriteConfig()
         ************************************************************/
        int WriteConfig(const char *filename, Uint32 levelMask);

        /************************************************************
         * CreateCmd()
         ************************************************************/
        static int CreateCmd(
                ClientData            clientData,
                Tcl_Interp           *interp,
                int                   argc,
                const char           *argv[]);

    /****************************************************************
     * Private Methods
     ****************************************************************/
    private:

        /************************************************************
         * InstCmd()
         ************************************************************/
        static int InstCmd(
                ClientData             clientData,
                Tcl_Interp            *interp,
                int                    objc,
                Tcl_Obj               *const objv[]) {
            return ((ConfigDB *)clientData)->InstCmd(objc, objv);
        }

        /************************************************************
         * ScanHelper()
         ************************************************************/
        static int ScanHelper(
                ClientData        clientData,
                Tcl_Interp       *interp,
                int               objc,
                Tcl_Obj          *const objv[]) {
            return ((ConfigDB *)clientData)->ScanHelper(objc, objv);
        }
        int ScanHelper(
                Uint32      objc,
                Tcl_Obj    *const objv[]);

        /************************************************************
         * ParseLevelSpec
         ************************************************************/
        Int32 ParseLevelSpec(const char *level);

        /************************************************************
         * FindFirstOpt()
         ************************************************************/
        Int32 FindFirstOpt(Int32 start_opt, Int32 optc, Tcl_Obj *const objv[]);

        /************************************************************
         * pushLevel()
         ************************************************************/
        void pushLevel(Int32 level);

        /************************************************************
         * popLevel()
         ************************************************************/
        Int32 popLevel();
        
        /************************************************************
         * topLevel()
         ************************************************************/
        Int32 topLevel();

    /****************************************************************
     * Private Data
     ****************************************************************/
    private:
        String                   instName;
        Tcl_Interp              *parentInterp;
        Tcl_Interp              *slaveInterp;
        Tcl_Command              cmdProc;

        ConfigDBSection         *d_root;
        ConfigDBSection         *d_currSection;
        Stack<ConfigDBSection>   d_parseStack;

        Int32                   *d_levelStack;
        Uint32                   d_currLevel;
        Uint32                   d_maxLevel;

        String                   pathTmp;
};

/*******************************************************************
 * ConfigDB_SetCurrent()
 *
 * Sets the value of an item in the config database... The itemName
 * must be a fully-qualified name.
 *******************************************************************/
void ConfigDB_SetCurrent(
        const char  *itemName,
        const char  *itemValue);

/*******************************************************************
 * ConfigDB_SetCurrentObj()
 *
 * Sets the value of an item in the config database... The itemName
 * must be a fully-qualified name.
 *
 * NOTE :: itemObjVal is caller-allocated and callee-freed.
 *******************************************************************/
void ConfigDB_SetCurrentObj(
        const char  *itemName,
        Tcl_Obj     *itemObjVal);

/*******************************************************************
 * ConfigDB_GetCurrent()
 *
 * Gets the value of an item in the config database... The itemName
 * must be a fully-qualified name.
 *******************************************************************/
const char *ConfigDB_GetCurrent(const char *itemName);

/*******************************************************************
 * ConfigDB_GetCurrentObj()
 *
 * Gets the value of an item in the config database... The itemName
 * must be a fully-qualified name.
 *******************************************************************/
Tcl_Obj *ConfigDB_GetCurrentObj(const char *itemName);

/*******************************************************************
 * ConfigDB_WriteConfig()
 *******************************************************************/
int ConfigDB_WriteConfig(const char *filename, Uint32 levelMask);

#endif /* INCLUDED_CONFIG_DB_H */


