/****************************************************************************
 *                               ConfigDB.cc
 *
 * Author: Matthew Ballance
 * Desc:   
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 ****************************************************************************/
#include "ConfigDB.h"
#include "CmdSwitcher.h"
#include <string.h>
#include <stdlib.h>

static ConfigDB    *Globl_ConfigDB = 0;

/****************************************************************************
 *                        ConfigDBCallback Methods
 ****************************************************************************/

/********************************************************************
 * ConfigDBCallback()
 ********************************************************************/
ConfigDBCallback::ConfigDBCallback(
        ConfigDBElem               *elem,
        Tcl_ObjCmdProc             *func,
        ClientData                  clientData)
{
    d_tclCmd   = 0;
    d_func     = func;
    d_userData = clientData;
    d_interp   = 0;
    d_parent = elem;

    d_parent->addCallback(this);
}

/********************************************************************
 * ConfigDBCallback()
 ********************************************************************/
ConfigDBCallback::ConfigDBCallback(
        ConfigDBElem               *elem,
        Tcl_Interp                 *interp,
        Tcl_Obj                    *cmd)
{

    d_tclCmd = Tcl_DuplicateObj(cmd);
    Tcl_IncrRefCount(d_tclCmd);

    d_func     = TclCmdFunc;
    d_userData = this;
    d_interp   = interp;
    d_parent   = elem;

    d_parent->addCallback(this);
}

/********************************************************************
 * ~ConfigDBCallback()
 ********************************************************************/
ConfigDBCallback::~ConfigDBCallback()
{
    d_parent->removeCallback(this);

    if (d_tclCmd) {
        Tcl_DecrRefCount(d_tclCmd);
    }
}

/********************************************************************
 * Invoke()
 ********************************************************************/
void ConfigDBCallback::Invoke(const char *elem_name)
{
    Tcl_Obj   *objv[4];
    Tcl_Obj   *name = Tcl_NewStringObj(elem_name, -1);

    objv[0] = name;

    Tcl_IncrRefCount(name);

    d_func(d_userData, d_interp, 1, objv);

    Tcl_DecrRefCount(name);
}

/********************************************************************
 * TclCmdFunc()
 ********************************************************************/
int ConfigDBCallback::TclCmdFunc(
        ClientData               clientData,
        Tcl_Interp              *interp,
        int                      objc,
        Tcl_Obj                 *const objv[])
{
    Tcl_Obj *list;
    int        lobjc;
    Tcl_Obj  **lobjv;

    ConfigDBCallback *cb = (ConfigDBCallback *)clientData;

    Tcl_ListObjGetElements(interp, cb->d_tclCmd, &lobjc, &lobjv);

    list = Tcl_NewListObj(lobjc, lobjv);
    Tcl_IncrRefCount(list);

    Tcl_ListObjAppendElement(interp, list, objv[0]);

    Tcl_EvalObjEx(interp, list, TCL_EVAL_GLOBAL);

    Tcl_DecrRefCount(list);

    return TCL_OK;
}

/****************************************************************************
 *                          ConfigDBElem Methods
 ****************************************************************************/

/********************************************************************
 * ConfigDBElem()
 ********************************************************************/
ConfigDBElem::ConfigDBElem(
        ConfigDBElem            *parent,
        const char              *name,
        ConfigDBElem::Type       type)
{
    d_parent = parent;

    d_elemName = new char [strlen(name)+1];
    strcpy(d_elemName, name);

    if (d_parent) {
        const char *parent_name = d_parent->getFullElemName();

        d_fullElemName = new char [strlen(parent_name)+1+strlen(d_elemName)+1];
        if (parent_name && parent_name[0]) {
            strcpy(d_fullElemName, parent_name);
            strcat(d_fullElemName, ".");
            strcat(d_fullElemName, d_elemName);
        } else {
            strcpy(d_fullElemName, d_elemName);
        }
    } else {
        d_fullElemName = new char [strlen(d_elemName)+1];
        strcpy(d_fullElemName, d_elemName);
    }

    d_type = type;
}

/********************************************************************
 * ~ConfigDBElem()
 ********************************************************************/
ConfigDBElem::~ConfigDBElem()
{
    delete [] d_elemName;
}

/********************************************************************
 * callCallbacks()
 ********************************************************************/
void ConfigDBElem::callCallbacks(const char *name)
{
    ConfigDBElem *this_level = this;

    /**** Call all callbacks at this level ***/
    while (this_level) {
        for (Uint32 i=0; i<this_level->d_callbacks.length(); i++) {
            this_level->d_callbacks.idx(i)->Invoke(name);
        }

        this_level = this_level->d_parent;
    }
}

/****************************************************************************
 *                          ConfigDBSection Methods
 ****************************************************************************/

/********************************************************************
 * ConfigDBSection()
 ********************************************************************/
ConfigDBSection::ConfigDBSection(
        ConfigDBSection            *parent,
        const char                 *name) : 
    ConfigDBElem(parent, name, ConfigDBElem::Type_Section)
{
    if (parent) {
        parent->addChild(this);
    }
}

/********************************************************************
 * ~ConfigDBSection()
 ********************************************************************/
ConfigDBSection::~ConfigDBSection()
{
    /**** Delete all sub-elements ****/
    for (Uint32 i=0; i<d_children.length(); i++) {
        delete d_children.idx(i);
    }

    /**** detach from parent ****/
    ((ConfigDBSection *)d_parent)->removeChild(this);
}

/********************************************************************
 * addChild()
 ********************************************************************/
void ConfigDBSection::addChild(ConfigDBElem *child)
{
    d_children.append(child);
}

/********************************************************************
 * removeChild()
 ********************************************************************/
void ConfigDBSection::removeChild(ConfigDBElem *child)
{
    d_children.remove(child);
}

/********************************************************************
 * findOrAddItem()
 ********************************************************************/
ConfigDBItem *ConfigDBSection::findOrAddItem(const char *name)
{
    char *p;

    /**** If there is a '.' here, then recurse... ****/
    if ((p = strchr(name, '.'))) {
        ConfigDBSection *this_elem = 0;
        char            *tmp_name = new char [p-name+1];

        memcpy(tmp_name, name, p-name);
        tmp_name[p-name] = 0;

        for (Uint32 i=0; i<d_children.length(); i++) {
            if (!strcmp(tmp_name, d_children.idx(i)->getElemName())) {
                this_elem = (ConfigDBSection *)d_children.idx(i);
                break;
            }
        }

        if (!this_elem) {
            char *n;

            if ((n = strchr(tmp_name, '.'))) {
                *n = 0;
            }

            this_elem = new ConfigDBSection(this, tmp_name);

            if (n) {
                *n = '.';
            }
        }

        delete [] tmp_name;

        return this_elem->findOrAddItem(p+1);
    } else {
        ConfigDBItem    *item = 0;
        /**** No '.', so we must be the last... ****/

        for (Uint32 i=0; i<d_children.length(); i++) {
            if (!strcmp(d_children.idx(i)->getElemName(), name)) {
                item = (ConfigDBItem *)d_children.idx(i);
                break;
            }
        }

        if (!item) {
            item = new ConfigDBItem(this, name);
        }

        return item;
    }
}

/********************************************************************
 * findOrAddSection()
 ********************************************************************/
ConfigDBSection *ConfigDBSection::findOrAddSection(const char *name)
{
    char *p;

    /**** If there is a '.' here, then recurse... ****/
    if ((p = strchr(name, '.'))) {
        ConfigDBSection *this_elem = 0;
        char            *tmp_name = new char [p-name+1];

        memcpy(tmp_name, name, p-name);
        tmp_name[p-name] = 0;

        for (Uint32 i=0; i<d_children.length(); i++) {
            if (!strcmp(tmp_name, d_children.idx(i)->getElemName())) {
                this_elem = (ConfigDBSection *)d_children.idx(i);
                break;
            }
        }

        if (!this_elem) {
            char *n;

            if ((n = strchr(tmp_name, '.'))) {
                *n = 0;
            }

            this_elem = new ConfigDBSection(this, tmp_name);

            if (n) {
                *n = '.';
            }
        }

        delete [] tmp_name;

        return this_elem->findOrAddSection(p+1);
    } else {
        ConfigDBSection    *sect = 0;
        /**** No '.', so we must be the last... ****/

        for (Uint32 i=0; i<d_children.length(); i++) {
            if (!strcmp(d_children.idx(i)->getElemName(), name)) {
                sect = (ConfigDBSection *)d_children.idx(i);
                break;
            }
        }

        if (!sect) {
            sect = new ConfigDBSection(this, name);
        }

        return sect;
    }
}

/********************************************************************
 * findItem()
 ********************************************************************/
ConfigDBItem *ConfigDBSection::findItem(const char *name)
{
    char *p;

    /**** If there is a '.' here, then recurse... ****/
    if ((p = strchr(name, '.'))) {
        ConfigDBSection *this_elem = 0;
        char            *tmp_name = new char [p-name+1];

        memcpy(tmp_name, name, p-name);
        tmp_name[p-name] = 0;

        for (Uint32 i=0; i<d_children.length(); i++) {
            if (!strcmp(tmp_name, d_children.idx(i)->getElemName())) {
                this_elem = (ConfigDBSection *)d_children.idx(i);
                break;
            }
        }

        delete [] tmp_name;

        if (!this_elem) {
            return 0;
        } else {
            return this_elem->findItem(p+1);
        }
    } else {
        ConfigDBItem    *item = 0;
        /**** No '.', so we must be the last... ****/

        for (Uint32 i=0; i<d_children.length(); i++) {
            if (!strcmp(d_children.idx(i)->getElemName(), name)) {
                item = (ConfigDBItem *)d_children.idx(i);
                break;
            }
        }

        return item;
    }
}

/********************************************************************
 * findSection()
 ********************************************************************/
ConfigDBSection *ConfigDBSection::findSection(const char *name)
{
    char *p;

    /**** If there is a '.' here, then recurse... ****/
    if ((p = strchr(name, '.'))) {
        ConfigDBSection *this_elem = 0;
        char            *tmp_name = new char [p-name+1];

        memcpy(tmp_name, name, p-name);
        tmp_name[p-name] = 0;

        for (Uint32 i=0; i<d_children.length(); i++) {
            if (!strcmp(tmp_name, d_children.idx(i)->getElemName())) {
                this_elem = (ConfigDBSection *)d_children.idx(i);
                break;
            }
        }

        delete [] tmp_name;

        if (!this_elem) {
            return 0;
        } else {
            return this_elem->findSection(p+1);
        }
    } else {
        ConfigDBSection    *sect = 0;
        /**** No '.', so we must be the last... ****/

        for (Uint32 i=0; i<d_children.length(); i++) {
            if (!strcmp(d_children.idx(i)->getElemName(), name)) {
                sect = (ConfigDBSection *)d_children.idx(i);
                break;
            }
        }

        return sect;
    }
}

/********************************************************************
 * findElem()
 ********************************************************************/
ConfigDBElem *ConfigDBSection::findElem(const char *name)
{
    char *p;

    /**** If there is a '.' here, then recurse... ****/
    if ((p = strchr(name, '.'))) {
        ConfigDBSection  *this_sect = 0;
        char             *tmp_name = new char [p-name+1];

        memcpy(tmp_name, name, p-name);
        tmp_name[p-name] = 0;

        for (Uint32 i=0; i<d_children.length(); i++) {
            if (!strcmp(tmp_name, d_children.idx(i)->getElemName()) &&
                (d_children.idx(i)->getType() == ConfigDBElem::Type_Section)) {
                this_sect = (ConfigDBSection *)d_children.idx(i);
                break;
            }
        }

        delete [] tmp_name;

        if (!this_sect) {
            return 0;
        } else {
            return this_sect->findElem(p+1);
        }
    } else {
        ConfigDBElem    *elem = 0;
        /**** No '.', so we must be the last... ****/

        for (Uint32 i=0; i<d_children.length(); i++) {
            if (!strcmp(d_children.idx(i)->getElemName(), name)) {
                elem = d_children.idx(i);
                break;
            }
        }

        return elem;
    }
}

/********************************************************************
 * write()
 */
/** Do a depth-first recursion. Then, look at our child elements...
 ********************************************************************/
bool ConfigDBSection::write(FILE *fp, Uint32 levelMask)
{
    Uint32 i;

    for (i=0; i<d_children.length(); i++) {
        /**** Recurse if this is a section node... ****/
        if (d_children.idx(i)->getType() == ConfigDBElem::Type_Section) {
            d_children.idx(i)->write(fp, levelMask);
        }
    }

    /**** Now, see if any elements in this section will be written ****/
    for (i=0; i<d_children.length(); i++) {
        if (d_children.idx(i)->getType() == ConfigDBElem::Type_Item) {
            if (d_children.idx(i)->write(0, levelMask)) {
                break;
            }
        }
    }

    /**** Looks like one of the elements should be written... ****/
    if (i < d_children.length()) {
        fprintf(fp, "section %s {\n", getFullElemName());
        for (i=0; i<d_children.length(); i++) {
            if (d_children.idx(i)->getType() == ConfigDBElem::Type_Item) {
                d_children.idx(i)->write(fp, levelMask);
            }
        }
        fprintf(fp, "}\n");
    }

    return true;
}

/****************************************************************************
 *                           ConfigDBItem Methods
 ****************************************************************************/

/********************************************************************
 * ConfigDBItem()
 ********************************************************************/
ConfigDBItem::ConfigDBItem(
        ConfigDBSection      *parent,
        const char           *name) : 
    ConfigDBElem(parent, name, ConfigDBElem::Type_Item)
{
    parent->addChild(this);
    d_values    = 0;
    d_numValues = 0;
}

/********************************************************************
 * ~ConfigDBItem()
 ********************************************************************/
ConfigDBItem::~ConfigDBItem()
{
}

/********************************************************************
 * setValue()
 ********************************************************************/
void ConfigDBItem::setValue(
        Uint32                  level,
        Tcl_Obj                *value)
{
    /**** Realloc the values array ****/
    if (level >= d_numValues) {
        Tcl_Obj     **tmp = d_values;

        d_values = new Tcl_Obj *[level+1];

        if (d_numValues) {
            memcpy(d_values, tmp, sizeof(Tcl_Obj *)*d_numValues);
            delete [] tmp;
        }

        /**** Zero out the new storage ****/
        memset(&d_values[d_numValues], 0, 
                sizeof(Tcl_Obj *)*((level+1)-d_numValues));
        d_numValues = level+1;
    }

    /**** If there was already a value here, release our hold on it ****/
    if (d_values[level]) {
        Tcl_DecrRefCount(d_values[level]);
    }

    Tcl_IncrRefCount(value);
    d_values[level] = value;

    callCallbacks(d_fullElemName);
}

/********************************************************************
 * getValue()
 ********************************************************************/
const Tcl_Obj *ConfigDBItem::getValue()
{
    if (d_numValues) {
        Uint32     idx = d_numValues;

        while (idx && !d_values[idx-1]) {
            idx--;
        }

        if (idx) {
            return d_values[idx-1];
        } else {
            return 0;
        }
    } else {
        return 0;
    }
}

/********************************************************************
 * write()
 ********************************************************************/
bool ConfigDBItem::write(FILE *fp, Uint32 levelMask)
{
    bool ret = false;

    /**** Find out if any of the bits in levelMask match a 
     **** present config value
     ****/
    for (Int32 i=d_numValues; i>=0; i--) {
        if ((levelMask & (1 << i)) && d_values[i]) {
            ret = true;

            if (fp) {
                fprintf(fp, "    current %s %s\n",
                        getElemName(), Tcl_GetString(d_values[i]));
                break;
            }
        }
    }
   
    return ret;
}

/*******************************************************************
 * ConfigDB_CreateCmd()
 *
 * Creates a new instance of a ConfigDB
 *
 * config_db ?-global? <name>
 *******************************************************************/
int ConfigDB::CreateCmd(
        ClientData        clientData,
        Tcl_Interp       *interp,
        int               argc,
        const char       *argv[])
{
    ConfigDB    *cDB;
    const char  *instName = 0;
    Int32        setGlob  = 0;

    if (argc < 2) {
        Tcl_AppendResult(interp, "too few args", 0);
        return TCL_ERROR;
    }

    if (argc > 2) {
        if (!strcmp(argv[1], "-global")) {
            if (argc < 3) {
                Tcl_AppendResult(interp, "too few args", 0);
                return TCL_ERROR;
            }
            setGlob = 1;
            instName = argv[2];
        }
    } else {
        instName = argv[1];
    }

    cDB = new ConfigDB(interp, (char *)instName); 
    Tcl_AppendResult(interp, instName, 0);

    if (setGlob) {
        Globl_ConfigDB = cDB;
    }

    return TCL_OK;
}

/**********************************************************
 * SetCurrent()
 **********************************************************/
void ConfigDB::SetCurrent(const char *itemName, Tcl_Obj *value)
{
    SetCurrent(itemName, 0, value);
}

/**********************************************************
 * SetCurrent()
 **********************************************************/
void ConfigDB::SetCurrent(const char *itemName, Uint32 level, Tcl_Obj *value)
{
    Int32                idx    = 0, nidx = 0;
    ConfigDBItem        *target = 0;

    pathTmp = itemName;

    if (!(target = d_root->findOrAddItem(itemName))) {
        fprintf(stderr, "ERROR: findOrAddItem failed\n");
        return;
    }

    target->setValue(level, Tcl_DuplicateObj(value));
}

/**********************************************************
 * GetCurrent()
 **********************************************************/
Tcl_Obj *ConfigDB::GetCurrent(const char *itemName)
{
    Int32        idx;
    Tcl_Obj     *ret = 0;

    pathTmp = itemName;

    if ((idx = pathTmp.strrchr('.')) < 0) {
        fprintf(stderr, "ERROR :: fully-qualified path required\n");
    } else {
        ConfigDBItem *item = d_root->findItem(itemName);

        if (item) {
            ret = (Tcl_Obj *)item->getValue();
        }
    }
    return ret;
}

/**********************************************************
 * WriteConfig()
 **********************************************************/
int ConfigDB::WriteConfig(const char *filename, Uint32 levelMask)
{
    FILE *fp;

    if (!(fp = fopen(filename, "w"))) {
        return TCL_ERROR;
    }

    d_root->write(fp, levelMask);

    fclose(fp);

    return TCL_OK;
}

typedef enum {
    CDC_Clear = 1,
    CDC_Load,
    CDC_Values,
    CDC_Current,
    CDC_SectNames,
    CDC_ValueNames,
    CDC_AddCallback,
    CDC_DelCallback,
    CDC_Delete,
    CDC_Write,
    CDC_NumCmds
} CDBCmd;

static CmdSwitchStruct cdb_cmds[] = {
    {"clear",             CDC_Clear        },
    {"load",              CDC_Load         },
    {"values",            CDC_Values       },
    {"current",           CDC_Current      },
    {"section_names",     CDC_SectNames    },
    {"value_names",       CDC_ValueNames   },
    {"add_callback",      CDC_AddCallback  },
    {"del_callback",      CDC_DelCallback  },
    {"delete",            CDC_Delete       },
    {"write",             CDC_Write        },
    {"",                  0                }
};

/*******************************************************************
 * InstCmd()
 *
 * This function handles the instance commands of a config database
 *******************************************************************/
int ConfigDB::InstCmd(
        Uint32         argc,
        Tcl_Obj       *const objv[])
{
    Int32              cmd;
    Int32              ret, idx;
    ConfigDBSection   *sect;
    Char              *varName; 
    Tcl_Obj           *value;

    if (argc < 2) {
        Tcl_AppendResult(parentInterp, "too few args - expect 2", 0);
        return TCL_ERROR;
    }

    cmd = CmdSwitch(cdb_cmds, Tcl_GetString(objv[1]));

    switch (cmd) {

        /************************************************************
         * Clear
         ************************************************************/
        case CDC_Clear:
            break;

        /************************************************************
         * Load
         *
         *   1    2      3
         * load <file> [options]
         ************************************************************/
        case CDC_Load: {
            if (argc < 3) {
                Tcl_AppendResult(parentInterp, "too few args", 0);
                return TCL_ERROR;
            }

            Int32 level = ParseLevelSpec("user");

            d_currSection = d_root;

            for (Uint32 i=3; i<argc; i++) {
                const char *opt = Tcl_GetString(objv[i]);
                const char *val;

                if (!strcmp(opt, "-level")) {
                    i++;

                    if ((level = ParseLevelSpec(Tcl_GetString(objv[i]))) < 0) {
                        Tcl_AppendResult(parentInterp, "bad level spec ",
                                Tcl_GetString(objv[i]), 0);
                        return TCL_ERROR;
                    }
                } else {
                    Tcl_AppendResult(parentInterp, "unknown option ", opt, 0);
                    return TCL_ERROR;
                }
            }

            pushLevel(level);

            ret = Tcl_EvalFile(slaveInterp, Tcl_GetString(objv[2]));
            if (ret != TCL_OK) {
                fprintf(stderr, "scan got error: %s\n", 
                        Tcl_GetStringResult(slaveInterp));
            }

            popLevel();
            } break;

        /************************************************************
         * Values
         ************************************************************/
        case CDC_Values:
            break;

        /************************************************************
         * Current
         *
         *    1       2      3      ...
         * current <path> <value> [options]
         * or
         * current <path> [options]
         ************************************************************/
        case CDC_Current: {
            int i, last_arg;
            if (argc < 3) {
                Tcl_AppendResult(parentInterp, "too few args", 0);
                return TCL_ERROR;
            }

            pathTmp = Tcl_GetString(objv[2]);
            if ((idx = pathTmp.strrchr('.')) < 0) {
                Tcl_AppendResult(parentInterp, "need fully-qualified path", 0);
                return TCL_ERROR;
            }

            Int32 first_opt = FindFirstOpt(3, argc, objv);

            /*** argc must be >3 and last_arg must be >3 to count as
             *** setting a value
             ***/
            if ((argc > 3) && (first_opt > 3)) {
                Int32 level = ParseLevelSpec("user");

                /*** If first_opt < argc, parse options ***/
                if (first_opt < argc) {
                    for (Int32 i=first_opt; i<argc; i++) {
                        const char *opt = Tcl_GetString(objv[i]);
                        const char *val;
                        if (!strcmp(opt, "-level")) {
                            i++;
                            val = Tcl_GetString(objv[i]);
                            if ((level = ParseLevelSpec(val)) < 0) {
                                Tcl_AppendResult(parentInterp, 
                                        "bad level spec ", val, 0);
                                return TCL_ERROR;
                            }
                        } else {
                            Tcl_AppendResult(parentInterp, "unknown 'current' ",
                                    "option ", opt, 0);
                            return TCL_ERROR;
                        }
                    }
                }
                SetCurrent(Tcl_GetString(objv[2]), 
                        level, Tcl_DuplicateObj(objv[3]));
            } else {
                /*** Probably getting a value ***/

                value = GetCurrent(Tcl_GetString(objv[2]));
                if (value) {
                    Tcl_SetObjResult(parentInterp, Tcl_DuplicateObj(value));
                } else {
                    Tcl_SetObjResult(parentInterp, Tcl_NewStringObj("", -1));
                }
            }
            pathTmp[idx] = 0;

            } break;

        /************************************************************
         * SectNames
         *
         * section_names <section>
         ************************************************************/
        case CDC_SectNames: {
            /**** Find section specified by 'section' ****/
            ConfigDBSection *sect;

            if (argc < 3) {
                Tcl_AppendResult(parentInterp, "too few args - expect 3", 0);
                return TCL_ERROR;
            }

            if (!strcmp(Tcl_GetString(objv[2]), ".")) {
                sect = d_root;
            } else {
                if (!(sect = d_root->findSection(Tcl_GetString(objv[2])))) {
                    Tcl_AppendResult(parentInterp, "no section named \"",
                            Tcl_GetString(objv[2]), "\"", 0);
                    return TCL_ERROR;
                }
            }

            Vector<ConfigDBElem>  *c = sect->getChildren();
            Tcl_Obj               *ret = Tcl_NewListObj(0, 0);

            for (Uint32 e=0; e<c->length(); e++) {

                if (c->idx(e)->getType() != ConfigDBElem::Type_Section) {
                    continue;
                }
                Tcl_ListObjAppendElement(parentInterp, ret,
                        Tcl_NewStringObj(c->idx(e)->getElemName(), -1));
            }

            Tcl_SetObjResult(parentInterp, ret);
         }  break;

        /************************************************************
         * ValueNames
         ************************************************************/
        case CDC_ValueNames: {
            /**** Find section specified by 'section' ****/
            ConfigDBSection *sect;

            if (argc < 3) {
                Tcl_AppendResult(parentInterp, "too few args - expect 3", 0);
                return TCL_ERROR;
            }

            if (!strcmp(Tcl_GetString(objv[2]), ".")) {
                sect = d_root;
            } else {
                if (!(sect = d_root->findSection(Tcl_GetString(objv[2])))) {
                    Tcl_AppendResult(parentInterp, "no section named \"",
                            Tcl_GetString(objv[1]), "\"", 0);
                    return TCL_ERROR;
                }
            }

            Vector<ConfigDBElem>  *c = sect->getChildren();
            Tcl_Obj               *ret = Tcl_NewListObj(0, 0);

            for (Uint32 e=0; e<c->length(); e++) {

                if (c->idx(e)->getType() != ConfigDBElem::Type_Item) {
                    continue;
                }
                Tcl_ListObjAppendElement(parentInterp, ret,
                        Tcl_NewStringObj(c->idx(e)->getElemName(), -1));
            }

            Tcl_SetObjResult(parentInterp, ret);

          } break;

        /************************************************************
         * AddCallback
         *
         * add_callback <name> <proc>
         ************************************************************/
        case CDC_AddCallback: {
            if (argc < 4) {
                Tcl_AppendResult(parentInterp, "too few args - expect 4", 0);
                return TCL_ERROR;
            }

            ConfigDBElem     *elem;

            if (!(elem = d_root->findElem(Tcl_GetString(objv[2])))) {
                Tcl_AppendResult(parentInterp, "no config elem ", 
                        Tcl_GetString(objv[2]), 0);
                return TCL_ERROR;
            }

            ConfigDBCallback *cb = new ConfigDBCallback(elem, 
                    parentInterp, objv[3]);

            char buf[20];
            sprintf(buf, "0x%08x", (Uint32)cb);

            Tcl_SetObjResult(parentInterp, Tcl_NewStringObj(buf, -1));
        } break;

        /************************************************************
         * DelCallback
         ************************************************************/
        case CDC_DelCallback: {
            if (argc < 3) {
                Tcl_AppendResult(parentInterp, "too few args - expect 3", 0);
                return TCL_ERROR;
            }

            ConfigDBCallback *cb = 
                (ConfigDBCallback *)strtoul(Tcl_GetString(objv[2]), 0, 0);

            delete cb;
        } break;

        /************************************************************
         * Delete
         * 
         * delete <name>
         ************************************************************/
        case CDC_Delete: {
            if (argc < 3) {
                Tcl_AppendResult(parentInterp, "too few args - expect 3", 0);
                return TCL_ERROR;
            }

            ConfigDBElem *elem = d_root->findElem(Tcl_GetString(objv[2]));

            if (!elem) {
                Tcl_AppendResult(parentInterp, "no elem \"", 
                        Tcl_GetString(objv[2]), "\"", 0);
                return TCL_ERROR;
            }

            delete elem;
        } break;
                              
        /************************************************************
         * write
         *
         * write <file>
         ************************************************************/
        case CDC_Write: {
            if (argc < 3) {
                Tcl_AppendResult(parentInterp, "too few args- expect 3", 0);
                return TCL_ERROR;
            }

            const  char *filename = Tcl_GetString(objv[2]);
            Uint32 levelMask = (1 << ParseLevelSpec("user"));

            /*** TODO: parse extra options... ****/

            return WriteConfig(filename, levelMask);
        } break;

        /************************************************************
         * default
         ************************************************************/
        default:
            Tcl_AppendResult(parentInterp, "Unknown sub-cmd ", 
                    Tcl_GetString(objv[0]), 0);
            return TCL_ERROR;
            break;
    }

    return TCL_OK;
}

typedef enum {
    CDH_Section  = 1,
    CDH_Current,
    CDH_Values,
    CDH_Comment,
    CDH_NumCmds
} CDH_Cmds;


static CmdSwitchStruct   cdh_cmds[] = {
    {"section",          CDH_Section     },
    {"current",          CDH_Current     },
    {"values",           CDH_Values      },
    {"#",                CDH_Comment     },
    {"",                 0               }
};

/*******************************************************************
 * ScanHelper()
 *
 * Used by the config-scanning routines...
 *******************************************************************/
int ConfigDB::ScanHelper(
        Uint32         argc,
        Tcl_Obj       *const objv[])
{
    Int32           cmd = CmdSwitch(cdh_cmds, Tcl_GetString(objv[0]));
    Uint32          i;
    Tcl_Obj        *sectName, *extendSpec;
    Tcl_Obj        *sectData;
    Int32           ret;
    ConfigDBItem   *item = 0;
    bool            isSet;
    Char      *varName;

    switch (cmd) {

        /************************************************************
         * Section
         ************************************************************/
        case CDH_Section:
            /****    section <SectName> { } 
             **** or
             ****    section <SectName> : <ExtendSpec> { }
             ****/
            if (argc > 2) {
                sectName = objv[1];
                sectData = objv[2];

                if (d_currSection) {
                    d_parseStack.push(d_currSection);
                    d_currSection = d_currSection->findOrAddSection(
                            Tcl_GetString(sectName));
                } else {
                    fprintf(stderr, "ERROR :: internal error\n");
                    return TCL_ERROR;
                }

                /**** See what form we have... ****/
                if (argc > 4) {
                    /**** Extended section ****/
                    if (!String::equal(Tcl_GetString(objv[2]), ":")) {
                        Tcl_AppendResult(slaveInterp, 
                                "bad extended section: expecting ':'", 0);
                        return TCL_ERROR;
                    }

                    sectData = objv[3];

                    /**** Load the base config from the specified sections
                     ****/
                } 

                ret = Tcl_EvalObjEx(slaveInterp, sectData, TCL_EVAL_GLOBAL);

                if (ret != TCL_OK) {
                    return ret;
                } else {
                    /**** Pop this tree-node off the stack... ****/
                    if (!(d_currSection = d_parseStack.pop())) {
                        fprintf(stderr, "ERROR :: internal error (pop)\n");
                        return TCL_ERROR;
                    }
                }
            }
            break;

        /************************************************************
         **** current <var> ?<val>? 
         ************************************************************/
        case CDH_Current: {
            if (argc < 2) {
                Tcl_AppendResult(slaveInterp, "too few args", 0);
                return TCL_ERROR;
            }

            if (!d_currSection) {
                fprintf(stderr, "internal error\n");
                return TCL_ERROR;
            }

            varName = Tcl_GetString(objv[1]);

            if (argc > 2) {
                isSet = true;
            } else {
                isSet = false;
            }


            if (strchr(varName, '.')) {
                /**** fully-qualified path ****/
                if (!(item = d_root->findItem(varName)) && isSet) {
                    item = d_root->findOrAddItem(varName);
                }
            } else {
                /**** local path... ****/
                if (!(item = d_currSection->findItem(varName)) && isSet) {
                    item = d_currSection->findOrAddItem(varName);
                }
            }

            Int32 level = topLevel();

            if (isSet) {
                item->setValue(level, objv[2]);
            } else {
                /**** Get value ****/
                if (item) {
                    Tcl_SetObjResult(slaveInterp, 
                            (Tcl_Obj *)item->getValue());
                } else {
                    Tcl_AppendResult(slaveInterp,
                            "unset var: ", Tcl_GetString(objv[1]), 0);
                }
            }
            } break;

        /************************************************************
         * Values
         ************************************************************/
        case CDH_Values:
            break;

        case CDH_Comment:
            fprintf(stderr, "NOTE :: Got comment\n");
            break;

        default:
            return TCL_ERROR;
    }

    return TCL_OK;
}

/********************************************************************
 * ParseLevelSpec()
 *
 * Parses a string specifying a config level. Returns the level,
 * or -1 in case of error
 ********************************************************************/
Int32 ConfigDB::ParseLevelSpec(const char *level)
{
    static struct {
        const char           *str;
        Int32                 level;
    } level_def[] = {
        { "system",               0  },
        { "system1",              1  },
        { "system2",              2  },
        { "system3",              3  },
        { "user",                 4  },
        { "user1",                5  },
        { "user2",                6  },
        { "user3",                7  },
        { "directory",            8  },
        { "directory1",           9  },
        { "directory2",           10 },
        { "directory3",           11 },
        { "project",              12 },
        { "project1",             13 },
        { "project2",             14 },
        { "project3",             15 },
        { (const char *)0,        -1 }
    };

    int idx=0;

    while (level_def[idx].str) {
        if (!strcmp(level_def[idx].str, level)) {
            return level_def[idx].level;
        }
        idx++;
    }

    /**** Okay, didn't find a predefined string... Expect an integer
     **** greater than 15 and less than 32
     ****/
    Int32 intLevel = strtol(level, 0, 0);

    if (intLevel < 32 && intLevel > 15) {
        return intLevel;
    } else {
        return -1;
    }
}

/********************************************************************
 * FindFirstOpt()
 ********************************************************************/
Int32 ConfigDB::FindFirstOpt(Int32 start_opt, Int32 optc, Tcl_Obj *const objv[])
{
    for (Int32 i=start_opt; i<optc; i++) {
        const char *opt = Tcl_GetString(objv[i]);

        if (opt[0] == '-') {
            return i;
        }
    }

    return optc;
}

/********************************************************************
 * pushLevel()
 ********************************************************************/
void ConfigDB::pushLevel(Int32 level)
{
    d_levelStack[d_currLevel++] = level;
}

/********************************************************************
 * popLevel()
 ********************************************************************/
Int32 ConfigDB::popLevel()
{
    if (d_currLevel) {
        return d_levelStack[--d_currLevel];
    } else {
        return 0;
    }
}

/********************************************************************
 * topLevel
 ********************************************************************/
Int32 ConfigDB::topLevel()
{
    if (d_currLevel) {
        return d_levelStack[d_currLevel-1];
    } else {
        return 0;
    }
}

/********************************************************************
 * ConfigDB()
 ********************************************************************/
ConfigDB::ConfigDB(
        Tcl_Interp        *interp,
        Char              *inst_name) : 
    instName(inst_name), parentInterp(interp), pathTmp("")
{
    Uint32 i = 0;

    slaveInterp = Tcl_CreateInterp();
    if (Tcl_Init(slaveInterp) != TCL_OK) {
        fprintf(stderr, "ERROR :: Tcl_Init()\n");
    }
    for (i=0; cdh_cmds[i].cmdString[0]; i++) {
        Tcl_CreateObjCommand(slaveInterp, cdh_cmds[i].cmdString,
                &ConfigDB::ScanHelper, this, 0);
    }

    cmdProc = Tcl_CreateObjCommand(parentInterp, inst_name, 
            &ConfigDB::InstCmd, this, 0);

    d_root = new ConfigDBSection(0, "");

    d_levelStack = new Int32[256];
    d_maxLevel   = 256;
    d_currLevel  = 0;
}

/*******************************************************************
 * ~ConfigDB()
 *******************************************************************/
ConfigDB::~ConfigDB(void)
{
    if (cmdProc) {
        Tcl_DeleteCommandFromToken(parentInterp, cmdProc);
    }
}

/*******************************************************************
 * ConfigDB_Init()
 *******************************************************************/
extern "C" int ConfigDB_Init(Tcl_Interp *interp)
{
    Tcl_CreateCommand(interp, "config_db", 
            (Tcl_CmdProc *)ConfigDB::CreateCmd, 0, 0);
    return TCL_OK;
}

/*******************************************************************
 * ConfigDB_SetCurrent()
 *
 * Sets the value of an item in the config database... The itemName
 * must be a fully-qualified name.
 *******************************************************************/
void ConfigDB_SetCurrent(
        const char  *itemName,
        const char  *itemValue)
{
    ConfigDB   *cdb = Globl_ConfigDB;

    cdb->SetCurrent(itemName, Tcl_NewStringObj(itemValue, -1));
}

/*******************************************************************
 * ConfigDB_GetCurrent()
 *
 * Gets the value of an item in the config database... The itemName
 * must be a fully-qualified name.
 *******************************************************************/
const char *ConfigDB_GetCurrent(const char *itemName)
{
    ConfigDB   *cdb = Globl_ConfigDB;
    Tcl_Obj    *cur;

    cur = cdb->GetCurrent(itemName);

    if (cur) {
        return Tcl_GetString(cur);
    } else {
        return 0;
    }
}

/*******************************************************************
 * ConfigDB_GetCurrentObj()
 *
 * Gets the value of an item in the config database... The itemName
 * must be a fully-qualified name.
 *******************************************************************/
Tcl_Obj *ConfigDB_GetCurrentObj(const char *itemName)
{
    ConfigDB   *cdb = Globl_ConfigDB;
    return cdb->GetCurrent(itemName);
}

/*******************************************************************
 * ConfigDB_SetCurrentObj()
 *
 * Sets the value of an item in the config database... The itemName
 * must be a fully-qualified name.
 *
 * NOTE :: itemObjVal is caller-allocated and callee-freed.
 *******************************************************************/
void ConfigDB_SetCurrentObj(const char *itemName, Tcl_Obj *itemObjVal)
{
    ConfigDB   *cdb = Globl_ConfigDB;
    cdb->SetCurrent(itemName, itemObjVal); 
}

/*******************************************************************
 * ConfigDB_WriteConfig()
 *******************************************************************/
int ConfigDB_WriteConfig(const char *filename, Uint32 levelMask)
{
    ConfigDB    *cdb = Globl_ConfigDB;

    return cdb->WriteConfig(filename, levelMask);
}


