/*****************************************************************************
 *                             CallbackMgr.cc 
 *
 * Author: Matthew Ballance
 * Desc:   Implements a callback interface for C and TCL.
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 *****************************************************************************/
#include "CallbackMgr.h"
#include "Callback.h"
#include "CallbackType.h"
#include "CallbackLeaf.h"
#include "CmdSwitcher.h"

#include <string.h>
#include <strings.h>
#include <unistd.h>
#include <stdlib.h>

#undef DEBUG_CALLBACK_MGR
#define FP stderr

#ifdef DEBUG_CALLBACK_MGR
#define DBG_MSG(x) fprintf x
#else
#define DBG_MSG(x) 
#endif

CallbackMgr *CallbackMgr::Globl_CbMgr = 0;

typedef enum {
    CBM_AddType = 1,
    CBM_AddCb,
    CBM_GetTypes,
    CBM_Invoke,
    CBM_Disable,
    CBM_Enable,
    CBM_Destroy,
    CBM_NumCmds
} CBMCmds;

static CmdSwitchStruct cbmCmds[] = {
    {"add_type",         CBM_AddType        },
    {"add",              CBM_AddCb          },
    {"get_types",        CBM_GetTypes       },
    {"invoke",           CBM_Invoke         },
    {"disable",          CBM_Disable        },
    {"enable",           CBM_Enable         },
    {"destroy",          CBM_Destroy        },
    {"",                 0                  }
};

typedef struct PrvTclCbStruct {
    PrvTclCbStruct(Tcl_Obj *const cmd) {
        cmdObj = Tcl_DuplicateObj(cmd);
    }

    Tcl_Obj   *cmdObj;

    voidPtr    cbId;
    String     cmdStr;
} PrvTclCbStruct, *PrvTclCbPtr;

/************************************************************************
 * CallbackMgr_TclCbProc()
 ************************************************************************/
static int CallbackMgr_TclCbProc(
        ClientData        clientData,
        Tcl_Interp       *interp,
        int               argc,
        Tcl_Obj          *const objv[])
{
    Uint32           len = 0, i;
    PrvTclCbStruct   *cbS = (PrvTclCbStruct *)clientData;
    Tcl_Obj          *cmd_list;
    int               ret;

    cmd_list = Tcl_NewListObj(0, 0);
    Tcl_ListObjAppendList(interp, cmd_list, cbS->cmdObj);

    for (i=0; i<argc; i++) {
        Tcl_ListObjAppendElement(interp, cmd_list, objv[i]);
    }

    ret = Tcl_EvalObjEx(interp, cmd_list, TCL_EVAL_GLOBAL);
    if (ret != TCL_OK) {
        fprintf(stderr, "ERROR :: While invoking cb \"%s\"\n",
                Tcl_GetStringResult(interp));
    }

    return ret;
}


/************************************************************************
 * CallbackMgr_CmdProc()
 ************************************************************************/
static int CallbackMgr_CmdProc(
        ClientData        clientData,
        Tcl_Interp       *interp,
        int               argc,
        Tcl_Obj          *const objv[])
{
    CallbackMgr      *cbm = (CallbackMgr *)clientData;
    Int32             cmd, i, ii, len;
    Vector<String>   *result;
    Char             *cmdStr = 0, *instStr = 0, *typeStr; 
    voidPtr           cbId;
    Char              buf[16];
    PrvTclCbStruct   *cbS = (PrvTclCbStruct *)clientData;
  

    if (argc < 2) {
        Tcl_AppendResult(interp, "too few args. expecting: ", 0);
        CmdSwitch_AppendAll(interp, cbmCmds);
        return TCL_ERROR;
    }

    cmd = CmdSwitch(cbmCmds, Tcl_GetString(objv[1]));

    switch (cmd) {
        case CBM_AddType:
            if (argc < 3) {
                Tcl_AppendResult(interp, "too few args", 0);
                return TCL_ERROR;
            }
            cbm->AddCbType(Tcl_GetString(objv[2]));
            break;

        case CBM_AddCb:
            if (argc < 4) {
                Tcl_AppendResult(interp, "too few args", 0);
                return TCL_ERROR;
            }

            typeStr = Tcl_GetString(objv[2]);
            instStr = Tcl_GetString(objv[3]);

            cbS = new PrvTclCbStruct(objv[4]);

            cbm->AddCb(typeStr, instStr, CallbackMgr_TclCbProc, cbS, &cbId);
            cbS->cbId = cbId;

            sprintf(buf, "0x%x", cbId);
            Tcl_AppendResult(interp, buf, 0);
            break;

        case CBM_GetTypes:
            if (argc < 3) {
                Tcl_AppendResult(interp, "too few args", 0);
                return TCL_ERROR;
            }
            cbm->GetCbTypes(Tcl_GetString(objv[2]), &result);

            if (result) {
                Tcl_Obj *list = Tcl_NewListObj(0, 0);

                for (i=0; i<result->length(); i++) {
                    Tcl_ListObjAppendElement(interp, list, 
                            Tcl_NewStringObj(result->idx(i)->value(), -1));
                }

                Tcl_SetObjResult(interp, list);
            }
            break;

        /**** $cbm invoke <type> [inst|"null"] <cb_args>
         ****/
        case CBM_Invoke:
            if (argc < 4) {
                Tcl_AppendResult(interp, "too few args", 0);
                return TCL_ERROR;
            }

            typeStr = Tcl_GetString(objv[2]);
            instStr = Tcl_GetString(objv[3]);

            cbm->Invoke(typeStr, instStr, 
                    (argc > 4)?argc-4:0, (argc > 4)?&objv[4]:0);
            break;

        case CBM_Disable:
            if (argc < 4) {
                Tcl_AppendResult(interp, "too few args", 0);
                return TCL_ERROR;
            }
            
            cbm->Disable(Tcl_GetString(objv[2]), Tcl_GetString(objv[3]));
            break;

        case CBM_Enable:
            if (argc < 4) {
                Tcl_AppendResult(interp, "too few args", 0);
                return TCL_ERROR;
            }
            
            cbm->Enable(Tcl_GetString(objv[2]), Tcl_GetString(objv[3]));
            break;

        case CBM_Destroy:
            cbId = (void *)strtoul(Tcl_GetString(objv[2]), 0, 0);
            CallbackMgr_DeleteCb(cbId);
            break;

        default:
            Tcl_AppendResult(interp, 
                    "unknown sub-command: ", Tcl_GetString(objv[1]), 
                    " possible: ", 0);
            CmdSwitch_AppendAll(interp, cbmCmds);
            return TCL_ERROR;
            break;
    }

    return TCL_OK;
}

static const char *prv_cbTypes[] = {
    "App.Construct",
    "App.Register",
    ""
};

/************************************************************************
 * CallbackMgr_CreateCallbacks()
 ************************************************************************/
static void CallbackMgr_CreateCallbacks(void)
{
    CallbackMgr *cbm = CallbackMgr::GetCbMgr();
    Uint32       idx = 0;

    while (prv_cbTypes[idx][0]) {
        cbm->AddCbType(prv_cbTypes[idx]);
        idx++;
    }
}

/************************************************************************
 * CallbackMgr_Init()
 ************************************************************************/
int CallbackMgr_Init(Tcl_Interp *interp)
{
    CallbackMgr::Init(interp);
    CallbackMgr_CreateCallbacks();

    return TCL_OK;
}


/************************************************************************
 * GetCbMgr()
 ************************************************************************/
CallbackMgr *CallbackMgr::GetCbMgr()
{
    if (!Globl_CbMgr) {
        fprintf(stderr, "FATAL ERROR: Cannot initialize CallbackMgr\n");
        exit(1);
    }

    return Globl_CbMgr;
}
/************************************************************************
 * CallbackMgr_AddCb()
 ************************************************************************/
int CallbackMgr_AddCb(
        const char      *cbTypeName,
        const char      *cbInstName,
        Tcl_ObjCmdProc   cbFunc,
        ClientData       cbUserData,
        ClientData      *cbId
        )
{
    Tcl_HashEntry    *hashEntry;

    CallbackMgr::GetCbMgr()->AddCb(cbTypeName, cbInstName, cbFunc,
            cbUserData, cbId);

    return TCL_OK;
}

/************************************************************************
 * CallbackMgr_AddCbType()
 ************************************************************************/
int CallbackMgr_AddCbType(
        const char   *cbTypeName
        )
{
    CallbackMgr::GetCbMgr()->AddCbType(cbTypeName);

    return TCL_OK;
}

/************************************************************************
 * GetCbList()
 ************************************************************************/
CallbackList *CallbackMgr::GetCbList(
        const char *cbTypeName, const char *cbInstName)
{
    Tcl_HashEntry     *hashEntry;
    CallbackMgr       *mgr = CallbackMgr::GetCbMgr();
    CallbackType      *cbType;

    if (!(hashEntry = Tcl_FindHashEntry(&mgr->cbTypeHash, cbTypeName))) {
        int tmpi;
        cbType = new CallbackType(cbTypeName, mgr);

        hashEntry = Tcl_CreateHashEntry(&mgr->cbTypeHash, cbTypeName, &tmpi);
        Tcl_SetHashValue(hashEntry, cbType);
    }

    cbType = (CallbackType *)Tcl_GetHashValue(hashEntry);

    return cbType->GetCbList(cbInstName);
}

/************************************************************************
 * CallbackMgr_DeleteCb()
 ************************************************************************/
int CallbackMgr_DeleteCb(ClientData cbId)
{
    Callback *cb = (Callback *)cbId;
    delete cb;
    return TCL_OK;
}

/************************************************************************
 * CallbackMgr_Invoke()
 ************************************************************************/
int CallbackMgr_Invoke(
        const char  *cbTypeName,
        const char  *cbInstName,
        Uint32       objc,
        Tcl_Obj     *const objv[])
{
    CallbackMgr *cbm = CallbackMgr::GetCbMgr();
    return cbm->Invoke(cbTypeName, cbInstName, objc, objv);
}

/************************************************************************
 * CallbackMgr_Disable()
 ************************************************************************/
int CallbackMgr_Disable(
        const char *cbTypeName,
        const char *cbInstName)
{
    CallbackMgr::GetCbMgr()->Disable(cbTypeName, cbInstName);
    return TCL_OK;
}

/************************************************************************
 * CallbackMgr_Enable()
 ************************************************************************/
int CallbackMgr_Enable(
        const char *cbTypeName,
        const char *cbInstName)
{
    CallbackMgr::GetCbMgr()->Enable(cbTypeName, cbInstName);
    return TCL_OK;
}

/************************************************************************
 * CallbackMgr(interp)
 ************************************************************************/
CallbackMgr::CallbackMgr(
        Tcl_Interp        *interp
        )
{
    this->interp = interp;
    retVect = new Vector<String>();
    strVect = new Vector<String>();
    numCBs  = 0;
    Tcl_InitHashTable(&cbTypeHash, TCL_STRING_KEYS);
}


/************************************************************************
 * AddCbType()
 ************************************************************************/
int CallbackMgr::AddCbType(const char *cbTypeName)
{
    const char      *ptr = 0, *nptr = 0; 
    char             tmp[1024];
    CallbackLeaf    *cbL;
    CallbackType    *cbType;
    Tcl_HashEntry   *hashEntry;
    /**** Scan through each element of the callback path.
     **** - if ${path-to-here}.* doesn't exist, add it...
     ****/

    /**** Now, do this for each following element in the path...
     ****/

    ptr = cbTypeName;
    while ((ptr && (nptr = strchr(ptr, '.'))) || ptr) { 

        /**** Get the full path-to-here... 
         **** 
         **** if (ptr == cbTypeName) { tmp = "*" } else {
         ****     if (nptr) {
         ****
         ****
         **** We will add a ${tmp}.* entry to the hash if it
         **** doesn't exist. This ${tmp}.* entry will contain
         **** 
         ****/
        if (ptr == cbTypeName) {
            strcpy(tmp, "*");
        } else {
            memcpy(tmp, cbTypeName, ptr-cbTypeName);
            tmp[ptr-cbTypeName] = 0;

            if (*(ptr-1) == '.') {
                strcat(tmp, "*");
            } else {
                strcat(tmp, ".*");
            }
        }

        DBG_MSG((FP, "Checking entry \"%s\"\n", tmp));

        if (!(hashEntry = Tcl_FindHashEntry(&cbTypeHash, tmp))) {
            int tmpi;
            cbL = new CallbackLeaf();
            DBG_MSG((FP, "\tAdding new has entry for \"%s\"\n", tmp));
            hashEntry = Tcl_CreateHashEntry(&cbTypeHash, tmp, &tmpi);
            Tcl_SetHashValue(hashEntry, cbL);
        } else {
            cbL = (CallbackLeaf *)Tcl_GetHashValue(hashEntry);
        }

        /**** Okay, now we add (ptr -> nptr) to the CallbackLeaf
         ****/
        if (nptr) {
            memcpy(tmp, ptr, nptr-ptr);
            tmp[nptr-ptr] = 0;
        } else {
            strcpy(tmp, ptr);
        }

        DBG_MSG((FP, "\tAdding \"%s\" to entry\n", tmp));

        cbL->addName(tmp);

        if (nptr) {
            ptr = nptr+1;
        } else {
            ptr = 0;
        }
    }

    /**** Okay, now we add the actual callback spec... ****/
    if (!(hashEntry = Tcl_FindHashEntry(&cbTypeHash, cbTypeName))) {
        int tmpi;

        cbType = new CallbackType(cbTypeName, this);
        hashEntry = Tcl_CreateHashEntry(&cbTypeHash, cbTypeName, &tmpi);
        Tcl_SetHashValue(hashEntry, cbType);
    } 

    return TCL_OK;
}

/************************************************************************
 * GetCbTypes()
 ************************************************************************/
int CallbackMgr::GetCbTypes(
        const char          *cbTypeSpec,
        Vector<String>     **result)
{
    Uint32 i;
    Tcl_HashEntry   *hashEntry;
    CallbackLeaf    *cbL;

    *result = 0;

    for (i=0; i<retVect->length(); i++) {
        strVect->append(retVect->idx(i));
    }
    retVect->empty();    

    if (cbTypeSpec[strlen(cbTypeSpec)-1] != '*') {
        return TCL_ERROR;
    }

    if (!(hashEntry = Tcl_FindHashEntry(&cbTypeHash, cbTypeSpec))) {
        return TCL_ERROR;
    }

    cbL = (CallbackLeaf *)Tcl_GetHashValue(hashEntry); 

    *result = cbL->elemList;
}

/************************************************************************
 * AddCb()
 ************************************************************************/
int CallbackMgr::AddCb(
        const char      *cbTypeName,
        const char      *cbInstName,
        Tcl_ObjCmdProc  *cbFunc,
        ClientData       cbClientData,
        ClientData      *cbId)
{
    Tcl_HashEntry    *hashEntry;
    CallbackType     *cbType;
    Callback         *cb;

    /**** Okay, first find the cbEntry ****/

    if (!(hashEntry = Tcl_FindHashEntry(&cbTypeHash, cbTypeName))) {
        /**** Uh-oh... We've got an error... ****/
        return TCL_ERROR;
    }

    cb = new Callback(cbFunc, cbClientData);
    cbType = (CallbackType *)Tcl_GetHashValue(hashEntry);
    cbType->AddCb(cbInstName, cb);

    *cbId = cb;

    return TCL_OK;
}

/************************************************************************
 * Invoke()
 ************************************************************************/
int CallbackMgr::Invoke(
        const char  *cbTypeName,
        const char  *cbInstName,
        Uint32       argc,
        Tcl_Obj     *const objv[])
{
    Tcl_HashEntry     *hashEntry;
    CallbackType      *cbType;

    /**** Okay, first lookup the cbTypeName ****/

    if (!(hashEntry = Tcl_FindHashEntry(&cbTypeHash, cbTypeName))) {
        return TCL_ERROR;
    }

    cbType = (CallbackType *)Tcl_GetHashValue(hashEntry);

    if (!cbType) {
        return TCL_ERROR;
    } else {
        cbType->Invoke(cbInstName, argc, objv);
    }

    return TCL_OK;
}

/************************************************************************
 * Disable()
 ************************************************************************/
int CallbackMgr::Disable(const char *cbTypeName, const char *cbInstName)
{
    Tcl_HashEntry     *hashEntry;
    CallbackType      *cbType;

    /**** Okay, first lookup the cbTypeName ****/

    if (!(hashEntry = Tcl_FindHashEntry(&cbTypeHash, cbTypeName))) {
        return TCL_ERROR;
    }

    cbType = (CallbackType *)Tcl_GetHashValue(hashEntry);

    cbType->Disable(cbInstName);

    return TCL_OK;
}

/************************************************************************
 * Init()
 ************************************************************************/
void CallbackMgr::Init(Tcl_Interp *interp)
{
    char *env = (char *)malloc(64);

    Globl_CbMgr = new CallbackMgr(interp);

    sprintf(env, "IVI_PRV_CB_MGR=0x%x", Globl_CbMgr);
    putenv(env);

    Tcl_CreateObjCommand(interp, "callback", CallbackMgr_CmdProc,
            Globl_CbMgr, NULL);
}

/************************************************************************
 * Enable()
 ************************************************************************/
int CallbackMgr::Enable(const char *cbTypeName, const char *cbInstName)
{
    Tcl_HashEntry     *hashEntry;
    CallbackType      *cbType;

    /**** Okay, first lookup the cbTypeName ****/

    if (!(hashEntry = Tcl_FindHashEntry(&cbTypeHash, cbTypeName))) {
        return TCL_ERROR;
    }

    cbType = (CallbackType *)Tcl_GetHashValue(hashEntry);

    cbType->Enable(cbInstName);

    return TCL_OK;
}
