/****************************************************************************
 *                          BitVectorABVal.cc
 *
 * Author: Matthew Ballance
 * Desc:   Implements a bitvector based on an 8-bit DFIOVal structure
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 ****************************************************************************/
#include "BitVectorABVal.h"
#include "DFIOValChg.h"
#include <string.h>
#include <stdio.h>


/********************************************************************
 * BitVectorABVal()
 ********************************************************************/
BitVectorABVal::BitVectorABVal(
        const DFIOVal8         *val,
        Uint32                 len)
{
    Uint32 valLen;

    d_bitLen = len;
    valLen = (len/8);
    if (len%8) {
        valLen++;
    }

    d_value = new DFIOVal8[valLen];
    d_valueLen = d_valueMaxLen = valLen;

    memcpy(d_value, val, sizeof(DFIOVal8)*valLen);
}

/********************************************************************
 * BitVectorABVal()
 ********************************************************************/
BitVectorABVal::BitVectorABVal(
        const DFIOVal32           *val,
        Uint32                     bitLen)
{
    Uint32 valLen = (bitLen/8);
    Uint32 val32Len = (bitLen/32);

    if (bitLen%8) {
        valLen++;
    }

    if (bitLen%32) {
        val32Len++;
    }

    d_value = new DFIOVal8[valLen];
    d_valueLen = d_valueMaxLen = valLen;

    /**** Want to take 4-change-wide data and pack it into 1-wide data
     ****/
    for (Uint32 i=0; i<val32Len; i++) {
        for (Uint32 x=0; x<4; x++) {
            d_value[(4*i)+x].aval = ((val[i].aval >> (8*x)) & 0xFF);
            d_value[(4*i)+x].bval = ((val[i].bval >> (8*x)) & 0xFF);
        }
    }
}

/****************************************************************************
 *                           Formatting Methods
 ****************************************************************************/

/********************************************************************
 * fmtAddAttributes()
 ********************************************************************/
void BitVectorABVal::fmtAddAttributes(
        char                    radixChar,
        Uint32                  in_len,
        String                 &out,
        Uint32                 &idx,
        Uint32                  flags)
{
    bool         isSigned = (flags & RadixFlags_Signed)?true:false;
    bool         isSized  = (flags & RadixFlags_Sized )?true:false;
    bool         isTyped  = (flags & RadixFlags_Typed )?true:false;
   
    if (isSized) {
        Uint32 tLen = in_len, ii=0;
        char   tmpb[16];

        do {
            tmpb[ii++] = "0123456789"[tLen%10];
            tLen /= 10;
        } while (tLen);

        for (Uint32 i=0; i<ii; i++) {
            out[idx++] = tmpb[ii-i-1];
        }
    }

    if (isSized || isTyped) { 
        out[idx++] = '\'';
    }

    if (isTyped) {
        out[idx++] = radixChar;
    }
}

/********************************************************************
 * fmtOct_WriteDigits()
 ********************************************************************/
void BitVectorABVal::fmtOct_WriteDigits(
        Uint32             value,
        Uint32             numDigits,
        String            &out,
        Uint32            &idx)
{
    for (Uint32 i=0; i<numDigits; i++) {
        out[idx++] = "01234567"[value & 3];
        value >>= 3;
    }
}

/********************************************************************
 * fmtBin()
 ********************************************************************/
void BitVectorABVal::fmtBin(
        const DFIOVal           *in,
        Uint32                   in_len,
        String                  &out,
        Uint32                   flags,
        Uint32                  &infoFlags)
{
    bool         isSigned = (flags & RadixFlags_Signed)?true:false;
    bool         isSized  = (flags & RadixFlags_Sized )?true:false;
    bool         isTyped  = (flags & RadixFlags_Typed )?true:false;
    Uint32       idx      = 0;
    Uint32       nV[4]    = {0, 0, 0, 0};

    out.chkSize(in_len+8);

    /**** If the output should be 'sized', then prepend the size 
     **** of the input vector
     ****/
    if (isSized) {
        Uint32 tLen = in_len, ii=0;
        char   tmpb[16];

        do {
            tmpb[ii++] = "0123456789"[tLen%10];
            tLen /= 10;
        } while (tLen);

        for (Uint32 i=0; i<ii; i++) {
            out[idx++] = tmpb[ii-i-1];
        }
    }

    if (isSized || isTyped) { out[idx++] = '\''; }

    if (isTyped) { out[idx++] = 'b'; }

    for (Int32 i=in_len-1; i>=0; i--) {
        DFIOBitVal dv = DFIOValChg_GetBit(in, i);
        out[idx++] = "01zx"[(Uint32)dv];

        nV[dv]++;
    }

    out[idx] = 0;
    out.setLen(idx);

    setFmtInfoFlags(infoFlags, 
            nV[(Uint32)DFIOBitVal_0], nV[(Uint32)DFIOBitVal_1],
            nV[(Uint32)DFIOBitVal_Z], nV[(Uint32)DFIOBitVal_X]);
}

/********************************************************************
 * fmtHex()
 ********************************************************************/
void BitVectorABVal::fmtHex(
        const DFIOVal                *in,
        Uint32                        in_len,
        String                       &out,
        Uint32                        flags,
        Uint32                       &infoFlags)
{
    Uint32 idx=0;
    Uint32 val;
    Uint32 nybLen, byteLen;
    Uint32 n0=0, n1=0, nz=0, nx=0;
    Int32  ii;
    bool   first=true;

    /**** top (0th) digit contains remainder ****/
    fmtAddAttributes('h', in_len, out, idx, flags);

    nybLen  = (in_len/4) + ((in_len % 4)?1:0);
    byteLen = (in_len/8) + ((in_len % 8)?1:0);

    out.chkSize(nybLen+8);

    /**** Output data ****/
    for (ii=(nybLen-1); ii>=0; ) {
        const DFIOVal8 *dv = &in[(ii>>1)];

        if (!first || (ii & 1)) {
            if (dv->bval & 0xF0) {
                if (!(dv->aval & 0xF0)) {
                    out[idx++] = 'z';
                    nz++;
                } else {
                    out[idx++] = 'x';
                    nx++;
                }
            } else {
                out[idx++] = "0123456789ABCDEF"[((dv->aval&0xF0) >> 4)];
                n1++;
            }
            ii--;
        }
        first=false;

        if (dv->bval & 0x0F) {
            if (!(dv->aval & 0x0F)) {
                out[idx++] = 'z';
                nz++;
            } else {
                out[idx++] = 'x';
                nx++;
            }
        } else {
            out[idx++] = "0123456789ABCDEF"[(dv->aval&0x0F)];
            n1++;
        }
        ii--;

    }

    out[idx] = 0;
    out.setLen(idx);

    setFmtInfoFlags(infoFlags, n0, n1, nz, nx);
}

/********************************************************************
 * fmtOct()
 ********************************************************************/
void BitVectorABVal::fmtOct(
        const DFIOVal           *in,
        Uint32                   in_len,
        String                  &out,
        Uint32                   flags,
        Uint32                  &infoFlags)
{
    Uint32       idx=0;
    Uint32       n0=0, n1=0, nz=0, nx=0;
    Int32        nDigits, nBytes;
    bool         first=true;
    DFIOVal32    accum = {0, 0};
    Uint32       numAccum=0;

    fmtAddAttributes('o', in_len, out, idx, flags);

    nDigits = (in_len/3) + ((in_len%3)?1:0);
    nBytes  = (in_len/8) + ((in_len%8)?1:0);

    for (Int32 ii=(in_len-1); ii>=0; ) {
        const DFIOVal8    *dv = &in[ii/8];
        Uint32             numTaken1, numTaken, shftAmt, mask;
   
        /**** Number of bits to take this pass ****/
        if (first) {
            if (in_len%3) {
                numTaken = (in_len%3);
            } else {
                numTaken = 3;
            }
        } else {
            numTaken1 = ((ii%8)+1);
            if ((3-numAccum) < numTaken1) { 
                numTaken = (3-numAccum); 
            } else {
                numTaken = numTaken1;
            }
        }
        mask = (1 << (numTaken+numAccum))-1;

        shftAmt = (((ii%8) - numTaken) + 1);

        accum.aval |= ((dv->aval >> shftAmt) & mask);
        accum.bval |= ((dv->bval >> shftAmt) & mask);

        numAccum += numTaken;

        /**** If this is the first iteration, then go ahead and 
         **** output whatever we got...
         ****/

        if (first || numAccum == 3) {
            if (accum.bval) {
                out[idx++] = 'x';
                nx++;
            } else {
                out[idx++] = "01234567"[accum.aval & 7];
                n1++;
                n0++;
            }

            accum.aval  = 0;
            accum.bval  = 0;
            numAccum    = 0;
        }

        ii -= numTaken;
        first = false;
    }

    infoFlags = 0;
    setFmtInfoFlags(infoFlags, n0, n1, nz, nx);

    out[idx] = 0;
    out.setLen(idx);
}

/********************************************************************
 * divTen()
 ********************************************************************/
void BitVectorABVal::divTen(
        Uint8                 *in,
        Uint32                &vlen,
        Uint32                &remainder)
{
    remainder = 0;

    for (Int32 i=(vlen-1); i>=0; i--) {
        Uint16 tmp = in[i];

        tmp += remainder;
        if (tmp >= 10) {
            in[i] = (tmp/10);
            remainder = (tmp%10);
        } else {
            remainder = in[i];
            in[i] = 0;
            if (i == (vlen-1)) {
                vlen--;
            }
        }

        if (i != 0) {
            remainder *= 256;
        }
    }
}

/********************************************************************
 * fmtDec()
 ********************************************************************/
void BitVectorABVal::fmtDec(
        const DFIOVal       *in,
        Uint32               in_len,
        String              &out,
        Uint32               flags,
        Uint32              &infoFlags)
{
    Uint32   idx=0, i;
    Uint32   vlen, byteLen;
    Uint32   n01=0, nx=0, nz=0;
    char    *sv;


    byteLen = (in_len/8) + ((in_len % 8)?1:0);

    if (byteLen > d_fmtDecTmpLen) {
        if (d_fmtDecTmp) {
            delete [] d_fmtDecTmp;
        }

        d_fmtDecTmp    = new Uint8[byteLen];
        d_fmtDecTmpLen = byteLen;
    }

    for (i=0; i<byteLen; i++) {
        const DFIOVal  *dv = &in[i];

        if (dv->bval) {
            if (dv->aval) {
                nx++;
            } else {
                nz++;
            }
        } else {
            n01++;
        }
        d_fmtDecTmp[i] = dv->aval;
    }

    infoFlags = 0;

    if (nz || nx) {
        setFmtInfoFlags(infoFlags, n01, n01, nz, nx);

        if (infoFlags & FmtInfo_ExclZ) {
            out[idx++] = 'z';
        } else {
            out[idx++] = 'x';
        }
        out[idx] = 0;
        return;
    }

    /**** ****/
    do {
        Uint32 remainder;
        divTen(d_fmtDecTmp, byteLen, remainder);

        out[idx++] = "0123456789"[remainder];
    } while (byteLen);

    sv = out.value();
    for (i=0; i<(idx/2); i++) {
        char tmp;
        tmp = sv[i];
        sv[i] = sv[idx-i-1];
        sv[idx-i-1] = tmp;
    }

    out[idx] = 0;
    out.setLen(idx);
}

/********************************************************************
 * fmtString()
 ********************************************************************/
void BitVectorABVal::fmtString(
        const DFIOVal       *in,
        Uint32               in_len,
        String              &out,
        Uint32               flags,
        Uint32              &infoFlags)
{
    Uint32 byteLen = (in_len/8) + ((in_len % 8)?1:0);
    Uint32 n0=0, n1=0, nz=0, nx=0;
    Uint32 idx=0;

    out.chkSize(byteLen+8);

    for (Int32 ii=byteLen-1; ii>=0; ii--) {
        const DFIOVal *dv = &in[ii];

        if (dv->bval) {
            if (dv->aval) {
                out[idx++] = 'x';
                nx+=4;
            } else {
                out[idx++] = 'z';
                nz+=4;
            }
        } else {
            out[idx++] = dv->aval;
            n0 += 4;
            n1 += 4;
        }
    }

    out[idx] = 0;
    out.setLen(idx);

    infoFlags = 0;
    setFmtInfoFlags(infoFlags, n0, n1, nz, nx);
}


/********************************************************************
 * setSize()
 ********************************************************************/
void BitVectorABVal::setSize(Uint32 bitLen)
{
    Uint32 valLen = (bitLen/8);

    if (bitLen%8) {
        valLen++;
    }

    if (valLen > d_valueMaxLen) {
        DFIOVal8    *tmp = d_value;

        d_value = new DFIOVal8[valLen];

        if (d_valueMaxLen) {
            memcpy(d_value, tmp, sizeof(DFIOVal8)*d_valueLen);
            delete [] tmp;
        }
    }

    d_bitLen   = bitLen;
    d_valueLen = valLen;
}

/********************************************************************
 * operator +
 ********************************************************************/
BitVectorABVal operator + (const BitVectorABVal       &op)
{
}

/********************************************************************
 * add
 ********************************************************************/
void BitVectorABVal::add(
        BitVectorABVal              &dest,
        const BitVectorABVal        &op1,
        const BitVectorABVal        &op2)
{
    const BitVectorABVal &max_obj = (op1.d_bitLen>op2.d_bitLen)?op1:op2;
    const BitVectorABVal &min_obj = (op1.d_bitLen>op2.d_bitLen)?op2:op1;
    Uint32                max_len = max_obj.d_valueLen;
    Uint32                min_len = min_obj.d_valueLen;
    Uint32                carry   = 0;
    Uint32                tmpVal;

    dest.setSize(max_len*8);
    for (Uint32 i=0; i<min_len; i++) {
        tmpVal = max_obj.d_value[i].aval + min_obj.d_value[i].aval + carry;
        dest.d_value[i].aval = (tmpVal & 0xFF);
        carry = ((tmpVal & 0xFF00) >> 8);
    }

}


/********************************************************************
 * sub()
 ********************************************************************/
static void sub(
        BitVectorABVal             &dest,
        const BitVectorABVal       &op1,
        const BitVectorABVal       &op2)
{

}

/********************************************************************
 * setFmtInfoFlags()
 ********************************************************************/
void BitVectorABVal::setFmtInfoFlags(
        Uint32               &flags,
        Uint32                n0,
        Uint32                n1,
        Uint32                nz,
        Uint32                nx)
{
    flags = 0;

    if (nz && !nx && !n0 && !n1) {
        flags |= FmtInfo_ExclZ;
    } else if (nx && !nz && !n0 && !n1) {
        flags |= FmtInfo_ExclX;
    } 

    if (nz || nx) {
        flags |= FmtInfo_HasXZ;
    }
}

Uint8  *BitVectorABVal::d_fmtDecTmp    = 0;
Uint32  BitVectorABVal::d_fmtDecTmpLen = 0;


