--------------------------------------------------------------------------------
-- Project:     fb-unisim: simplified behavioral version of Xilinx Unisim library
-- File:        fb-unisim_VITAL.vhd
-- Description: equivalent to unisim_VITAL.vhd, behavioral models
-- History:     2003vi24, FB, created.
--------------------------------------------------------------------------------

----- CELL BUFG -----
library IEEE;
use IEEE.STD_LOGIC_1164.all;
use IEEE.NUMERIC_STD.all;

-- entity declaration --
entity BUFG is
   port(
      O                              :	out   STD_ULOGIC;
      I                              :	in    STD_ULOGIC);
end BUFG;

architecture BUFG_V of BUFG is
begin
	O <= I; -- after 100 ps;
end BUFG_V;

--------------------------------------------------------------------------------
----- CELL SRL16E -----
library IEEE;
use IEEE.STD_LOGIC_1164.all;
use IEEE.NUMERIC_STD.all;

entity SRL16E is
  generic (
       INIT :     bit_vector := X"0000");
  port (D   : in  STD_ULOGIC;
        CE  : in  STD_ULOGIC;
        CLK : in  STD_ULOGIC;
        A0  : in  STD_ULOGIC;
        A1  : in  STD_ULOGIC;
        A2  : in  STD_ULOGIC;
        A3  : in  STD_ULOGIC;
        Q   : out STD_ULOGIC); 
end SRL16E;

architecture SRL16E_V of SRL16E is
	signal SHIFT_REG : std_logic_vector (16 downto 0) := ('X' & To_StdLogicVector(INIT));
	signal LENGTH    : integer := 0;
	signal ADDR      : std_logic_vector(3 downto 0) := "0000";
begin -- architecture

	ADDR  <= (A3, A2, A1, A0);
	LENGTH<= TO_INTEGER(UNSIGNED(ADDR));
	Q     <= SHIFT_REG(LENGTH); -- after 100 ps;
	
	process(CLK)
	begin
		if rising_edge(CLK) then
			if (CE = '1') then
				for I in 15 downto 1 loop
					SHIFT_REG(I) <= SHIFT_REG(I-1);
				end loop;
				SHIFT_REG(0) <= D;
			end if;
		end if;
	end process;

end SRL16E_V;

--------------------------------------------------------------------------------
----CELL ramb4_s16_s16----

library IEEE;
use IEEE.STD_LOGIC_1164.all;
use IEEE.NUMERIC_STD.all;

entity RAMB4_S16_S16 is
	generic (
		INIT_00 : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
		INIT_01 : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
		INIT_02 : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
		INIT_03 : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
		INIT_04 : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
		INIT_05 : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
		INIT_06 : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
		INIT_07 : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
		INIT_08 : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
		INIT_09 : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
		INIT_0A : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
		INIT_0B : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
		INIT_0C : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
		INIT_0D : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
		INIT_0E : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000";
		INIT_0F : bit_vector := X"0000000000000000000000000000000000000000000000000000000000000000");
	port (
		DIA    : in STD_LOGIC_VECTOR (15 downto 0);
		DIB    : in STD_LOGIC_VECTOR (15 downto 0);
		ENA    : in STD_ULOGIC;
		ENB    : in STD_ULOGIC;
		WEA    : in STD_ULOGIC;
		WEB    : in STD_ULOGIC;
		RSTA   : in STD_ULOGIC;
		RSTB   : in STD_ULOGIC;
		CLKA   : in STD_ULOGIC;
		CLKB   : in STD_ULOGIC;
		ADDRA  : in STD_LOGIC_VECTOR (7 downto 0);
		ADDRB  : in STD_LOGIC_VECTOR (7 downto 0);
		DOA    : out STD_LOGIC_VECTOR (15 downto 0);
		DOB    : out STD_LOGIC_VECTOR (15 downto 0)); 
end ramb4_s16_s16; 

architecture RAMB4_S16_S16_V of RAMB4_S16_S16 is
	signal ADDRESS_A: integer;
	signal ADDRESS_B: integer;
begin

	ADDRESS_A<= TO_INTEGER(UNSIGNED(ADDRA));
	ADDRESS_B<= TO_INTEGER(UNSIGNED(ADDRB));

	process(CLKA, RSTA, CLKB, RSTB)
		variable MEM       : std_logic_vector(4095 downto 0) := 
			To_StdLogicVector(INIT_0F) &
			To_StdLogicVector(INIT_0E) &
			To_StdLogicVector(INIT_0D) &
			To_StdLogicVector(INIT_0C) &
			To_StdLogicVector(INIT_0B) &
			To_StdLogicVector(INIT_0A) &
			To_StdLogicVector(INIT_09) &
			To_StdLogicVector(INIT_08) &
			To_StdLogicVector(INIT_07) &
			To_StdLogicVector(INIT_06) &
			To_StdLogicVector(INIT_05) &
			To_StdLogicVector(INIT_04) &
			To_StdLogicVector(INIT_03) &
			To_StdLogicVector(INIT_02) &
			To_StdLogicVector(INIT_01) &
			To_StdLogicVector(INIT_00);
		constant DIAW   : integer := 16;
		constant DIAW_1 : integer := 15;
		constant DIBW   : integer := 16;
		constant DIBW_1 : integer := 15;
	begin
		--
		-- Port A
		--
		if RSTA= '1' then
			DOA <= (others => '0');
		elsif rising_edge(CLKA) then
			if ENA= '1' then
				if (WEA = '1') then
					MEM((ADDRESS_A*DIAW + DIAW_1) downto (ADDRESS_A*DIAW)) := DIA;
					--DOA <= DIA;
				end if;
				DOA <= MEM((ADDRESS_A*DIAW + DIAW_1) downto (ADDRESS_A*DIAW));-- after 100 ps;
			end if;
		end if;
		--
		-- Port B
		--
		if RSTB= '1' then
			DOB <= (others => '0');
		elsif rising_edge(CLKB) then
			if ENB= '1' then
				if (WEB = '1') then
					MEM((ADDRESS_B*DIBW + DIBW_1) downto (ADDRESS_B*DIBW)) := DIB;
					--DOB <= DIB;
				end if;
				DOB <= MEM((ADDRESS_B*DIBW + DIBW_1) downto (ADDRESS_B*DIBW));-- after 100 ps;
			end if;
		end if;
  end process;
end RAMB4_S16_S16_V;

--------------------------------------------------------------------------------
----- CELL CLKDLL -----
library IEEE;
use IEEE.std_logic_1164.all;

entity CLKDLL is
	generic (
		FACTORY_JF : bit_vector := X"00";
		STARTUP_WAIT : boolean := false;             
		
		DUTY_CYCLE_CORRECTION : boolean := TRUE;
		CLKDV_DIVIDE : real := 2.0); 
	port (
		CLKIN   : in  std_ulogic := '0';
		CLKFB   : in  std_ulogic := '0';
		RST     : in  std_ulogic := '0';
		CLK0    : out std_ulogic := '0';
		CLK90   : out std_ulogic := '0';
		CLK180  : out std_ulogic := '0';
		CLK270  : out std_ulogic := '0';
		CLK2X   : out std_ulogic := '0';
		CLKDV   : out std_ulogic := '0';
		LOCKED  : out std_ulogic := '0');
end CLKDLL;
 
architecture CLKDLL_V of CLKDLL is
	signal period:      time:= 10 ns;
	signal lastperiod:  time:= 10 ns;
	signal lastposedge: time:= 10 ns;
	signal phaseshift:  time:= 1 ns;
	signal clk1x: std_logic;
	signal locked_ii: std_logic;
	signal clk0_ii, clk90_ii, clk180_ii, clk270_ii, clk2x_ii: std_logic;
begin
	process(CLKIN)
	begin
		if RST= '1' then
			period     <= 10 ns;
			lastperiod <= 10 ns;
			lastposedge<= 10 ns;
		elsif rising_edge(CLKIN) then
			period<= now - lastposedge;
			lastposedge<= now;
			if    lastperiod-period> 100 ps then locked_ii<= '0';
			elsif period-lastperiod> 100 ps then locked_ii<= '0';
			else                                 locked_ii<= '1'; end if;
			lastperiod<= period;
		end if;			
	end process;
	LOCKED<= locked_ii;
	
	process(CLKFB)
	begin
		if rising_edge(CLKFB) then
			--if (now-lastposedge < period/2) then 
			if (now-lastposedge < period/4) then 
				phaseshift<= now - lastposedge;
			end if;
		end if;
	end process;
	
	process(CLKIN)
	begin
		if rising_edge(CLKIN) then
			clk1x<= '0' after period/2-phaseshift, '1' after period-phaseshift;
		end if;
	end process;
	CLK0  <= clk1x;
	CLK90 <= transport clk1x after period/4;
	CLK180<= transport clk1x after period/2;
	CLK270<= transport clk1x after 3*period/4;

	process(CLKIN)
		variable phase1, phase2, phase3, phase4: time;
	begin
		if rising_edge(CLKIN) then
			if 1*period/4-phaseshift> 0 ns then phase1:= 1*period/4-phaseshift; 
			else                                phase1:= 0 ns; end if;
			if 2*period/4-phaseshift> 0 ns then phase2:= 2*period/4-phaseshift; 
			else                                phase2:= 0 ns; end if;
			if 3*period/4-phaseshift> 0 ns then phase3:= 3*period/4-phaseshift; 
			else                                phase3:= 0 ns; end if;
			if 4*period/4-phaseshift> 0 ns then phase4:= 4*period/4-phaseshift; 
			else                                phase4:= 0 ns; end if;
			clk2x_ii<= '0' after phase1, 
				'1' after phase2, 
				'0' after phase3,
				'1' after phase4;
		end if;
	end process;
	CLK2X <= CLKIN when locked_ii= '0' else clk2x_ii;
end CLKDLL_V;

--------------------------------------------------------------------------------
----- CELL FDC -----
library IEEE;
use IEEE.STD_LOGIC_1164.all;

entity FDC is
   generic(
      INIT                           :  bit := '0');
   port(
      Q                              :	out   STD_ULOGIC;
      D                              :	in    STD_ULOGIC;
      C                              :	in    STD_ULOGIC;
      CLR                            :	in    STD_ULOGIC);
end FDC;

architecture FDC_V of FDC is
begin

	process(C, CLR) 
		VARIABLE mem : STD_ULOGIC := TO_X01(INIT);
   begin
		if CLR= '1' then
			mem:= '0';
		elsif rising_edge(C) then
			mem:= D;
		end if;
		Q<= mem;-- after 100 ps;
	end process;

end FDC_V;

--------------------------------------------------------------------------------
----- CELL FDR -----
library IEEE;
use IEEE.STD_LOGIC_1164.all;

entity FDR is
   generic(
      INIT                           :  bit := '0'  );
   port(
      Q                              :	out   STD_ULOGIC;
      D                              :	in    STD_ULOGIC;
      C                              :	in    STD_ULOGIC;
      R                              :	in    STD_ULOGIC);
end FDR;

architecture FDR_V of FDR is
begin

	process(C, R) 
		VARIABLE mem : STD_ULOGIC := TO_X01(INIT);
	begin
		if rising_edge(C) then
			if R= '1' then
				mem:= '0';
			else
				mem:= D;
			end if;
		end if;
		Q<= mem;-- after 100 ps;
	end process;

end FDR_V;

--------------------------------------------------------------------------------
----- CELL FDRE -----
library IEEE;
use IEEE.STD_LOGIC_1164.all;

entity FDRE is
   generic(
      INIT                           :  bit := '0'  );
   port(
      Q                              :	out   STD_ULOGIC;
      D                              :	in    STD_ULOGIC;
      C                              :	in    STD_ULOGIC;
      CE                             :	in    STD_ULOGIC;
      R                              :	in    STD_ULOGIC);
end entity;

architecture FDRE_V of FDRE is
begin

	process(C, R) 
		VARIABLE mem : STD_ULOGIC := TO_X01(INIT);
	begin
		if rising_edge(C) then
			if R= '1' then
				mem:= '0';
			elsif CE= '1' then
				mem:= D;
			end if;
		end if;
		Q<= mem;-- after 100 ps;
	end process;

end FDRE_V;

-----------------------------------------------------------------------------
----- CELL SRL16 -----

library IEEE;
use IEEE.STD_LOGIC_1164.all;

--library UNISIM;
--use UNISIM.vcomponents.all;

entity SRL16 is
	generic (
		INIT : bit_vector := X"0000");
	port (
		D   : in STD_ULOGIC;
		CLK : in STD_ULOGIC;
		A0  : in STD_ULOGIC;
		A1  : in STD_ULOGIC;
		A2  : in STD_ULOGIC;
		A3  : in STD_ULOGIC;
		Q   : out STD_ULOGIC); 
end SRL16;

architecture SRL16_V of SRL16 is
	component SRL16E 
		generic (
			INIT : bit_vector := X"0000");
		port (
			D   : in STD_ULOGIC;
			CE  : in STD_ULOGIC;
			CLK : in STD_ULOGIC;
			A0  : in STD_ULOGIC;
			A1  : in STD_ULOGIC;
			A2  : in STD_ULOGIC;
			A3  : in STD_ULOGIC;
			Q   : out STD_ULOGIC); 
	end component;

	signal VCC: std_ulogic;
begin
	VCC<= '1';
	
	U: SRL16E generic map (
		INIT => INIT)
	port map (
		D   => D,
		CE  => VCC,
		CLK => CLK,
		A0  => A0,
		A1  => A1,
		A2  => A2,
		A3  => A3,
		Q   => Q);
end SRL16_V;

--------------------------------------------------------------------------------
----- CELL RAM16X1D -----

library IEEE;
use IEEE.STD_LOGIC_1164.all;
use IEEE.NUMERIC_STD.all;

entity RAM16X1D is
	generic (
		INIT : bit_vector(15 downto 0) := X"0000");
	port (
		D     : in std_ulogic;
		WE    : in std_ulogic;
		WCLK  : in std_ulogic;
		A0    : in std_ulogic;
		A1    : in std_ulogic;
		A2    : in std_ulogic;
		A3    : in std_ulogic;
		DPRA0 : in std_ulogic;
		DPRA1 : in std_ulogic;
		DPRA2 : in std_ulogic;
		DPRA3 : in std_ulogic;
		
		SPO   : out std_ulogic;
		DPO   : out std_ulogic);
end RAM16X1D;

architecture RAM16X1D_V of RAM16X1D is
	signal MEM       : std_logic_vector( 16 downto 0 ) := ('X' & To_StdLogicVector(INIT) );
	signal Index_SP  : integer:= 0;
	signal Index_DP  : integer:= 0;
	signal ADDR      : std_logic_vector(3 downto 0);
	signal DPRA      : std_logic_vector(3 downto 0);
begin
	ADDR<= A3 & A2 & A1 & A0;
	DPRA<= DPRA3 & DPRA2 & DPRA1 & DPRA0;
	
	Index_SP <= TO_INTEGER(UNSIGNED(ADDR));
	Index_DP <= TO_INTEGER(UNSIGNED(DPRA));

	SPO <= MEM(Index_SP);-- after 100 ps;
	DPO <= MEM(Index_DP);-- after 100 ps;

	process(WCLK)
	begin
		if rising_edge(WCLK) then
			if WE= '1' then
				MEM(Index_SP)<= D;
			end if;
		end if;
  end process;

end RAM16X1D_V;

--------------------------------------------------------------------------------
-- end of file
