/*
 * main.c - main-file for iTALC-key-generator
 *
 * iTALC
 * Copyright (c) 2004-2005 Tobias Doerffel <tobias@doerffel.de>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program (see COPYING); if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 */


#include <string.h>
#include <stdio.h>
#include <unistd.h>
#include <getopt.h>
#include <pwd.h>
#include <sys/types.h>

#include "rsa_crypt_lib.h"
#include "paths.h"

void displayHelp( void );


int main( int argc, char * * argv )
{
	int c;			/* return value from getopt_long */
	int * longindex = NULL;
	char force = 0;
	char * user_home = NULL;
	char * user_name = NULL;
	char * public_key_dir = NULL;
	char * priv_key_file = NULL;
	char * pub_key_file = NULL;
	RSA * private_key = NULL;
	RSA * public_key = NULL;
	struct passwd * user_info = NULL;

	/* parse command line options */
	c = EOF+1;

	while( c != EOF )
	{
		static struct option cmd_options[] =
		{
			{ "help", 0, NULL, 'h' },
			{ "user", 1, NULL, 'u' },
			{ "force", 0, NULL, 'f' },
			{ "default-public-key-dir", 1, NULL, 'd' },
			{ "public-key-dir", 1, NULL, 'p' }
		} ;

		c = getopt_long( argc, argv, "hu:fdp:", cmd_options, longindex );

		switch( c )
		{
			case 'u':
				if( strlen( optarg ) <= 0 )
				{
					fprintf( stderr, "Missing argument for -u\n" );
					displayHelp();
					exit( 1 );
				}
				user_name = malloc( strlen( optarg ) + 1 );
				strcpy( user_name, optarg );
				break;

			case 'f':
				force = 1;
				break;

			case 'd':
				public_key_dir = malloc( strlen( ITALC_CLIENT_PUBLIC_KEYS ) + 1 );
				strcpy( public_key_dir, ITALC_CLIENT_PUBLIC_KEYS );
				break;

			case 'p':
				if( strlen( optarg ) <= 0 )
				{
					fprintf( stderr, "Missing argument for -p\n" );
					displayHelp();
					exit( 1 );
				}
				public_key_dir = malloc( strlen( optarg ) + 2 );
				strcpy( public_key_dir, optarg );
				strcat( public_key_dir, "/" );
				break;

			case 'h':
			case '?':
			case ':':
				displayHelp();
				exit( 1 );
				break;

		} /* end switch( c ) */

	} /* end of option parsing loop */

	if( optind < argc )
	{
		fprintf( stderr, "Too many parameters.\n" );
		displayHelp();
		exit( 1 );
	}

	if( user_name == NULL )
	{
		user_name = getenv( "USER" );
		if( user_name == NULL )
		{
			fprintf( stderr, "No user specified and nothing found in $USER.\n" );
			exit( 1 );
		}
		fprintf( stdout, "Warning: no user specified. Will use contents of $USER.\n" );
	}
	user_info = getpwnam( user_name );
	if( user_info == NULL )
	{
		fprintf( stderr, "Could not obtain information about given user.\n" );
		exit( 1 );
	}
	user_home = user_info->pw_dir;	

	/* constuct private-key-filename */
	priv_key_file = malloc( strlen( user_home ) + 1 + strlen( ITALC_CONFIG_PATH ) + strlen( "/id_rsa" ) + 1 );
	strcpy( priv_key_file, user_home );
	strcat( priv_key_file, "/" );
	strcat( priv_key_file, ITALC_CONFIG_PATH );
	/* remove file ".italc" if it exists */
	remove( priv_key_file );
	/* and now try to make a directory if it doesn't already exist */
	mkdir( priv_key_file, 0744 );
	chown( priv_key_file, user_info->pw_uid, user_info->pw_gid );

	/* now concatenate the rest of the filename */
	strcat( priv_key_file, "/id_rsa" );

	/* construct public-key-filename */
	pub_key_file = malloc( strlen( priv_key_file ) + strlen( ".public" ) + 1 );
	strcpy( pub_key_file, priv_key_file );
	strcat( pub_key_file, ".public" );

	/* try to read private key */
	private_key = readPrivateKey( priv_key_file );
	if( private_key != NULL )
	{
		/* worked, so there are already some keys */
		RSA_free( private_key );
		if( !force )
		{
			fprintf( stderr, "Keys are already existing. Won't overwrite them until specified by --force\n" );
			exit( 1 );
		}
	}

	fprintf( stdout, "Generating key-pair... " );
	fflush( stdout );

	/* generate 1024bit RSA-key and save it */
	private_key = rsaGeneratePrivateKey( 1024 );
	fprintf( stdout, "done.\n" );

	writePrivateKey( private_key, priv_key_file );

	/* derive public key from private key and save it */
	public_key = publicKeyFromPrivateKey( private_key );
	writePublicKey( public_key, pub_key_file );

	/* make sure, created files are owned by specified user */
	chown( priv_key_file, user_info->pw_uid, user_info->pw_gid );
	chown( pub_key_file, user_info->pw_uid, user_info->pw_gid );
	fprintf( stdout, "The keys were saved in %s/%s/\n", user_home, ITALC_CONFIG_PATH );

	/* if specified, we also store the public-key into the public-key-directory */
	if( public_key_dir != NULL )
	{
		/* construct public-key-filename */
		char * public_key_in_pkd = malloc( strlen( public_key_dir ) + strlen ( user_name ) + strlen( ".public" ) + 1 );
		strcpy( public_key_in_pkd, public_key_dir );
		strcat( public_key_in_pkd, user_name );
		strcat( public_key_in_pkd, ".public" );
		writePublicKey( public_key, public_key_in_pkd );
		fprintf( stdout, "The public-key was saved in %s\n", public_key_in_pkd );
		free( public_key_in_pkd );
		free( public_key_dir );
	}
	/* cleanup and exit */
	RSA_free( public_key );
	RSA_free( private_key );
	free( priv_key_file );
	free( pub_key_file );
}




void displayHelp( void )
{
	fprintf( stdout, "\nUsage: italc-keygen [options]\n\n" );
	fprintf( stdout, "Generate public-private-key-pair for iTALC-users.\n\n" );
	fprintf( stdout, "Options:\n");
	fprintf( stdout, "  -h, --help                    display this text\n" );
	fprintf( stdout, "  -u, --user=USER               generate keys for user USER and save\n" );
	fprintf( stdout, "                                them in USER's home-directory\n" );
	fprintf( stdout, "  -f, --force                   overwrite existing keys\n" );
	fprintf( stdout, "  -d, --default-public-key-dir  write public-key-file into default-directory\n" );
	fprintf( stdout, "                                "ITALC_CLIENT_PUBLIC_KEYS"\n" );
	fprintf( stdout, "  -p, --public-key-dir[=DIR]    directory to store public-key-file in\n" );
	fprintf( stdout, "\nExample:\n" );
	fprintf( stdout, "  italc-keygen -u foo -f -p\n\n" );
	fprintf( stdout, "Generate a key-pair for user 'foo' and store the public-key-file in the\n" );
	fprintf( stdout, "default-public-key-directory.\n\n" );
}
