/*
 ui-commands.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"

static gchar *ret_texts[] =
{
    "Invalid parameter",
    "Not enough parameters given",
    "Not connected to IRC server yet",
    "Not joined to any channels yet",
    "Error: getsockname() failed",
    "Error: listen() failed",
    "Multiple matches found, be more specific",
    "Nick not found",
    "Not joined to such channel",
    "Server not found",
    "Channel not fully synchronized yet, try again after a while",
    "Doing this is not a good idea. Add -YES if you really mean it",
};

static GString *tmpstr;

static gint commands_compare(COMMAND_REC *rec, COMMAND_REC *rec2)
{
    if (rec->category == NULL && rec2->category != NULL)
	return -1;
    if (rec2->category == NULL && rec->category != NULL)
	return 1;

    return strcmp(rec->cmd, rec2->cmd);
}

static void help_category(GList *cmdlist, gint items, gint max)
{
    COMMAND_REC *rec, *last;
    GString *str;
    GList *tmp;
    gint lines, cols, line, col, skip;
    gchar *cmdbuf;

    str = g_string_new(NULL);

    cols = max > 60 ? 1 : (60 / max);
    lines = items <= cols ? 1 : items / cols+1;

    last = NULL; cmdbuf = g_malloc(max+1); cmdbuf[max] = '\0';
    for (line = 0, col = 0, skip = 1, tmp = cmdlist; line < lines; last = rec, tmp = tmp->next)
    {
	rec = tmp->data;

	if (--skip == 0)
	{
	    skip = lines;
	    memset(cmdbuf, ' ', max);
	    memcpy(cmdbuf, rec->cmd, strlen(rec->cmd));
	    g_string_sprintfa(str, "%s ", cmdbuf);
	    cols++;
	}

	if (col == cols || tmp->next == NULL)
	{
	    printtext(NULL, NULL, MSGLEVEL_CLIENTCRAP, str->str);
	    g_string_truncate(str, 0);
	    col = 0; line++;
	    tmp = g_list_nth(cmdlist, line-1); skip = 1;
	}
    }
    if (str->len != 0)
	printtext(NULL, NULL, MSGLEVEL_CLIENTCRAP, str->str);
    g_string_free(str, TRUE);
    g_free(cmdbuf);
}

static gboolean show_help(COMMAND_REC *cmd)
{
    GString *out, *buf;
    gchar *path;
    gint f;

    /* helpdir/command or helpdir/category/command */
    if (cmd->category == NULL)
	path = g_strdup_printf("%s/%s", HELPDIR, cmd->cmd);
    else
	path = g_strdup_printf("%s/%s/%s", HELPDIR, cmd->category, cmd->cmd);
    f = open(path, O_RDONLY);
    g_free(path);

    if (f == -1)
	return FALSE;

    /* just print to screen whatever is in the file */
    out = g_string_new(NULL);
    buf = g_string_new(NULL);
    while (read_line(FALSE, f, out, buf) > 0)
	printtext(NULL, NULL, MSGLEVEL_NEVER, out->str);
    g_string_free(out, TRUE);
    g_string_free(buf, TRUE);

    close(f);
    return TRUE;
}

static gboolean cmd_help(gchar *data)
{
    COMMAND_REC *rec, *last, *helpitem;
    GList *tmp, *cmdlist;
    gint len, max, items, findlen;
    gboolean header;

    g_return_val_if_fail(data != NULL, FALSE);

    /* sort the commands list */
    commands = g_list_sort(commands, (GCompareFunc) commands_compare);

    /* print command, sort by category */
    cmdlist = NULL; last = NULL; header = FALSE; helpitem = NULL;
    max = items = 0; findlen = strlen(data);
    for (tmp = commands; tmp != NULL; last = rec, tmp = tmp->next)
    {
	rec = tmp->data;

	if (last != NULL && rec->category != NULL &&
	    (last->category == NULL || strcmp(rec->category, last->category) != 0))
	{
	    /* category changed */
	    if (items > 0)
	    {
		if (!header)
		{
		    printtext(NULL, NULL, MSGLEVEL_CLIENTCRAP, "Irssi commands:");
		    header = TRUE;
		}
		if (last->category != NULL)
		{
		    printtext(NULL, NULL, MSGLEVEL_CLIENTCRAP, "");
		    printtext(NULL, NULL, MSGLEVEL_CLIENTCRAP, "%s:", last->category);
		}
		help_category(cmdlist, items, max);
	    }

	    g_list_free(cmdlist); cmdlist = NULL;
	    items = 0; max = 0;
	}

	if (last != NULL && g_strcasecmp(rec->cmd, last->cmd) == 0)
	    continue; /* don't display same command twice */

	if (strlen(rec->cmd) >= findlen && g_strncasecmp(rec->cmd, data, findlen) == 0)
	{
	    if (rec->cmd[findlen] == '\0')
	    {
		helpitem = rec;
		break;
	    }
	    else if (strchr(rec->cmd+findlen+1, ' ') == NULL)
	    {
		/* not a subcommand (and matches the query) */
		len = strlen(rec->cmd);
		if (max < len) max = len;
		items++;
		cmdlist = g_list_append(cmdlist, rec);
	    }
	}
    }

    if ((helpitem == NULL && items == 0) || (helpitem != NULL && !show_help(helpitem)))
	printtext(NULL, NULL, MSGLEVEL_CLIENTCRAP, "No help for %s", data);

    if (items != 0)
    {
	/* display the last category */
	if (!header)
	{
	    printtext(NULL, NULL, MSGLEVEL_CLIENTCRAP, "Irssi commands:");
	    header = TRUE;
	}

	if (last->category != NULL)
	{
	    printtext(NULL, NULL, MSGLEVEL_CLIENTCRAP, "");
	    printtext(NULL, NULL, MSGLEVEL_CLIENTCRAP, "%s:", last->category);
	}
	help_category(cmdlist, items, max);
	g_list_free(cmdlist);
    }

    return TRUE;
}

static gboolean cmd_servers(gchar *data)
{
    GList *tmp;
    gchar *tag, *next_connect;
    gint left;

    for (tmp = servers; tmp != NULL; tmp = tmp->next)
    {
        SERVER_REC *rec = tmp->data;

	printformat(NULL, NULL, MSGLEVEL_CRAP, IRCTXT_SERVER_LIST,
		    rec->tag, rec->address, rec->port,
		    rec->ircnet == NULL ? "" : rec->ircnet, rec->nick);
    }

    for (tmp = lookup_servers; tmp != NULL; tmp = tmp->next)
    {
        SERVER_REC *rec = tmp->data;

	printformat(NULL, NULL, MSGLEVEL_CRAP, IRCTXT_SERVER_LOOKUP_LIST,
		    rec->tag, rec->address, rec->port,
		    rec->ircnet == NULL ? "" : rec->ircnet, rec->nick);
    }

    for (tmp = reconnects; tmp != NULL; tmp = tmp->next)
    {
        RECONNECT_REC *rec = tmp->data;

	tag = g_strdup_printf("RECON-%d", rec->tag);
	left = rec->next_connect-time(NULL);
	next_connect = g_strdup_printf("%02d:%02d", left/60, left%60);
	printformat(NULL, NULL, MSGLEVEL_CRAP, IRCTXT_SERVER_RECONNECT_LIST,
		    tag, rec->address, rec->port,
		    rec->ircnet == NULL ? "" : rec->ircnet,
		    rec->nick, next_connect);
	g_free(next_connect);
	g_free(tag);
    }

    return TRUE;
}

static gboolean cmd_unquery(gchar *data, SERVER_REC *server, CHANNEL_REC *cur_channel)
{
    CHANNEL_REC *channel;

    g_return_val_if_fail(data != NULL, FALSE);

    if (*data == '\0')
        channel = cur_channel; /* remove current query */
    else
    {
        channel = channel_find(*data == '=' ? NULL : server, data);
        if (channel == NULL)
        {
            printformat(server, NULL, MSGLEVEL_CLIENTERROR, IRCTXT_NO_QUERY, data);
            return TRUE;
        }
    }

    /* remove query */
    if (channel == NULL || (channel->type != CHANNEL_TYPE_QUERY &&
                            channel->type != CHANNEL_TYPE_DCC_CHAT))
    {
        /* not a query! */
        return TRUE;
    }

    channel->left = TRUE;
    channel_destroy(channel);
    return TRUE;
}

static gboolean cmd_query(gchar *data, SERVER_REC *server, CHANNEL_REC *cur_channel)
{
    CHANNEL_REC *channel;

    g_return_val_if_fail(data != NULL, FALSE);

    if (*data == '\0')
    {
        /* remove current query */
        cmd_unquery("", server, cur_channel);
        return TRUE;
    }

    if (*data != '=' && (server == NULL || !server->connected))
        cmd_return_error(CMDERR_NOT_CONNECTED);

    channel = channel_find(*data == '=' ? NULL : server, data);
    if (channel != NULL)
    {
        /* query already existed - change to query window */
        signal_emit("gui channel open", 1, channel);
        return TRUE;
    }

    channel_create(server, data, CHANNEL_TYPE_QUERY, FALSE);
    return TRUE;
}

static gboolean cmd_msg(gchar *data, SERVER_REC *server)
{
    CHANNEL_REC *channel;
    NICK_REC *nickrec;
    gchar *params, *target, *msg, *nickmode;

    g_return_val_if_fail(data != NULL, FALSE);

    params = cmd_get_params(data, 2 | PARAM_FLAG_GETREST, &target, &msg);
    if (*target == '\0' || *msg == '\0') cmd_param_error(CMDERR_NOT_ENOUGH_PARAMS);

    if (*target == '=')
    {
        /* dcc msg - handled in ui-dcc.c */
        g_free(params);
        return TRUE;
    }

    if (server == NULL || !server->connected) cmd_param_error(CMDERR_NOT_CONNECTED);
    channel = channel_find(server, target);

    if (*target == '@' && ischannel(target[1]))
	target++; /* Hybrid 6 feature, send msg to all ops in channel */

    if (ischannel(*target))
    {
	/* msg to channel */
	nickrec = channel == NULL ? NULL : nicklist_find(channel, server->nick);
	nickmode = !setup_get_bool("toggle_show_nickmode") || nickrec == NULL ? "" :
	    nickrec->op ? "@" : nickrec->voice ? "+" : " ";

	if (channel != NULL && CHANNEL_PARENT(channel)->active == channel)
	{
	    printformat(server, target, MSGLEVEL_PUBLIC | MSGLEVEL_NOHILIGHT,
			IRCTXT_OWN_MSG, server->nick, msg, nickmode);
	}
	else
	{
	    printformat(server, target, MSGLEVEL_PUBLIC | MSGLEVEL_NOHILIGHT,
			IRCTXT_OWN_MSG_CHANNEL, server->nick, target, msg, nickmode);
	}
    }
    else
    {
        /* private message */
	printformat(server, target, MSGLEVEL_MSGS | MSGLEVEL_NOHILIGHT,
		    channel == NULL ? IRCTXT_OWN_MSG_PRIVATE : IRCTXT_OWN_MSG_PRIVATE_QUERY, target, msg, server->nick);
    }

    g_free(params);
    return TRUE;
}

static gboolean cmd_notice(gchar *data, SERVER_REC *server)
{
    gchar *params, *target, *msg;

    g_return_val_if_fail(data != NULL, FALSE);
    if (server == NULL || !server->connected) cmd_return_error(CMDERR_NOT_CONNECTED);

    params = cmd_get_params(data, 2 | PARAM_FLAG_GETREST, &target, &msg);
    if (*target == '\0' || *msg == '\0') cmd_param_error(CMDERR_NOT_ENOUGH_PARAMS);

    if (*target == '@' && ischannel(target[1]))
	target++; /* Hybrid 6 feature, send notice to all ops in channel */

    printformat(server, target, MSGLEVEL_NOTICES | MSGLEVEL_NOHILIGHT,
		IRCTXT_OWN_NOTICE, target, msg);

    g_free(params);
    return TRUE;
}

static gboolean cmd_me(gchar *data, SERVER_REC *server, CHANNEL_REC *cur_channel)
{
    g_return_val_if_fail(data != NULL, FALSE);

    if (cur_channel->type == CHANNEL_TYPE_DCC_CHAT)
    {
        /* DCC action - handled by ui-dcc.c */
        return TRUE;
    }

    if (server == NULL || !server->connected) cmd_return_error(CMDERR_NOT_CONNECTED);

    printformat(server, cur_channel->name, MSGLEVEL_ACTIONS,
		IRCTXT_OWN_ME, server->nick, data);

    if (cur_channel->type == CHANNEL_TYPE_CHANNEL ||
        cur_channel->type == CHANNEL_TYPE_QUERY)
    {
        g_string_sprintf(tmpstr, "PRIVMSG %s :\001ACTION %s\001", cur_channel->name, data);
        irc_send_cmd(server, tmpstr->str);
    }

    return TRUE;
}

static gboolean cmd_action(gchar *data, SERVER_REC *server)
{
    gchar *params, *target, *text;

    g_return_val_if_fail(data != NULL, FALSE);
    if (server == NULL || !server->connected) cmd_return_error(CMDERR_NOT_CONNECTED);
    if (*data == '=')
    {
	/* DCC action - handled by ui-dcc.c */
	return TRUE;
    }

    params = cmd_get_params(data, 3 | PARAM_FLAG_GETREST, &target, &text);
    if (*target == '\0' || *text == '\0') cmd_param_error(CMDERR_NOT_ENOUGH_PARAMS);

    printformat(server, target, MSGLEVEL_ACTIONS, IRCTXT_OWN_ME, server->nick, text);

    g_string_sprintf(tmpstr, "PRIVMSG %s :\001ACTION %s\001", target, text);
    irc_send_cmd(server, tmpstr->str);

    g_free(params);
    return TRUE;
}

static gboolean cmd_ctcp(gchar *data, SERVER_REC *server)
{
    gchar *params, *target, *ctcpcmd, *ctcpdata;

    g_return_val_if_fail(data != NULL, FALSE);
    if (server == NULL || !server->connected) cmd_return_error(CMDERR_NOT_CONNECTED);

    params = cmd_get_params(data, 3 | PARAM_FLAG_GETREST, &target, &ctcpcmd, &ctcpdata);
    if (*target == '\0' || *ctcpcmd == '\0') cmd_param_error(CMDERR_NOT_ENOUGH_PARAMS);

    if (*target == '@' && ischannel(target[1]))
	target++; /* Hybrid 6 feature, send ctcp to all ops in channel */

    g_strup(ctcpcmd);
    printformat(server, target, MSGLEVEL_CTCPS, IRCTXT_OWN_CTCP, target, ctcpcmd, ctcpdata);

    g_free(params);
    return TRUE;
}

static gboolean cmd_nctcp(gchar *data, SERVER_REC *server)
{
    gchar *params, *target, *ctcpcmd, *ctcpdata;

    g_return_val_if_fail(data != NULL, FALSE);
    if (server == NULL || !server->connected) cmd_return_error(CMDERR_NOT_CONNECTED);

    params = cmd_get_params(data, 3 | PARAM_FLAG_GETREST, &target, &ctcpcmd, &ctcpdata);
    if (*target == '\0' || *ctcpcmd == '\0') cmd_param_error(CMDERR_NOT_ENOUGH_PARAMS);

    if (*target == '@' && ischannel(target[1]))
	target++; /* Hybrid 6 feature, send notice to all ops in channel */

    g_strup(ctcpcmd);
    printformat(server, target, MSGLEVEL_NOTICES, IRCTXT_OWN_NOTICE, target, ctcpcmd, ctcpdata);

    g_free(params);
    return TRUE;
}

static gboolean cmd_banstat(gchar *data, SERVER_REC *server, CHANNEL_REC *cur_channel)
{
    CHANNEL_REC *channel;
    GList *tmp;

    g_return_val_if_fail(data != NULL, FALSE);
    if (server == NULL || !server->connected) cmd_return_error(CMDERR_NOT_CONNECTED);

    if (strcmp(data, "*") == 0 || *data == '\0')
        channel = cur_channel;
    else
    {
        channel = channel_find(server, data);
        if (channel == NULL)
        {
            /* not joined to such channel, but ask ban lists from server */
            GString *str;

            str = g_string_new(NULL);
            g_string_sprintf(str, "%s b", data);
            signal_emit("command mode", 3, str->str, server, cur_channel);
            g_string_sprintf(str, "%s e", data);
            signal_emit("command mode", 3, str->str, server, cur_channel);
            g_string_free(str, TRUE);
            return TRUE;
        }
    }

    if (channel == NULL) cmd_return_error(CMDERR_CHAN_NOT_FOUND);
    if (channel->type != CHANNEL_TYPE_CHANNEL) cmd_return_error(CMDERR_NOT_JOINED);

    /* show bans.. */
    for (tmp = g_list_first(channel->banlist); tmp != NULL; tmp = tmp->next)
    {
        BAN_REC *rec;

        rec = (BAN_REC *) tmp->data;
        if (*rec->setby == '\0')
            printformat(server, channel->name, MSGLEVEL_CRAP, IRCTXT_BANLIST, channel->name, rec->ban);
        else
            printformat(server, channel->name, MSGLEVEL_CRAP, IRCTXT_BANLIST,
                        channel->name, rec->ban, rec->setby, (gint) (time(NULL)-rec->time));
    }

    /* ..and show ban exceptions.. */
    for (tmp = g_list_first(channel->ebanlist); tmp != NULL; tmp = tmp->next)
    {
        BAN_REC *rec;

        rec = (BAN_REC *) tmp->data;
        if (*rec->setby == '\0')
            printformat(server, channel->name, MSGLEVEL_CRAP, IRCTXT_EBANLIST, channel->name, rec->ban);
        else
            printformat(server, channel->name, MSGLEVEL_CRAP, IRCTXT_EBANLIST,
                        channel->name, rec->ban, rec->setby, (gint) (time(NULL)-rec->time));
    }

    return TRUE;
}

static gboolean cmd_invitelist(gchar *data, SERVER_REC *server, CHANNEL_REC *cur_channel)
{
    CHANNEL_REC *channel;
    GList *tmp;

    g_return_val_if_fail(data != NULL, FALSE);
    if (server == NULL || !server->connected) cmd_return_error(CMDERR_NOT_CONNECTED);

    if (strcmp(data, "*") == 0 || *data == '\0')
        channel = cur_channel;
    else
        channel = channel_find(server, data);
    if (channel == NULL) cmd_return_error(CMDERR_CHAN_NOT_FOUND);
    if (channel->type != CHANNEL_TYPE_CHANNEL) cmd_return_error(CMDERR_NOT_JOINED);

    for (tmp = g_list_first(channel->invitelist); tmp != NULL; tmp = tmp->next)
        printformat(server, channel->name, MSGLEVEL_CRAP, IRCTXT_INVITELIST, channel->name, tmp->data);

    return TRUE;
}

static gboolean cmd_echo(gchar *data, SERVER_REC *server, CHANNEL_REC *channel)
{
    g_return_val_if_fail(data != NULL, FALSE);

    printtext(server, channel == NULL ? NULL : channel->name, MSGLEVEL_CRAP, "%s", data);
    return TRUE;
}

static void cmd_log_create(gchar *fname, gchar *spec)
{
    LOG_REC *log;
    gboolean opened;

    log = log_create(fname, spec);
    if (log == NULL || log->handle != -1) return;

    /* need to open the log file */
    opened = log_file_open(log);
    printformat(NULL, NULL, MSGLEVEL_CLIENTNOTICE,
                opened ? IRCTXT_LOG_OPENED :
                IRCTXT_LOG_OPEN_FAILED, fname);
    if (!opened)
        log_file_close(log);
}

static void cmd_log_close(gchar *fname)
{
    LOG_REC *log;

    log = log_file_find(fname);
    if (log == NULL)
        printformat(NULL, NULL, MSGLEVEL_CLIENTERROR, IRCTXT_LOG_NOT_OPEN, fname);
    else
    {
        log_file_destroy(log);
        printformat(NULL, NULL, MSGLEVEL_CLIENTNOTICE, IRCTXT_LOG_CLOSED, fname);
    }
}

static void cmd_log_start(gchar *fname)
{
    LOG_REC *log;

    log = log_file_find(fname);
    if (log != NULL)
    {
        log_file_open(log);
        printformat(NULL, NULL, MSGLEVEL_CLIENTNOTICE, IRCTXT_LOG_OPENED, fname);
    }
}

static void cmd_log_stop(gchar *fname)
{
    LOG_REC *log;

    log = log_file_find(fname);
    if (log == NULL || log->handle == -1)
        printformat(NULL, NULL, MSGLEVEL_CLIENTERROR, IRCTXT_LOG_NOT_OPEN, fname);
    else
    {
        log_file_close(log);
        printformat(NULL, NULL, MSGLEVEL_CLIENTNOTICE, IRCTXT_LOG_CLOSED, fname);
    }
}

static void cmd_log_list(void)
{
    GList *tmp, *sub;

    for (tmp = g_list_first(logs); tmp != NULL; tmp = tmp->next)
    {
        LOG_REC *rec = tmp->data;

        printformat(NULL, NULL, MSGLEVEL_CLIENTNOTICE, IRCTXT_LOG_LISTENTRY,
                    rec->fname, bits2level(rec->level));
        for (sub = g_list_first(rec->items); sub != NULL; sub = sub->next)
        {
            LOG_ITEM_REC *subrec = sub->data;

            printformat(NULL, NULL, MSGLEVEL_CLIENTNOTICE, IRCTXT_LOG_SUBLISTENTRY,
                        subrec->name, bits2level(subrec->level));
        }
    }
}

static gboolean cmd_log(gchar *data)
{
    gchar *params, *cmd, *fname, *spec;

    g_return_val_if_fail(data != NULL, FALSE);

    params = cmd_get_params(data, 3 | PARAM_FLAG_GETREST, &cmd, &fname, &spec);

    if (g_strcasecmp(cmd, "CREATE") == 0)
        cmd_log_create(fname, spec);
    else if (g_strcasecmp(cmd, "CLOSE") == 0)
        cmd_log_close(fname);
    else if (g_strcasecmp(cmd, "START") == 0)
        cmd_log_start(fname);
    else if (g_strcasecmp(cmd, "STOP") == 0)
        cmd_log_stop(fname);
    else if (g_strcasecmp(cmd, "LIST") == 0)
        cmd_log_list();

    g_free(params);
    return TRUE;
}

static gboolean cmd_version(gchar *data, SERVER_REC *server, CHANNEL_REC *channel)
{
    gchar *str;

    g_return_val_if_fail(data != NULL, FALSE);

    if (*data == '\0')
	printtext(NULL, NULL, MSGLEVEL_CLIENTNOTICE, "Client: "PACKAGE" " VERSION);

    str = g_strdup_printf("VERSION %s", data);
    irc_send_cmd(server, str);
    g_free(str);
    return TRUE;
}

static gboolean cmd_sv(gchar *data, SERVER_REC *server, CHANNEL_REC *channel)
{
    gchar *params, *target, *str;
    gchar *sysname, *release;
    struct utsname un;

    g_return_val_if_fail(data != NULL, FALSE);

    if (uname(&un) == 0)
    {
	sysname = un.sysname;
	release = un.release;
    }
    else
    {
	sysname = "??";
	release = "??";
    }

    params = cmd_get_params(data, 1, &target);
    if (*target == '\0' &&
	channel->type != CHANNEL_TYPE_CHANNEL &&
	channel->type != CHANNEL_TYPE_QUERY &&
	channel->type != CHANNEL_TYPE_DCC_CHAT)
	cmd_param_error(CMDERR_NOT_JOINED);

    str = g_strdup_printf("%s "PACKAGE" v"VERSION " running in %s %s - "IRSSI_WEBSITE,
			  *target == '\0' ? channel->name : target, sysname, release);
    signal_emit("command msg", 3, str, server, channel);
    g_free(str);

    g_free(params);
    return TRUE;
}

static gboolean cmd_ver(gchar *data, SERVER_REC *server, CHANNEL_REC *channel)
{
    gchar *params, *target, *str;

    g_return_val_if_fail(data != NULL, FALSE);

    params = cmd_get_params(data, 1, &target);
    if (*target == '\0' &&
	channel->type != CHANNEL_TYPE_CHANNEL &&
	channel->type != CHANNEL_TYPE_QUERY &&
	channel->type != CHANNEL_TYPE_DCC_CHAT)
	cmd_param_error(CMDERR_NOT_JOINED);

    str = g_strdup_printf("%s VERSION", *target == '\0' ? channel->name : target);
    signal_emit("command ctcp", 3, str, server, channel);
    g_free(str);

    g_free(params);
    return TRUE;
}

static gboolean cmd_ts(gchar *data, SERVER_REC *server, CHANNEL_REC *channel)
{
    GList *tmp;

    g_return_val_if_fail(data != NULL, FALSE);

    for (tmp = channels; tmp != NULL; tmp = tmp->next)
    {
	CHANNEL_REC *rec = tmp->data;

	if (rec->type != CHANNEL_TYPE_CHANNEL)
	    continue;

	printformat(NULL, NULL, MSGLEVEL_CLIENTNOTICE, IRCTXT_TOPIC,
		    rec->name, rec->topic == NULL ? "" : rec->topic);
    }

    return TRUE;
}

static gboolean cmd_unknown(gchar *data, SERVER_REC *server, CHANNEL_REC *channel)
{
    data = g_strdup(data); g_strup(data);
    printtext(server, channel == NULL ? NULL : channel->name, MSGLEVEL_CRAP, "Unknown command: %s", data);
    g_free(data);
    return FALSE;
}

static gboolean event_cmderror(gpointer error)
{
    printtext(NULL, NULL, MSGLEVEL_CLIENTERROR, ret_texts[GPOINTER_TO_INT(error)]);
    return TRUE;
}

void ui_commands_init(void)
{
    tmpstr = g_string_new(NULL);

    command_bind("help", NULL, (SIGNAL_FUNC) cmd_help);
    command_bind("servers", NULL, (SIGNAL_FUNC) cmd_servers);
    command_bind("query", NULL, (SIGNAL_FUNC) cmd_query);
    command_bind("unquery", NULL, (SIGNAL_FUNC) cmd_unquery);
    command_bind("msg", NULL, (SIGNAL_FUNC) cmd_msg);
    command_bind("notice", NULL, (SIGNAL_FUNC) cmd_notice);
    command_bind("me", NULL, (SIGNAL_FUNC) cmd_me);
    command_bind("action", NULL, (SIGNAL_FUNC) cmd_action);
    command_bind("ctcp", NULL, (SIGNAL_FUNC) cmd_ctcp);
    command_bind("nctcp", NULL, (SIGNAL_FUNC) cmd_nctcp);
    command_bind("banstat", NULL, (SIGNAL_FUNC) cmd_banstat);
    command_bind("invitelist", NULL, (SIGNAL_FUNC) cmd_invitelist);
    command_bind("echo", NULL, (SIGNAL_FUNC) cmd_echo);
    command_bind("log", NULL, (SIGNAL_FUNC) cmd_log);
    command_bind("version", NULL, (SIGNAL_FUNC) cmd_version);
    command_bind("sv", NULL, (SIGNAL_FUNC) cmd_sv);
    command_bind("ver", NULL, (SIGNAL_FUNC) cmd_ver);
    command_bind("ts", NULL, (SIGNAL_FUNC) cmd_ts);

    signal_add("unknown command", (SIGNAL_FUNC) cmd_unknown);
    signal_add("error command", (SIGNAL_FUNC) event_cmderror);
    /*signal_add("default command", (SIGNAL_FUNC) cmd_unknown);*/
}

void ui_commands_deinit(void)
{
    command_unbind("help", (SIGNAL_FUNC) cmd_help);
    command_unbind("servers", (SIGNAL_FUNC) cmd_servers);
    command_unbind("query", (SIGNAL_FUNC) cmd_query);
    command_unbind("unquery", (SIGNAL_FUNC) cmd_unquery);
    command_unbind("msg", (SIGNAL_FUNC) cmd_msg);
    command_unbind("notice", (SIGNAL_FUNC) cmd_notice);
    command_unbind("me", (SIGNAL_FUNC) cmd_me);
    command_unbind("action", (SIGNAL_FUNC) cmd_action);
    command_unbind("ctcp", (SIGNAL_FUNC) cmd_ctcp);
    command_unbind("nctcp", (SIGNAL_FUNC) cmd_nctcp);
    command_unbind("banstat", (SIGNAL_FUNC) cmd_banstat);
    command_unbind("invitelist", (SIGNAL_FUNC) cmd_invitelist);
    command_unbind("echo", (SIGNAL_FUNC) cmd_echo);
    command_unbind("log", (SIGNAL_FUNC) cmd_log);
    command_unbind("version", (SIGNAL_FUNC) cmd_version);
    command_unbind("sv", (SIGNAL_FUNC) cmd_sv);
    command_unbind("ver", (SIGNAL_FUNC) cmd_ver);
    command_unbind("ts", (SIGNAL_FUNC) cmd_ts);

    signal_remove("unknown command", (SIGNAL_FUNC) cmd_unknown);
    signal_remove("error command", (SIGNAL_FUNC) event_cmderror);
    /*signal_remove("default command", (SIGNAL_FUNC) cmd_unknown);*/

    g_string_free(tmpstr, TRUE);
}
