/*
 mainloop.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "../common.h"
#include "../irc-base/irc-base.h"

#define MIN_TIMEOUT 50
#define MAX_TIMEOUT 10000

typedef struct
{
    gint handle;
    gint mode;
    gint tag;
    GUIInputFunction func;
    gpointer data;
}
INPUT_REC;

typedef struct
{
    gint ms;
    struct timeval last;
    gint tag;
    GUITimeoutFunction func;
    gpointer data;
}
TIMEOUT_REC;

static GSList *inputlist, *timelist;
static GSList *run_inputlist, *run_timelist;
static gint maxhandle, mintimeout, incount, timecount;

static gboolean quit_program;

gint gui_input_add(gint handle, GUIInputCondition condition,
                   GUIInputFunction function, gpointer data)
{
    INPUT_REC *inp;
#ifdef MEM_DEBUG
    gchar str[100];

    sprintf(str, "handle = %d cond = %d data = %p", handle, condition, data);
    ig_set_data(str);
#endif
    inp = g_new(INPUT_REC, 1);
#ifdef MEM_DEBUG
    ig_set_data("");
#endif
    inp->handle = handle;
    inp->mode = condition;
    inp->func = function;
    inp->data = data;
    inp->tag = ++incount;
    if (handle > maxhandle) maxhandle = handle;

    inputlist = g_slist_append(inputlist, inp);
    return incount;
}

void gui_input_remove(gint tag)
{
    GSList *tmp, *rec;
    gint maxhandle;

    rec = NULL; maxhandle = -1;
    for (tmp = inputlist; tmp != NULL; tmp = tmp->next)
    {
        INPUT_REC *inp = tmp->data;

        if (inp->tag == tag)
            rec = tmp;
        else if (inp->handle > maxhandle)
            maxhandle = inp->handle;
    }

    if (rec != NULL)
    {
        g_free(rec->data);
        run_inputlist = g_slist_remove(run_inputlist, rec->data);
        inputlist = g_slist_remove(inputlist, rec->data);
    }
}

guint gui_timeout_add(guint32 interval, GUITimeoutFunction function, gpointer data)
{
    TIMEOUT_REC *t;
#ifdef MEM_DEBUG
    gchar str[100];

    sprintf(str, "interval = %d data = %p", interval, data);
    ig_set_data(str);
#endif
    t = g_new0(TIMEOUT_REC, 1);
#ifdef MEM_DEBUG
    ig_set_data("");
#endif
    if (gettimeofday(&t->last, NULL) != 0)
        g_error("mainloop() : gettimeofday() failed\n");

    t->ms = interval;
    t->func = function;
    t->data = data;
    t->tag = ++timecount;
    if (interval < mintimeout && interval >= MIN_TIMEOUT) mintimeout = interval;
    timelist = g_slist_append(timelist, t);
    return timecount;
}

void gui_timeout_remove(gint tag)
{
    GSList *tmp, *rec;

    rec = NULL; mintimeout = MIN_TIMEOUT;
    for (tmp = timelist; tmp != NULL; tmp = tmp->next)
    {
        TIMEOUT_REC *t = tmp->data;

        if (t->tag == tag)
            rec = tmp;
        else if (t->ms < mintimeout)
            mintimeout = t->ms;
    }

    if (rec != NULL)
    {
        g_free(rec->data);
        timelist = g_slist_remove(timelist, rec->data);
        run_timelist = g_slist_remove(run_timelist, rec->data);
        timecount--;
    }
}

void main_loop(void)
{
    GSList *tmp;
    struct timeval tv;
    fd_set infd, outfd, exfd;
    long secs, usecs;

    while (!quit_program)
    {
        FD_ZERO(&infd); FD_ZERO(&outfd); FD_ZERO(&exfd);
        for (tmp = inputlist; tmp != NULL; tmp = tmp->next)
        {
            INPUT_REC *inp;

            inp = (INPUT_REC *) tmp->data;

            if (inp->mode & GUI_INPUT_READ)
                FD_SET(inp->handle, &infd);
            if (inp->mode & GUI_INPUT_WRITE)
                FD_SET(inp->handle, &outfd);
            if (inp->mode & GUI_INPUT_EXCEPTION)
                FD_SET(inp->handle, &exfd);
        }

        tv.tv_sec = mintimeout/1000;
        tv.tv_usec = (mintimeout%1000)*1000;
        if (select(maxhandle+1, &infd, &outfd, NULL, &tv) > 0)
        {
            if (quit_program) break;

            for (tmp = inputlist; tmp != NULL; tmp = tmp->next)
                run_inputlist = g_slist_append(run_inputlist, tmp->data);

            while (run_inputlist != NULL)
            {
                INPUT_REC *inp = run_inputlist->data;

                run_inputlist = g_slist_remove(run_inputlist, inp);
                if ((inp->mode & GUI_INPUT_READ && FD_ISSET(inp->handle, &infd)) ||
                    (inp->mode & GUI_INPUT_WRITE && FD_ISSET(inp->handle, &outfd)) ||
                    (inp->mode & GUI_INPUT_EXCEPTION && FD_ISSET(inp->handle, &exfd)))
                {
                    inp->func(inp->data, inp->handle, inp->mode);
                    if (quit_program) break;
                }
            }
        }
        if (quit_program) break;

        for (tmp = timelist; tmp != NULL; tmp = tmp->next)
            run_timelist = g_slist_append(run_timelist, tmp->data);

        if (gettimeofday(&tv, NULL) != 0)
            g_error("mainloop() : gettimeofday() failed\n");

        while (run_timelist != NULL)
        {
            TIMEOUT_REC *t = run_timelist->data;

            run_timelist = g_slist_remove(run_timelist, t);

            secs = tv.tv_sec - t->last.tv_sec;
            usecs = tv.tv_usec - t->last.tv_usec;
            if (usecs < 0)
            {
                usecs += 1000000;
                secs--;
            }

            if (t->ms < secs*1000+usecs/1000)
            {
                if (gettimeofday(&t->last, NULL) != 0)
                    g_error("mainloop() : gettimeofday() failed\n");

                t->func(t->data);
                if (quit_program) break;
            }
        }
    }

    quit_program = FALSE;
}

static gboolean gui_exit(void)
{
    quit_program = TRUE;
    return TRUE;
}

void mainloop_first_init(void)
{
    inputlist = timelist = NULL;
    run_inputlist = run_timelist = NULL;
    maxhandle = -1; mintimeout = MAX_TIMEOUT;
    incount = 0; timecount = 0;
    quit_program = FALSE;
}

void mainloop_init(void)
{
    signal_add("gui exit", (SIGNAL_FUNC) gui_exit);
}

void mainloop_deinit(void)
{
    signal_remove("gui exit", (SIGNAL_FUNC) gui_exit);

    g_slist_free(run_inputlist);
    g_slist_free(run_timelist);

    g_slist_foreach(inputlist, (GFunc) g_free, NULL);
    g_slist_free(inputlist);

    g_slist_foreach(timelist, (GFunc) g_free, NULL);
    g_slist_free(timelist);

    inputlist = NULL;
    timelist = NULL;
}
