/*
 channels-setup.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"

GList *setupchannels;

SETUP_CHANNEL_REC *channels_setup_find(gchar *channel, SERVER_REC *server)
{
    GList *tmp;

    g_return_val_if_fail(channel != NULL, NULL);
    g_return_val_if_fail(server != NULL, NULL);

    for (tmp = setupchannels; tmp != NULL; tmp = tmp->next)
    {
        SETUP_CHANNEL_REC *rec = tmp->data;

        if (g_strcasecmp(rec->name, channel) == 0 &&
            (*rec->ircnet == '\0' || (server->ircnet != NULL && g_strcasecmp(rec->ircnet, server->ircnet) == 0)))
        {
            return rec;
        }
    }

    return NULL;
}

void channels_setup_destroy(SETUP_CHANNEL_REC *channel)
{
    g_return_if_fail(channel != NULL);

    g_free(channel->name);
    g_free(channel->ircnet);
    if (channel->password != NULL) g_free(channel->password);
    if (channel->botmasks != NULL) g_free(channel->botmasks);
    if (channel->autosendcmd != NULL) g_free(channel->autosendcmd);
    if (channel->background != NULL) g_free(channel->background);
    if (channel->font != NULL) g_free(channel->font);
    g_free(channel);

    setupchannels = g_list_remove(setupchannels, channel);
}

/* connected to server, autojoin to channels. */
static gboolean event_connected(SERVER_REC *server)
{
    GString *str;
    GList *tmp;

    g_return_val_if_fail(server != NULL, FALSE);

    if (server->autojoin_channels)
    {
	signal_emit("command join", 4, server->autojoin_channels, server,
		    cur_channel, GINT_TO_POINTER(TRUE));
	return TRUE;
    }

    /* join to the channels marked with autojoin in setup */
    str = g_string_new(NULL);
    for (tmp = setupchannels; tmp != NULL; tmp = tmp->next)
    {
        SETUP_CHANNEL_REC *rec = tmp->data;

        if (rec->autojoin && (*rec->ircnet == '\0' || (server->ircnet != NULL && g_strcasecmp(rec->ircnet, server->ircnet) == 0)))
        {
	    if (str->len != 0)
		g_string_append_c(str, ',');

	    if (rec->password != NULL)
		g_string_sprintfa(str, "%s %s", rec->name, rec->password);
	    else
		g_string_append(str, rec->name);
        }
    }
    if (str->len != 0)
	signal_emit("command join", 4, str->str, server, cur_channel, GINT_TO_POINTER(TRUE));
    g_string_free(str, TRUE);
    return TRUE;
}

/* channel wholist received: send the auto send command */
static gboolean channel_wholist(CHANNEL_REC *channel)
{
    SETUP_CHANNEL_REC *rec;
    NICK_REC *nick;
    GList *tmp, *bots;

    g_return_val_if_fail(channel != NULL, FALSE);

    rec = channels_setup_find(channel->name, channel->server);
    if (rec != NULL && rec->botmasks != NULL && *rec->botmasks != '\0')
    {
        /* find first available bot.. */
	bots = str2list(rec->botmasks, ' ');
        for (tmp = bots; tmp != NULL; tmp = tmp->next)
        {
            nick = nicklist_find(channel, tmp->data);
            if (nick != NULL)
            {
                /* got one! */
                gchar *str;

                str = g_strdup_printf(rec->autosendcmd, nick->nick);
                signal_emit("send command", 3, str, channel->server, channel);
                g_free(str);
                break;
            }
        }
	if (bots != NULL)
	{
	    g_free(bots->data);
	    g_list_free(bots);
	}
    }
    else if (rec != NULL && rec->autosendcmd != NULL && *rec->autosendcmd != '\0')
    {
        /* just send the command. */
	signal_emit("send command", 3, rec->autosendcmd, channel->server, channel);
    }

    return TRUE;
}

void channels_setup_init(void)
{
    signal_add("event connected", (SIGNAL_FUNC) event_connected);
    signal_add("channel wholist", (SIGNAL_FUNC) channel_wholist);
}

void channels_setup_deinit(void)
{
    while (setupchannels != NULL)
        channels_setup_destroy(setupchannels->data);

    signal_remove("event connected", (SIGNAL_FUNC) event_connected);
    signal_remove("channel wholist", (SIGNAL_FUNC) channel_wholist);
}
