/*
 setup.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"
#include "setup-int.h"

#include "setup-appearance.h"
#include "setup-lookandfeel.h"
#include "setup-completion.h"
#include "setup-flood.h"
#include "setup-history.h"
#include "setup-misc.h"
#include "setup-dcc.h"

#include "setup-colors.h"
#include "setup-txt.h"
#include "setup-themes.h"
#include "setup-bg.h"

#include "setup-servers.h"
#include "setup-ircnets.h"
#include "setup-ircproxy.h"
#include "setup-session.h"
#include "setup-lists.h"
#include "setup-keyboard.h"

GdkColor setup_colors[100+EXTRA_COLORS];
GdkFont *font_normal, *font_bold;

gchar *font_name;

static void sig_close(GtkWidget *dialog)
{
    GtkToggleButton *button;
    proplist_t aprop;

    button = gtk_object_get_data(GTK_OBJECT(dialog), "button");

    aprop = config_section(&cprop, "appearance");
    config_set_bool(aprop, "toggle_buggy_gtkthemes", gtk_toggle_button_get_active(button));
    if (!readonly) PLSave(cprop, TRUE);

    gtk_widget_destroy(dialog);
}

static gboolean sig_ask_gtkthemes(void)
{
    GtkWidget *dialog, *label, *button;

    signal_remove("irssi init finished", (SIGNAL_FUNC) sig_ask_gtkthemes);

    dialog = gnome_dialog_new(PACKAGE, GNOME_STOCK_BUTTON_OK, NULL);
    gtk_window_set_policy(GTK_WINDOW(dialog), FALSE, TRUE, FALSE);
    gtk_signal_connect(GTK_OBJECT(dialog), "delete_event",
		       GTK_SIGNAL_FUNC(sig_close), NULL);

    label = gtk_label_new(_("GTK pixmap themes are a bit buggy, if you really want to\n"
			    "use them, you should enable this workaround. Without it\n"
			    "X server's memory usage grows all the time. If you enable\n"
			    "it, you won't see the nicks' gone status in nicklist"));
    gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), label, TRUE, TRUE, 0);

    button = gtk_check_button_new_with_label(_("Enable workaround"));
    gtk_object_set_data(GTK_OBJECT(dialog), "button", button);
    gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), button, FALSE, FALSE, 0);

    gnome_dialog_button_connect_object(GNOME_DIALOG(dialog), 0, GTK_SIGNAL_FUNC(sig_close), GTK_OBJECT(dialog));
    gtk_widget_show_all(dialog);

    return TRUE;
}

void setup_init(void)
{
    proplist_t aprop;
    
    settings_init();
    init_lists();

    /* servers dialog */
    init_ircproxy();
    init_ircnets();
    init_servers();

    /* themes dialog */
    init_themes();
    init_txt();
    init_bg();
    init_colors();

    init_session();
    init_keyboard();

    /* preferences */
    init_misc();
    init_history();
    init_flood();
    init_completion();
    init_dcc();
    init_appearance();
    init_lookandfeel();

    aprop = config_get_prop(cprop, "appearance");
    if (aprop == NULL || config_get_prop(aprop, "toggle_buggy_gtkthemes") == NULL)
	signal_add("irssi init finished", (SIGNAL_FUNC) sig_ask_gtkthemes);
}

void setup_deinit(void)
{
    deinit_lists();

    deinit_servers();
    deinit_ircnets();
    deinit_ircproxy();

    deinit_themes();
    deinit_txt();
    deinit_bg();
    deinit_colors();

    deinit_session();
    deinit_keyboard();

    deinit_misc();
    deinit_history();
    deinit_flood();
    deinit_completion();
    deinit_appearance();
    deinit_dcc();
    deinit_lookandfeel();
}

static void sig_setup_apply(GtkWidget *dialog, gint page)
{
    gchar *str, *signal, *data;

    if (page != -1) return;

    signal = gtk_object_get_data(GTK_OBJECT(dialog), "signal");
    data = gtk_object_get_data(GTK_OBJECT(dialog), "setup_data");

    str = g_strconcat(signal, " apply", NULL);
    signal_emit(str, 2, dialog, data);
    g_free(str);

    if (!readonly) PLSave(cprop, TRUE);

    signal_emit("setup changed", 0);
}

static void sig_setup_destroyed(GtkWidget *widget, GtkWidget **dialog)
{
    gchar *str, *signal, *data;

    signal = gtk_object_get_data(GTK_OBJECT(*dialog), "signal");
    data = gtk_object_get_data(GTK_OBJECT(*dialog), "setup_data");

    str = g_strconcat(signal, " destroyed", NULL);
    signal_emit(str, 2, *dialog, data);
    g_free(str);

    *dialog = NULL;
}

void setup(gchar *signal, GtkWidget **dialog, gpointer data)
{
    if (*dialog != NULL)
    {
        /* Setup already open */
        gdk_window_raise((*dialog)->window);
        return;
    }

    *dialog = gnome_property_box_new();
    gtk_window_set_policy(GTK_WINDOW(*dialog), TRUE, TRUE, FALSE);

    gtk_object_set_data(GTK_OBJECT(*dialog), "signal", signal);
    gtk_object_set_data(GTK_OBJECT(*dialog), "setup_data", data);
    signal_emit(signal, 2, *dialog, data);

#ifdef HAVE_GNOME
    gtk_signal_connect(GTK_OBJECT(*dialog), "apply",
                       GTK_SIGNAL_FUNC(sig_setup_apply), data);
#else
    gnome_property_box_set_apply_func(*dialog, (GtkFunction) sig_setup_apply, *dialog);
#endif
    gtk_signal_connect(GTK_OBJECT(*dialog), "destroy",
                       GTK_SIGNAL_FUNC(sig_setup_destroyed), dialog);

    gtk_widget_show_all(*dialog);
}

void setup_finish_init(GtkWidget *dialog, SETUP_REC *options)
{
    gint num;

    for (num = 0; options[num].tag != NULL; num++)
    {
	if (options[num].widget == NULL) continue;

        switch (options[num].type)
        {
            case SETUP_TYPE_TOGGLEBUTTON:
                gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(options[num].widget), *((gboolean *) options[num].data));
                gtk_signal_connect_object(GTK_OBJECT(options[num].widget), "clicked",
                                          GTK_SIGNAL_FUNC(gnome_property_box_changed), GTK_OBJECT(dialog));
                break;
            case SETUP_TYPE_ENTRY:
                gtk_entry_set_text(GTK_ENTRY(options[num].widget), *((gchar **) options[num].data));
            case SETUP_TYPE_SPIN:
                gtk_signal_connect_object(GTK_OBJECT(options[num].widget), "changed",
                                          GTK_SIGNAL_FUNC(gnome_property_box_changed), GTK_OBJECT(dialog));
                break;
        }
    }
}

void setup_write(gchar *section, SETUP_REC *options)
{
    proplist_t sprop;
    gint num;

    sprop = config_section(&cprop, section);
    for (num = 0; options[num].tag != NULL; num++)
    {
	if (options[num].widget == NULL) continue;

        switch (options[num].type)
        {
            case SETUP_TYPE_TOGGLEBUTTON:
                *((gboolean *) options[num].data) =
                    gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(options[num].widget));
                config_set_bool(sprop, options[num].tag, *((gboolean *) options[num].data));
                break;
            case SETUP_TYPE_SPIN:
                *((gint *) options[num].data) =
                    gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(options[num].widget));
                config_set_int(sprop, options[num].tag, *((gint *) options[num].data));
                break;
            case SETUP_TYPE_INT_OBJECT:
                *((gint *) options[num].data) =
                    GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(options[num].widget), "value"));
                config_set_int(sprop, options[num].tag, *((gint *) options[num].data));
                break;
            case SETUP_TYPE_ENTRY:
                if (*((gchar **) options[num].data) != NULL)
                    g_free(*((gchar **) options[num].data));
                *((gchar **) options[num].data) =
                    g_strdup(gtk_entry_get_text(GTK_ENTRY(options[num].widget)));
                config_set_str(sprop, options[num].tag, *((gchar **) options[num].data));
                break;
        }
    }
}

/* signal: Allow pressing Ctrl-b/c/g/v/- in entry */
gint setup_sig_keypress(GtkEntry *entry, GdkEventKey *event)
{
    gint pos;
    gchar c;

    if ((event->state & GDK_CONTROL_MASK) == 0)
	return 0;

    c = 0;
    switch (event->keyval)
    {
	case 'b': /* bold */
	    c = 2;
	    break;
	case 'c': /* color */
	    c = 3;
	    break;
	case 'g': /* bell */
	    c = 7;
	    break;
	case 'v': /* reverse */
	    c = 22;
	    break;
	case '-': /* underline */
	    c = 31;
	    break;
    }

    if (c == 0)
	return 0;

    pos = gtk_editable_get_position(GTK_EDITABLE(entry));
    gtk_editable_insert_text(GTK_EDITABLE(entry), &c, 1, &pos);
    gtk_editable_set_position(GTK_EDITABLE(entry), pos);

    gtk_signal_emit_stop_by_name(GTK_OBJECT(entry), "key_press_event");
    return 1;
}

