/*
 window-menus.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"
#include "gui-menu-commands.h"
#include "gui-nicklist-popup.h"
#include "wordclick.h"

static GList *servermenus;

static void sig_change_server(GtkWidget *menuitem, SERVER_REC *server)
{
    MAIN_WINDOW_REC *window;

    g_return_if_fail(menuitem != NULL);

    window = gtk_object_get_data(GTK_OBJECT(menuitem->parent), "window");
    if (window == NULL)
    {
	/* some unknown server change somewhere.. */
	signal_emit("servermenu changed", 2, gtk_object_get_data(GTK_OBJECT(menuitem->parent), "servermenu"), server);
    }
    else
    {
	/* window's server changed */
	window->active->active->server = server;
	signal_emit("channel server changed", 1, window->active->active);
    }
}

void gui_servermenu_create(GtkWidget *servermenu, MAIN_WINDOW_REC *window)
{
    GList *tmp;
    GtkWidget *menu, *menuitem;

    g_return_if_fail(servermenu != NULL);

    menu = gtk_menu_new();
    gtk_object_set_data(GTK_OBJECT(menu), "window", window);
    gtk_object_set_data(GTK_OBJECT(menu), "servermenu", servermenu);
    gtk_option_menu_set_menu(GTK_OPTION_MENU(servermenu), menu);

    servermenus = g_list_append(servermenus, servermenu);
    for (tmp = g_list_first(servers); tmp != NULL; tmp = tmp->next)
    {
        SERVER_REC *server = tmp->data;

        menuitem = gtk_menu_item_new_with_label(server->tag);
	gtk_object_set_data(GTK_OBJECT(menuitem), "server", server);
        gtk_signal_connect(GTK_OBJECT(menuitem), "activate",
                           GTK_SIGNAL_FUNC(sig_change_server), server);
        gtk_widget_show(menuitem);
        gtk_menu_append(GTK_MENU(menu), menuitem);
    }
}

void gui_servermenu_destroy(GtkWidget *servermenu)
{
    servermenus = g_list_remove(servermenus, servermenu);
}

static gboolean gui_servermenu_add(SERVER_REC *server)
{
    GList *tmp;
    GtkWidget *menu, *menuitem;
    MAIN_WINDOW_REC *window;

    g_return_val_if_fail(server != NULL, FALSE);

    /* Add server to each server menu */
    for (tmp = servermenus; tmp != NULL; tmp = tmp->next)
    {
	GtkWidget *servermenu = tmp->data;

        menu = gtk_option_menu_get_menu(GTK_OPTION_MENU(servermenu));

	menuitem = gtk_menu_item_new_with_label(server->tag);
	gtk_object_set_data(GTK_OBJECT(menuitem), "server", server);
        gtk_signal_connect(GTK_OBJECT(menuitem), "activate",
                           GTK_SIGNAL_FUNC(sig_change_server), server);
        gtk_widget_show(menuitem);
        gtk_menu_append(GTK_MENU(menu), menuitem);

	window = gtk_object_get_data(GTK_OBJECT(menu), "window");
        if (window != NULL)
            gtk_option_menu_set_history(GTK_OPTION_MENU(servermenu), g_list_index(servers, window->active->active->server));
    }

    return TRUE;
}

static gboolean gui_servermenu_remove(SERVER_REC *server)
{
    GList *tmp, *menus;
    GtkWidget *menu;

    g_return_val_if_fail(server != NULL, FALSE);

    /* Remove server from all server menus */
    for (tmp = servermenus; tmp != NULL; tmp = tmp->next)
    {
	menu = gtk_option_menu_get_menu(GTK_OPTION_MENU(tmp->data));
	for (menus = GTK_MENU_SHELL(menu)->children; menus != NULL; menus = menus->next)
	{
	    if (server == gtk_object_get_data(GTK_OBJECT(menus->data), "server"))
	    {
		/* This is it, remove it. */
		gtk_container_remove(GTK_CONTAINER(menu), menus->data);
		gtk_widget_queue_resize(menu);
		break;
	    }
	}
        /*menu = gtk_option_menu_get_menu(GTK_OPTION_MENU(servermenu));
        data = gtk_object_get_data(GTK_OBJECT(menu), "window");

        menu = gtk_menu_new();
        gtk_object_set_data(GTK_OBJECT(menu), "window", data);
        gui_servermenu_create(menu);
        gtk_option_menu_remove_menu(GTK_OPTION_MENU(servermenu));
        gtk_option_menu_set_menu(GTK_OPTION_MENU(servermenu), menu);*/
    }

    return TRUE;
}

/* add menuitem to menu */
GtkWidget *popup_add(GtkWidget *menu, gchar *label, gpointer func, gpointer data)
{
    GtkWidget *menu_item;

    g_return_val_if_fail(menu != NULL, NULL);

    if (label != NULL)
        menu_item = gtk_menu_item_new_with_label(label);
    else
        menu_item = gtk_menu_item_new();

    gtk_menu_append(GTK_MENU (menu), menu_item);

    if (func != NULL)
    {
	gtk_object_set_data(GTK_OBJECT(menu_item), "menu", menu);
        gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
                           GTK_SIGNAL_FUNC(func), data);
    }
    else
        gtk_widget_set_sensitive(menu_item, FALSE);

    gtk_widget_show(menu_item);
    return menu_item;
}

/* add submenu to menu */
void popup_add_sub(GtkWidget *menu, gchar *label, GtkWidget *submenu)
{
    GtkWidget *menu_item;

    g_return_if_fail(menu != NULL);
    g_return_if_fail(label != NULL);

    menu_item = gtk_menu_item_new_with_label(label);

    gtk_menu_append(GTK_MENU(menu), menu_item);
    gtk_menu_item_set_submenu(GTK_MENU_ITEM(menu_item), submenu);
    gtk_widget_show(menu_item);
}

void gui_menus_insert(MAIN_WINDOW_REC *window, gchar *path, GnomeUIInfo *menu, gpointer data)
{
    GtkWidget *popup;

    popup = gtk_object_get_data(GTK_OBJECT(window->window), "popup");
    if (popup == NULL)
    {
        gnome_app_insert_menus_with_data(GNOME_APP(window->window),
            path != NULL ? path : LAST_MENU, menu, data);
    }
    else
    {
        GtkWidget *w;
        gint pos;

        w = gnome_app_find_menu_pos(popup, path != NULL ? path : LAST_MENU, &pos);
        gnome_app_fill_menu_with_data(GTK_MENU_SHELL(w), menu, NULL, TRUE, pos, data);
    }
}

void gui_menus_remove(MAIN_WINDOW_REC *window, gchar *path)
{
    GtkWidget *popup;

    popup = gtk_object_get_data(GTK_OBJECT(window->window), "popup");
    if (popup == NULL)
        gnome_app_remove_menus(GNOME_APP(window->window), path, 1);
    else
    {
#ifdef HAVE_GNOME
        GtkWidget *parent;
        GList *children;
        gint pos;

        parent = gnome_app_find_menu_pos(popup, path, &pos);
        if (parent != NULL)
        {
            children = g_list_nth(GTK_MENU_SHELL(parent)->children, pos-1);
            gtk_container_remove(GTK_CONTAINER(parent), children->data);
            gtk_widget_queue_resize(parent);
        }
#else
        gnome_app_remove_from_menu(popup, path, 1);
#endif
    }
}

void gui_init_popupmenu(GtkWidget *menu, CHANNEL_REC *channel, GdkEventButton *event)
{
    g_return_if_fail(channel != NULL);

    switch (channel->type)
    {
        case CHANNEL_TYPE_CHANNEL:
            gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(channel_submenu[2].widget), GTK_WIDGET_VISIBLE(WINDOW_GUI(CHANNEL_PARENT(channel))->nlist_widget));
            break;

        case CHANNEL_TYPE_QUERY:
        case CHANNEL_TYPE_DCC_CHAT:
            gui_nicklist_popup(menu, channel, channel->name, event);
            break;

        case CHANNEL_TYPE_EMPTY:
            if (channel->level & MSGLEVEL_MSGS)
            {
                gchar *word;

                word = gtk_object_get_data(GTK_OBJECT(menu), "word");
                if (word != NULL) gui_nicklist_popup(menu, channel, word, event);
            }
            break;
    }
}

void gui_popupmenu_create(CHANNEL_REC *channel, gchar *word, GdkEventButton *event)
{
    GtkWidget *popup;
    GnomeUIInfo *sub;

    g_return_if_fail(channel != NULL);
    g_return_if_fail(event != NULL);

    switch (channel->type)
    {
        case CHANNEL_TYPE_CHANNEL:
            sub = channel_submenu;
            break;

        case CHANNEL_TYPE_QUERY:
            sub = query_submenu;
            break;

        case CHANNEL_TYPE_DCC_CHAT:
            sub = dccchat_menu;
            break;

        default:
            return;
    }

    popup = gnome_popup_menu_new(sub);
    if (word != NULL)
    {
        word = g_strdup(word);
        gtk_signal_connect_object(GTK_OBJECT(popup), "selection_done",
                                  GTK_SIGNAL_FUNC(g_free), (GtkObject *) word);
    }
    popup_add(popup, NULL, NULL, NULL);

    gtk_object_set_data(GTK_OBJECT(popup), "word", word);
    gtk_object_set_data(GTK_OBJECT(popup), "channel", channel);

    if (word != NULL)
        word_clicked(CHANNEL_PARENT(channel), word, popup, event);

    gui_init_popupmenu(popup, channel, event);
    gnome_popup_menu_do_popup(popup, NULL, NULL, event, channel);
}

static gboolean gui_server_changed(CHANNEL_REC *channel)
{
    MAIN_WINDOW_REC *mainwin;

    g_return_val_if_fail(channel != NULL, FALSE);

    mainwin = WINDOW_GUI(CHANNEL_PARENT(channel))->parent;
    if (mainwin->active->active == channel)
    {
	gtk_option_menu_set_history(GTK_OPTION_MENU(mainwin->servermenu),
				    g_list_index(servers, channel->server));
    }

    return TRUE;
}

static gboolean signal_window_focused(WINDOW_REC *window)
{
    CHANNEL_REC *channel;

    g_return_val_if_fail(window != NULL, FALSE);

    channel = window->active;
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(WINDOW_GUI(window)->parent->autoraise),
                                   channel == NULL ? FALSE : CHANNEL_GUI(channel)->autoraise);
    return TRUE;
}

void gui_menus_init(void)
{
    servermenus = NULL;
    signal_add_last("server connected", (SIGNAL_FUNC) gui_servermenu_add);
    signal_add("server disconnected", (SIGNAL_FUNC) gui_servermenu_remove);
    signal_add("channel server changed", (SIGNAL_FUNC) gui_server_changed);
    signal_add("window focused", (SIGNAL_FUNC) signal_window_focused);
}

void gui_menus_deinit(void)
{
    g_list_free(servermenus);

    signal_remove("server connected", (SIGNAL_FUNC) gui_servermenu_add);
    signal_remove("server disconnected", (SIGNAL_FUNC) gui_servermenu_remove);
    signal_remove("channel server changed", (SIGNAL_FUNC) gui_server_changed);
    signal_remove("window focused", (SIGNAL_FUNC) signal_window_focused);
}
