/*
 gui-menu-commands.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"

#include "dialog-about.h"
#include "dialog-connect.h"
#include "dialog-disconnect.h"
#include "dialog-channels.h"
#include "dialog-dcc.h"
#include "dialog-links.h"
#include "dialog-log.h"
#include "dialog-modes.h"
#include "dialog-notifylist.h"
#include "dialog-plugins.h"
#include "dialog-rawlog.h"
#include "dialog-status.h"
#include "dialog-textwidget.h"
#include "dialog-window.h"

#include "gui-dcc.h"

#include "setup-servers.h"
#include "setup-lists.h"
#include "setup-keyboard.h"

static void menu_quit(void);

static void menu_edit_cut(GtkWidget *menuitem, MAIN_WINDOW_REC *window);
static void menu_edit_copy(GtkWidget *menuitem, MAIN_WINDOW_REC *window);
static void menu_edit_paste(GtkWidget *menuitem, MAIN_WINDOW_REC *window);

static void menu_chan_part(GtkWidget *menuitem, CHANNEL_REC *channel);
static void menu_chan_modes(GtkWidget *menuitem, CHANNEL_REC *channel);
static void menu_chan_toggle_nicklist(GtkWidget *menuitem, CHANNEL_REC *channel);
static void menu_query_close(GtkWidget *menuitem, CHANNEL_REC *channel);
static void menu_dcc_mirc_ctcps(GtkWidget *menuitem, CHANNEL_REC *channel);

static void menu_links(GtkWidget *menuitem, MAIN_WINDOW_REC *window);
static void menu_usermode_away(GtkWidget *menuitem, MAIN_WINDOW_REC *window);
static void menu_usermode_invisible(GtkWidget *menuitem, MAIN_WINDOW_REC *window);
static void menu_usermode_wallops(GtkWidget *menuitem, MAIN_WINDOW_REC *window);
static void menu_usermode_snotes(GtkWidget *menuitem, MAIN_WINDOW_REC *window);

static void menu_dcc_chat(GtkWidget *menuitem, MAIN_WINDOW_REC *window);
static void menu_dcc_send(GtkWidget *menuitem, MAIN_WINDOW_REC *window);

static void menu_window_new(GtkWidget *menuitem, MAIN_WINDOW_REC *window);
static void menu_window_new_window(GtkWidget *menuitem, MAIN_WINDOW_REC *window);
static void menu_window_new_tabbed(GtkWidget *menuitem, MAIN_WINDOW_REC *window);
static void menu_window_new_split(GtkWidget *menuitem, MAIN_WINDOW_REC *window);
static void menu_window_auto_raise(GtkWidget *menuitem, MAIN_WINDOW_REC *window);
static void menu_window_close(GtkWidget *menuitem, MAIN_WINDOW_REC *window);

static void menu_settings_preferences(void);
static void menu_settings_servers(void);
static void menu_settings_themes(void);
static void menu_settings_session(void);
static void menu_settings_keyboard(void);

static void menu_irssi_website(void);

GnomeUIInfo file_menu[] = {
    { GNOME_APP_UI_ITEM, N_("_New"), NULL, menu_window_new, NULL, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_NEW, 0, 0, NULL },
    { GNOME_APP_UI_ITEM, N_("_Save Text As..."), NULL, dialog_textwidget_save, NULL, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_SAVE, 0, 0, NULL },
    GNOMEUIINFO_SEPARATOR,
    { GNOME_APP_UI_ITEM, N_("_Close"), NULL, menu_window_close, NULL, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_CLOSE, 0, 0, NULL },
    { GNOME_APP_UI_ITEM, N_("_Quit"), NULL, menu_quit, NULL, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_QUIT, 0, 0, NULL },
    { GNOME_APP_UI_ENDOFINFO }
};

GnomeUIInfo server_menu[] = {
    { GNOME_APP_UI_ITEM, N_("_Connect..."), NULL, dialog_connect, NULL, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_JUMP_TO, 0, 0, NULL },
    { GNOME_APP_UI_ITEM, N_("_Disconnect..."), NULL, dialog_disconnect, NULL, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_CLOSE, 0, 0, NULL },
    { GNOME_APP_UI_ITEM, N_("C_hannels..."), NULL, dialog_channels, NULL, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BOOK_YELLOW, 0, 0, NULL },
    GNOMEUIINFO_SEPARATOR,
    { GNOME_APP_UI_ITEM, N_("_Status Window..."), NULL, dialog_status, NULL, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BOOK_OPEN, 0, 0, NULL },
    { GNOME_APP_UI_ITEM, N_("N_otify list..."), NULL, dialog_notifylist, NULL, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_PREF, 0, 0, NULL },
    { GNOME_APP_UI_ITEM, N_("_Raw Log..."), NULL, dialog_rawlog, NULL, NULL,
      GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL },
    { GNOME_APP_UI_ITEM, N_("_Links..."), NULL, menu_links, NULL, NULL,
      GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL },
    GNOMEUIINFO_SEPARATOR,
#define USERMENU_FIRST 9
    { GNOME_APP_UI_TOGGLEITEM, N_("_Away..."), NULL, menu_usermode_away, NULL, NULL,
      GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL },
    GNOMEUIINFO_SEPARATOR,
    { GNOME_APP_UI_TOGGLEITEM, N_("_Invisible"), NULL, menu_usermode_invisible, NULL, NULL,
      GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL },
    { GNOME_APP_UI_TOGGLEITEM, N_("Receive _Wallops"), NULL, menu_usermode_wallops, NULL, NULL,
      GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL },
    { GNOME_APP_UI_TOGGLEITEM, N_("Receive Server _Notices"), NULL, menu_usermode_snotes, NULL, NULL,
      GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL },
    { GNOME_APP_UI_ENDOFINFO }
};

GnomeUIInfo edit_menu[] = {
    { GNOME_APP_UI_ITEM, N_("C_ut"), NULL, menu_edit_cut, NULL, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_CUT, 0, 0, NULL },
    { GNOME_APP_UI_ITEM, N_("_Copy"), NULL, menu_edit_copy, NULL, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_COPY, 0, 0, NULL },
    { GNOME_APP_UI_ITEM, N_("_Paste"), NULL, menu_edit_paste, NULL, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_PASTE, 0, 0, NULL },
    GNOMEUIINFO_SEPARATOR,
    { GNOME_APP_UI_ITEM, N_("_Find..."), NULL, dialog_textwidget_find, NULL, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_SEARCH, 0, 0, NULL },
    { GNOME_APP_UI_ENDOFINFO }
};

GnomeUIInfo dcc_menu[] = {
    { GNOME_APP_UI_ITEM, N_("C_onnections..."), NULL, dialog_dcc, NULL, NULL,
      GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL },
    GNOMEUIINFO_SEPARATOR,
    { GNOME_APP_UI_ITEM, N_("_Send File..."), NULL, menu_dcc_send, NULL, NULL,
      GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL },
    { GNOME_APP_UI_ITEM, N_("_Chat..."), NULL, menu_dcc_chat, NULL, NULL,
      GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL },
    { GNOME_APP_UI_ENDOFINFO }
};

GnomeUIInfo settings_menu[] = {
    { GNOME_APP_UI_ITEM, N_("_Server List..."), NULL, menu_settings_servers, NULL, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BOOK_BLUE, 0, 0, NULL },
    { GNOME_APP_UI_ITEM, N_("_Themes..."), NULL, menu_settings_themes, NULL, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_COLORSELECTOR, 0, 0, NULL },
    { GNOME_APP_UI_ITEM, N_("_Automation..."), NULL, setup_aliases, NULL, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_EXEC, 0, 0, NULL },
    { GNOME_APP_UI_ITEM, N_("_Logging..."), NULL, dialog_log, NULL, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_PROP, 0, 0, NULL },
    { GNOME_APP_UI_ITEM, N_("Plu_gins..."), NULL, dialog_plugins, NULL, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_INDEX, 0, 0, NULL },
    { GNOME_APP_UI_ITEM, N_("_Keyboard..."), NULL, menu_settings_keyboard, NULL, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_PREF, 0, 0, NULL },
    GNOMEUIINFO_SEPARATOR,
    { GNOME_APP_UI_ITEM, N_("_Preferences..."), NULL, menu_settings_preferences, NULL, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_PREF, 0, 0, NULL },
/*    { GNOME_APP_UI_ITEM, N_("_Session..."), NULL, menu_settings_session, NULL, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_PREF, 0, 0, NULL },*/
    { GNOME_APP_UI_ENDOFINFO }
};

GnomeUIInfo windows_menu[] = {
    { GNOME_APP_UI_ITEM, N_("New _Window"), NULL, menu_window_new_window, NULL, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_NEW, 0, 0, NULL },
    { GNOME_APP_UI_ITEM, N_("_New Tab"), NULL, menu_window_new_tabbed, NULL, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_NEW, 0, 0, NULL },
    { GNOME_APP_UI_ITEM, N_("New _Split Window"), NULL, menu_window_new_split, NULL, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_NEW, 0, 0, NULL },
    GNOMEUIINFO_SEPARATOR,
#define MENU_AUTORAISE_POS 4
    { GNOME_APP_UI_TOGGLEITEM, N_("Automatically _Raise"), NULL, menu_window_auto_raise, NULL, NULL,
      GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL },
    { GNOME_APP_UI_ITEM, N_("_Save Size and Position..."), NULL, dialog_window_save, NULL, NULL,
      GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL },
    GNOMEUIINFO_SEPARATOR,
    { GNOME_APP_UI_ITEM, N_("_Close"), NULL, menu_window_close, NULL, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_CLOSE, 0, 0, NULL },
    { GNOME_APP_UI_ENDOFINFO }
};

GnomeUIInfo help_menu[] = {
#ifdef HAVE_GNOME
    GNOMEUIINFO_HELP("irssi"),
    GNOMEUIINFO_SEPARATOR,
#endif
    { GNOME_APP_UI_ITEM, N_("Irssi _Website"), NULL, menu_irssi_website, NULL, NULL,
      GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL },
    GNOMEUIINFO_SEPARATOR,
#ifdef HAVE_GNOME
    GNOMEUIINFO_MENU_ABOUT_ITEM(dialog_about,NULL),
#else
    { GNOME_APP_UI_ITEM, N_("_About..."), NULL, dialog_about, NULL, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_ABOUT, 0, 0, NULL },
#endif
    { GNOME_APP_UI_ENDOFINFO }
};

GnomeUIInfo main_menu[] = {
    { GNOME_APP_UI_SUBTREE, N_("_File"), NULL, file_menu, NULL, NULL,
      GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL },
    { GNOME_APP_UI_SUBTREE, N_("_Edit"), NULL, edit_menu, NULL, NULL,
      GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL },
    { GNOME_APP_UI_SUBTREE, N_("_DCC"), NULL, dcc_menu, NULL, NULL,
      GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL },
    { GNOME_APP_UI_SUBTREE, N_("_Server"), NULL, server_menu, NULL, NULL,
      GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL },
    { GNOME_APP_UI_SUBTREE, N_("Se_ttings"), NULL, settings_menu, NULL, NULL,
      GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL },
    { GNOME_APP_UI_SUBTREE, N_("_Windows"), NULL, windows_menu, NULL, NULL,
      GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL },
    { GNOME_APP_UI_SUBTREE, N_("_Help"), NULL, help_menu, NULL, NULL,
      GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL },
    { GNOME_APP_UI_ENDOFINFO }
};

/* menus to insert afterwards */
GnomeUIInfo channel_submenu[] = {
    { GNOME_APP_UI_ITEM, N_("_Leave channel"), NULL, menu_chan_part, NULL, NULL,
      0, 0, 0, 0, NULL },
    { GNOME_APP_UI_ITEM, N_("Channel _Modes..."), NULL, menu_chan_modes, NULL, NULL,
      0, 0, 0, 0, NULL },
    { GNOME_APP_UI_TOGGLEITEM, N_("Show _Nick List"), NULL, menu_chan_toggle_nicklist, NULL, NULL,
      0, 0, 0, 0, NULL },
    GNOMEUIINFO_END
};

GnomeUIInfo channel_menu[] = {
    { GNOME_APP_UI_SUBTREE, N_("Channel"), NULL, channel_submenu, NULL, NULL,
      GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL },
    GNOMEUIINFO_END
};

GnomeUIInfo query_submenu[] = {
    { GNOME_APP_UI_ITEM, N_("Close _Query"), NULL, menu_query_close, NULL, NULL,
      0, 0, 0, 0, NULL },
    GNOMEUIINFO_END
};

GnomeUIInfo query_menu[] = {
    { GNOME_APP_UI_SUBTREE, N_("Query"), NULL, query_submenu, NULL, NULL,
      GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL },
    GNOMEUIINFO_END
};

GnomeUIInfo dccchat_menu[] = {
    { GNOME_APP_UI_ITEM, N_("_Close DCC chat"), NULL, menu_query_close, NULL, NULL,
      0, 0, 0, 0, NULL },
    { GNOME_APP_UI_TOGGLEITEM, N_("_Mirc CTCPs"), NULL, menu_dcc_mirc_ctcps, NULL, NULL,
      0, 0, 0, 0, NULL },
    GNOMEUIINFO_END
};

GnomeUIInfo toolbar[] = {
    { GNOME_APP_UI_ITEM, N_("Connect"), NULL, dialog_connect, NULL, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_JUMP_TO, 0, 0, NULL },
    { GNOME_APP_UI_ITEM, N_("Disconnect"), NULL, dialog_disconnect, NULL, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_CLOSE, 0, 0, NULL },
    { GNOME_APP_UI_ITEM, N_("Channels"), NULL, dialog_channels, NULL, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BOOK_YELLOW, 0, 0, NULL },
    { GNOME_APP_UI_ITEM, N_("Status"), NULL, dialog_status, NULL, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BOOK_OPEN, 0, 0, NULL },
    GNOMEUIINFO_SEPARATOR,
    { GNOME_APP_UI_ITEM, N_("Find"), NULL, dialog_textwidget_find, NULL, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_SEARCH, 0, 0, NULL },
    { GNOME_APP_UI_ITEM, N_("New"), NULL, menu_window_new, NULL, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_NEW, 0, 0, NULL },
    { GNOME_APP_UI_ITEM, N_("Close"), NULL, menu_window_close, NULL, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_CLOSE, 0, 0, NULL },
    GNOMEUIINFO_END
};

static void menu_quit(void)
{
    signal_emit("command quit", 1, "");
}

static void menu_edit_cut(GtkWidget *menuitem, MAIN_WINDOW_REC *window)
{
    gtk_editable_cut_clipboard(GTK_EDITABLE(window->entry));
}

static void menu_edit_copy(GtkWidget *menuitem, MAIN_WINDOW_REC *window)
{
    WINDOW_VIEW_REC *view;

    if (GTK_EDITABLE(window->entry)->selection_start_pos !=
	GTK_EDITABLE(window->entry)->selection_end_pos)
    {
	/* if entry line has selection, get it.. */
	gtk_editable_claim_selection(GTK_EDITABLE(window->entry), TRUE, 0);
    }
    else
    {
        /* if not, get the selection from active text widget */
	view = WINDOW_VIEW(window->active);
	if (view->itext)
	    gtk_itext_get_selection(GTK_ITEXT(view->text));
	else
	    gtk_editable_claim_selection(GTK_EDITABLE(view->text), TRUE, 0);
    }
}

static void menu_edit_paste(GtkWidget *menuitem, MAIN_WINDOW_REC *window)
{
    gtk_editable_paste_clipboard(GTK_EDITABLE(window->entry));
}

static void menu_chan_part(GtkWidget *menuitem, CHANNEL_REC *channel)
{
    g_return_if_fail(channel != NULL);

    signal_emit("command part", 3, channel->name, channel->server, channel);
}

static void menu_chan_modes(GtkWidget *menuitem, CHANNEL_REC *channel)
{
    g_return_if_fail(channel != NULL);

    dialog_modes(channel);
}

static void menu_chan_toggle_nicklist(GtkWidget *menuitem, CHANNEL_REC *channel)
{
    GUI_WINDOW_REC *window;

    g_return_if_fail(menuitem != NULL);
    if (channel == NULL) return;

    window = WINDOW_GUI(CHANNEL_PARENT(channel));
    if (window->nlist_widget == NULL)
        return; /* this function was called from window_create().. */

    CHANNEL_GUI(channel)->nicklist = GTK_CHECK_MENU_ITEM(menuitem)->active;
    if (CHANNEL_GUI(channel)->nicklist)
	gtk_widget_show(window->nlist_widget);
    else
	gtk_widget_hide(window->nlist_widget);
}

static void menu_query_close(GtkWidget *menuitem, CHANNEL_REC *channel)
{
    g_return_if_fail(channel != NULL);

    signal_emit("command unquery", 3, channel->name, channel->server, channel);
}

static void menu_dcc_mirc_ctcps(GtkWidget *menuitem, CHANNEL_REC *channel)
{
    DCC_REC *dcc;

    g_return_if_fail(channel != NULL);

    if (channel->type == CHANNEL_TYPE_DCC_CHAT)
    {
        dcc = dcc_find_item(DCC_TYPE_CHAT, channel->name+1, NULL);
        if (dcc != NULL)
        {
            dcc->mirc_ctcp = !dcc->mirc_ctcp;
        }
    }
}

static void sig_away_ok(GtkWidget *widget)
{
    GtkWidget *entry;
    gchar *reason;

    g_return_if_fail(widget != NULL);

    entry = gtk_object_get_data(GTK_OBJECT(widget), "entry");
    g_return_if_fail(entry != NULL);

    reason = gtk_entry_get_text(GTK_ENTRY(entry));
    if (*reason != '\0')
    {
        signal_emit("command awayall", 1, reason);
    }

    gtk_widget_destroy(widget);
}

static void menu_links(GtkWidget *menuitem, MAIN_WINDOW_REC *window)
{
    g_return_if_fail(menuitem != NULL);
    if (window == NULL) return;

    if (window->active->active->server == NULL)
	gui_dialog(DIALOG_ERROR, _("Not connected to server"));
    else
	dialog_links(window->active->active->server);
}

static void menu_usermode_away(GtkWidget *menuitem, MAIN_WINDOW_REC *window)
{
    static gboolean toggling = FALSE;
    SERVER_REC *server;

    g_return_if_fail(menuitem != NULL);
    if (window == NULL) return;

    toggling = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(menuitem), "toggling"));
    if (toggling) return;

    server = window->active->active->server;
    if (server == NULL)
    {
	gui_dialog(DIALOG_ERROR, _("Not connected to server"));
	return;
    }
    gtk_object_set_data(GTK_OBJECT(menuitem), "toggling", GINT_TO_POINTER(TRUE));
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menuitem),
                                  server == NULL ? FALSE : server->usermode_away);
    gtk_object_set_data(GTK_OBJECT(menuitem), "toggling", GINT_TO_POINTER(FALSE));

    if (server != NULL)
    {
        if (server->usermode_away)
            signal_emit("command awayall", 1, "");
        else
            gui_entry_dialog(_("Away reason:"), NULL, "AwayReason", GTK_SIGNAL_FUNC(sig_away_ok), NULL);
    }
}

static void usermode_set(GtkWidget *menuitem, gchar modechar, gboolean mode, CHANNEL_REC *channel)
{
    static gboolean toggling = FALSE;
    GString *str;

    g_return_if_fail(menuitem != NULL);

    toggling = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(menuitem), "toggling"));
    if (toggling) return;

    gtk_object_set_data(GTK_OBJECT(menuitem), "toggling", GINT_TO_POINTER(TRUE));
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menuitem), mode);
    gtk_object_set_data(GTK_OBJECT(menuitem), "toggling", GINT_TO_POINTER(FALSE));

    if (channel->server != NULL)
    {
        str = g_string_new(NULL);
        g_string_sprintf(str, "%s %c%c", channel->server->nick, mode ? '-' : '+', modechar);
        signal_emit("command mode", 3, str->str, channel->server, channel);
        g_string_free(str, TRUE);
    }
}

static void menu_usermode_invisible(GtkWidget *menuitem, MAIN_WINDOW_REC *window)
{
    SERVER_REC *server;

    g_return_if_fail(menuitem != NULL);
    if (window == NULL) return;

    server = window->active->active->server;
    if (server != NULL)
	usermode_set(menuitem, 'i', server == NULL ? FALSE :
		     server->usermode_invisible, window->active->active);
}

static void menu_usermode_wallops(GtkWidget *menuitem, MAIN_WINDOW_REC *window)
{
    SERVER_REC *server;

    g_return_if_fail(menuitem != NULL);
    if (window == NULL) return;

    server = window->active->active->server;
    if (server != NULL)
	usermode_set(menuitem, 'w', server == NULL ? FALSE :
		     server->usermode_wallops, window->active->active);
}

static void menu_usermode_snotes(GtkWidget *menuitem, MAIN_WINDOW_REC *window)
{
    SERVER_REC *server;

    g_return_if_fail(menuitem != NULL);
    if (window == NULL) return;

    server = window->active->active->server;
    if (server != NULL)
	usermode_set(menuitem, 's', server == NULL ? FALSE :
		     server->usermode_servernotes, window->active->active);
}

static void menu_dcc_chat(GtkWidget *menuitem, MAIN_WINDOW_REC *window)
{
    if (window->active->active->server == NULL)
	gui_dialog(DIALOG_ERROR, _("Not connected to server"));
    else
	gui_dcc_chat(window->active->active->server);
}

static void menu_dcc_send(GtkWidget *menuitem, MAIN_WINDOW_REC *window)
{
    if (window->active->active->server == NULL)
	gui_dialog(DIALOG_ERROR, _("Not connected to server"));
    else
	gui_dcc_send(window->active->active->server, NULL, NULL);
}

static void menu_window_new(GtkWidget *menuitem, MAIN_WINDOW_REC *window)
{
    channel_create(window == NULL ? NULL : window->active->active->server,
		   _("(empty)"), CHANNEL_TYPE_EMPTY, FALSE);
}

static void menu_window_new_window(GtkWidget *menuitem, MAIN_WINDOW_REC *window)
{
    signal_emit("gui window create override", 1, GINT_TO_POINTER(0));
    menu_window_new(menuitem, window);
}

static void menu_window_new_tabbed(GtkWidget *menuitem, MAIN_WINDOW_REC *window)
{
    signal_emit("gui window create override", 1, GINT_TO_POINTER(1));
    menu_window_new(menuitem, window);
}

static void menu_window_new_split(GtkWidget *menuitem, MAIN_WINDOW_REC *window)
{
    signal_emit("gui window create override", 1, GINT_TO_POINTER(2));
    menu_window_new(menuitem, window);
}

static void menu_window_auto_raise(GtkWidget *menuitem, MAIN_WINDOW_REC *window)
{
    static gboolean toggling = FALSE;
    GUI_CHANNEL_REC *gui;

    g_return_if_fail(menuitem != NULL);
    if (window == NULL) return;

    toggling = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(menuitem), "toggling"));
    if (toggling) return;

    gui = CHANNEL_GUI(window->active->active);
    gui->autoraise = GTK_CHECK_MENU_ITEM(menuitem)->active; /* FIXME: better way for this? gtk needs a function for this.. */

    gtk_object_set_data(GTK_OBJECT(menuitem), "toggling", GINT_TO_POINTER(TRUE));
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menuitem), gui->autoraise);
    gtk_object_set_data(GTK_OBJECT(menuitem), "toggling", GINT_TO_POINTER(FALSE));
}

static void menu_window_close(GtkWidget *menuitem, MAIN_WINDOW_REC *window)
{
    g_return_if_fail(menuitem != NULL);
    if (window == NULL) return;

    if (channels->next != NULL)
    {
	/* destroy window unless it's the last one */
	ui_window_destroy(window->active);
    }
}

static void menu_settings_preferences(void)
{
    static GtkWidget *dialog = NULL;

    setup("setup preferences", &dialog, NULL);
}

static void menu_settings_servers(void)
{
    static GtkWidget *dialog = NULL;

    setup("setup servers", &dialog, NULL);
}

static void menu_settings_themes(void)
{
    static GtkWidget *dialog = NULL;

    setup("setup themes", &dialog, NULL);
}

#if 0
static void menu_settings_session(void)
{
    static GtkWidget *dialog = NULL;

    setup("setup session", &dialog, NULL);
}
#endif

static void menu_settings_keyboard(void)
{
    static GtkWidget *dialog = NULL;

    setup("setup keyboard", &dialog, NULL);
}

static void menu_irssi_website(void)
{
    gchar *str;

    str = g_strdup_printf(setup_get_str("url_www_client"), IRSSI_WEBSITE);
    execute(str);
    g_free(str);
}

void gui_menus_create(MAIN_WINDOW_REC *window, GnomeUIInfo *menu)
{
    if (setup_get_bool("toggle_show_menubar"))
        gnome_app_create_menus_with_data(GNOME_APP(window->window), menu, window);
    else
    {
        GtkWidget *popup;

        popup = gnome_popup_menu_new(menu);
        gtk_object_set_data(GTK_OBJECT(window->window), "popup", popup);
    }

    window->usermodes[0] = server_menu[USERMENU_FIRST].widget;
    window->usermodes[1] = server_menu[USERMENU_FIRST+2].widget;
    window->usermodes[2] = server_menu[USERMENU_FIRST+3].widget;
    window->usermodes[3] = server_menu[USERMENU_FIRST+4].widget;
    window->autoraise = windows_menu[MENU_AUTORAISE_POS].widget;
}
