/*
 gtk-specific.c : functions to get irssi build without GNOME libraries. these
                  functions are pretty slow and dum, if you want better, use
                  GNOME libraries!

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"

#include <pwd.h>

static void sig_property_apply(GtkWidget *widget, GnomePropertyBox *box)
{
    g_return_if_fail(box != NULL);

    gtk_widget_set_sensitive(box->ok_button, FALSE);
    gtk_widget_set_sensitive(box->apply_button, FALSE);

    box->applyfunc(box->applyfuncdata, -1);
}

static void sig_property_ok(GtkWidget *widget, GnomePropertyBox *box)
{
    g_return_if_fail(box != NULL);

    sig_property_apply(widget, box);
    gtk_widget_destroy(GTK_WIDGET(box->dialog));
}

GtkWidget *gnome_property_box_new(void)
{
    GnomePropertyBox *box;

    box = g_new(GnomePropertyBox, 1);

    box->dialog = GTK_DIALOG(gtk_dialog_new());
    gtk_signal_connect_object(GTK_OBJECT(box->dialog), "destroy",
                              GTK_SIGNAL_FUNC(g_free), (GtkObject *) box);

    box->notebook = gtk_notebook_new();
    gtk_box_pack_start(GTK_BOX(box->dialog->vbox), box->notebook, TRUE, TRUE, 0);

    box->ok_button = gtk_button_new_with_label(_("Ok"));
    gtk_signal_connect(GTK_OBJECT(box->ok_button), "clicked",
                       GTK_SIGNAL_FUNC(sig_property_ok), box);
    gtk_box_pack_start(GTK_BOX(box->dialog->action_area), box->ok_button, TRUE, TRUE, 10);

    box->apply_button = gtk_button_new_with_label(_("Apply"));
    gtk_signal_connect(GTK_OBJECT(box->apply_button), "clicked",
                       GTK_SIGNAL_FUNC(sig_property_apply), box);
    gtk_box_pack_start(GTK_BOX(box->dialog->action_area), box->apply_button, TRUE, TRUE, 10);

    box->close_button = gtk_button_new_with_label(_("Cancel"));
    gtk_signal_connect_object(GTK_OBJECT(box->close_button), "clicked",
                              GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(box->dialog));
    gtk_box_pack_start(GTK_BOX(box->dialog->action_area), box->close_button, TRUE, TRUE, 10);

    gtk_widget_set_sensitive(box->apply_button, FALSE);
    gtk_widget_set_sensitive(box->ok_button, FALSE);

    gtk_object_set_data(GTK_OBJECT(box->dialog), "propertybox", box);
    return GTK_WIDGET(box->dialog);
}

void gnome_property_box_set_apply_func(GtkWidget *box, GtkFunction func, gpointer data)
{
    GnomePropertyBox *rec;

    g_return_if_fail(box != NULL);

    rec = gtk_object_get_data(GTK_OBJECT(box), "propertybox");
    rec->applyfunc = (APPLY_FUNC) func;
    rec->applyfuncdata = data;
}

void gnome_property_box_changed(GtkWidget *box)
{
    GnomePropertyBox *rec;

    g_return_if_fail(box != NULL);

    rec = gtk_object_get_data(GTK_OBJECT(box), "propertybox");
    gtk_widget_set_sensitive(rec->ok_button, TRUE);
    gtk_widget_set_sensitive(rec->apply_button, TRUE);
}

gint gnome_property_box_append_page(GtkWidget *box, GtkWidget *child, GtkWidget *tab_label)
{
    GnomePropertyBox *rec;

    g_return_val_if_fail(box != NULL, 0);

    rec = gtk_object_get_data(GTK_OBJECT(box), "propertybox");
    gtk_notebook_append_page(GTK_NOTEBOOK(rec->notebook), child, tab_label);
    return g_list_length(GTK_NOTEBOOK(rec->notebook)->children) - 1;
}

GtkNotebook *gnome_property_box_get_notebook(GtkWidget *box)
{
    GnomePropertyBox *rec;

    g_return_val_if_fail(box != NULL, 0);

    rec = gtk_object_get_data(GTK_OBJECT(box), "propertybox");
    return GTK_NOTEBOOK(rec->notebook);
}

GtkWidget *gnome_app_new(gchar *id, gchar *title)
{
    GtkWidget *window, *vbox;

    g_return_val_if_fail(title != NULL, NULL);

    window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title(GTK_WINDOW(window), title);

    vbox = gtk_vbox_new(FALSE, 0);
    gtk_widget_show(vbox);
    gtk_container_add(GTK_CONTAINER(window), vbox);

    return window;
}

void gnome_app_set_contents(GtkWidget *window, GtkWidget *child)
{
    g_return_if_fail(window != NULL);
    g_return_if_fail(child != NULL);

    gtk_box_pack_end(GTK_BOX(gtk_container_children(GTK_CONTAINER(window))->data),
                     child, TRUE, TRUE, 0);
}

GtkWidget *gnome_app_find_menu_pos(GtkWidget *parent, gchar *path, gint *pos)
{
    GtkWidget *ret;
    GList *tmp;
    gchar *ptr;
    gint num;

    g_return_val_if_fail(parent != NULL, NULL);
    g_return_val_if_fail(path != NULL, NULL);
    g_return_val_if_fail(pos != NULL, NULL);

    if (*path == '/') path++;
    if (*path == '\0') return parent;

    path = g_strdup(dgettext(PACKAGE, path));
    do
    {
        ptr = strrchr(path, '_');
        if (ptr != NULL)
            memmove(ptr, ptr+1, strlen(ptr));
    }
    while (ptr != NULL);

    ptr = strchr(path, '/');
    if (ptr != NULL) *ptr++ = '\0';

    ret = NULL;
    for (num = 1, tmp = g_list_first(GTK_MENU_SHELL(parent)->children); tmp != NULL; tmp = tmp->next, num++)
    {
        GtkContainer *rec = tmp->data;
        gchar *str;

        gtk_label_get(GTK_LABEL(gtk_container_children(rec)->data), &str);
        if (strcmp(str, path) == 0)
        {
            *pos = num;
            ret = ptr == NULL ? parent :
                gnome_app_find_menu_pos(GTK_MENU_ITEM(rec)->submenu, ptr, pos);
            break;
        }
    }
    g_free(path);
    if (ret == NULL) *pos = 0;

    return ret;
}

void popupfunc(GtkWidget *widget, gpointer data)
{
    GtkWidget *menu;
    gpointer data2;
    void (*func) (GtkWidget *, gpointer);

    func = gtk_object_get_data(GTK_OBJECT(widget), "menu_activate_func");
    if (func == NULL) return;

    menu = gtk_object_get_data(GTK_OBJECT(gtk_object_get_data(GTK_OBJECT(widget), "menu")), "parent");

    data2 = gtk_object_get_data(GTK_OBJECT(menu), "menu_activate_data");
    func(widget, data != NULL ? data : data2);
}

void gnome_app_fill_menu_with_data(GtkMenuShell *menu, GnomeUIInfo *uiinfo, GtkAccelGroup *accel_group, gboolean insert_shortcuts, gint pos, gpointer data)
{
    gchar *str, *ptr;
    gint n;

    g_return_if_fail(menu != NULL);
    g_return_if_fail(uiinfo != NULL);

    for (n = 0; uiinfo[n].type != GNOME_APP_UI_ENDOFINFO; n++)
    {
        if (uiinfo[n].label == NULL)
            str = NULL;
        else
        {
            str = g_strdup(dgettext(PACKAGE, uiinfo[n].label));
            ptr = strchr(str, '_');
            if (ptr != NULL)
                memmove(ptr, ptr+1, strlen(ptr));
        }

        switch (uiinfo[n].type)
        {
            case GNOME_APP_UI_SEPARATOR:
                uiinfo[n].widget = gtk_menu_item_new();
                break;
            case GNOME_APP_UI_TOGGLEITEM:
                uiinfo[n].widget = gtk_check_menu_item_new_with_label(str);
                gtk_check_menu_item_set_show_toggle(GTK_CHECK_MENU_ITEM(uiinfo[n].widget), TRUE);
                break;
            case GNOME_APP_UI_ITEM:
            case GNOME_APP_UI_SUBTREE:
                uiinfo[n].widget = gtk_menu_item_new_with_label(str);
                break;
        }
        gtk_menu_shell_insert(menu, uiinfo[n].widget, pos++);
        if (str != NULL) g_free(str);

        if (uiinfo[n].type == GNOME_APP_UI_SUBTREE)
        {
            GtkWidget *submenu;

	    submenu = gtk_menu_new();
	    gtk_object_set_data(GTK_OBJECT(submenu), "parent", gtk_object_get_data(GTK_OBJECT(menu), "parent"));
            gtk_menu_item_set_submenu(GTK_MENU_ITEM(uiinfo[n].widget), submenu);
            gnome_app_fill_menu_with_data(GTK_MENU_SHELL(submenu), (GnomeUIInfo *) uiinfo[n].moreinfo,
                                          NULL, TRUE, pos, data);
        }
        else
        {
            gtk_object_set_data(GTK_OBJECT(uiinfo[n].widget), "menu_activate_func", uiinfo[n].moreinfo);
	    gtk_object_set_data(GTK_OBJECT(uiinfo[n].widget), "menu", menu);
            gtk_signal_connect(GTK_OBJECT(uiinfo[n].widget), "activate",
                               GTK_SIGNAL_FUNC(popupfunc), data != NULL ? data : uiinfo[n].user_data);
        }
    }

    gtk_widget_show_all(GTK_WIDGET(menu));
}

void gnome_app_insert_menus_with_data(GtkWidget *window, gchar *label, GnomeUIInfo *uiinfo, gpointer data)
{
    GtkWidget *menubar, *submenu;
    gint position;

    g_return_if_fail(window != NULL);
    g_return_if_fail(uiinfo != NULL);

    menubar = gtk_object_get_data(GTK_OBJECT(window), "menubar");
    g_return_if_fail(menubar != NULL);

    submenu = gnome_app_find_menu_pos(menubar, dgettext(PACKAGE, label), &position);
    if (submenu == NULL) submenu = menubar;
    gtk_object_set_data(GTK_OBJECT(submenu), "parent", menubar);
    gnome_app_fill_menu_with_data(GTK_MENU_SHELL(submenu), uiinfo, NULL, TRUE, position, data);
}

void gnome_app_create_menus_with_data(GtkWidget *window, GnomeUIInfo *uiinfo, gpointer data)
{
    GtkWidget *menubar;

    g_return_if_fail(window != NULL);
    g_return_if_fail(uiinfo != NULL);

    menubar = gtk_menu_bar_new();
    gtk_object_set_data(GTK_OBJECT(window), "menubar", menubar);
    gtk_widget_show(menubar);
    gtk_box_pack_start(GTK_BOX(gtk_container_children(GTK_CONTAINER(window))->data),
                       menubar, FALSE, FALSE, 0);

    gtk_object_set_data(GTK_OBJECT(menubar), "parent", menubar);
    gnome_app_fill_menu_with_data(GTK_MENU_SHELL(menubar), uiinfo, NULL, TRUE, 0, data);
}

void gnome_app_remove_from_menu(GtkWidget *menu, gchar *path, gint items)
{
    GtkWidget *child;
    GList *children;
    gint position;
    gchar *ptr;

    g_return_if_fail(menu != NULL);
    g_return_if_fail(path != NULL);

    path = g_strdup(dgettext(PACKAGE, path));
    ptr = strchr(path, '_');
    if (ptr != NULL)
        memmove(ptr, ptr+1, strlen(ptr));

    menu = gnome_app_find_menu_pos(menu, path, &position); position--;
    g_return_if_fail(menu != NULL);

    g_free(path);

    children = g_list_nth(GTK_MENU_SHELL(menu)->children, position);
    while (children != NULL && items > 0)
    {
        child = children->data;
        children = children->next;

        gtk_container_remove(GTK_CONTAINER(menu), child);
        items--;
    }

    gtk_widget_queue_resize(menu);
}

void gnome_app_remove_menus(GtkWidget *window, gchar *path, gint items)
{
    GtkWidget *menubar;

    menubar = gtk_object_get_data(GTK_OBJECT(window), "menubar");
    g_return_if_fail(menubar != NULL);

    gnome_app_remove_from_menu(menubar, path, items);
}

GtkWidget *gnome_popup_menu_new(GnomeUIInfo *uiinfo)
{
    GtkWidget *menu;

    menu = gtk_menu_new();
    gtk_widget_show(menu);

    gtk_object_set_data(GTK_OBJECT(menu), "parent", menu);
    gnome_app_fill_menu_with_data(GTK_MENU_SHELL(menu), uiinfo, NULL, TRUE, 0, NULL);
    return menu;
}

static gint popup_button_pressed(GtkWidget *widget, GdkEventButton *event, gpointer data)
{
    gpointer user_data;

    if (event->button != 3)
        return FALSE;

    user_data = gtk_object_get_data(GTK_OBJECT(widget), "menu_activate_data");
    gtk_signal_emit_stop_by_name (GTK_OBJECT (widget), "button_press_event");

    gnome_popup_menu_do_popup(data, NULL, NULL, event, user_data);
    return TRUE;
}

static void popup_attach_widget_destroyed(GtkWidget *widget, gpointer data)
{
    gtk_object_unref(GTK_OBJECT(data));
}

void gnome_popup_menu_attach(GtkWidget *menu, GtkWidget *widget, gpointer data)
{
    gtk_object_ref(GTK_OBJECT(menu));
    gtk_object_sink(GTK_OBJECT(menu));

    gtk_object_set_data(GTK_OBJECT(menu), "menu_activate_data", data);
    gtk_widget_add_events(widget, GDK_BUTTON_PRESS_MASK);

    gtk_signal_connect(GTK_OBJECT(widget), "button_press_event",
                       (GtkSignalFunc) popup_button_pressed, menu);

    gtk_signal_connect(GTK_OBJECT(widget), "destroy",
                       (GtkSignalFunc) popup_attach_widget_destroyed, menu);
}

void gnome_popup_menu_do_popup(GtkWidget *menu, GtkMenuPositionFunc posfunc, gpointer posdata, GdkEventButton *event, gpointer data)
{
    if (data != NULL)
	gtk_object_set_data(GTK_OBJECT(menu), "menu_activate_data", data);

    gtk_menu_popup(GTK_MENU(menu), NULL, NULL, posfunc, posdata,
                   event == NULL ? 0 : event->button, event == NULL ? GDK_CURRENT_TIME : event->time);
}

void gnome_app_create_toolbar_with_data(GtkWidget *window, GnomeUIInfo *uiinfo, gpointer data)
{
    GtkWidget *toolbar;
    gint n;

    g_return_if_fail(window != NULL);
    g_return_if_fail(uiinfo != NULL);

    toolbar = gtk_toolbar_new(GTK_ORIENTATION_HORIZONTAL, GTK_TOOLBAR_BOTH);
    gtk_box_pack_start(GTK_BOX(gtk_container_children(GTK_CONTAINER(window))->data),
                       toolbar, FALSE, FALSE, 0);

    for (n = 0; uiinfo[n].type != GNOME_APP_UI_ENDOFINFO; n++)
    {
        uiinfo[n].widget =
            gtk_toolbar_append_item(GTK_TOOLBAR(toolbar), dgettext(PACKAGE, uiinfo[n].label), dgettext(PACKAGE, uiinfo[n].label), NULL, NULL,
                                    GTK_SIGNAL_FUNC(uiinfo[n].moreinfo), data != NULL ? data : uiinfo[n].user_data);
    }

    gtk_widget_show_all(toolbar);
}

void sig_file_entry_browse_ok(GtkWidget *widget)
{
    GtkWidget *fsel, *entry;

    g_return_if_fail(widget != NULL);

    fsel = gtk_object_get_data(GTK_OBJECT(widget), "fsel");
    entry = gtk_object_get_data(GTK_OBJECT(widget), "entry");
    gtk_entry_set_text(GTK_ENTRY(entry), gtk_file_selection_get_filename(GTK_FILE_SELECTION(fsel)));

    gtk_widget_destroy(fsel);
}

void sig_file_entry_browse(GtkWidget *widget)
{
    GtkWidget *fsel, *entry;
    gchar *path;

    g_return_if_fail(widget != NULL);

    entry = gtk_object_get_data(GTK_OBJECT(widget), "entry");
    path = gtk_entry_get_text(GTK_ENTRY(entry));
    if (*path != '\0')
        path = g_strdup(path);
    else
    {
        path = gtk_object_get_data(GTK_OBJECT(widget), "browse path");
        if (path != NULL)
            path = g_strconcat(path, "/", NULL);
    }

    fsel = gtk_file_selection_new(gtk_object_get_data(GTK_OBJECT(widget), "title"));
    gtk_object_set_data(GTK_OBJECT(widget), "fsel", fsel);
    if (path != NULL) gtk_file_selection_set_filename(GTK_FILE_SELECTION(fsel), path);
    g_free(path);
    gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(fsel)->ok_button), "clicked",
                              GTK_SIGNAL_FUNC(sig_file_entry_browse_ok), GTK_OBJECT(widget));
    gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(fsel)->cancel_button), "clicked",
                              GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(fsel));
    gtk_widget_show(fsel);
}

static void file_entry_destroyed(GtkWidget *widget)
{
    gchar *path;

    path = gtk_object_get_data(GTK_OBJECT(widget), "browse path");
    if (path != NULL) g_free(path);
}

GtkWidget *gnome_file_entry_new(gchar *id, gchar *title)
{
    GtkWidget *hbox, *entry, *button;

    g_return_val_if_fail(title != NULL, NULL);

    hbox = gtk_hbox_new(FALSE, 5);
    gtk_signal_connect(GTK_OBJECT(hbox), "destroy",
                       GTK_SIGNAL_FUNC(file_entry_destroyed), NULL);

    entry = gtk_entry_new();
    gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);

    button = gtk_button_new_with_label(_("Browse"));
    gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
                              GTK_SIGNAL_FUNC(sig_file_entry_browse), GTK_OBJECT(hbox));
    gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);

    gtk_object_set_data(GTK_OBJECT(hbox), "entry", entry);
    gtk_object_set_data(GTK_OBJECT(hbox), "title", title);

    return hbox;
}

gchar *gnome_file_entry_get_full_path(GtkWidget *fileentry, gboolean must_exist)
{
    struct stat statbuf;
    GtkWidget *entry;
    gchar *path;

    entry = gtk_object_get_data(GTK_OBJECT(fileentry), "entry");
    g_return_val_if_fail(entry != NULL, NULL);

    path = gtk_entry_get_text(GTK_ENTRY(entry));
    return stat(path, &statbuf) != 0 ? NULL : path;
}

GtkWidget *gnome_file_entry_gtk_entry(GtkWidget *fileentry)
{
    return gtk_object_get_data(GTK_OBJECT(fileentry), "entry");
}

void gnome_file_entry_set_default_path(GtkWidget *widget, gchar *path)
{
    path = g_strdup(path);
    gtk_object_set_data(GTK_OBJECT(widget), "browse path", path);
    gtk_signal_connect_object(GTK_OBJECT(widget), "destroy",
                              GTK_SIGNAL_FUNC(g_free), (GtkObject *) path);
}

GtkWidget *gnome_dialog_new(gchar *title, ...)
{
    GtkWidget *dialog, *button;
    va_list args;
    gchar *butlabel, butstr[30];
    gint buts;

    dialog = gtk_dialog_new();
    gtk_container_border_width(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), 5);
    gtk_window_set_title(GTK_WINDOW(dialog), title);

    va_start(args, title);
    buts = 0;
    while ((butlabel = va_arg(args, gchar *)) != NULL)
    {
        button = gtk_button_new_with_label(butlabel);
        gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->action_area), button, TRUE, TRUE, 0);
        sprintf(butstr, "dialog_button_%d", buts++);
        gtk_object_set_data(GTK_OBJECT(dialog), butstr, button);
    }
    va_end(args);

    return dialog;
}

void gnome_dialog_button_connect(GtkDialog *dialog, gint button, GtkSignalFunc func, gpointer data)
{
    GtkWidget *but;
    gchar butstr[30];

    sprintf(butstr, "dialog_button_%d", button);
    but = gtk_object_get_data(GTK_OBJECT(dialog), butstr);
    g_return_if_fail(but != NULL);

    gtk_signal_connect(GTK_OBJECT(but), "clicked", func, data);
}

void gnome_dialog_button_connect_object(GtkDialog *dialog, gint button, GtkSignalFunc func, GtkObject *object)
{
    GtkWidget *but;
    gchar butstr[30];

    sprintf(butstr, "dialog_button_%d", button);
    but = gtk_object_get_data(GTK_OBJECT(dialog), butstr);
    g_return_if_fail(but != NULL);

    gtk_signal_connect_object(GTK_OBJECT(but), "clicked", func, object);
}

GtkWidget *gnome_entry_new(gchar *history_id)
{
    return gtk_entry_new();
}

GtkWidget *gnome_entry_gtk_entry(GtkWidget *gentry)
{
    return gentry;
}
