/*
 dialog-disconnect.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"

static GtkWidget *clist = NULL;

static void clist_append_server(GtkWidget *clist, SERVER_REC *server)
{
    gchar *titles[4];
    gchar *str;

    g_return_if_fail(clist != NULL);
    g_return_if_fail(server != NULL);

    str = g_strdup_printf("%d", server->port);

    titles[0] = server->tag;
    titles[1] = server->ircnet == NULL ? "" : server->ircnet;
    titles[2] = server->address;
    titles[3] = str;

    gtk_clist_set_row_data(GTK_CLIST(clist), gtk_clist_append(GTK_CLIST(clist), titles), server);

    g_free(str);
}

static void sig_join(GtkWidget *clist)
{
    SERVER_REC *server;
    gchar *channel;
    gint selection;

    g_return_if_fail(clist != NULL);

    selection = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(clist), "selection"));
    if (selection == -1)
    {
        gui_dialog(DIALOG_ERROR, _("No server selected"));
        return;
    }

    channel = gtk_entry_get_text(gtk_object_get_data(GTK_OBJECT(clist), "entry"));
    if (*channel == '\0')
    {
        gui_dialog(DIALOG_ERROR, _("No channel specified"));
        return;
    }

    server = gtk_clist_get_row_data(GTK_CLIST(clist), selection);
    signal_emit("command join", 3, channel, server, cur_channel);
}

static void sig_select_row(GtkWidget *clist, gint row)
{
    g_return_if_fail(clist != NULL);

    gtk_object_set_data(GTK_OBJECT(clist), "selection", GINT_TO_POINTER(row));
}

static void sig_unselect_row(GtkWidget *clist)
{
    g_return_if_fail(clist != NULL);

    gtk_object_set_data(GTK_OBJECT(clist), "selection", GINT_TO_POINTER(-1));
}

static void sig_destroy(GtkWidget *dialog)
{
    g_return_if_fail(dialog != NULL);

    gtk_widget_destroy(dialog);
    clist = NULL;
}

static gboolean redraw_servers(void)
{
    GList *tmp;

    if (clist != NULL)
    {
        gtk_clist_freeze(GTK_CLIST(clist));
        gtk_clist_clear(GTK_CLIST(clist));
        for (tmp = g_list_first(servers); tmp != NULL; tmp = tmp->next)
            clist_append_server(clist, tmp->data);
        gtk_clist_thaw(GTK_CLIST(clist));
    }
    return TRUE;
}

static void clist_click_column(GtkCList *clist, gint column)
{
    if (column != clist->sort_column)
	gtk_clist_set_sort_column(clist, column);
    else
    {
	if (clist->sort_type == GTK_SORT_ASCENDING)
	    clist->sort_type = GTK_SORT_DESCENDING;
	else
	    clist->sort_type = GTK_SORT_ASCENDING;
    }

    gtk_clist_sort(clist);
}

void dialog_join(void)
{
    GtkWidget *dialog, *scrollbox, *entry;
    gchar *titles[4];

    titles[0] = _("Server Tag"); titles[1] = _("IRC Network");
    titles[2] = _("Address"); titles[3] = _("Port");

    dialog = gnome_dialog_new(PACKAGE, _("Join"), GNOME_STOCK_BUTTON_CLOSE, NULL);
    gtk_widget_set_usize(dialog, 350, 250);
    gtk_window_set_policy(GTK_WINDOW(dialog), FALSE, TRUE, FALSE);
    gtk_signal_connect(GTK_OBJECT(dialog), "destroy",
                       GTK_SIGNAL_FUNC(sig_destroy), NULL);
    gtk_signal_connect(GTK_OBJECT(dialog), "delete_event",
                       GTK_SIGNAL_FUNC(gtk_widget_destroy), NULL);

    /* Create server clist widget */
    scrollbox = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrollbox),
                                   GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), scrollbox, TRUE, TRUE, 0);

    clist = gtk_clist_new_with_titles(4, titles);
    gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 0, TRUE);
    gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 1, TRUE);
    gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 2, TRUE);
    gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 3, TRUE);
    gtk_container_add(GTK_CONTAINER(scrollbox), clist);
    gtk_object_set_data(GTK_OBJECT(clist), "selection", GINT_TO_POINTER(-1));
    gtk_clist_set_selection_mode(GTK_CLIST(clist), GTK_SELECTION_BROWSE);
    gtk_signal_connect(GTK_OBJECT(clist), "select_row",
                       GTK_SIGNAL_FUNC(sig_select_row), NULL);
    gtk_signal_connect(GTK_OBJECT(clist), "unselect_row",
                       GTK_SIGNAL_FUNC(sig_unselect_row), NULL);
    gtk_signal_connect(GTK_OBJECT(clist), "click_column",
		       GTK_SIGNAL_FUNC(clist_click_column), NULL);

    entry = gui_create_labelentry(GNOME_DIALOG(dialog)->vbox, _("Channel"), NULL, FALSE);
    gtk_object_set_data(GTK_OBJECT(clist), "entry", entry);

    redraw_servers();
    gtk_clist_select_row(GTK_CLIST(clist), 0, 0);

    gnome_dialog_button_connect_object(GNOME_DIALOG(dialog), 0, GTK_SIGNAL_FUNC(sig_join), GTK_OBJECT(clist));
    gnome_dialog_button_connect_object(GNOME_DIALOG(dialog), 1, GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(dialog));

    /* Make columns resizeable */
    gtk_clist_set_column_resizeable(GTK_CLIST(clist), 0, TRUE);
    gtk_clist_set_column_resizeable(GTK_CLIST(clist), 1, TRUE);
    gtk_clist_set_column_resizeable(GTK_CLIST(clist), 2, TRUE);
    gtk_widget_show_all(dialog);
}

void dialog_join_init(void)
{
    signal_add("server connected", (SIGNAL_FUNC) redraw_servers);
    signal_add("server disconnected", (SIGNAL_FUNC) redraw_servers);
    signal_add("server connect failed", (SIGNAL_FUNC) redraw_servers);
}

void dialog_join_deinit(void)
{
    signal_remove("server connected", (SIGNAL_FUNC) redraw_servers);
    signal_remove("server disconnected", (SIGNAL_FUNC) redraw_servers);
    signal_remove("server connect failed", (SIGNAL_FUNC) redraw_servers);
}
