/*************************************************************************
 * 
 * irmp3 - Multimedia Audio Jukebox for Linux
 * http://irmp3.sourceforge.net
 *
 * $Source: /cvsroot/irmp3/irmp3/src/irmp3/irmp3.c,v $ -- command line client
 * $Id: irmp3.c,v 1.7 2003/11/29 21:31:31 boucman Exp $
 *
 * Copyright (C) by Jeremy Rosen, Volker Wegert
 *
 * Please contact the current maintainer, Jeremy Rosen <jeremy.rosen@enst-bretagne.fr>
 * for information and support regarding irmp3.
 *
 *
 */

#include <sys/types.h> 	/* for sockets */
#include <sys/stat.h>   /* for sockets */
#include <sys/socket.h> /* for sockets */
#include <netinet/in.h>
#include <netdb.h>      /* for protocol entries */
#include <signal.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <stdio.h>
#include <argp.h>
#include <stdlib.h>
#include <error.h>
#include <glob.h>

#include "config.h"
#include "irmp3log.h"
#include "irmp3config.h"
 
/****** Default parameters *******/
int   used_port = 0;
char* used_host = NULL;
char* used_pipe = NULL;
int pipe_pos = 0; //position of pipe in command line
int net_pos = 0; // position of net arg in command line


char* arg_command = NULL;
char* input_file = NULL;

char * config_file_name = CONFFILE;
int  verbosity = LOG_NORMAL;
/**********************************/

char do_default_input = 1;
const char *argp_program_version     = "irmp3 0.1";
const char *argp_program_bug_address = "Jeremy rosen <jeremy.rosen@enst-bretagne.fr>";
static char doc[] = 
"irmp3 -- a command line client to send commands to an imrp3d server \
\v \
This program is used to send commands to an irmp3d server. \
It will try all input sources specified on the command line and use the \
first one available. If no output destination is specified, irmp3 will \
attempt to use the default pipe, then the default network connection, \
and if both fail, will fail as well. If no input option is specified, \
the program will read the commands from the standard input.";

static char args_doc[]= "[COMMAND ; COMMAND...]";

static struct argp_option options[] = {
	{0,0,0,0,"Output options: ",0},
	{"network",'n',"HOST[:PORT]",OPTION_ARG_OPTIONAL,"Host and port to connect to.\n   Defaults to localhost and port defined in config file",0}, // todo : document default values
	{"pipe",'p',"PIPENAME",OPTION_ARG_OPTIONAL,"name of the pipe file to connect to.\nDefaults is read from config file",0},
	{0,0,0,0,"Input options : ",0},
	{"file",'f',"FILENAME",0,"read commands from this file.\nA '-' will read from standard input (defaulte behaviour).",0},
	{"command [; command...]",0,0,OPTION_DOC,"A list of commands to execute.",0},
	{0,0,0,0,"Miscellaneous options : ",0},
	{"verbose",'v',0,0,"More verbose output. Can be specified multiple times",0},
	{"config",'c',"CONFIG_FILE",0,"The irmp3 config file to use",0},
	{0,0,0,0,NULL,0}};

static error_t parse_opt (int key, char *arg, struct argp_state *state);
static struct argp our_argp =  { options,parse_opt,args_doc,doc,NULL,NULL,0};


/*
 * connect or reconnect to the server
 */
static int connect_to_server(int port, char * host) 
{
	struct protoent* protocol;
	struct sockaddr_in dest_addr;
	int sin_size;
	int sockdesc;
	struct hostent* server_data;
	
	log_printf(LOG_VERBOSE,"connecting to %s:%d...\n",host,port);
	protocol = getprotobyname("tcp");
	sockdesc = socket( AF_INET, SOCK_STREAM, protocol->p_proto);
	dest_addr.sin_family = AF_INET;
	dest_addr.sin_port = htons(port);
	memset(&(dest_addr.sin_zero),'\0',8);
	if (sockdesc == -1) 
	{
		sockdesc = -1;
		perror("socket");
		return -1;
	};
	if((server_data=gethostbyname(host)) == NULL )
	{
		sockdesc = -1;
		perror("gethostbyname");
		return -1;
	}
	dest_addr.sin_addr = (*((struct in_addr *) server_data->h_addr));
	if (connect( sockdesc, (struct sockaddr *)&dest_addr, sizeof(struct sockaddr)) == -1)
	{
		perror("connect");
		return -1;
	};
	sin_size = sizeof(struct sockaddr_in);
	if(getpeername(sockdesc, (struct sockaddr *)&dest_addr,&sin_size ) == -1)
	{
		perror("getpeername");
		return -1;
	};
	/*
	 * OK, connection is established
	 * */
	return sockdesc;
}

/*
 * main parser for options. it will also open all file descriptors,
 * so all we have to do after that is read from input_fd 
 * and write to output_fr
 * */
 static error_t parse_opt (int key, char *arg, struct argp_state *state) 
{
  char commands[4000000];
  int i;
  char* tmp_port;
  char* tmp_host;
  
  memset(commands,0,256);
  switch(key) {
  case 'c':
    config_file_name = strdup(arg);
    return 0;
  case 'v':
    verbosity++;
    return 0;
  case 'n':
    // use network connection
    // note current position in arg list
    net_pos =  state->next -1;
    log_printf(LOG_VERBOSE,"network argument detected at position %d\n", net_pos);
    if (arg == NULL) {
	    return 0;
    } else {
	    // read connection information from command line
	    tmp_host = strdup(arg);
	    if ((tmp_port = strchr(tmp_host,':')) == 0) { 
		    used_host = tmp_host;
		    log_printf(LOG_VERBOSE,"host detected: %s\n", used_host);
		    return 0;
	    } else {
		    // evaluate complete host:port syntax
		    *tmp_port= '\0';
		    tmp_port++;
		    used_port =  atoi(tmp_port);
		    if(*tmp_host == '\0') {
			    log_printf(LOG_VERBOSE,"port detected: %d\n", used_port);
			    return 0;
		    } else {
			    used_host = tmp_host;
			    log_printf(LOG_VERBOSE,"host detected: %s\n", used_host);
			    log_printf(LOG_VERBOSE,"port detected: %d\n", used_port);
			    return 0;
		    }
	    }
    }
    log_printf(LOG_ERROR,"this point should never be reached: %s %s\n", __FILE__, __LINE__);
    exit(EXIT_FAILURE);
  case 'p':
	    // use pipe
	    pipe_pos =  state->next -1;
	    log_printf(LOG_VERBOSE,"pipe argument detected at position %d\n", pipe_pos);
	    if (arg == NULL) {
		    return 0;
	    } else {
		    used_pipe = strdup(arg);
		    log_printf(LOG_VERBOSE,"pipe name detected: %s\n", used_pipe);
	    }
    return 0;
  case 'f':
    // read input from file

    input_file =strdup(arg);
    return 0;
  case ARGP_KEY_ARGS: // all other arguments
    for(i=state->next ;i < state->argc; i++) {
	    strcat(commands," ");
	    strcat(commands,state->argv[i]);
    }
    log_printf(LOG_VERBOSE,"Commands read on command line: %s\n",commands);
    arg_command = strdup(commands);
    state->next = state->argc;
    return 0;
  default:
    return ARGP_ERR_UNKNOWN;
  }
}	
/*
 * Main function, once options are parsed choose the right input and ouput, 
 * then just read one and write to the other
 * */
int main(int argc , char* argv[])
{
	char input_stream[256];
	char* tmp_char;
	int temp;
	int input_fd =-1;
	int output_fd =-1;
	int pipe_fd[2];
	glob_t glob_result;
	log_open(NULL,verbosity); //open log with default verbosity
	argp_parse(&our_argp,argc,argv,0,0,NULL);
	config_read(config_file_name );
	log_open(NULL,verbosity); //open log, with command line verbosity
	// Guess input
	if(input_file && input_fd == -1) {
		input_fd = open(input_file, 0);
		if (input_fd != -1) { 
			log_printf(LOG_VERBOSE,"Reading from input file %s...\n",input_file);
		} else {
			log_printf(LOG_NORMAL,"Unable to open input file %s! \n",input_file);
		}
	} else if (arg_command && input_fd == -1) {
		// cut command line in pieces
		for(tmp_char=arg_command;*tmp_char;tmp_char++) {
			if(*tmp_char==';') {
				if (*(tmp_char-1) == ' ') {
					*(tmp_char-1)='\n';
					*tmp_char=' ';
				} else {
					*tmp_char='\n';
				}
			}
		}
		// create a pipe to store commands	
		pipe(pipe_fd);
		write(pipe_fd[1],arg_command,strlen(arg_command));
		write(pipe_fd[1],"\n",1);
		close(pipe_fd[1]);
		input_fd = pipe_fd[0];
		log_printf(LOG_VERBOSE,"Reading from command line\n %s\n",arg_command);
	} else if (input_fd == -1){
		log_printf(LOG_VERBOSE,"Reading from standard input\n");
		input_fd = STDIN_FILENO;
	}

	
	if (input_fd == -1) {
		log_printf(LOG_ERROR,"given input is not valid, will not proceed\n");
		exit(EXIT_FAILURE);
	}
	// guess output according to config file
	if(!net_pos && !pipe_pos) { // no output option on command line
		if(!strcmp(config_getstr("fifo_enabled","yes"),"yes")) {
			pipe_pos = 1;
		} 
		if(!strcmp(config_getstr("netctl_enabled","yes"),"yes")) {
			net_pos = 2;
		} 
	}
	// find correct params if not set yet
	if(pipe_pos) {
		if(!used_pipe) {
			used_pipe =  malloc(512);
			sprintf(used_pipe,"%s/%s",config_getstr("fifo_directory","/tmp/"),\
					config_getstr("fifo_name","irmp3-%u-%p.tmp"));
		}
	}
	if(net_pos) {
		if(!used_port) {
			used_port = config_getnum("netctl_port",9232);
		}
		if(!used_host) {
			used_host = "localhost";
		}
	}
	// replace %p and %u in pipe name
	if(pipe_pos) {
		while ((tmp_char =strchr(used_pipe,'%'))) {
			*tmp_char = '*';
			*(tmp_char+1) = '*';
		}
		glob(used_pipe,0,NULL,&glob_result);
		if(glob_result.gl_pathc <1 ) {
			log_printf(LOG_NORMAL,"no file corresponds to %s, won't connect via a pipe\n",used_pipe);
			pipe_pos = 0;
		} else if (glob_result.gl_pathc > 1) {
			log_printf(LOG_VERBOSE,"more than one file corresponds to %s, will use the first one\n",used_pipe);
			used_pipe = glob_result.gl_pathv[0];
		} else {
			used_pipe = glob_result.gl_pathv[0];
		}
			
			
	}

	// try to open the different output in the right order
	for(temp = 1 ; temp <= argc && output_fd == -1; temp++) {
		if (net_pos == temp) {
			output_fd = connect_to_server(used_port,used_host);
			if (output_fd != -1) {
				log_printf(LOG_VERBOSE,"Connected to irmp3d on host  %s port %d\n",used_host,used_port);
			} else {
				log_printf(LOG_NORMAL,"Connection failed on %s:%d\n",used_host,used_port);
			}
		} else if (pipe_pos == temp) {
			output_fd = open(used_pipe, O_WRONLY);
			if (output_fd != -1) {
				log_printf(LOG_VERBOSE,"Connected to irmp3d via %s\n",used_pipe);
			} else {
				log_printf(LOG_NORMAL,"No valid FIFO found at %s\n",used_pipe);
			}

			
		}
	}
	if (output_fd == -1) {
		log_printf(LOG_ERROR,"No valid output found. Won't send command\n");
		exit(EXIT_FAILURE);
	}

	while (1) {
		temp = read(input_fd,input_stream,256);
		if(temp == 0) {
			exit(0);
		} else if (temp == -1) {
			perror("read ");
			exit(1);
		}
		temp = write(output_fd,input_stream,temp);
		if (temp == -1) {
			perror("write ");
			exit(1);
		}
	}
	return 0;
}
