#include <stdio.h>
#include <ctype.h>
#include <gnome.h>
#include "gipsc.h"
#include "../ifc.h"
#include "../../../prips/prips.h"

#define DEFAULT_SAVE_FILE "ipsc.txt"

/* 
 * Global variables
 */
const char *APP_ID = "gipsc";
const char *APP_VERSION = "0.4.2";
const char *APP_AUTHORS = "Dan Kelly (dan@vertekcorp.com)";
const char *APP_NAME = "IP Subnet Calculator";

Network n;

GipscMain mc;
GipscSubnets si;
GipscGeneral gi;
GipscHost hi;
GipscCidr ci;

/* 
 * Function prototypes
 */
GtkWidget *gipsc_create_main(GipscMain *mw);

void gipsc_update_main(const GipscMain *mc, const Network *n);
void update_tabs(const Network *n);

void create_save_dialog_cb(void);
void subnet_bits_value_changed_cb(GtkWidget *widget, GtkSpinButton *sbtn);
void class_button_pressed_cb(GtkWidget *class_button, gpointer class);
void save_dialog_ok_button_cb(GtkObject *obj);
void specify_network_cb(void);
void about_cb(GtkWidget *widget, gpointer data); 
int exit_cb(GtkWidget *widget, gpointer data);
GnomeUIInfo *get_interface_menu(void);

void specify_network_cb(void)
{
	GnomeDialog *dialog;
	GtkWidget *addr_entry;
	GtkWidget *help_label;
	gboolean reply;
	GString *default_addr;
	gchar *help_text = "Please enter a network address. "
		"The valid formats for the network "
		"address are CIDR notation, address/"
		"subnet mask, or simply the address. "
		"If only an IP address is entered, the "
		"network information will be determined "
		"based upon the first three bits of the "
		"address.";

	dialog = GNOME_DIALOG( gnome_dialog_new(APP_NAME,
				  GNOME_STOCK_BUTTON_OK,
				  GNOME_STOCK_BUTTON_CANCEL,
				  NULL));
	
	help_label = gtk_label_new(help_text);
	gtk_label_set_line_wrap( GTK_LABEL(help_label), TRUE);
	gtk_box_pack_start( GTK_BOX(dialog->vbox), 
			    help_label,
			    FALSE, FALSE, 0);	

	addr_entry = gtk_entry_new();
	gtk_box_pack_start( GTK_BOX(dialog->vbox), 
			    addr_entry,
			    FALSE, FALSE, 0);	
	
	default_addr = g_string_new(NULL);
	g_string_sprintf(default_addr, "%s/%d", denumberize(n.host_addr),
		ipsc_network_get_prefix_bits(&n));
	gtk_entry_set_text( GTK_ENTRY(addr_entry), default_addr->str);
	g_string_free(default_addr, TRUE);

	gnome_dialog_set_parent(dialog, GTK_WINDOW(mc.app));
	gnome_dialog_set_default(dialog, GNOME_OK);
	gnome_dialog_editable_enters(dialog, 
		GTK_EDITABLE(addr_entry)); 

	gtk_widget_show_all( GTK_WIDGET(dialog));
	gtk_widget_grab_focus(addr_entry);

GET_REPLY:
	reply = gnome_dialog_run(dialog);
	if(reply == GNOME_OK)
	{
		gchar *addr;

		addr = gtk_entry_get_text( GTK_ENTRY(addr_entry));
		addr = g_strstrip(addr);

		if(ipsc_network_init_parse_text(&n, addr) < 0)
		{
			const char *err_mesg = "Incorrect network address";
			GtkWidget *mesg_box = gnome_message_box_new(err_mesg, 
							GNOME_MESSAGE_BOX_ERROR,
							GNOME_STOCK_BUTTON_OK,
							NULL);
			gnome_dialog_run( GNOME_DIALOG(mesg_box));
			goto GET_REPLY;
		}
		gipsc_update_main(&mc, &n);
		update_tabs(&n);
	}

	gtk_widget_destroy( GTK_WIDGET(dialog));
}

/* 
 * About box
 */
void about_cb(GtkWidget *widget, gpointer data)                   
{                       
	static GtkWidget *box = NULL;
	const char *authors[] = {               
		APP_AUTHORS,
		NULL    
	};              

	if(box != NULL)
	{
		gdk_window_show(box->window);
		gdk_window_raise(box->window);
	}
	else
	{
		box = gnome_about_new( 	
			APP_NAME,
			APP_VERSION,
			"(C) 1999, under the GNU General Public License",
			authors,
			"The IP Subnet Calculator is a tool that allows "
			"network administrators to make calculations that "
			"will assist in subnetting a network.  It also has a "
			"number of other useful functions.",
			NULL);

		gtk_signal_connect( GTK_OBJECT(box),
				    "destroy",
				    GTK_SIGNAL_FUNC(gtk_widget_destroyed),
				    &box);
		gtk_widget_show(box);
	}
}                       

int exit_cb(GtkWidget *widget, gpointer data)
{
	gtk_main_quit();
	return(FALSE);
}

void notebook_switch_page_cb(GtkWidget *widget, GtkNotebookPage *page, 
			     guint page_num, gpointer *data)
{
	if( GTK_LABEL(page->tab_label) == ci.tab_label)
	{
		gtk_label_set_text(mc.subnet_bits_label, "Supernet bits");
		mc.subnet_bits_adj->upper = 31;
	}
	else
	{
		gtk_label_set_text(mc.subnet_bits_label, "Subnet bits");
		mc.subnet_bits_adj->upper = n.subnet_bits_max;
	}
} 

/* When the OK button in the save dialog is clicked this function can
 * be called up to five times (there must be a better way!).  The first 
 * time it is called we find the file name where we are going to save.  
 * Subsequent calls check to see if the checkboxes we've added to the 
 * fselect main vbox are checked.  This function is called one additional
 * time for each check box.  If a particular check box is checked, we 
 * call the save routing for that check box.
 */
void save_dialog_ok_button_cb(GtkObject *obj)
{
	GtkLabel *label;
	gchar *label_text;
	static gchar *file_name;
	FILE *f;
	
	if( GTK_IS_FILE_SELECTION(obj))
	{
		file_name = gtk_file_selection_get_filename(
			GTK_FILE_SELECTION(obj)
		);
		return;
	}	
	else
	{
		if(!file_name)
			return;

		if((f = fopen((const char *) file_name, "a")) == NULL)
		{
			fprintf(stderr, "error opening %s\n", file_name);
			return;
		}

		if( GTK_TOGGLE_BUTTON(obj)->active)
		{
			label = GTK_LABEL( GTK_BIN(obj)->child);
			gtk_label_get(label, &label_text);
		
			if( strcmp(label_text, gi.tab_label->label) == 0)
			{
				ipsc_network_fprint_general(&n, f);
			}
			else if( strcmp(label_text, si.tab_label->label) == 0)
			{
				ipsc_network_fprint_subnets(&n, f);
			}
			else if( strcmp(label_text, hi.tab_label->label) == 0)
			{
				ipsc_network_fprint_host(&n, f);
			}
			else if( strcmp(label_text, ci.tab_label->label) == 0)
			{
				ipsc_network_fprint_cidr(&n, f);
			}
			else
			{
				g_assert_not_reached();
				return;	
			}
		}
		
		if(fclose(f) == EOF)
		{
			perror("fclose");
			return;
		}
	}
}

/* Create a file selection window.  Add our own functionality to
 * the selection's main vbox.  The user will be able to choose
 * what he/she'd like to save.
 */
void create_save_dialog_cb(void)
{
	GtkFileSelection *fselect;
	GtkWidget *general_check, *subnets_check, *host_check, *cidr_check; 
	GtkWidget *vbox, *mbox;
	GtkWidget *frame;

	general_check = gtk_check_button_new_with_label(gi.tab_label->label);
	subnets_check = gtk_check_button_new_with_label(si.tab_label->label);
	host_check = gtk_check_button_new_with_label(hi.tab_label->label);
	cidr_check = gtk_check_button_new_with_label(ci.tab_label->label);

	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(general_check), TRUE);
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(subnets_check), TRUE);
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(host_check), TRUE);
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(cidr_check), TRUE);
 
	frame = gtk_frame_new("Selected Information");
 	vbox = gtk_vbox_new(FALSE, 0);

	fselect = GTK_FILE_SELECTION( gtk_file_selection_new("Save..."));
	mbox = fselect->main_vbox;
 
	gtk_widget_hide(fselect->ok_button);
	gtk_widget_hide(fselect->cancel_button);

	gtk_signal_connect_object( GTK_OBJECT(fselect->ok_button),
		"clicked", save_dialog_ok_button_cb, GTK_OBJECT(fselect));

	gtk_signal_connect_object( GTK_OBJECT(fselect->ok_button),
		"clicked", save_dialog_ok_button_cb, GTK_OBJECT(general_check));
	
	gtk_signal_connect_object( GTK_OBJECT(fselect->ok_button),
		"clicked", save_dialog_ok_button_cb, GTK_OBJECT(subnets_check));

	gtk_signal_connect_object( GTK_OBJECT(fselect->ok_button),
		"clicked", save_dialog_ok_button_cb, GTK_OBJECT(host_check));	

	gtk_signal_connect_object( GTK_OBJECT(fselect->ok_button),
		"clicked", save_dialog_ok_button_cb, GTK_OBJECT(cidr_check));

	gtk_signal_connect_object( GTK_OBJECT(fselect->ok_button),
		"clicked", gtk_widget_destroy, GTK_OBJECT(fselect));
        
	gtk_signal_connect_object( GTK_OBJECT(fselect->cancel_button),
		"clicked", gtk_widget_destroy, GTK_OBJECT(fselect));
         
	gtk_file_selection_set_filename(fselect, DEFAULT_SAVE_FILE);

	gtk_box_pack_start( GTK_BOX(mbox), frame, FALSE, FALSE, 0);
	gtk_box_pack_start( GTK_BOX(vbox), general_check, FALSE, FALSE, 0);
	gtk_box_pack_start( GTK_BOX(vbox), subnets_check, FALSE, FALSE, 0);
	gtk_box_pack_start( GTK_BOX(vbox), host_check, FALSE, FALSE, 0);
	gtk_box_pack_start( GTK_BOX(vbox), cidr_check, FALSE, FALSE, 0);      

	gtk_container_add(GTK_CONTAINER(frame), vbox);
 
	gtk_widget_show_all(GTK_WIDGET(fselect));
}

void subnet_bits_value_changed_cb(GtkWidget *widget, GtkSpinButton *sbtn)
{
	n.subnet_bits = gtk_spin_button_get_value_as_int(sbtn);
	ipsc_network_init(&n);
	update_tabs(&n);
}

void update_tabs(const Network *n)
{
	gipsc_update_general(&gi, n);
	gipsc_update_subnets(&si, n);
	gipsc_update_host(&hi, n);
	gipsc_update_cidr(&ci, n);
}

void gipsc_update_main(const GipscMain *mw, const Network *n)
{
	switch( tolower(n->class))
	{
	case 'a':
		gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(mw->class_a_button), 
					      TRUE);
		break;
	case 'b':
		gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(mw->class_b_button), 
					      TRUE);
		break;
	case 'c':	
		gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(mw->class_c_button), 
					      TRUE);
		break;
	}	
	

	gtk_signal_handler_block( GTK_OBJECT(mw->subnet_bits_adj), 
				  mw->subnet_bits_adj_handler_id);
	mw->subnet_bits_adj->upper = n->subnet_bits_max;
	gtk_spin_button_set_value( GTK_SPIN_BUTTON(mw->subnet_bits), 
				   n->subnet_bits);
	gtk_signal_handler_unblock( GTK_OBJECT(mw->subnet_bits_adj), 
				    mw->subnet_bits_adj_handler_id);
}

void set_interface_cb(GtkWidget *widget, gpointer if_name)
{
	ipsc_network_init_by_interface(&n, (char *) if_name);
	gipsc_update_main(&mc, &n);
	update_tabs(&n);
}

void class_button_pressed_cb(GtkWidget *class_button, gpointer class)
{
	int current_bits;

	ipsc_network_set_class_info(&n, GPOINTER_TO_INT(class), TRUE);
	ipsc_network_init(&n);
	update_tabs(&n);
	
	mc.subnet_bits_adj->upper = n.subnet_bits_max;
	current_bits = gtk_spin_button_get_value_as_int(mc.subnet_bits);
	
	if(current_bits > n.subnet_bits_max)
	{
		gtk_spin_button_set_value(mc.subnet_bits, 
			n.subnet_bits_max);
	}
}

GtkWidget *gipsc_create_main(GipscMain *mc)
{
	GtkWidget *frame, *widgets;
	GtkWidget *vbox, *hbox, *vbox_any;
	GSList *group;
	
	vbox = gtk_vbox_new(FALSE, 0);
	
	hbox = gtk_hbox_new(FALSE, 0);
 	gtk_box_pack_start( GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

	frame = gtk_frame_new("Address Type");
	gtk_container_border_width( GTK_CONTAINER(frame), 10);
	gtk_box_pack_start( GTK_BOX(hbox), frame, FALSE, FALSE, 0);

	vbox_any = gtk_vbox_new(FALSE, 5);
	gtk_container_border_width( GTK_CONTAINER(vbox_any), 5);
	gtk_container_add( GTK_CONTAINER(frame), vbox_any);

	/* 
	 * Create the class radio buttons 
	 */ 
	mc->class_a_button = gtk_radio_button_new_with_label(NULL, 
		"Class A (1.x.x.x - 126.x.x.x)");

	group = gtk_radio_button_group( GTK_RADIO_BUTTON(mc->class_a_button));
	mc->class_b_button = gtk_radio_button_new_with_label(group, 
		"Class B (128.x.x.x - 191.x.x.x)");

	group = gtk_radio_button_group( GTK_RADIO_BUTTON(mc->class_b_button));
	mc->class_c_button = gtk_radio_button_new_with_label(group, 
		"Class C (192.x.x.x - 223.x.x.x)");

	gtk_signal_connect( GTK_OBJECT(mc->class_a_button), "pressed", 
		GTK_SIGNAL_FUNC(class_button_pressed_cb), (gpointer) 'a');

	gtk_signal_connect( GTK_OBJECT(mc->class_b_button), "pressed", 
		GTK_SIGNAL_FUNC(class_button_pressed_cb), (gpointer) 'b');

	gtk_signal_connect( GTK_OBJECT(mc->class_c_button), "pressed", 
		GTK_SIGNAL_FUNC(class_button_pressed_cb), (gpointer) 'c');

	gtk_box_pack_start( GTK_BOX(vbox_any), mc->class_a_button, FALSE, FALSE, 0);
	gtk_box_pack_start( GTK_BOX(vbox_any), mc->class_b_button, FALSE, FALSE, 0);
	gtk_box_pack_start( GTK_BOX(vbox_any), mc->class_c_button, FALSE, FALSE, 0);

	/*
	 * Widgets to the right of the "Address Type" frame 
	 */
	vbox_any = gtk_vbox_new(FALSE, 5);
	gtk_container_border_width( GTK_CONTAINER(vbox_any), 10);
	gtk_box_pack_start( GTK_BOX(hbox), vbox_any, FALSE, FALSE, 0);

	/* 
	 * Subnet bits 
 	 */
	mc->subnet_bits_label = GTK_LABEL( gtk_label_new("Subnet bits"));
	gtk_box_pack_start(GTK_BOX(vbox_any), GTK_WIDGET(mc->subnet_bits_label),
		FALSE, FALSE, 0);
	
	mc->subnet_bits_adj = GTK_ADJUSTMENT( gtk_adjustment_new(0, 0, 31, 1, 1, 0));
	
	mc->subnet_bits = GTK_SPIN_BUTTON( gtk_spin_button_new(mc->subnet_bits_adj, 0, 0));
	gtk_box_pack_start( GTK_BOX(vbox_any), 
			    GTK_WIDGET(mc->subnet_bits), 
			    FALSE, 
			    FALSE, 0);
 	
	mc->subnet_bits_adj_handler_id =  
		gtk_signal_connect( GTK_OBJECT(mc->subnet_bits_adj), 
			    "value_changed",
			    GTK_SIGNAL_FUNC(subnet_bits_value_changed_cb),
			    (gpointer) mc->subnet_bits);

	mc->notebook = gtk_notebook_new();
	gtk_notebook_set_tab_pos( GTK_NOTEBOOK(mc->notebook), GTK_POS_TOP);
	gtk_box_pack_start( GTK_BOX(vbox), mc->notebook, FALSE, FALSE, 0);
	gtk_widget_set_usize(mc->notebook, 350, 300);
	gtk_signal_connect( GTK_OBJECT(mc->notebook),
			    "switch_page",
			    GTK_SIGNAL_FUNC(notebook_switch_page_cb),
			    NULL);

	widgets = gipsc_create_general(&gi);	
	gtk_notebook_append_page( GTK_NOTEBOOK(mc->notebook), 
				  GTK_WIDGET(widgets), 
				  GTK_WIDGET(gi.tab_label));

	widgets = gipsc_create_subnets(&si);
	gtk_notebook_append_page( GTK_NOTEBOOK(mc->notebook), 
				  GTK_WIDGET(widgets), 
				  GTK_WIDGET(si.tab_label));

	widgets = gipsc_create_host(&hi);
	gtk_notebook_append_page( GTK_NOTEBOOK(mc->notebook), 
				  GTK_WIDGET(widgets), 
				  GTK_WIDGET(hi.tab_label));

	widgets = gipsc_create_cidr(&ci);
	gtk_notebook_append_page( GTK_NOTEBOOK(mc->notebook), 
				  GTK_WIDGET(widgets), 
				  GTK_WIDGET(ci.tab_label));
	return(vbox);
}

GnomeUIInfo *get_interface_menu(void)
{
	GnomeUIInfo *ui_info;
	GnomeUIInfo ui_info_end = GNOMEUIINFO_END;
	static char **if_names = NULL;
	int if_count, i;

	if_names = ifc_get_names(&if_count);

	ui_info = malloc((if_count +1) 	
			* sizeof(GnomeUIInfo));

	for(i = 0; i < if_count; i++)
	{
  		GnomeUIInfo ui_info_tmp = 
			      { 
				GNOME_APP_UI_ITEM, 
			       	if_names[i], 
				NULL,
    			       	set_interface_cb, 
			       	(gpointer) if_names[i],
			       	NULL, 
			       	GNOME_APP_PIXMAP_NONE, 
			       	NULL, 0, 0, NULL 
			      };

		ui_info[i] = ui_info_tmp;
	}
	ui_info[i] = ui_info_end;

	return(ui_info);
}

int main(int argc, char *argv[])
{
	GtkWidget *app_bar;
	GtkWidget *widgets;

	GnomeUIInfo file_menu [] = {                     
		GNOMEUIINFO_MENU_SAVE_AS_ITEM(create_save_dialog_cb, NULL),
		GNOMEUIINFO_MENU_EXIT_ITEM(exit_cb, NULL),      
		GNOMEUIINFO_END         
	};                      

	GnomeUIInfo help_menu [] = {                     
		GNOMEUIINFO_MENU_ABOUT_ITEM(about_cb, NULL),          
		GNOMEUIINFO_END         
	};                      

	GnomeUIInfo opts_menu[] = {                     
		{
			GNOME_APP_UI_ITEM, "_Specify Network",
			"Specify network information",
			specify_network_cb, NULL, 
			NULL, 0, 0, 'n', GDK_CONTROL_MASK
		},
		GNOMEUIINFO_END         
	};     	                 
                             
	GnomeUIInfo interface_menu [] = {
		GNOMEUIINFO_RADIOLIST(get_interface_menu()),
		GNOMEUIINFO_END
	};

	GnomeUIInfo main_menu [] = {                     
		GNOMEUIINFO_MENU_FILE_TREE(file_menu),
		GNOMEUIINFO_SUBTREE("_Options", opts_menu),
		GNOMEUIINFO_SUBTREE("_Interface", interface_menu),
		GNOMEUIINFO_MENU_HELP_TREE(help_menu),          
		GNOMEUIINFO_END         
	};                      

	gnome_init(APP_ID, APP_VERSION, argc, argv);
	
	mc.app = gnome_app_new(APP_ID, APP_NAME);    
	gtk_signal_connect(GTK_OBJECT(mc.app), "delete_event",
		GTK_SIGNAL_FUNC(gtk_main_quit),
		NULL);

	widgets = gipsc_create_main(&mc);
	app_bar = gnome_appbar_new(FALSE, TRUE, GNOME_PREFERENCES_USER);              

	gnome_app_set_contents(GNOME_APP(mc.app), widgets);
 	gnome_app_create_menus(GNOME_APP(mc.app), main_menu);  
	gnome_app_set_statusbar(GNOME_APP(mc.app), app_bar); 
	gnome_app_install_menu_hints(GNOME_APP(mc.app), main_menu);      
	
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(mc.class_c_button), TRUE);
	
	gtk_signal_emit_by_name( GTK_OBJECT(mc.class_c_button), "pressed");

	gtk_widget_show_all(mc.app);

	gtk_main();
	return(0);
}
