/*
 * Configurable ps-like program.
 * Linux-specific process information collection routines.
 *
 * Copyright (c) 2008 David I. Bell
 * Permission is granted to use, distribute, or modify this source,
 * provided that this copyright notice remains intact.
 */

#include <dirent.h>
#include <sys/stat.h>
#include <errno.h>

#include "ips.h"


#define	PROCDIR		"/proc"		/* path for proc filesystem */
#define	BEGNAMECHAR	'('		/* begin char for program name */
#define	ENDNAMECHAR	')'		/* end char for program name */

/*
 * A generous size of a buffer which will hold the /proc file names
 * that we are interested in.  These file names only contain small fixed
 * components along with one or two possible integer values.
 */
#define	PROC_FILE_LEN	80


/*
 * Static variables.
 */
static	DIR *	procdir;		/* opendir for /proc */


/*
 * Local procedures.
 */
static	void	ExamineProcessId(int pid);
static	void	GetProcessCommandLine(PROC * proc);
static	void	GetProcessEnvironment(PROC * proc);
static	void	GetProcessOpenFileCount(PROC * proc);
static	void	GetProcessCurrentDirectory(PROC * proc);
static	void	GetProcessRootDirectory(PROC * proc);
static	void	GetProcessExecInode(PROC * proc);
static	void	GetProcessStdioDescriptors(PROC * proc);
static	void	GetProcessWaitSymbol(PROC * proc);

static	BOOL	ReadLinkPath(const char * name, char ** retpath,
			int * retpathlength);


/*
 * Initialize things that we need up front for process status collection.
 * Returns TRUE if successful.
 */
BOOL
InitializeProcessData(void)
{
	CollectStaticSystemInfo();

	if (use_user_names)
		CollectUserNames();

	if (use_group_names)
		CollectGroupNames();

	if (use_device_names)
		CollectDeviceNames();

	/*
	 * Open the /proc directory so that the names in it can be found.
	 */
	procdir = opendir(PROCDIR);

	if (procdir == NULL)
	{
		fprintf(stderr, "Cannot open %s\n", PROCDIR);

		return FALSE;
	}

	ancient_flag = TRUE;

	return TRUE;
}


/*
 * Collect system information that we need.
 * This information collected here is static.
 */
void
CollectStaticSystemInfo(void)
{
	char *	cp;
	int	fd;
	int	cc;
	char	buf[256];

	/*
	 * Collect the amount of memory on the system.
	 */
	fd = open(PROCDIR "/meminfo", O_RDONLY);

	if (fd < 0)
		return;

	cc = read(fd, buf, sizeof(buf) - 1);

	(void) close(fd);

	if (cc <= 0)
		return;

	buf[cc] = '\0';

	cp = strstr(buf, "MemTotal:");

	if (cp == NULL)
		return;

	cp += 9;

	total_memory_clicks = GetDecimalNumber(&cp) / (CLICK_SIZE / 1024);

	/*
	 * Get the starting uptime and time of day.
	 * This will be used to determine the age of processes.
	 */
	start_uptime = GetUptime();
	start_time = time(NULL);
}


/*
 * Get the uptime of the system in jiffies.  Since the /proc data format
 * is floating point seconds, we have to convert it.
 * Returns 0 if something is wrong.
 */
ULONG
GetUptime(void)
{
	char *	cp;
	int	fd;
	int	cc;
	ULONG	intval;
	ULONG	fracval;
	ULONG	fracscale;
	char	buf[128];

	fd = open(PROCDIR "/uptime", O_RDONLY);

	if (fd < 0)
		return 0;

	cc = read(fd, buf, sizeof(buf) - 1);

	(void) close(fd);

	if (cc <= 0)
		return 0;

	buf[cc] = '\0';

	cp = buf;

	while (isblank(*cp))
		cp++;

	intval = 0;
	fracval = 0;
	fracscale = 1;

	while (isdigit(*cp))
		intval = intval * 10 + *cp++ - '0';

	if (*cp == '.')
		cp++;

	while (isdigit(*cp))
	{
		fracval = fracval * 10 + *cp++ - '0';
		fracscale = fracscale * 10;
	}

	if ((*cp != ' ') && (*cp != '\n'))
		return 0;

	return (intval * TICKS) + ((fracval * TICKS) / fracscale);
}


/*
 * Scan all processes and set their new state.
 */
void
ScanProcesses(void)
{
	const struct dirent *	dp;
	const char *		name;
	int			pid;
	int			i;

	UpdateTimes();

	/*
	 * If we require our own process information, then get that now.
	 */
	if (use_self)
		ExamineProcessId(my_pid);

	/*
	 * If there were no specific pids given, then scan them all by
	 * reading the numeric entries from the "/proc" directory.
	 * Otherwise, examine just the specified processes.
	 */
	if (pid_count == 0)
	{
		seekdir(procdir, 0);

		while ((dp = readdir(procdir)) != NULL)
		{
			name = dp->d_name;

			pid = 0;

			while (isdigit(*name))
				pid = pid * 10 + *name++ - '0';

			if (*name)
				continue;

			if ((pid != my_pid) || !use_self)
				ExamineProcessId(pid);
		}
	}
	else
	{
		for (i = 0; i < pid_count; i++)
		{
			if ((pid_list[i] != my_pid) || !use_self)
				ExamineProcessId(pid_list[i]);
		}
	}

	RemoveDeadProcesses();

	SortProcesses();

	UpdateProcessCounts();

	ancient_flag = FALSE;
}


/*
 * Collect data about the specified process id.
 * This allocates a new PROC structure if necessary.
 * If the process is successfully examined, the valid flag is set.
 */
static void
ExamineProcessId(int pid)
{
	PROC *		proc;
	int		fd;
	int		cc;
	int		i;
	char *		begname;
	char *		endname;
	char *		cp;
	long		ticks_from_start;
	long		seconds_from_start;
	BOOL		okskip;
	struct	stat	statbuf;
	char		buf[512];
	char		name[PROC_FILE_LEN];

	proc = FindProcess(pid);

	proc->isvalid = FALSE;
	proc->isshown = FALSE;

	okskip = ((pid != my_pid) || !use_self);

	if (okskip && no_self && (pid == my_pid))
		return;

	sprintf(name, "%s/%d/stat", PROCDIR, proc->pid);

	fd = open(name, O_RDONLY);

	if (fd < 0)
	{
		if (errno != ENOENT)
			perror(name);

		return;
	}

	if (fstat(fd, &statbuf) < 0)
	{
		(void) close(fd);

		return;
	}

	proc->deathTime = 0;

	if (okskip)
	{
		if (my_procs && (statbuf.st_uid != my_uid))
		{
			(void) close(fd);

			return;
		}

		if (no_root && (statbuf.st_uid == 0))
			return;

		if ((user_count > 0))
		{
			for (i = 0; i < user_count; i++)
			{
				if (statbuf.st_uid == user_list[i])
					break;
			}

			if (i == user_count)
			{
				(void) close(fd);

				return;
			}
		}

		if ((group_count > 0))
		{
			for (i = 0; i < group_count; i++)
			{
				if (statbuf.st_gid == group_list[i])
					break;
			}

			if (i == group_count)
			{
				(void) close(fd);

				return;
			}
		}
	}

	proc->uid = statbuf.st_uid;
	proc->gid = statbuf.st_gid;

	/*
	 * Read the process status into a buffer.
	 */
	cc = read(fd, buf, sizeof(buf));

	if (cc < 0)
	{
		(void) close(fd);

		return;
	}

	(void) close(fd);

	if (cc == sizeof(buf))
	{
		fprintf(stderr, "status buffer overflow");

		return;
	}

	buf[cc] = '\0';

	/*
	 * The program name begins after a left parenthesis.
	 * Break the status string into two at that point.
	 */
	begname = strchr(buf, BEGNAMECHAR);

	if (begname == NULL)
	{
		fprintf(stderr, "Cannot find start of program name\n");

		return;
	}

	*begname++ = '\0';

	/*
	 * The program name ends after a right parenthesis.
	 * But, look for the rightmost one in case the program name
	 * itself contains a parenthesis!
	 */
	endname = strchr(begname, ENDNAMECHAR);

	if (endname == NULL)
	{
		fprintf(stderr, "Cannot find end of program name\n");

		return;
	}

	while ((cp = strchr(endname + 1, ENDNAMECHAR)) != NULL)
		endname = cp;

	MakePrintable(begname, endname - begname);

	*endname++ = '\0';

	strncpy(proc->program, begname, MAX_PROGRAM_LEN);

	proc->program[MAX_PROGRAM_LEN] = '\0';

	/*
	 * Find the process state character, and then parse the numbers on
	 * the rest of the line to collect the remaining state information.
	 */
	cp = endname;

	while (isblank(*cp))
		cp++;

	if ((*cp == '\0') || (*cp == '\n') || isdigit(*cp))
	{
		fprintf(stderr, "Bad proc state character\n");

		return;
	}

	proc->state = *cp++;

	proc->parent_pid = GetDecimalNumber(&cp);
	proc->process_group = GetDecimalNumber(&cp);
	proc->session_id = GetDecimalNumber(&cp);
	proc->tty_dev = GetDecimalNumber(&cp);
	proc->tty_pgrp = GetDecimalNumber(&cp);
	proc->flags = GetDecimalNumber(&cp);
	proc->min_flt = GetDecimalNumber(&cp);
	proc->cmin_flt = GetDecimalNumber(&cp);
	proc->maj_flt = GetDecimalNumber(&cp);
	proc->cmaj_flt = GetDecimalNumber(&cp);
	proc->utime = GetDecimalNumber(&cp);
	proc->stime = GetDecimalNumber(&cp);
	proc->cutime = GetDecimalNumber(&cp);
	proc->cstime = GetDecimalNumber(&cp);
	proc->priority = GetDecimalNumber(&cp);
	proc->nice = GetDecimalNumber(&cp);
	proc->threadCount = GetDecimalNumber(&cp);
	proc->it_real_value = GetDecimalNumber(&cp);
	proc->start_time_ticks = GetDecimalNumber(&cp);
	proc->vsize = GetDecimalNumber(&cp);
	proc->rss = GetDecimalNumber(&cp);
	proc->rss_limit = GetDecimalNumber(&cp);
	proc->start_code = GetDecimalNumber(&cp);
	proc->end_code = GetDecimalNumber(&cp);
	proc->start_stack = GetDecimalNumber(&cp);
	proc->esp = GetDecimalNumber(&cp);
	proc->eip = GetDecimalNumber(&cp);
	proc->signal = GetDecimalNumber(&cp);
	proc->sigblock = GetDecimalNumber(&cp);
	proc->sigignore = GetDecimalNumber(&cp);
	proc->sigcatch = GetDecimalNumber(&cp);
	proc->wchan = GetDecimalNumber(&cp);
	proc->nswap = GetDecimalNumber(&cp);
	proc->cnswap = GetDecimalNumber(&cp);
	proc->exitsignal = GetDecimalNumber(&cp);
	proc->processor = GetDecimalNumber(&cp);
	proc->realTimePriority = GetDecimalNumber(&cp);
	proc->policy = GetDecimalNumber(&cp);

	/*
	 * Convert the processes start time into clock time and age.
	 * Get the number of ticks after we started when the specified
	 * process started and convert that to elapsed seconds.
	 * This can be positive or negative according to whether the
	 * process started before or after us.
	 */
	ticks_from_start = proc->start_time_ticks - start_uptime;

	if (ticks_from_start >= 0)
		seconds_from_start = ticks_from_start / TICKS;
	else
		seconds_from_start = -((-ticks_from_start) / TICKS);

	/*
	 * Add the elapsed seconds to the clock time when we started
	 * to get the clock time the process started.
	 */
	proc->start_time_clock = start_time + seconds_from_start;

	/*
	 * See if the process has changed state, and is therefore active.
	 */
	CheckActiveProcess(proc);

	/*
	 * Get several pieces of extra data, but only if the process
	 * has changed state, and only if these data are required.
	 * However, get the extra data always if it is older than
	 * the specified sync time.
	 */
	if (proc->changed ||
		((proc->last_sync_time + sync_time) <= current_time))
	{
		proc->last_sync_time = current_time;

		GetProcessOpenFileCount(proc);
		GetProcessStdioDescriptors(proc);
		GetProcessCurrentDirectory(proc);
		GetProcessRootDirectory(proc);
		GetProcessExecInode(proc);
		GetProcessCommandLine(proc);
		GetProcessEnvironment(proc);
		GetProcessWaitSymbol(proc);
	}

	proc->live_counter = live_counter;
	proc->isvalid = TRUE;

	if (IsShownProcess(proc))
		proc->isshown = TRUE;
}


/*
 * Get the wait channel symbol name for the process.
 */
static void
GetProcessWaitSymbol(PROC * proc)
{
	int	fd;
	int	len;
	char	name[PROC_FILE_LEN];

	/*
	 * If the wait channel is 0 or is not used then
	 * set an empty symbol name.
	 */
	if (!use_wchan || (proc->wchan == 0))
	{
		proc->wchan_symbol[0] = '\0';

		return;
	}

	/*
	 * Open the wait channel file and read it into a buffer
	 * including trying to read one extra character.
	 */
	sprintf(name, "%s/%d/wchan", PROCDIR, proc->pid);

	fd = open(name, O_RDONLY);
	len = -1;

	if (fd >= 0)
	{
		len = read(fd, proc->wchan_symbol, MAX_WCHAN_LEN + 1);

		(void) close(fd);
	}

	/*
	 * If the symbol wasn't found then store a dash for the symbol.
	 */
	if (len < 0)
	{
		proc->wchan_symbol[0] = '-';
		proc->wchan_symbol[1] = '\0';

		return;
	}

	/*
	 * If we read one more character than our limit, then we missed
	 * some of the symbol name line, so flag that by replacing the
	 * last character of the allowed length with a vertical bar.
	 */
	if (len > MAX_WCHAN_LEN)
	{
		len = MAX_WCHAN_LEN;
		proc->wchan_symbol[MAX_WCHAN_LEN - 1] = '|';
	}

	/*
	 * Null terminate the wait symbol name and make it printable.
	 */
	proc->wchan_symbol[len] = '\0';

	MakePrintable(proc->wchan_symbol, len);
}


/*
 * Get the command line for the specified process.
 * If there isn't one, then use the program name as the command line,
 * surrounded by parenthesis.  If the command line is small, then it
 * fits within the proc structure, otherwise we have to malloc it.
 */
static void
GetProcessCommandLine(PROC * proc)
{
	int	fd;
	int	len;
	char	name[PROC_FILE_LEN];
	char	buffer[MAX_COMMAND_LEN + 2];

	len = 0;

	if (!use_command)
	{
		proc->hascommand = FALSE;
		proc->command_length = 0;
		proc->command[0] = '\0';

		return;
	}

	proc->hascommand = TRUE;

	/*
	 * Open the command line file and read it into a large buffer,
	 * including trying to read one extra character.
	 */
	sprintf(name, "%s/%d/cmdline", PROCDIR, proc->pid);

	fd = open(name, O_RDONLY);

	if (fd >= 0)
	{
		len = read(fd, buffer, MAX_COMMAND_LEN + 1);

		(void) close(fd);
	}

	/*
	 * If we could not get the command line, or if there was none
	 * there, then use the program name surrounded by parenthesis.
	 * Remember that there is no real command line for user tests.
	 */
	if ((fd < 0) || (len <= 0))
	{
		proc->hascommand = FALSE;
		len = strlen(proc->program);

		buffer[0] = '(';
		memcpy(&buffer[1], proc->program, len);
		buffer[len + 1] = ')';

		len += 2;
	}

	/*
	 * If we read one more character than our limit, then we missed
	 * some of the command line, so flag that by replacing the last
	 * character of the allowed length with a vertical bar.
	 */
	if (len > MAX_COMMAND_LEN)
	{
		len = MAX_COMMAND_LEN;
		buffer[MAX_COMMAND_LEN - 1] = '|';
	}

	/*
	 * Null terminate the command line and make it printable.
	 */
	buffer[len] = '\0';

	MakePrintable(buffer, len);

	/*
	 * See if the command line is the same as last time.
	 * If so, then we are done.
	 */
	if ((len == proc->command_length) &&
		(buffer[0] == proc->command[0]) &&
		(buffer[len - 1] == proc->command[len - 1]) &&
		(memcmp(buffer, proc->command, len) == 0))
	{
		return;
	}

	/*
	 * Free any old command line buffer that was there, and point
	 * back to the space already allocated in the proc structure.
	 */
	if (proc->command != proc->command_buffer)
		free(proc->command);

	proc->command = proc->command_buffer;

	/*
	 * If the command line is too large for the proc buffer,
	 * then allocate a new one.
	 */
	if (len > BUF_COMMAND_LEN)
	{
		proc->command = malloc(len + 1);

		if (proc->command == NULL)
		{
			fprintf(stderr, "No memory\n");

			exit(1);
		}
	}

	/*
	 * Copy the command line into the command buffer and set its length.
	 */
	proc->command_length = len;
	memcpy(proc->command, buffer, len + 1);
}


/*
 * Get the environment for the specified process.
 * This could be very large, so it is allocated dynamically and we
 * attempt to share strings among processes.
 */
static void
GetProcessEnvironment(PROC * proc)
{
	int	fd;
	int	len;
	char	name[PROC_FILE_LEN];
	char	buffer[MAX_ENVIRON_LEN + 2];

	len = 0;

	if (!use_environment)
		return;

	/*
	 * Open the environment file and read it into a large buffer,
	 * including trying to read one extra character.
	 */
	sprintf(name, "%s/%d/environ", PROCDIR, proc->pid);

	fd = open(name, O_RDONLY);

	if (fd >= 0)
	{
		len = read(fd, buffer, MAX_ENVIRON_LEN + 1);

		(void) close(fd);
	}

	/*
	 * If we could not open the file, or if there was nothing there,
	 * then free any old environment string and point to a null string.
	 */
	if ((fd < 0) || (len <= 0))
	{
		if (proc->environment != empty_string)
			FreeSharedString(proc->environment);

		proc->environment = empty_string;
		proc->environment_length = 0;

		return;
	}

	/*
	 * If we read one more character than our limit, then we missed
	 * some of the environment, so flag that by replacing the last
	 * character of the allowed length with a vertical bar.
	 */
	if (len > MAX_ENVIRON_LEN)
	{
		len = MAX_ENVIRON_LEN;
		buffer[MAX_ENVIRON_LEN - 1] = '|';
	}

	/*
	 * Null terminate the environment string and make it printable.
	 */
	buffer[len] = '\0';

	MakePrintable(buffer, len);

	/*
	 * Compare the string against what is already there.
	 * If they are the same, then we don't need to do anything.
	 */
	if ((proc->environment_length == len) &&
		(proc->environment[0] == buffer[0]) &&
		(memcmp(proc->environment, buffer, len) == 0))
	{
		return;
	}

	/*
	 * Free the old string, and copy the buffer into a new shared string.
	 * If the string is the same as the one from another process (which
	 * is likely), then it will be shared.
	 */
	if (proc->environment != empty_string)
		FreeSharedString(proc->environment);

	proc->environment = AllocateSharedString(buffer, len);
	proc->environment_length = len;

	if (proc->environment == NULL)
	{
		proc->environment = empty_string;
		proc->environment_length = 0;
	}
}


/*
 * Get the number of open files for the process.
 * This is expensive and so is only gotten if the column is actually in use.
 * The permissions only allow this information to be gotten for the same user
 * id or if you are running as root.
 */
static void
GetProcessOpenFileCount(PROC * proc)
{
	DIR *			dir;
	const struct dirent *	dp;
	const char *		cp;
	int			count;
	char			name[PROC_FILE_LEN];

	proc->openfiles = -1;

	if (!use_open_files)
		return;

	sprintf(name, "%s/%d/fd", PROCDIR, proc->pid);

	dir = opendir(name);

	if (dir == NULL)
		return;

	count = 0;

	while ((dp = readdir(dir)) != NULL)
	{
		cp = dp->d_name;

		while (isdigit(*cp))
			cp++;

		if (*cp == '\0')
			count++;
	}

	closedir(dir);

	proc->openfiles = count;
}


/*
 * Get the current working directory of the specified process.
 * This is expensive and so is only gotten if the column is actually in use.
 * The permissions only allow this information to be gotten for the same user
 * id or if you are running as root.
 */
static void
GetProcessCurrentDirectory(PROC * proc)
{
	char	name[PROC_FILE_LEN];

	if (!use_current_directory)
		return;

	sprintf(name, "%s/%d/cwd", PROCDIR, proc->pid);

	ReadLinkPath(name, &proc->cwd_path, &proc->cwd_path_length);
}


/*
 * Get the current root directory of the specified process.
 * This is expensive and so is only gotten if the column is actually in use.
 * The permissions only allow this information to be gotten for the same user
 * id or if you are running as root.
 */
static void
GetProcessRootDirectory(PROC * proc)
{
	char	name[PROC_FILE_LEN];

	if (!use_root_directory)
		return;

	sprintf(name, "%s/%d/root", PROCDIR, proc->pid);

	ReadLinkPath(name, &proc->root_path, &proc->root_path_length);
}


/*
 * Get the device and inode of the executable file for specified process.
 * This is expensive and so is only gotten if the column is actually in use.
 * The permissions only allow this information to be gotten for the same user
 * id or if you are running as root.
 */
static void
GetProcessExecInode(PROC * proc)
{
	char	name[PROC_FILE_LEN];

	if (!use_exec_inode)
		return;

	sprintf(name, "%s/%d/exe", PROCDIR, proc->pid);

	ReadLinkPath(name, &proc->exec_path, &proc->exec_path_length);
}


/*
 * Get information about processes three standard file descriptors.
 * This is expensive and so is only gotten if the columns are actually in use.
 * The permissions only allow this information to be gotten for the same user
 * id or if you are running as root.
 */
static void
GetProcessStdioDescriptors(PROC * proc)
{
	int	fd;
	int	dummy;
	char	name[PROC_FILE_LEN];

	for (fd = 0; fd <= 2; fd++)
	{
		if (!use_stdio[fd])
			continue;

		sprintf(name, "%s/%d/fd/%d", PROCDIR, proc->pid, fd);

		ReadLinkPath(name, &proc->stdio_paths[fd], &dummy);
	}
}


/*
 * Get the destination path and length for the specified symbolic link.
 * The path is returned into the indicated variables, which are updated.
 * The returned string is allocated within a shared pool of strings and
 * so can only be freed by calling the appropriate routine.  Returns TRUE
 * if the information was able to be obtained.
 */
static BOOL
ReadLinkPath(const char * name, char ** retpath, int * retpathlength)
{
	char *	newpath;
	char *	oldpath;
	int	len;
	int	oldlength;
	char	buffer[MAX_PATH_LEN];

	/*
	 * Save the values for the old path.
	 */
	oldpath = *retpath;
	oldlength = *retpathlength;

	/*
	 * Read the value of the symbolic link.
	 */
	len = readlink(name, buffer, sizeof(buffer));

	if ((len <= 0) || (len == sizeof(buffer)))
	{
		FreeSharedString(oldpath);

		*retpath = empty_string;
		*retpathlength = 0;

		return FALSE;
	}

	buffer[len] = '\0';

	/*
	 * If the path is the same as the existing one then do nothing.
	 */
	if ((len == oldlength) && (strcmp(oldpath, buffer) == 0))
		return TRUE;

	/*
	 * The value has changed.  Delete the old string.
	 */
	FreeSharedString(oldpath);

	*retpath = empty_string;
	*retpathlength = 0;

	/*
	 * Allocate a new shared string to store the new value.
	 */
	newpath = AllocateSharedString(buffer, len);

	if (newpath == NULL)
		return FALSE;

	/*
	 * Return the new string value.
	 */
	*retpath = newpath;
	*retpathlength = len;

	return TRUE;
}

/* END CODE */
