/*
 * Configurable ps-like program.
 * Main program.
 *
 * Copyright (c) 2008 David I. Bell
 * Permission is granted to use, distribute, or modify this source,
 * provided that this copyright notice remains intact.
 */

#include <signal.h>
#include <sys/stat.h>

#include "ips.h"


/*
 * List of columns being shown.
 */
int	show_count;
COLUMN *show_list[MAX_COLUMNS];


/*
 * Other global variables.
 */
ULONG		start_uptime;		/* uptime jiffies at start */
time_t		start_time;		/* clock time at start */
time_t		current_time;		/* current clock time */
long		total_memory_clicks;	/* amount of total memory */
ULONG		live_counter;		/* counter for live procs */
BOOL		ancient_flag;		/* seeing pre-existing procs */
BOOL		no_self;		/* don't show myself */
BOOL		no_root;		/* don't show root procs */
BOOL		info;			/* show info line at top */
BOOL		no_header;		/* don't show column header */
BOOL		my_procs;		/* only show my procs */
BOOL		active_only;		/* only show active procs */
BOOL		clear_screen;		/* clear screen each loop */
BOOL		loop;			/* loop showing status */
BOOL		running;		/* we still want to run */
BOOL		frozen;			/* data collection is frozen */
BOOL		update;			/* update once even if frozen */
BOOL		needRefresh;		/* need to refresh display */
BOOL		useCurses;		/* use curses display */
BOOL		useX11;			/* use X11 display */
BOOL		vertical;		/* vertical output format */
BOOL		top_set;		/* top option was used */
BOOL		top_auto;		/* autosize height for top */
BOOL		use_open_files;		/* using open file info */
BOOL		use_current_directory;	/* using current dir info */
BOOL		use_root_directory;	/* using root dir info */
BOOL		use_exec_inode;		/* using executable info */
BOOL		use_device_names;	/* using device name info */
BOOL		use_user_names;		/* using user name info */
BOOL		use_group_names;	/* using group name info */
BOOL		use_init;		/* using initial sleep */
BOOL		use_command;		/* using command line info */
BOOL		use_self;		/* using my own proc info */
BOOL		use_environment;	/* using environment info */
BOOL		use_wchan;		/* using wait channel symbol */
BOOL		use_stdio[3];		/* using various stdio info */
pid_t		my_pid;			/* my pid */
uid_t		my_uid;			/* my real user id */
gid_t		my_gid;			/* my real group id */
dev_t		null_dev;		/* device of /dev/null */
ino_t		null_inode;		/* inode of /dev/null */
int		proc_alloc_count;	/* allocated proc structures */
int		deathTime;		/* seconds for dead processes */
int		active_time;		/* seconds for active procs */
int		pid_count;		/* pids in pid_list */
int		user_count;		/* users in user_list */
int		group_count;		/* groups in group_list */
int		program_count;		/* programs in program_list */
int		width;			/* width of output */
int		height;			/* height of output */
int		separation;		/* blanks between columns */
int		sleep_time_ms;		/* milliseconds between loops */
int		sync_time;		/* seconds between syncs */
int		init_time;		/* seconds for initial sleep */
int		top_count;		/* number of procs for top */
int		scrollSeconds;		/* seconds between scrolling */
int		overlapLines;		/* lines of overlap */
int		skipCount;		/* lines to skip in display */
int		procShowCount;		/* processes wanting showing */
int		procTotalCount;		/* count of all processes */
DISPLAY *	display;		/* the output display device */
PROC *		process_list;		/* list of existent procs */
PROC *		free_process_list;	/* free proc structure list */
char *		geometry;		/* window geometry string */
char *		fontName;		/* font name */
char *		foregroundName;		/* foreground color name */
char *		backgroundName;		/* background color name */
char *		displayName;		/* display name */
const char *	displayType = DISPLAY_TYPE_TTY;		/* display type */
char		empty_string[4];	/* empty string */
char		root_string[4] = "/";	/* root path string */
pid_t		pid_list[MAX_PIDS];	/* pids to be shown */
uid_t		user_list[MAX_USERS];	/* user ids to be shown */
gid_t		group_list[MAX_GROUPS];	/* group ids to be shown */
char		program_list[MAX_PROGRAMS][MAX_PROGRAM_LEN + 2];


/*
 * Static procedures.
 */
static	void	GetTerminalSize(void);
static	void	SetUseFlags(void);
static	void	VerifyDescriptors(void);
static	void	HandleSigPipe(int);


int
main(int argc, char ** argv)
{
	ARGS *	ap;
	ARGS	args;

	/*
	 * Make sure someone hasn't closed off our standard file
	 * descriptors so that we don't accidentally overwrite
	 * something with our output.
	 */
	VerifyDescriptors();

	ap = &args;
	ap->table = ++argv;
	ap->count = --argc;

	my_pid = getpid();
	my_uid = getuid();
	my_gid = getgid();

	DpySetDisplay(NULL);

	DefaultAllOptions();

	/*
	 * Parse the system definition file.
	 */
	if (!ParseSystemInitFile())
		return 1;

	/*
	 * Check specially for the -noinit option before reading the user's
	 * option file.  This option MUST be immediately after the program
	 * name.  This check is a bit of a hack.
	 */
	if ((ap->count > 0) && (strcmp(ap->table[0], "-noinit") == 0))
	{
		ap->table++;
		ap->count--;
	}
	else if (!ParseUserInitFile())
		return 1;

	/*
	 * Look up and execute the initial macros if they exist.
	 */
	if (MacroExists(MACRO_TYPE_OPTION, SYSTEM_INIT_MACRO) &&
		!ExpandOptionName(SYSTEM_INIT_MACRO, 0))
	{
		fprintf(stderr,
			"Problem expanding system initial macro \"%s\"\n",
			SYSTEM_INIT_MACRO);

		return 1;
	}

	if (MacroExists(MACRO_TYPE_OPTION, USER_INIT_MACRO) &&
		!ExpandOptionName(USER_INIT_MACRO, 0))
	{
		fprintf(stderr,
			"Problem expanding system initial macro \"%s\"\n",
			USER_INIT_MACRO);

		return 1;
	}

	/*
	 * Parse the command line options possibly expanding them.
	 * Indicate that we are level zero so that bare macro names on the
	 * command line will be recognized even if they are lower case.
	 */
	if (!ParseOptions(ap, 0))
		return 1;

	/*
	 * Set the flags for what items we need collecting based on
	 * the columns that have been referenced.
	 */
	SetUseFlags();

	/*
	 * Initialize for collecting of process data.
	 */
	if (!InitializeProcessData())
		return 1;

	/*
	 * Catch SIGPIPE so we can be clean if a window is closed.
	 */
	signal(SIGPIPE, HandleSigPipe);

	/*
	 * Initialize for displaying of process status.
	 */
	if (!InitializeDisplay())
		return 1;

	/*
	 * If we require a time interval before displaying results
	 * (such as for calculating percentage cpu), then collect
	 * the initial process status and sleep a while.
	 */
	InitialProcessScan();

	/*
	 * Here is the main loop.  It terminates after one iteration
	 * if we just want a snapshot.
	 */
	running = TRUE;

	while (running)
	{
		/*
		 * Release any string storage used during the last loop.
		 */
		FreeTempStrings();

		/*
		 * Collect the new process status if we aren't frozen
		 * unless the update once flag was also set.
		 */
		if (!frozen || update)
			ScanProcesses();

		update = FALSE;

		/*
		 * Recalculate the window size information.
		 */
		GetTerminalSize();

		/*
		 * Show the selected processes.
		 */
		ShowSelectedProcesses();

		/*
		 * If we don't want to loop, then we are done.
		 */
		if (!loop)
			break;

		/*
		 * Sleep while handling commands if any.
		 * If we are frozen then sleep a long time (i.e, 10 minutes).
		 */
		WaitForCommands(frozen ? 60*10*1000 : sleep_time_ms);
	}

	/*
	 * Close the display and return success.
	 */
	DpyClose();

	return 0;
}


/*
 * Set the use variables according to the items required by the columns
 * that have been referenced.  These variables are used to avoid
 * collecting expensive-to-obtain data when it is not required.
 */
static void
SetUseFlags(void)
{
	int	i;
	USEFLAG	flags;

	flags = USE_NONE;

	/*
	 * Scan the columns that are being shown.
	 */
	for (i = 0; i < show_count; i++)
		flags |= show_list[i]->useflag;

	/*
	 * Add in the columns that are being sorted by.
	 */
	flags |= GetSortingUseFlags();

	/*
	 * Add in the columns referenced by conditions.
	 */
	flags |= GetConditionUseFlags();

	/*
	 * Now set the boolean variables according to the flags.
	 */
	if (flags & USE_INIT)
		use_init = TRUE;

	if (flags & USE_USER_NAME)
		use_user_names = TRUE;

	if (flags & USE_GROUP_NAME)
		use_group_names = TRUE;

	if (flags & USE_DEV_NAME)
		use_device_names = TRUE;

	if (flags & USE_OPEN_FILE)
		use_open_files = TRUE;

	if (flags & USE_CURR_DIR)
		use_current_directory = TRUE;

	if (flags & USE_ROOT_DIR)
		use_root_directory = TRUE;

	if (flags & USE_EXEC_INODE)
		use_exec_inode = TRUE;

	if (flags & USE_COMMAND)
		use_command = TRUE;

	if (flags & USE_SELF)
		use_self = TRUE;

	if (flags & USE_ENVIRON)
		use_environment = TRUE;

	if (flags & USE_WCHAN)
		use_wchan = TRUE;

	if (flags & USE_STDIN)
		use_stdio[0] = TRUE;

	if (flags & USE_STDOUT)
		use_stdio[1] = TRUE;

	if (flags & USE_STDERR)
		use_stdio[2] = TRUE;
}


/*
 * Routine called to get the new terminal size from the display device.
 */
static void
GetTerminalSize(void)
{
	width = DpyGetCols();
	height = DpyGetRows();

	if (width <= 0)
		width = 1;

	if (height <= 0)
		height = 1;

	/*
	 * If we are automatically doing the top few processes
	 * then set the top count to the number of lines.
	 */
	if (top_auto)
	{
		top_count = height;

		if (DpyDoesScroll())
			top_count--;

		if (!no_header)
			top_count--;

		if (info)
			top_count--;

		if (top_count <= 0)
			top_count = 1;
	}
}


/*
 * Make sure that the standard file descriptors are opened.
 * If not, then stop right now without doing anything.
 * This check is required so that we can't be tricked into
 * writing error output into opened files.
 */
static void
VerifyDescriptors(void)
{
	struct stat	statbuf;

	if ((fstat(STDIN, &statbuf) < 0) || (fstat(STDOUT, &statbuf) < 0)
		|| (fstat(STDERR, &statbuf) < 0))
	{
		_exit(99);
	}
}


/*
 * Routine to catch SIGPIPE so we can exit cleanly.
 */
static void
HandleSigPipe(int arg)
{
	_exit(2);
}

/* END CODE */
