// --------------------------------------------------------------------
// Canvas Painter
// --------------------------------------------------------------------
/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (C) 1993-2007  Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "ipetext.h"
#include "ipepdfparser.h"
#include "ipecanvas.h"
#include "ipecanvaspainter.h"
#include "ipefonts.h"

#include <QPainter>
#include <QImage>
#include <QPaintDevice>
#include <QApplication>
#include <QPixmap>
#include <QPainterPath>

// What a mess is Windows!
#ifdef GetObject
#undef GetObject
#endif

inline QPointF QPt(const IpeVector &v)
{
  return QPointF(v.iX, v.iY);
}

inline IpeString IpeQ(const QString &str)
{
  return IpeString(str.toUtf8());
}

inline QString QIpe(const IpeString &str)
{
  return QString::fromUtf8(str.CString());
}

// --------------------------------------------------------------------

class IpeRenderData : public IpeBitmap::MRenderData {
public:
  virtual ~IpeRenderData() { /* Nothing */ }
  QImage iImage;
};

// --------------------------------------------------------------------

IpeCanvasPainter::IpeCanvasPainter(const IpeStyleSheet *sheet,
				   IpeCanvasFonts *fonts, QPainter *painter,
				   double zoom, bool pretty,
				   int maxBitmapSize)
  : IpePainter(sheet), iFonts(fonts),
    iPainter(painter), iZoom(zoom), iPretty(pretty),
    iMaxBitmapSize(maxBitmapSize)
{
  iDimmed = false;
  iPainter->setFont(QApplication::font());
  iFont = 0;
  // these symbolic dash styles are on fixed indices
  for (int i = 0; i < 4; ++i)
    iDash[i] =
      StyleSheet()->Find(IpeAttribute(IpeAttribute::EDashStyle, true, i+1));
}

void IpeCanvasPainter::DoMoveTo(const IpeVector &v)
{
  if (iType.size() > 0 && iType.back() != EEndClosedPath)
    iType.push_back(EEndPath);
  iV.push_back(QPt(Matrix() * v));
  iType.push_back(EMoveTo);
}

void IpeCanvasPainter::DoLineTo(const IpeVector &v)
{
  iV.push_back(QPt(Matrix() * v));
  iType.push_back(ELineTo);
}

void IpeCanvasPainter::DoCurveTo(const IpeVector &v1, const IpeVector &v2,
				 const IpeVector &v3)
{
  iV.push_back(QPt(Matrix() * v1));
  iV.push_back(QPt(Matrix() * v2));
  iV.push_back(QPt(Matrix() * v3));
  iType.push_back(ECurveTo);
}

void IpeCanvasPainter::DoClosePath()
{
  iType.push_back(EEndClosedPath);
}

void IpeCanvasPainter::DoDrawPath()
{
  if (iType.size() > 0 && iType.back() != EEndClosedPath)
    iType.push_back(EEndPath);

  const IpeRepository *rep = StyleSheet()->Repository();

  // build path object
  QPainterPath path;
  uint k = 0;
  for (uint i = 0; i < iType.size(); ++i) {
    switch (iType[i]) {
    case EMoveTo:
      path.moveTo(iV[k++]);
      break;
    case ELineTo:
      path.lineTo(iV[k++]);
      break;
    case ECurveTo:
      path.cubicTo(iV[k], iV[k+1], iV[k+2]);
      k += 3;
      break;
    case EEndPath:
      break;
    case EEndClosedPath:
      path.closeSubpath();
      break;
    }
  }
  IpeAttribute windRule = WindRule();
  if (!windRule)
    windRule = StyleSheet()->WindRule();
  Qt::FillRule wind = (windRule.Index() != 0) ?
    Qt::WindingFill : Qt::OddEvenFill;
  path.setFillRule(wind);

  // clear path
  iType.clear();
  iV.clear();

  IpeAttribute fill = Fill();
  IpeAttribute stroke = Stroke();
  IpeAttribute dash = DashStyle();

  // nothing to draw?
  if (fill.IsNullOrVoid() && dash.IsVoid())
    return;

  if (!fill.IsNullOrVoid()) {
    assert(fill.IsAbsolute());
    IpeColor fillColor = rep->ToColor(fill);
    QColor qfill(int(fillColor.iRed * 255),
		 int(fillColor.iGreen * 255),
		 int(fillColor.iBlue * 255));
    QBrush brush;
    DimColor(qfill);
    brush.setColor(qfill);
    brush.setStyle(Qt::SolidPattern);
    iPainter->setBrush(brush);
  } else
    iPainter->setBrush(Qt::NoBrush);

  if (!dash.IsVoid() && !stroke.IsNull()) {
    assert(stroke.IsAbsolute());
    IpeColor strokeColor = rep->ToColor(stroke);
    QPen pen;
    QColor qstroke(int(strokeColor.iRed * 255),
		   int(strokeColor.iGreen * 255),
		   int(strokeColor.iBlue * 255));
    DimColor(qstroke);
    pen.setColor(qstroke);
    // width
    if (LineWidth()) {
      assert(LineWidth().IsAbsolute());
      double wid = iZoom * rep->ToScalar(LineWidth()).ToDouble();
      pen.setWidth(uint(wid + 0.5));
    }
    IpeAttribute join = LineJoin();
    if (!join)
      join = StyleSheet()->LineJoin();
    switch (join.Index()) {
    case IpeStrokeStyle::EMiterJoin:
      pen.setJoinStyle(Qt::MiterJoin);
      break;
    case IpeStrokeStyle::ERoundJoin:
      pen.setJoinStyle(Qt::RoundJoin);
      break;
    case IpeStrokeStyle::EBevelJoin:
      pen.setJoinStyle(Qt::BevelJoin);
      break;
    }
    IpeAttribute cap = LineCap();
    if (!cap)
      cap = StyleSheet()->LineCap();
    switch (cap.Index()) {
    case IpeStrokeStyle::EButtCap:
      pen.setCapStyle(Qt::FlatCap);
      break;
    case IpeStrokeStyle::ERoundCap:
      pen.setCapStyle(Qt::RoundCap);
      break;
    case IpeStrokeStyle::ESquareCap:
      pen.setCapStyle(Qt::SquareCap);
      break;
    }
    if (!dash.IsNullOrSolid()) {
      // now comes the tricky part
      if (dash == iDash[0])
	pen.setStyle(Qt::DashLine);
      else if (dash == iDash[1])
	pen.setStyle(Qt::DotLine);
      else if (dash == iDash[2])
	pen.setStyle(Qt::DashDotLine);
      else if (dash == iDash[3])
	pen.setStyle(Qt::DashDotDotLine);
      else
	pen.setStyle(Qt::DotLine);
    }
    iPainter->setPen(pen);
  } else
    iPainter->setPen(Qt::NoPen);

  iPainter->drawPath(path);
}

void IpeCanvasPainter::DoDrawBitmap(IpeBitmap bitmap)
{
  if (!bitmap.RenderData()) {
    // todo: should we remember if this failed?
    IpeBuffer data = bitmap.PixelData();
    if (data.size() > 0) {
      IpeRenderData *render = new IpeRenderData;
      // warning: data buffer must remain alive as long as img is
      QImage img((uchar *) data.data(), bitmap.Width(), bitmap.Height(),
		 QImage::Format_RGB32);
      if (img.width() > iMaxBitmapSize || img.height() > iMaxBitmapSize) {
	int wid = img.width();
	int ht = img.height();
	double factor;
	if (wid > ht)
	  factor = wid / double(iMaxBitmapSize);
	else
	  factor = ht / double(iMaxBitmapSize);
	wid = int(wid / factor + 0.5);
	ht = int(ht / factor + 0.5);
	render->iImage = img.scaled(wid, ht, Qt::KeepAspectRatio,
				    Qt::FastTransformation);
      } else
	render->iImage = img.copy();
      // img no longer needed
      bitmap.SetRenderData(render);
    }
  }

  if (bitmap.RenderData()) {
    IpeRenderData *render = static_cast<IpeRenderData *>(bitmap.RenderData());
    IpeMatrix tf = Matrix() * IpeMatrix(1.0 / render->iImage.width(), 0.0,
					0.0, -1.0 / render->iImage.height(),
					0.0, 1.0);
    QMatrix m;
    m.setMatrix(tf.iA[0], tf.iA[1], tf.iA[2], tf.iA[3], tf.iA[4], tf.iA[5]);
    iPainter->setMatrix(m);
    iPainter->drawImage(0, 0, render->iImage);
    iPainter->resetMatrix();
  }
}

void IpeCanvasPainter::DoDrawText(const IpeText *text)
{
  // Current origin is lower left corner of text box

  // Draw bounding box rectangle
  if (!iPretty && !iDimmed && !text->isInternal()) {
    QPen pen;
    pen.setColor(Qt::green);
    pen.setStyle(Qt::DotLine);
    iPainter->setPen(pen);
    iPainter->setBrush(Qt::NoBrush);
    QPolygonF poly;
    poly.append(QPt(Matrix() * IpeVector(0,0)));
    poly.append(QPt(Matrix() * IpeVector(0, text->TotalHeight())));
    poly.append(QPt(Matrix() * IpeVector(text->Width(), text->TotalHeight())));
    poly.append(QPt(Matrix() * IpeVector(text->Width(), 0)));
    iPainter->drawConvexPolygon(poly);
    iPainter->setPen(Qt::NoPen);
    iPainter->setBrush(Qt::green);
    QPointF ref = QPt(Matrix() * text->Align());
    QRectF r(ref.x() - 3, ref.y() - 3, 6, 6);
    iPainter->drawRect(r);
  }

  IpeAttribute str = Stroke();
  if (str.IsNull()) str = IpeAttribute::Black();
  assert(str.IsAbsolute());
  IpeColor stroke = StyleSheet()->Repository()->ToColor(str);
  QColor col(int(stroke.iRed * 255), int(stroke.iGreen * 255),
	     int(stroke.iBlue * 255));
  DimColor(col);

  const IpeText::XForm *xf = text->getXForm();
  if (!xf || !iFonts) {
    if (!text->isInternal()) {
      QString s = QIpe(text->Text());
      int olen = s.length();
      int i = s.indexOf(QLatin1Char('\n'));
      if (i >= 0)
	s.truncate(i);
      s.truncate(30);
      if (s.length() < olen)
	s += QLatin1String("...");

      QPointF pt = QPt(Matrix().Translation());
      pt.setY(pt.y() - iPainter->fontMetrics().descent());
      iPainter->setPen(col);
      iPainter->drawText(pt, s);
    }
  } else {
    iTextRgb = col.rgb();
    Transform(IpeMatrix(xf->iStretch, 0, 0, xf->iStretch, 0, 0));
    Execute(xf->iStream);
  }
}

// --------------------------------------------------------------------

//! If dimming, replace color by dimmed version.
void IpeCanvasPainter::DimColor(QColor &col)
{
  if (iDimmed) {
    int h, s, v;
    col.getHsv(&h, &s, &v);
    v += 150;
    if (v > 255)
      col = col.light(175);
    else
      col.setHsv(h, s, v);
  }
}

// --------------------------------------------------------------------

//! Clear PDF argument stack
void IpeCanvasPainter::ClearArgs()
{
  while (!iArgs.empty()) {
    delete iArgs.back();
    iArgs.pop_back();
  }
}

void IpeCanvasPainter::Execute(const IpeBuffer &buffer)
{
  IpeBufferSource source(buffer);
  IpePdfParser parser(source);
  iFont = 0;
  iFillRgb = 0x000000;
  iStrokeRgb = 0x000000;
  while (!parser.Eos()) {
    IpePdfToken tok = parser.Token();
    if (tok.iType != IpePdfToken::EOp) {
      const IpePdfObj *obj = parser.GetObject();
      if (!obj)
	break; // no further parsing attempted
      iArgs.push_back(obj);
    } else {
      // its an operator, execute it
      IpeString op = tok.iString;
      parser.GetToken();
      if (op == "cm")
	Opcm();
      else if (op == "Tf")
	OpTf();
      else if (op == "Td")
	OpTd();
      else if (op == "TJ")
	OpTJ();
      else if (op == "rg")
	Oprg(false);
      else if (op == "RG")
	Oprg(true);
      else if (op == "g")
	Opg(false);
      else if (op == "G")
	Opg(true);
      else if (op == "k")
	Opk(false);
      else if (op == "K")
	Opk(true);
      else if (op == "w")
	Opw();
      else if (op == "m")
	Opm();
      else if (op == "l")
	Opl();
      else if (op == "BT")
	OpBT();
      else if (op != "ET") {
	IpeString a;
	for (uint i = 0; i < iArgs.size(); ++i)
	  a += iArgs[i]->Repr() + " ";
	ipeDebug("Op %s (%s)", op.CString(), a.CString());
      }
      ClearArgs();
    }
  }
  ClearArgs();
}

void IpeCanvasPainter::Opg(bool stroke)
{
  if (iArgs.size() != 1 || !iArgs[0]->Number())
    return;
  int gr = int(iArgs[0]->Number()->Value() * 255);
  if (stroke)
    iStrokeRgb = qRgb(gr, gr, gr);
  else
    iFillRgb = qRgb(gr, gr, gr);
}


void IpeCanvasPainter::Oprg(bool stroke)
{
  if (iArgs.size() != 3 || !iArgs[0]->Number() || !iArgs[1]->Number()
      || !iArgs[2]->Number())
    return;
  int r = int(iArgs[0]->Number()->Value() * 255);
  int g = int(iArgs[1]->Number()->Value() * 255);
  int b = int(iArgs[2]->Number()->Value() * 255);
  QColor col(r, g, b);
  DimColor(col);
  if (stroke)
    iStrokeRgb = col.rgb();
  else
    iFillRgb = col.rgb();
}

void IpeCanvasPainter::Opk(bool stroke)
{
  if (iArgs.size() != 4 || !iArgs[0]->Number() || !iArgs[1]->Number()
      || !iArgs[2]->Number() || !iArgs[3]->Number())
    return;
  // ignore values, but use text object stroke color
  if (stroke)
    iStrokeRgb = iTextRgb;
  else
    iFillRgb = iTextRgb;
}

void IpeCanvasPainter::Opcm()
{
  if (iArgs.size() != 6)
    return;
  IpeMatrix m;
  for (int i = 0; i < 6; ++i) {
    if (!iArgs[i]->Number())
      return;
    m.iA[i] = iArgs[i]->Number()->Value();
  }
  Transform(m);
}

void IpeCanvasPainter::Opw()
{
  if (iArgs.size() != 1 || !iArgs[0]->Number())
    return;
  iLineWid = iArgs[0]->Number()->Value();

}

void IpeCanvasPainter::Opm()
{
  if (iArgs.size() != 2 || !iArgs[0]->Number() || !iArgs[1]->Number())
    return;
  IpeVector t(iArgs[0]->Number()->Value(), iArgs[1]->Number()->Value());
  iMoveTo = t;
}

void IpeCanvasPainter::Opl()
{
  if (iArgs.size() != 2 || !iArgs[0]->Number() || !iArgs[1]->Number())
    return;
  IpeVector t(iArgs[0]->Number()->Value(), iArgs[1]->Number()->Value());

  QPen pen;
  pen.setColor(QRgb(iStrokeRgb));
  pen.setWidth(uint(iZoom * iLineWid + 0.5));
  iPainter->setPen(pen);
  iPainter->drawLine(QPt(Matrix() * iMoveTo), QPt(Matrix() * t));
}

void IpeCanvasPainter::OpBT()
{
  iFont = 0;
  iTextPos = IpeVector::Zero;
}

void IpeCanvasPainter::OpTf()
{
  if (iArgs.size() != 2 || !iArgs[0]->Name() || !iArgs[1]->Number())
    return;
  IpeString name = iArgs[0]->Name()->Value();
  iFontSize = iArgs[1]->Number()->Value();
  if (name[0] != 'F')
    return;
  int font = IpeLex(name.substr(1)).GetInt();
  // ipeDebug("Setting font %d at %g", font, iFontSize);
  IpeLinear m = Matrix().Linear() * IpeLinear(iFontSize, 0, 0, iFontSize);
  iFont = iFonts->GetSize(font, m);
}

void IpeCanvasPainter::OpTd()
{
  if (iArgs.size() != 2 || !iArgs[0]->Number() || !iArgs[1]->Number())
    return;
  IpeVector t(iArgs[0]->Number()->Value(), iArgs[1]->Number()->Value());
  iTextPos = iTextPos + t;
}

void IpeCanvasPainter::OpTJ()
{
  if (!iFont || iArgs.size() != 1 || !iArgs[0]->Array())
    return;
  // flush painted data
  // iPainter->end();
  IpeVector pos = iTextPos;
  for (int i = 0; i < iArgs[0]->Array()->Count(); ++i) {
    const IpePdfObj *obj = iArgs[0]->Array()->Obj(i, 0);
    if (obj->Number()) {
      pos.iX -= 0.001 * iFontSize * obj->Number()->Value();
    } else if (obj->String()) {
      IpeString s = obj->String()->Value();
      for (int j = 0; j < s.size(); ++j) {
	uchar ch = s[j];
	IpeVector pt = Matrix() * pos;
	DrawChar(ch, iFillRgb, int(pt.iX + 0.5), int(pt.iY + 0.5));
	pos.iX += 0.001 * iFontSize * iFont->Face()->Width(ch);
      }
    }
  }
}

// --------------------------------------------------------------------

//! Draw a glyph.
/*! Glyph is drawn with hotspot at position (x,y) (on the pixmap) */
bool IpeCanvasPainter::DrawChar(int ch, QRgb rgb, int x, int y)
{
  if (!iFont)
    return false;

  // generate the glyph pixmap

  int xOffset, yOffset, gw, gh;
  uchar *p = iFont->GetGlyph(ch, xOffset, yOffset, gw, gh);
  if (!p)
    return false;

  QImage image(gw, gh, QImage::Format_ARGB32);

  if (iFonts->AntiAlias()) {
    rgb &= 0x00ffffff; // take out alpha
    for (int yy = 0; yy < gh; ++yy) {
      uint *dst = (uint *) image.scanLine(yy);
      for (int xx = 0; xx < gw; ++xx) {
	uint pix = (*p++ & 0xff) << 24;
	*dst++ = pix | rgb;
      }
    }
  } else {
    image.fill(0x00ffffff);
    rgb = 0xff000000 | rgb;
    // one color
    for (int yy = 0; yy < gh; ++yy) {
      uint *dst = (uint *) image.scanLine(yy);
      for (int xx = 0; xx < gw; xx += 8) {
	int pix = *p++;
	for (int xx1 = xx; xx1 < xx + 8 && xx1 < gw; ++xx1) {
	  if (pix & 0x80)
	    *dst = rgb;
	  pix <<= 1;
	  ++dst;
	}
      }
    }
  }
  iPainter->drawImage(x - xOffset, y - yOffset, image);
  return true;
}

// --------------------------------------------------------------------
