// --------------------------------------------------------------------
// Ipelet for handling template objects
// --------------------------------------------------------------------
/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (C) 1993-2004  Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "ipelib.h"
#include "ipestyle.h"
#include "iperef.h"

#include "dlguse.h"
#include <qstringlist.h>
#include <qcombobox.h>

inline QString QIpe(const IpeString &str)
{
  return QString::fromUtf8(str.CString());
}

// --------------------------------------------------------------------

class TemplateIpelet : public Ipelet {
public:
  virtual int IpelibVersion() const { return IPELIB_VERSION; }
  virtual int NumFunctions() const { return 5; }
  virtual const char *Label() const { return "Templates"; }
  virtual const char *SubLabel(int function) const;
  virtual void Run(int function, IpePage *page, IpeletHelper *helper);
private:
  void CreateTemplate(IpePage *page, IpeletHelper *helper, bool newSheet);
  void UseTemplate(IpePage *page, IpeletHelper *helper, bool clone);
  void InsertBackground(IpeletHelper *helper);
};

// --------------------------------------------------------------------

const char * const sublabel[] = {
  "Create template (in new style sheet)",
  "Create template (in top style sheet)",
  "Use template",
  "Clone template (for editing it)",
  "Insert background on all pages"
};

const char *TemplateIpelet::SubLabel(int function) const
{
  return sublabel[function];
}

// --------------------------------------------------------------------

void TemplateIpelet::CreateTemplate(IpePage *page, IpeletHelper *helper,
				    bool newSheet)
{
  IpePage::iterator it = page->PrimarySelection();
  if (it == page->end()) {
    helper->Message("No object selected");
    return;
  }

  IpeDocument *doc = helper->EditDocument();
  if (!newSheet && doc->StyleSheet()->IsStandard()) {
    helper->MessageBox("I cannot add template to standard style sheet.",
		       "Dismiss", 0, 0);
    return;
  }

  IpeString name;
  if (!helper->GetString("Choose name for template:", name))
    return;

  if (newSheet) {
    IpeStyleSheet *ns = new IpeStyleSheet(doc->Repository());
    ns->SetCascade(doc->GetStyleSheet());
    doc->SetStyleSheet(ns);
  }

  IpeStyleSheet *sheet = doc->GetStyleSheet();
  IpeAttribute tpl =
    doc->Repository()->MakeSymbol(IpeAttribute::ETemplate, name);
  sheet->AddTemplate(tpl, it->Object()->Clone());
}

// --------------------------------------------------------------------

void TemplateIpelet::UseTemplate(IpePage *page, IpeletHelper *helper,
				 bool clone)
{
  const IpeStyleSheet *sheet = helper->StyleSheet();
  IpeAttributeSeq seq;
  sheet->AllNames(IpeAttribute::ETemplate, seq);
  if (seq.empty()) {
    helper->MessageBox("Your document's style sheet contains no templates",
		       "Dismiss", 0, 0);
    return;
  }
  QStringList sl;
  for (IpeAttributeSeq::const_iterator it = seq.begin(); it != seq.end(); ++it)
    sl.append(QIpe(sheet->Repository()->String(*it)));

  DlgUseTemplate dlg(0, 0, true);
  dlg.iTemplate->insertStringList(sl);
  if (dlg.exec() == QDialog::Accepted) {
    IpeAttribute tpl = seq[dlg.iTemplate->currentItem()];
    IpeObject *obj = new IpeReference(sheet, tpl);
    if (clone) {
      const IpeObject *obj = sheet->FindTemplate(tpl);
      page->push_back(IpePgObject(IpePgObject::ESecondary,
				  helper->CurrentLayer(), obj->Clone()));
      helper->Message("Inserted a copy of the template");
    } else {
      page->push_back(IpePgObject(IpePgObject::ESecondary,
				  helper->CurrentLayer(), obj));
      helper->Message("Inserted reference to template");
    }
  }
}

// --------------------------------------------------------------------

void TemplateIpelet::InsertBackground(IpeletHelper *helper)
{
  IpeDocument *doc = helper->EditDocument();
  IpeAttribute bga =
    doc->Repository()->MakeSymbol(IpeAttribute::ETemplate, "background");
  if (!doc->StyleSheet()->FindTemplate(bga)) {
    helper->MessageBox("No template 'background' is defined in "
		       "the current style sheet",
		       "Dismiss", 0, 0);
    return;
  }

  if (helper->MessageBox("Do you wish to add a 'background' layer with "
			 "a reference to the 'background' template to "
			 "every page of your document?",
			 "Yes, please", "No, thank you", 0) != 0)
    return;

  for (IpeDocument::iterator it = doc->begin(); it != doc->end(); ++it) {
    IpePage *page = *it;
    if (page->FindLayer("background") >= 0)
      continue;

    IpeLayer bglayer("background");
    bglayer.SetLocked(true);
    page->AddLayer(bglayer, 0);

    IpeObject *obj = new IpeReference(doc->StyleSheet(), bga);
    page->push_back(IpePgObject(IpePgObject::ENone, 0, obj));
  }
  helper->Message("Background added to all pages");
}

// --------------------------------------------------------------------

void TemplateIpelet::Run(int function, IpePage *page, IpeletHelper *helper)
{
  switch (function) {
  case 0:
    CreateTemplate(page, helper, true);
    break;
  case 1:
    CreateTemplate(page, helper, false);
    break;
  case 2:
    UseTemplate(page, helper, false);
    break;
  case 3:
    UseTemplate(page, helper, true);
    break;
  case 4:
    InsertBackground(helper);
  default:
    break;
  }
}

// --------------------------------------------------------------------

IPELET_DECLARE Ipelet *NewIpelet()
{
  return new TemplateIpelet;
}

// --------------------------------------------------------------------
