/*  Simple toggle button example.
 *
 *  button.cc
 *
 *  Inti Translation of the C example written by Naofumi Yasufuku <naofumi@users.sourceforge.net>
 */

#include "button.h"
#include <inti/gtk/label.h>
#include <inti/gdk/gl/config.h>
#include <inti/gdk/gl/context.h>
#include <inti/gtk/gl/init.h>
#include <inti/gdk/gl/shapes.h>
#include <inti/gdk/gl/window.h>
#include <GL/gl.h>
#include <GL/glu.h>
#include <iostream>
#include <cstdlib>
#include <cmath>

namespace {

const int TIMEOUT_INTERVAL = 10;

} // namespace

GLDrawingArea::GLDrawingArea(Gdk::GL::Config *glconfig)
: Gtk::GL::DrawingArea(*glconfig, 200, 200), animate(true), angle(0.0), pos_y(0.0)
{
}

GLDrawingArea::~GLDrawingArea()
{
}

void
GLDrawingArea::on_realize()
{
	Gtk::GL::DrawingArea::on_realize();

	static float ambient[]  = { 0.0, 0.0, 0.0, 1.0 };
	static float diffuse[]  = { 1.0, 1.0, 1.0, 1.0 };
	static float position[] = { 1.0, 1.0, 1.0, 0.0 };
	static float lmodel_ambient[] = {0.2, 0.2, 0.2, 1.0};
	static float local_view[] = {0.0};

	if (!gl_begin()) // OpenGL begin
		return;

	glLightfv(GL_LIGHT0, GL_AMBIENT, ambient);
	glLightfv(GL_LIGHT0, GL_DIFFUSE, diffuse);
	glLightfv(GL_LIGHT0, GL_POSITION, position);
	glLightModelfv(GL_LIGHT_MODEL_AMBIENT, lmodel_ambient);
	glLightModelfv(GL_LIGHT_MODEL_LOCAL_VIEWER, local_view);
	glEnable(GL_LIGHTING);
	glEnable(GL_LIGHT0);
	glEnable(GL_DEPTH_TEST);

	glClearColor(1.0, 1.0, 1.0, 1.0);
	glClearDepth(1.0);

	gl_end(); // OpenGL end
}

bool
GLDrawingArea::on_configure_event(const Gdk::EventConfigure& event)
{

	float width = get_allocation().width();
	float height = get_allocation().height();

	if (!gl_begin()) // OpenGL begin
		return false;

	glViewport(0, 0, (int)width, (int)height);
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();

	float aspect;
	if (width > height)
	{
		aspect = width / height;
		glFrustum(-aspect, aspect, -1.0, 1.0, 5.0, 60.0);
	}
	else
	{
		aspect = height / width;
		glFrustum(-1.0, 1.0, -aspect, aspect, 5.0, 60.0);
	}

	glMatrixMode(GL_MODELVIEW);
	gl_end(); // OpenGL end
	return true;
}

bool
GLDrawingArea::on_expose_event(const Gdk::EventExpose& event)
{
	// Brass color
	static float ambient[4] = { 0.329412, 0.223529, 0.027451, 1.0 };
	static float diffuse[4] = { 0.780392, 0.568627, 0.113725, 1.0 };
	static float specular[4] = { 0.992157, 0.941176, 0.807843, 1.0 };
	static float shininess = 0.21794872 * 128.0;

	if (!gl_begin()) // OpenGL begin
		return false;

	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	glLoadIdentity();
	glTranslatef(0.0, 0.0, -10.0);

	glPushMatrix();
	glTranslatef(0.0, pos_y, 0.0);
	glRotatef(angle, 0.0, 1.0, 0.0);
	glMaterialfv(GL_FRONT, GL_AMBIENT, ambient);
	glMaterialfv(GL_FRONT, GL_DIFFUSE, diffuse);
	glMaterialfv(GL_FRONT, GL_SPECULAR, specular);
	glMaterialf(GL_FRONT, GL_SHININESS, shininess);
	Gdk::GL::draw_torus(true, 0.3, 0.6, 30, 30);
	glPopMatrix();

	if (is_double_buffered())
		swap_buffers();
	else
		glFlush();

	gl_end(); // OpenGL end
	return true;
}

bool
GLDrawingArea::on_map_event(const Gdk::EventAny& event)
{
	if (animate)
		timeout_connect();
	return true;
}

bool
GLDrawingArea::on_unmap_event(const Gdk::EventAny& event)
{
	timeout_disconnect();
	return true;
}

void
GLDrawingArea::on_unrealize()
{
	Gtk::GL::DrawingArea::on_unrealize();
	timeout_disconnect();
}

bool
GLDrawingArea::on_visibility_notify_event(const Gdk::EventVisibility& event)
{
	if (animate)
	{
		if (event.state() == Gdk::VISIBILITY_FULLY_OBSCURED)
			timeout_disconnect();
		else
			timeout_connect();
	}
	return true;
}

void
GLDrawingArea::on_toggle_animation()
{
	animate = !animate;

	if (animate)
	{
		timeout_connect();
	}
	else
	{
		timeout_disconnect();
		queue_draw();
	}
}

bool
GLDrawingArea::on_timeout()
{
	angle += 3.0;
	if (angle >= 360.0)
		angle -= 360.0;

	float t = angle * G_PI / 180.0;
	if (t > G_PI)
		t = 2.0 * G_PI - t;

	pos_y = 2.0 * (sin(t) + 0.4 * sin(3.0*t)) - 1.0;
	queue_draw();
	return true;
}

void
GLDrawingArea::timeout_connect()
{
	if (!timeout_connection.is_connected())
		timeout_connection = Main::timeout_signal.connect(slot(this, &GLDrawingArea::on_timeout), TIMEOUT_INTERVAL);
}

void
GLDrawingArea::timeout_disconnect()
{
	if (timeout_connection.is_connected())
		timeout_connection.disconnect();
}

/*  GLToggleButton
 */

GLToggleButton::GLToggleButton(Gdk::GL::Config *glconfig)
{
	Gtk::VBox *vbox = new Gtk::VBox;
	vbox->set_border_width(10);

	// Create an OpenGL-capable drawing area.
	GLDrawingArea *drawing_area = new GLDrawingArea(glconfig);
	vbox->pack_start(*drawing_area);
	drawing_area->show();

	// Create a label.
	Gtk::Label *label = new Gtk::Label("Toggle Animation");
	vbox->pack_start(*label, false, false, 10);
	label->show();

	// Create a toggle button.
	//Gtk::ToggleButton *button = new Gtk::ToggleButton;
	sig_toggled().connect(slot(drawing_area, &GLDrawingArea::on_toggle_animation));

	// Add vbox to button.
	add(*vbox);
	vbox->show();
}

GLToggleButton::~GLToggleButton()
{
}

Window::Window(Gdk::GL::Config *glconfig)
{
	set_title("Button Example");

	// Perform the resizes immediately.
	set_resize_mode(Gtk::RESIZE_IMMEDIATE);

	// Get automatically redrawn if any of their children changed allocation.
	set_reallocate_redraws(true);

	// Set border width.
	set_border_width(10);

	// Create Open-GL toggle button.
	Gtk::Button *button = new GLToggleButton(glconfig);
	add(*button);
	button->show();
	show();
}

Window::~Window()
{
}

int main (int argc, char *argv[])
{
	using namespace Main;

	init(&argc, &argv);

	// Initialize GtkGLExt
	Gtk::GL::init(&argc, &argv);

	// Configure OpenGL-capable visual.
	Pointer<Gdk::GL::Config> glconfig = Gdk::GL::Config::create(Gdk::GL::MODE_RGB | Gdk::GL::MODE_DEPTH | Gdk::GL::MODE_DOUBLE);
	if (!glconfig)
	{
		std::cout << "*** Cannot find the double-buffered visual." << std::endl;
		std::cout << "*** Trying single-buffered visual." << std::endl;

		// Try single-buffered visual
		glconfig = Gdk::GL::Config::create(Gdk::GL::MODE_RGB | Gdk::GL::MODE_DEPTH);
		if (!glconfig)
		{
			std::cout << "*** No appropriate OpenGL-capable visual found." << std::endl;
			return 1;
		}
	}

	Window window(glconfig);
	window.sig_destroy().connect(slot(&Inti::Main::quit));

	run();
	return 0;
}

