/***************************************************************************
                         entity.h  -  description                              
                            -------------------                                         
   begin                : Sat Aug 14 1999                                           
   copyright            : (C) 1999 by Jon Anderson                         
   email                : janderson@onelink.com                                     
***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   * 
 *                                                                         *
 ***************************************************************************/

#ifndef ENTITY_H
#define ENTITY_H

//QT includes
#include <time.h>
#include <string>
#include <math.h>

//local includes
#include <Math/matrix44.h>
#include <Math/vector3.h>
#include <Math/vector4.h>
#include "controller.h"
#include "boundingbox.h"
#include "idata.h"

class Controller;

class Texture;

class Material;

class Property;

class SubObject;


/**
  * Any base class for any transformable object
  */

class Transformable
{

   public:
      Transformable();
      virtual ~Transformable() {};

      virtual void getCompleteMatrix( Matrix44 * ) = 0;
      virtual void getParentMatrix( Matrix44 * ) = 0;
      virtual void getBoundingMin( Vector4 * ) = 0;
      virtual void getBoundingMax( Vector4 * ) = 0;
      virtual void getTransformedPosition( Vector4 * ) = 0;

      virtual void setOrientation( float d, float x, float y, float z ) = 0;
      virtual void setOrientation( Quat &q ) = 0;
      virtual void setPosition( float x, float y, float z ) = 0;
      virtual void setPosition( Vector4 &p );


      virtual Vector4 & getPosition() = 0;
      virtual Quat & getOrientation() = 0;

      virtual void getPosition( Vector4 * ) = 0;
      virtual void getOrientation( Quat * ) = 0;

      virtual void move( float x, float y, float z ) = 0;
      virtual void rotate( float amount, float x, float y, float z, float px, float py, float pz ) = 0;
      virtual void scale( float x, float y, float z, float ox = 0, float oy = 0, float oz = 0 ) = 0;

      virtual void scale( Vector4 &amt, Vector4 &pt );
      virtual void move( Vector4 &p );
      virtual void rotate( Vector4 &amt, Vector4 &pt );

      int drawBoundingBox();

};

class IDed
{
   public:
      static int ID;
      static int getNewID() { return ID++; };

      IDed() { m_id = getNewID(); };

      int getID() { return m_id; };

   protected:
      int m_id;
};


class Typed
{
   public:
      Typed();
      ~Typed();
      /*
       * Type functions
       */
      int getType() const;
      void setType( int _type );

      int getBaseType();
      int getRootType();

      void addType( int b );
      bool isA( int p_type );

      static int m_uid;
      static int getUID();

      int *type_list;  //maximum of 8 type ids per entity!
      int num_types;

};

class Selectable : public Typed, public IDed, public Transformable
{
   public:
      Selectable();
      bool isSelected();
      void setSelected( bool flag );

   protected:
      bool selected;
};

class Renderable
{
   public:
      virtual ~Renderable() {};

      virtual int draw( int d_options = 0 ) = 0;
      virtual int render( int d_options = 0 ) = 0;
};


class Animatable
{
   public:
      Animatable();
      virtual ~Animatable();
      virtual void loadState() = 0;
      virtual void saveState() = 0;
      bool isAnimatable();
      void setAnimatable( bool );
      Controller * getController();
      void setController( Controller * );
   protected:
      Controller * m_controller;
      bool m_animatable;
};

class SelectMode
{
   public:
      static void clear();
      static bool is( int );
      static void add( int );
      static void remove( int );
      static void set( int );
   protected:
      static int m_mode[ 8 ];
      static int m_size;
};


/**This is the base class from which all other manipulatable
 * objects  are derived.  All properties of an entity are animatable.  This is
 * done through the Controller object.  The entity is a passive listener to
 * the controller.  An entity can be manipulated by the user, and states can
 * be saved as keyframes.  
 * Whenever a controller changes state, it will automatically update the entity
 * to the new state, thus it can drive the entities animation.
 * @author Jonathan E. Anderson
 */

class Entity : public Animatable, public Selectable, public Renderable
{
   public:
      Entity();
      Entity( float x, float y, float z, Entity *parent = 0 );
      Entity( Vector4 &p, Entity *parent = 0 );
      Entity( Entity *parent );

      void init( float, float, float, Entity *parent = 0 );

      virtual void copyFrom( Entity * );

      virtual ~Entity();


      virtual void setOrientation( float d, float x, float y, float z );
      virtual void setOrientation( Quat &q );
      virtual void setPosition( float x, float y, float z );
      virtual void setPosition( Vector4 &p ) {setPosition( p.x, p.y, p.z );};


      virtual Vector4 & getPosition() { return pos;};

      virtual Quat & getOrientation() { return quat;};

      virtual void getPosition( Vector4 * );
      virtual void getOrientation( Quat * );


      virtual void move( float x, float y, float z );
      virtual void rotate( float amount, float x, float y, float z, float px, float py, float pz );
      virtual void scale( float x, float y, float z, float ox = 0, float oy = 0, float oz = 0 );

      virtual void scale( Vector4 &amt, Vector4 &pt ) {scale( amt.x, amt.y, amt.z, pt.x, pt.y, pt.z );};

      virtual void move( Vector4 &p ) {move( p.x, p.y, p.z );};

      virtual void rotate( Vector4 &amt, Vector4 &pt ) {rotate( amt.w, amt.x, amt.y, amt.z, pt.x, pt.y, pt.z );};


      virtual void loadState();
      virtual void saveState();

      virtual int draw( int d_options = 0 );
      virtual int render( int d_options = 0 );

      virtual void setParent( Entity *_parent );
      Entity * getParent() { return parent;};

      void setDirty( bool flag ) {dirty = flag;};

      virtual void setDirtyBox( bool flag ) {dirty_box = flag;};

      bool isDirty() { return dirty;};

      void toggleDirty() {dirty = !dirty;};

      void setVisible( bool flag ) {visible = flag;};
      bool isVisible();

      void setVisibilityGroup( int i ) { m_vis_group = i; };
      int getVisibilityGroup( ) { return m_vis_group; };

      string getName() { return name;};

      string getNotes() { return notes;};


      /* Transformable methods
       */
      virtual void getCompleteMatrix( Matrix44 * );
      virtual void getLocalMatrix( Matrix44 * );
      virtual void getParentMatrix( Matrix44 * );
      virtual void getBoundingMin( Vector4 * );
      virtual void getBoundingMax( Vector4 * );
      virtual void getTransformedPosition( Vector4 * );

      virtual Entity & operator=( Entity &rhs );

      virtual Entity * clone();

      bool isDrawable() { return drawable;};

      virtual bool isDeletable() { return true;};


      void setName( string _name ) {name = _name;};

      void setNotes( string _notes ) {notes = _notes;};

      virtual SubObject * getSubObject( int mode, int index );

      string getBaseDescription();


      virtual void preDraw() {};

      virtual void postDraw() {};

      IData * getData( string );
      void putData( string, IData * );

      IDataMap * getAllData( ) { return & m_data; };

      static bool * initVisGroups();

   public:


      static int TYPE;


   protected:

      IDataMap m_data;
      Vector4 pos;
      Quat quat;

      enum { POSITION_CHANNEL, ORIENTATION_CHANNEL };

      Entity * parent;

      bool dirty;
      bool dirty_box;
      bool drawable;
      bool visible;                  // if cleared this object is hidden, not drawn or selectable in the 3D display
      int m_vis_group;

      // this only affects the display of the object, it will still be rendered
      string name;
      string notes;

};


#endif










































