/***************************************************************************
                         subobject.cpp  -  description
                            -------------------
   begin                : Sun Aug 13 2000
   copyright            : (C) 2000 by Jon Anderson
   email                : janderson@onelink.com
***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "subobject.h"
#include "vertex.h"
#include "face.h"
#include "edge.h"
#include "uvcoord.h"
#include "object.h"

int SubObject::TYPE = Typed::getUID();


SubObject::SubObject( Object *o ) :
      vlist(),
      elist(),
      flist(),
      uvlist()
{
   addType( TYPE );
   m_parent = o;
   visible = true;
}

SubObject::~SubObject()
{
   vlist.clear();
   elist.clear();
   flist.clear();
   uvlist.clear();
}

SubObject & SubObject::operator=( SubObject &rhs )
{
   vlist.clear();
   elist.clear();
   flist.clear();
   uvlist.clear();


   vlist = rhs.vlist;
   elist = rhs.elist;
   flist = rhs.flist;
   uvlist = rhs.uvlist;
   visible = rhs.isVisible();
   return *this;

}

/** From it's parent
  */
void SubObject::remove()
{
   getParentObject() ->removeSubObject( this );
}

Vector4 SubObject::getCenter()
{
   Vector4 c( 0, 0, 0, 0 );
   int n = ( int ) vlist.size();

   for ( int i = 0; i < n; i++ )
   {
      c += VERT( vlist[ i ] ) -> getPosition();
   }

   c.x /= n;
   c.y /= n;
   c.z /= n;
   c.w = 1;

   return c;

}

/*Removes any connections this subobject may have, and puts it in a
 * queue to be removed when all operations are finished.
 */
void SubObject::detach( bool remove_me = true )
{
   for ( int i = 0; i < ( int ) elist.size(); i++ )
   {
      EDGE( elist[ i ] ) ->removeSubObject( getType(), index );
   }

   for ( int i = 0; i < ( int ) flist.size(); i++ )
   {
      FACE( flist[ i ] ) ->removeSubObject( getType(), index );
   }

   for ( int i = 0; i < ( int ) vlist.size(); i++ )
   {
      VERT( vlist[ i ] ) ->removeSubObject( getType(), index );
   }


   //remove selected objects if they are isolated.
   for ( int i = 0; i < ( int ) vlist.size(); i++ )
   {
      if ( VERT( vlist[ i ] ) ->lonely() )
         VERT( vlist[ i ] ) -> detach( remove_me );
   }

   for ( int i = 0; i < ( int ) elist.size(); i++ )
   {
      if ( EDGE( elist[ i ] ) ->lonely() )
         EDGE( elist[ i ] ) -> detach( remove_me );
   }

   for ( int i = 0; i < ( int ) flist.size(); i++ )
   {
      if ( FACE( flist[ i ] ) ->lonely() )
         FACE( flist[ i ] ) -> detach( remove_me );
   }


   elist.clear();
   flist.clear();
   vlist.clear();

   if ( remove_me )
      getParentObject() -> markForDeletion( this );

}



bool SubObject::lonely()
{
   int ne = elist.size();
   int nf = flist.size();
   int nv = vlist.size();

   if ( ne + nf + nv == 0 )
      return true;
   else
      return false;



}

void SubObject::init()
{

}

vector<int> * SubObject::getSubObjectList( int t )
{

   if ( t == Vertex::TYPE )
      return & vlist;
   else if ( t == Edge::TYPE )
      return & elist;
   else if ( t == Face::TYPE )
      return & flist;
   else if ( t == UVCoord::TYPE )
      return & uvlist;

   cerr << "Invalid subobject type list requested" << endl;

   return 0;

}

int SubObject::getSubObj( int t, int x )
{
   vector<int> *list = getSubObjectList( t );

   return ( *list ) [ x ];

}

SubObject * SubObject::getSubObjPtr( int t, int x )
{
   vector<int> *list = getSubObjectList( t );
   int i = ( *list ) [ x ];
   return getParentObject() -> getSubObject( t, i );


}

int SubObject::getNumSubObjs( int t )
{
   vector<int> *list = getSubObjectList( t );

   return ( int ) list -> size();

}

void SubObject::addSubObject( int t, int x )
{

   vector<int> *list = getSubObjectList( t );

   list->push_back( x );
}

string SubObject::getDescription( int tp = 0 )
{
   int t;
   string desc;

   if ( tp == 0 )
      t = getType();
   else
      t = tp;

   if ( t == Vertex::TYPE )
      desc = "Vertex";

   if ( t == Edge::TYPE )
      desc = "Edge";

   if ( t == Face::TYPE )
      desc = "Face";

   return desc;


}

void SubObject::removeSubObject( int t, int x )
{


   vector<int> *list = getSubObjectList( t );

   vector<int>::iterator removeIter = find( list->begin(), list->end(), x );

   if ( removeIter == list->end() )
   {
      cerr << getDescription() << ":" << index << " couldn't find " << getDescription( t ) << ":" << x << " for removal." << endl;
      dump();
      return ;
   }

   list->erase( removeIter );
}

/**Checks to see if this subobject is adjacent to the given one.
  *They are adjacent if they share an edge. (Verts and Faces)
  *Edges cannot be adjacent.
  */
bool SubObject::adjacent( SubObject *so )
{
   vector<int> common;

   set_intersection( elist.begin(), elist.end(),
                     so->getEdges() ->begin(), so->getEdges() ->end(),
                     common.begin() );

   if ( common.size() > 0 )
      return true;
   else
      return false;

}

int SubObject::findSubObject( int type, int so )
{
   vector<int> *list = getSubObjectList( type );

   for ( int i = 0; i < ( int ) list->size(); i++ )
   {
      if ( ( *list ) [ i ] == so )
         return i;
   }

   cerr << getDescription() << ":" << index << " couldn't find " << getDescription( type ) << ":" << so << endl;
   return -1;

}

bool SubObject::hasSubObject( int type, int s )
{

   vector<int> *list = getSubObjectList( type );

   for ( int i = 0; i < ( int ) list->size(); i++ )
   {
      if ( ( *list ) [ i ] == s )
         return true;
   }

   return false;

}

/**
  * Replaces and old index with a new index
  * in a subobject list of type 'type'
  */
void SubObject::swapSubObject( int type, int old_index, int new_index )
{
   bool swapped = false;
   vector<int> *list = getSubObjectList( type );

   for ( int i = 0; i < ( int ) list->size(); i++ )
   {
      if ( ( *list ) [ i ] == old_index )
      {
         ( *list ) [ i ] = new_index;
         swapped = true;
      }
   }

   if ( ! swapped )
   {
      cerr << getDescription() << ": Unable to swap " << getDescription( type ) << " from " << old_index << " to " << new_index << endl;
      dump();
   }
}

/**
  * Replaces this subobject with the given
  * by looping through all connected subobjects
  * and replacing their index to this with
  * the new index.
  */
void SubObject::replace( int newIndex )
{
   //loop through all the connect subobjects and replace the old with the new.
   SubObject * s = getParentObject() -> getSubObject( getType(), newIndex );

   vector<int> *list = getEdges();

   for ( int i = 0; i < ( int ) list->size(); i++ )
   {
      EDGE( ( *list ) [ i ] ) ->swapSubObject( getType(), getParentIndex(), newIndex );
      s -> addEdge( ( *list ) [ i ] );
   }

   list = getVerts();

   for ( int i = 0; i < ( int ) list->size(); i++ )
   {
      VERT( ( *list ) [ i ] ) ->swapSubObject( getType(), getParentIndex(), newIndex );
      s -> addVert( ( *list ) [ i ] );
   }


   list = getFaces();

   for ( int i = 0; i < ( int ) list->size(); i++ )
   {
      FACE( ( *list ) [ i ] ) ->swapSubObject( getType(), getParentIndex(), newIndex );
      s -> addFace( ( *list ) [ i ] );
   }

   elist.clear();
   flist.clear();
   vlist.clear();

   getParentObject() -> markForDeletion( this );

   cerr << "Done replacing" << endl;
}


void SubObject::reindex( int oldIndex, int newIndex )
{

   //loop through all the connect subobjects and replace the old with the new.
   vector<int> *list = getEdges();

   for ( int i = 0; i < ( int ) list->size(); i++ )
      EDGE( ( *list ) [ i ] ) ->swapSubObject( getType(), oldIndex, newIndex );

   list = getVerts();

   for ( int i = 0; i < ( int ) list->size(); i++ )
      VERT( ( *list ) [ i ] ) ->swapSubObject( getType(), oldIndex, newIndex );


   list = getFaces();

   for ( int i = 0; i < ( int ) list->size(); i++ )
      FACE( ( *list ) [ i ] ) ->swapSubObject( getType(), oldIndex, newIndex );

   index = newIndex;

}

void SubObject::setIndex( int newIndex )
{
   reindex( index, newIndex );
}

bool SubObject::sharesSubObject( int type, int index, int sub )
{
   vector<int> list;
   list = getCommonSubObjects( type, sub );

   vector<int>::iterator finder;
   finder = find( list.begin(), list.end(), index );

   if ( finder == list.end() )
      return false;
   else
      return true;
}

vector<int> SubObject::getCommonSubObjects( int type, int sub )
{
   //get the other subobject
   SubObject * so = getParentObject() -> getSubObject( getType(), sub );

   //get the lists
   vector<int> *list1 = getSubObjectList( type );
   vector<int> *list2 = so -> getSubObjectList( type );

   vector<int> combined;

   combined.reserve( 10 );
   //print them out...


   //find the union of the two.
   /*set_intersection ( list1 -> begin(), list1 -> end(),
             list2 -> begin(), list2 -> end(),
             combined.begin()            );
   */

   for ( int i = 0; i < ( int ) list1->size(); i++ )
   {
      int x = ( *list1 ) [ i ];
      vector<int>::iterator finder;
      finder = find( list2 -> begin(), list2 -> end(), x );

      if ( finder != list2 -> end() )
         combined.push_back( x );
   }

   return combined;

}

bool SubObject::hasCommonSubObjects( int type, int sub )
{
   vector<int> common;

   common = getCommonSubObjects( type, sub );

   return ! common.empty();

}


void SubObject::dump()
{
   cerr << getDescription() << ":" << index << endl;
   cerr << "\tVLIST: ";

   for ( int i = 0; i < ( int ) vlist.size(); i++ )
      cerr << vlist[ i ] << " ";

   cerr << endl;

   cerr << "\tELIST: ";

   for ( int i = 0; i < ( int ) elist.size(); i++ )
      cerr << elist[ i ] << " ";

   cerr << endl;

   cerr << "\tFLIST: ";

   for ( int i = 0; i < ( int ) flist.size(); i++ )
      cerr << flist[ i ] << " ";

   cerr << endl;

}

void SubObject::getBoundingMin( Vector4 *v )
{
   Vector4 p;
   v -> assign( 10000, 10000, 10000, 10000 );

   for ( int i = 0; i < ( int ) vlist.size(); i++ )
   {
      VERT( vlist[ i ] ) -> getBoundingMin( &p );
      v -> min( p );
   }

}
void SubObject::removeDuplicateSubObjects( int type )
{
	vector<int> * slist = getSubObjectList( type );
	
	std::sort( slist -> begin(), slist -> end() );
	
	
	vector<int>::iterator dups = std::unique( slist -> begin(), slist -> end() );
	slist -> erase( dups, slist -> end() );
	
}

void SubObject::getBoundingMax( Vector4 *v )
{
   Vector4 p;
   v -> assign( -10000, -10000, -10000, -10000 );

   for ( int i = 0; i < ( int ) vlist.size(); i++ )
   {
      VERT( vlist[ i ] ) -> getBoundingMax( &p );
      v -> max( p );
   }
}

void SubObject::move( float x, float y, float z )
{
   for ( int i = 0; i < ( int ) vlist.size(); i++ )
   {
      VERT( vlist[ i ] ) -> move( x, y, z );
   }
}

void SubObject::rotate( float amount, float x, float y, float z, float px = 0, float py = 0, float pz = 0 )
{
   //rotate about the center.
   Vector4 center;

   for ( int i = 0; i < ( int ) vlist.size(); i++ )
   {
      center += VERT( vlist[ i ] ) -> getPosition();
   }

   center /= vlist.size();

   for ( int i = 0; i < ( int ) vlist.size(); i++ )
   {
      VERT( vlist[ i ] ) -> rotate( amount, x, y, z, center.x, center.y, center.z );
   }
}

void SubObject::scale( float x, float y, float z, float ox = 0, float oy = 0, float oz = 0 )
{
   //scale about the center.
   Vector4 center;
   Vector4 tmp;

   for ( int i = 0; i < ( int ) vlist.size(); i++ )
   {
      VERT( vlist[ i ] ) -> getTransformedPosition( &tmp );
      center += tmp;
   }

   center /= vlist.size();


   for ( int i = 0; i < ( int ) vlist.size(); i++ )
   {
      VERT( vlist[ i ] ) -> scale( x, y, z, center.x, center.y, center.z );
   }
}

bool SubObject::isVisible(){

    return visible;

}

inline void SubObject::getCompleteMatrix( Matrix44 *m ) { getParentMatrix( m ); };

inline void SubObject::getParentMatrix( Matrix44 *m ) { m_parent->getCompleteMatrix( m ); };

inline Quat & SubObject::getOrientation() {static Quat q; return q;};

inline Vector4 & SubObject::getPosition() {static Vector4 p; return p;};

inline void SubObject::addFace( int x ) {addSubObject( Face::TYPE, x );};

inline void SubObject::addEdge( int x ) {addSubObject( Edge::TYPE, x );};

inline void SubObject::addVert( int x ) {addSubObject( Vertex::TYPE, x );};

inline void SubObject::removeFace( int x ) {removeSubObject( Face::TYPE, x );};

inline void SubObject::removeEdge( int x ) {removeSubObject( Edge::TYPE, x );};

inline void SubObject::removeVert( int x ) {removeSubObject( Vertex::TYPE, x );};

inline int SubObject::getFace( int x ) { return getSubObj( Face::TYPE, x );};

inline int SubObject::getEdge( int x ) { return getSubObj( Edge::TYPE, x );};

inline int SubObject::getVert( int x ) { return getSubObj( Vertex::TYPE, x );};

inline Face * SubObject::getFacePtr( int x ) { return ( Face * ) getSubObjPtr( Face::TYPE, x );};

inline Edge * SubObject::getEdgePtr( int x ) { return ( Edge * ) getSubObjPtr( Edge::TYPE, x );};

inline UVCoord * SubObject::getUVPtr( int x ) { return ( UVCoord * ) getSubObjPtr( UVCoord::TYPE, x );};

inline Vertex * SubObject::getVertPtr( int x ) { return ( Vertex * ) getSubObjPtr( Vertex::TYPE, x );};

inline int SubObject::getNumFaces() { return getNumSubObjs( Face::TYPE );};

inline int SubObject::getNumEdges() { return getNumSubObjs( Edge::TYPE );};

inline int SubObject::getNumVerts() { return getNumSubObjs( Vertex::TYPE );};

inline void SubObject::swapFace( int ox, int nx ) {swapSubObject( Face::TYPE, ox, nx );};

inline void SubObject::swapVert( int ox, int nx ) {swapSubObject( Face::TYPE, ox, nx );};

inline void SubObject::swapEdge( int ox, int nx ) {swapSubObject( Face::TYPE, ox, nx );};

inline bool SubObject::sharesVert( int v, int i ) { return sharesSubObject( Vertex::TYPE, v, i );};

inline bool SubObject::sharesEdge( int e, int i ) { return sharesSubObject( Edge::TYPE, e, i );};

inline bool SubObject::sharesFace( int f, int i ) { return sharesSubObject( Face::TYPE, f, i );};

inline vector<int> SubObject::getCommonVerts( int i ) { return getCommonSubObjects( Vertex::TYPE, i );};

inline vector<int> SubObject::getCommonEdges( int i ) { return getCommonSubObjects( Edge::TYPE, i );};

inline vector<int> SubObject::getCommonFaces( int i ) { return getCommonSubObjects( Face::TYPE, i );};

inline bool SubObject::hasCommonVerts( int i ) { return hasCommonSubObjects( Vertex::TYPE, i );};

inline bool SubObject::hasCommonEdges( int i ) { return hasCommonSubObjects( Edge::TYPE, i );};

inline bool SubObject::hasCommonFaces( int i ) { return hasCommonSubObjects( Face::TYPE, i );};

inline bool SubObject::hasFace( int f ) { return hasSubObject( Face::TYPE, f );};

inline bool SubObject::hasEdge( int e ) { return hasSubObject( Edge::TYPE, e );};

inline bool SubObject::hasVert( int v ) { return hasSubObject( Vertex::TYPE, v );};

inline int SubObject::findFace( int f ) { return findSubObject( Face::TYPE, f );};

inline int SubObject::findVert( int v ) { return findSubObject( Vertex::TYPE, v );};

inline int SubObject::findEdge( int e ) { return findSubObject( Edge::TYPE, e );};

inline vector<int> * SubObject::getFaces() { return getSubObjectList( Face::TYPE );};

inline vector<int> * SubObject::getEdges() { return getSubObjectList( Edge::TYPE );};

inline vector<int> * SubObject::getVerts() { return getSubObjectList( Vertex::TYPE );};

inline vector<int> * SubObject::getUVs() { return getSubObjectList( UVCoord::TYPE );};
inline void SubObject::removeDuplicateVerts() { removeDuplicateSubObjects( Vertex::TYPE ); };
inline void SubObject::removeDuplicateFaces() { removeDuplicateSubObjects( Face::TYPE ); };
inline void SubObject::removeDuplicateEdges() { removeDuplicateSubObjects( Edge::TYPE ); };
inline void SubObject::removeDuplicateUVs() { removeDuplicateSubObjects( UVCoord::TYPE ); };

