/***************************************************************************
                        face.cpp  -  description                              
                           -------------------                                         
  begin                : Wed Dec 15 1999                                           
  copyright            : (C) 1999 by Jon Anderson                         
  email                : janderson@onelink.com                                     
***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   * 
 *                                                                         *
 ***************************************************************************/


#include "face.h"
#include "vertex.h"
#include "uvcoord.h"
#include "edge.h"
#include "mesh.h"
#include <Util/util.h>

#include "tesselator.h"

bool Face::filled = true;
int Face::TYPE = Typed::getUID();


Face::Face( Object *p ) : SubObject( p ),
      normal()
{
   addType( TYPE );
   usetest = false;

}

void Face::init( vector<int> &v )
{

   float quad[ 4 ][ 2 ] = {{0, 1}, {0, 0}, {1, 0}, {1, 1}};

   vlist.reserve( v.size() );
   uvlist.reserve( v.size() );
   elist.reserve( v.size() );


   //set up each vertex in the face.
   int max_i = 3;

   for ( int i = 0; i < ( int ) v.size(); i++ )
   {
      max_i = i;

      if ( max_i > 3 )
         max_i = 0;

      Vertex *vx = VERT( v[ i ] );

      //setup the vertex
      vlist.push_back( v[ i ] );

      vx->addFace( index );

      //setup the UV's here
      UVCoord *uc = getParentObject() -> createUVCoord( );

      uc->setPosition( quad[ max_i ][ 0 ], quad[ max_i ][ 1 ], 0 );

      uvlist.push_back( uc->getParentIndex() );

      uc -> addFace( index );

   }

   reEdge();

}



Face::~Face()
{}

void Face::extrude( Vector4 &v )
{
   extrude();
   move( v.x, v.y, v.z );
}

void Face::extrude()
{

   vector<int> tvlist = vlist;

   vector<int> telist = elist;


   //Remove the current face from the border edges.
   for ( int i = 0; i < ( int ) telist.size(); i++ )
   {
      EDGE( telist[ i ] ) ->removeFace( index );
   }

   elist.clear();

   //Remove the current vertices from the face, and create new ones to take their place.
   for ( int i = 0; i < ( int ) tvlist.size(); i++ )
   {
      Vertex * v = VERT( tvlist[ i ] );
      v->removeFace( index );

      Vertex *nv = getParentObject() -> createVertex( v -> getPosition() );
      nv->addFace( index );
      vlist[ i ] = nv->getParentIndex();
   }

   //create a series of quads to connect old to new verts.
   int num = vlist.size();

   for ( int i = 0; i < num; i++ )
   {
      int j = i + 1;

      if ( j == num ) j = 0;

      getParentObject() -> createFace( vlist[ i ], tvlist[ i ], tvlist[ j ], vlist[ j ] );
   }

   getParentObject() ->normalize();
   reEdge();

}

/** Inserts a new vertex into the face
  * between ov1 and ov2.  It assumes they are connected by an edge.
  */
void Face::addVertex( int ov1, int ov2, int nv )
{

   int cindex;

   int v1 = findVert( ov1 );
   int v2 = findVert( ov2 );

   //special case if one is 0 and the other isn't 1
   if ( v1 == 0 && v2 != 1 )
   {
      cindex = v2;
   }

   else if ( v2 == 0 && v1 != 1 )
   {
      cindex = v1;
   }

   else if ( v1 < v2 )
      cindex = v1;
   else
      cindex = v2;

   Vector4 uvp1, uvp2, u;

   uvp1 = UV( uvlist[ v1 ] ) -> getPosition();

   uvp2 = UV( uvlist[ v2 ] ) -> getPosition();

   u = uvp1 + uvp2;

   u.x /= 2;

   u.y /= 2;


   vlist.insert( vlist.begin() + cindex + 1, nv );


   UVCoord *uc = getParentObject() -> createUVCoord( u.x, u.y );

   uc -> addFace( index );

   uvlist.insert( uvlist.begin() + cindex + 1, uc->getParentIndex() );


   Vertex *vx = VERT( nv );

   vx->addFace( index );

}

Vector4 Face::getNormal( int v = 0 )
{
   //this needs to be over written for strips...
   //it's hardcoded for the first 3 verts right now.

   Vector3 normal3;

   Vector4 &v3 = VERT( vlist[ 2 ] ) ->getPosition();
   Vector4 &v2 = VERT( vlist[ 1 ] ) ->getPosition();
   Vector4 &v1 = VERT( vlist[ 0 ] ) ->getPosition();

   Vector4 a, b;
   a = v3 - v1;
   b = v2 - v1;

   normal3 = b.ToVector3() * a.ToVector3();

   normal3.normalize();
   normal.assign( normal3.x, normal3.y, normal3.z, 0 );
   return normal;
}

void Face::move( float amount )
{
   for ( int i = 0; i < ( int ) vlist.size(); i++ )
      VERT( vlist[ i ] ) ->move( normal.x * amount, normal.y * amount, normal.z * amount );

}

/**Subdivides the face from a vertex to an edge
  *Edges start at zero, and it contains verts e and e+1
  *This is only valid for triangles
 */
void Face::subdivideEV( int e, int v )
{
   //only allow edge to vertex subdivision if it's a tri
   if ( vlist.size() != 3 )
      return ;

   //Create new Verts half way on each edge
   Mesh *mp = ( Mesh * ) m_parent;

   int ev2 = e + 1;

   int ev1 = e;

   if ( ev2 > 2 )
      ev2 = 0;

   Vector4 ep1, ep2;

   ep1 = VERT( vlist[ ev1 ] ) ->getPosition();

   ep2 = VERT( vlist[ ev2 ] ) ->getPosition();


   Vertex *nv = mp -> createVertex( ( ep1.x + ep2.x ) / 2,
                                    ( ep1.y + ep2.y ) / 2,
                                    ( ep1.z + ep2.z ) / 2 );

   // verts->push_back(nv);
   int nv_index = nv->getParentIndex();




   mp -> createFace( vlist[ v ], vlist[ ev1 ], nv_index );

   mp -> createFace( vlist[ v ], nv_index, vlist[ ev2 ] );

   mp -> createFace( vlist[ ev1 ], vlist[ ev2 ], nv_index ); //filler triangle.


   VERT( vlist[ 0 ] ) ->removeFace( index );

   VERT( vlist[ 1 ] ) ->removeFace( index );

   VERT( vlist[ 2 ] ) ->removeFace( index );

   mp->removeFace( index );


   ( ( Object * ) m_parent ) ->normalize();
}

/**Subdivides the face through the given edges. This is only
  *valid for quads.  Edges are identified by their first vertex.
 */
void Face::subdivideE( int e1, int e2 )
{
   //only allow edge to edge subdivision if it's a quad
   if ( vlist.size() != 4 )
      return ;

   int e1v1, e1v2, e2v1, e2v2;

   e1v1 = e1;

   e2v1 = e2;

   e2v2 = e2 + 1;

   e1v2 = e1 + 1;

   if ( e2v2 > 3 )
      e2v2 = 0;

   if ( e1v2 > 3 )
      e1v2 = 0;

   //Create new Verts half way on each edge
   Mesh *mp = ( Mesh * ) m_parent;


   Vector4 &e1p1 = VERT( vlist[ e1v1 ] ) ->getPosition();

   Vector4 &e1p2 = VERT( vlist[ e1v2 ] ) ->getPosition();

   Vector4 &e2p1 = VERT( vlist[ e2v1 ] ) ->getPosition();

   Vector4 &e2p2 = VERT( vlist[ e2v2 ] ) ->getPosition();


   Vertex *nv1 = mp -> createVertex( ( e1p1.x + e1p2.x ) / 2,
                                     ( e1p1.y + e1p2.y ) / 2,
                                     ( e1p1.z + e1p2.z ) / 2 );

   Vertex *nv2 = mp -> createVertex( ( e2p1.x + e2p2.x ) / 2,
                                     ( e2p1.y + e2p2.y ) / 2,
                                     ( e2p1.z + e2p2.z ) / 2 );

   int nv1_index = nv1->getParentIndex();

   int nv2_index = nv2->getParentIndex();


   mp -> createFace( vlist[ e1v1 ], nv1_index, nv2_index, vlist[ e2v2 ] );

   mp -> createFace( nv1_index, vlist[ e1v2 ], vlist[ e2v1 ], nv2_index );

   mp -> createFace( vlist[ e1v1 ], vlist[ e1v2 ], nv1_index );

   mp -> createFace( vlist[ e2v1 ], vlist[ e2v2 ], nv2_index );


   VERT( vlist[ 0 ] ) ->removeFace( index );

   VERT( vlist[ 1 ] ) ->removeFace( index );

   VERT( vlist[ 2 ] ) ->removeFace( index );

   VERT( vlist[ 3 ] ) ->removeFace( index );

   mp->removeFace( index );

   m_parent -> normalize();

}

/**Overloaded operator
  *copies the vertex indexes into the face
  */
Face & Face::operator=( Face &rhs )
{
   SubObject::operator=( rhs );

   normal = rhs.normal;

   return *this;
}

void Face::copyFrom( SubObject * s )
{
   *this = * static_cast<Face *> ( s );
}

/**Returns a pointer to a clone of this face
 */
Face * Face::clone()
{
   return 0;
}

/**Sets the UV of the given vertex index to be the vector uv
 */
void Face::setUVCoord( int i, Vector4 &uv )
{
   UV( uvlist[ i ] ) ->setPosition( uv.x, uv.y, 1 );
}


/**Sets the UV of the given vertex index
 */
void Face::setUVCoord( int i, float u, float v )
{
   UV( uvlist[ i ] ) ->setPosition( u, v, 1 );
}


/**Reverse the ordering of the face.
  */
void Face::reverse()
{
   vector<int> rlist;
   vector<int> ulist;

   ulist.reserve( uvlist.size() );
   rlist.reserve( vlist.size() );

   for ( int i = vlist.size() - 1; i >= 0; i-- )
   {
      rlist.push_back( vlist[ i ] );
      ulist.push_back( uvlist[ i ] );
   }

   uvlist = ulist;
   vlist = rlist;

}


/** Ensures that the texture uv's wrap appropriately for cylinder and
  * sphere mapping.
  * It does this by comparing each UV against all others in the face.
  * If the distance is greater than 0.5, it adds +1 to the lower value.
  */
void Face::normalizeUVs( bool wrap_u = false, bool wrap_v = false )
{

   for ( int i = 0; i < ( int ) uvlist.size(); i++ )
   {
      for ( int j = i + 1; j < ( int ) uvlist.size(); j++ )
      {
         Vector4 p1;
         p1 = UV( uvlist[ i ] ) ->getPosition();

         Vector4 p2;
         p2 = UV( uvlist[ j ] ) ->getPosition();
         //check if we need to add 1 to coord inorder to wrap.

         if ( wrap_u )
         {
            if ( ( p2.x > ( p1.x + .5 ) ) && ( p2.x < 1 ) )
            {
               UV( uvlist[ i ] ) ->move( 1, 0, 0 );
            }

            if ( ( p1.x > ( p2.x + .5 ) ) && ( p1.x < 1 ) )
            {
               UV( uvlist[ j ] ) ->move( 1, 0, 0 );
            }

         }

         if ( wrap_v )
         {

            if ( ( p2.y > ( p1.y + .5 ) ) && ( p2.y < 1 ) )
               UV( uvlist[ i ] ) ->move( 0, 1, 0 );

            if ( ( p1.y > ( p2.y + .5 ) ) && ( p1.y < 1 ) )
               UV( uvlist[ j ] ) ->move( 0, 1, 0 );

         }
      }
   }

}

void Face::drawSelectedFace()
{
   float v[ 4 ];

   Vector4 p;

   //draw edges as deep blue.
   glEnable( GL_POLYGON_OFFSET_LINE );
   glPolygonOffset( 1, 1 );
   glLineWidth( 3 );
   glBegin( GL_LINE_LOOP );
   glColor4f( 0, 0, 1, 1 );

   for ( int i = 0; i < ( int ) vlist.size(); i++ )
   {
      VERT( vlist[ i ] ) ->getPosition( &p );
      glVertex3fv( p.v( v ) );
   }

   glEnd();
   glLineWidth( 1 );
   glDisable( GL_POLYGON_OFFSET_LINE );

   glColor4f( 1, 0, 0, 1 );

}


/**Draws the geometry of the face.*/
void Face::drawFace( int mode )
{
   GLenum glmode = GL_POLYGON;

   if ( vlist.size() == 3 )
      glmode = GL_TRIANGLES;

   if ( vlist.size() == 4 )
      glmode = GL_QUADS;

   float v[ 4 ];

   Vector4 uv;

   glBegin( glmode );

   vector<int>::iterator v_it = vlist.begin();

   vector<int>::iterator uv_it = uvlist.begin();

   while ( v_it != vlist.end() )
   {

      UV( *uv_it ) ->getPosition( &uv );
      glTexCoord2fv( uv.v( v ) );
      VERT( *v_it ) ->draw( mode );
      ++v_it;
      ++uv_it;
   }

   glEnd();


}


int Face::draw( int d_options = 0 )
{

   // float v[4];

   //Vertex *tuv;
   glPushAttrib( GL_COLOR_BUFFER_BIT );

   if ( SelectMode::is( Face::TYPE ) )
   {

      int mode = 0;

      if ( isSelected() )
      {
         glPushAttrib( GL_LIGHTING_BIT | GL_TEXTURE_BIT | GL_COLOR_BUFFER_BIT );

         glDisable( GL_LIGHTING );
         glDisable( GL_TEXTURE_2D );
         glColor4f( 1, 0, 0, 1 );

         drawBoundingBox();
         drawSelectedFace();
         mode = VERTEX_NORMAL | VERTEX_COORD;

      }

      else
      {

         mode = VERTEX_COLOR | VERTEX_NORMAL | VERTEX_COORD;

      }

      glPushName( index );
      drawFace( mode );
      glPopName();

      if ( isSelected() )
      {
         glPopAttrib();
      }

   }

   else
   {
      drawFace( VERTEX_COLOR | VERTEX_NORMAL | VERTEX_COORD );
   }

   glPopAttrib();

   return 0;
}

/**How can we 'smart' divide? */
void Face::subdivideTriangles()
{
   //find the center of this face, and create a vertex.
   Vector4 c( 0, 0, 0, 0 );
   int n = ( int ) vlist.size();

   for ( int i = 0; i < ( int ) vlist.size(); i++ )
   {
      c += VERT( vlist[ i ] ) -> getPosition();
   }

   c.x /= n;
   c.y /= n;
   c.z /= n;

   Vertex *v = getParentObject() -> createVertex( c );
   int nv = v -> getParentIndex();

   for ( int i = 0; i < ( int ) vlist.size(); i++ )
   {
      int v1 = vlist[ i ];
      int v2 = 0;

      if ( i + 1 == ( int ) vlist.size() )
         v2 = vlist[ 0 ];
      else
         v2 = vlist[ i + 1 ];

      //create a new face...
      getParentObject() -> createFace( v1, v2, nv );
   }

   //remove this face...
   detach();

}


bool Face::triangulate()
{
   if ( vlist.size() < 4 )
      return false;

   vector<int> nvlist = getFaceAsTris();

   int n = nvlist.size() / 3;

   for ( int i = 0; i < n; i++ )
   {
      int v1, v2, v3;
      v1 = findVert( nvlist[ i * 3 ] );
      v2 = findVert( nvlist[ i * 3 + 1 ] );
      v3 = findVert( nvlist[ i * 3 + 2 ] );

      Face * f = getParentObject() -> createFace( nvlist[ i * 3 ], nvlist[ i * 3 + 1 ], nvlist[ i * 3 + 2 ] );
      //create new texture coords.

      UVCoord *uv1 = UV( uvlist[ v1 ] );
      UVCoord *uv2 = UV( uvlist[ v2 ] );
      UVCoord *uv3 = UV( uvlist[ v3 ] );

      f->setUVCoord( 0, uv1 -> getPosition() );
      f->setUVCoord( 1, uv2 -> getPosition() );
      f->setUVCoord( 2, uv3 -> getPosition() );
   }

   detach();

   //for( int i=0; i<(int)uvlist.size(); i++ ) {
   // getParentObject() -> markForDeletion( UV( uvlist[i] ) );
   //}
   //
   return true;
}

vector<int> Face::getFaceAsTris()
{
   vector<Vector4> fverts;
   fverts.reserve( vlist.size() );

   for ( unsigned int i = 0; i < vlist.size(); i++ )
   {
      fverts.push_back( VERT( vlist[ i ] ) ->getPosition() );
   }

   Tesselator t;
   vector<int> nvlist;

   nvlist = t.tesselate( fverts, POLY_CCW );

   for ( unsigned int i = 0; i < nvlist.size(); i++ )
      nvlist[ i ] = vlist[ nvlist[ i ] ];

   return nvlist;

}

/**Returns the orientation of this face determined
  *by the given verts.  Returns true if they are in the right order,
  *false otherwise.
  */
bool Face::isOrientated( int ov1, int ov2 )
{
   int v1 = findVert( ov1 );
   int v2 = findVert( ov2 );

   if ( ! adjacentVerts( v1, v2 ) )
      return false;

   if ( v1 == ( ( int ) vlist.size() - 1 ) && v2 == 0 )
      return true;

   if ( v2 > v1 )
      return true;
   else
      return false;


}


/*Splits a face.
 *The split is ignored if the vertices are adjacent, or if
 *the resulting split would result in faces that have differently signed orientations.
 *
 */

int Face::splitByVerts( int ov1, int ov2 )
{

   // cerr<<"Index:"<<index<<endl;

   int v1 = findVert( ov1 );
   int v2 = findVert( ov2 );

   if ( adjacentVerts( v1, v2 ) )
      return false;

   int min, max;

   if ( v1 < v2 )
   {
      min = v1;
      max = v2;
   }

   else
   {
      min = v2;
      max = v1;
   }

   vector<int> newface1;
   vector<int> newface2;
   newface1.reserve( vlist.size() );
   newface2.reserve( vlist.size() );

   for ( int i = 0; i < ( int ) vlist.size(); i++ )
   {
      if ( i < min )
         newface1.push_back( vlist[ i ] );

      if ( i == min || i == max )
      {
         newface1.push_back( vlist[ i ] );
         newface2.push_back( vlist[ i ] );
      }

      if ( i > min && i < max )
         newface2.push_back( vlist[ i ] );

      if ( i > max )
         newface1.push_back( vlist[ i ] );
   }

   //for now, divide no matter what.  This can result in some ugly faces...
   //also ignores texture coords.

   //remove any references to this face, and mark it for deletion.
   detach();

   getParentObject() -> createFace( newface1 );

   getParentObject() -> createFace( newface2 );


   return index;

}

bool Face::adjacentVerts( int v1, int v2 )
{
   if ( abs( v2 - v1 ) == 1 )
      return true;

   if ( ( v1 == 0 ) && ( v2 == ( int ) vlist.size() - 1 ) )
      return true;

   return false;
}

/**Clear out the old edges, and setup new ones.
  */
bool Face::reEdge()
{

   for ( int i = 0; i < ( int ) elist.size(); i++ )
   {
      EDGE( elist[ i ] ) ->removeFace( index );
   }

   elist.clear();
   elist.reserve( vlist.size() );

   //set up Edges in the face.
   for ( int i = 0; i < ( int ) vlist.size(); i++ )
   {
      int next_i = 0;

      if ( i + 1 != ( int ) vlist.size() )
         next_i = i + 1;

      //setup the edge;
      int e_index = m_parent -> getEdge( vlist[ i ], vlist[ next_i ] );

      elist.push_back( e_index );

      EDGE( e_index ) ->addFace( index );

   }

   return true;

}

/**Merges this face with the given one.
  */
void Face::merge( int s1 )
{
   /*
    Face *f = FACE(s1);
    vector<int> *fedges = f->getEdges(); 
    
    int ei = 0;
    
    //find the edge that joins the faces.
    for(int i=0; i<(int)elist.size(); i++){
     for(int j=0; j<(int)fedges->size(); j++){
      if(elist[i] == (*fedges)[j])
       ei = elist[i];
     } 
    }
    
    Edge *e = EDGE(ei);
    
    //if we are the first edge
    
    //determine which vertex to re-order around
    int begin_vertex1 = e->v1;
    int begin_vertex2 = e->v2;
    int size = (int)vlist.size();
    
    int test_vertex = findVert(begin_vertex1);
    //if it's the last vertex, we know we should use the other one.
    if(test_vertex == size-1){
     begin_vertex1 = e->v2;
     begin_vertex2 = e->v1;
    } else{
     //if it's the first edge.
     if(vlist[begin_vertex1 + 1] == e->v2){
      begin_vertex1 = e->v2;
      begin_vertex2 = e->v1;  
     }
    }
    
    rotateOrder(begin_vertex1);
    f->rotateOrder(begin_vertex2);
    
    vector<int> *fverts = f->getVerts();
    for(int i=1; i<(int)fverts->size()-1; i++){
     addVert((*fverts)[i]);
     VERT((*fverts)[i])->addFace(index); 
    }
    
    reEdge(); 
    */
}

/**Rotates the ordering of the vertices in the face so that it starts with v1
  */
void Face::rotateOrder( int v )
{
   vector<int> nvlist;
   vector<int> nuvlist;

   int index = findVert( v );

   for ( int i = 0; i < index; i++ )
   {
      nvlist.push_back( vlist[ i ] );
      nuvlist.push_back( uvlist[ i ] );
   }

   for ( int i = ( int ) vlist.size() - 1; i >= index; i-- )
   {
      nvlist.insert( nvlist.begin(), vlist[ i ] );
      nuvlist.insert( nuvlist.begin(), uvlist[ i ] );
   }

   vlist = nvlist;
   uvlist = nuvlist;

}


int Face::getEdgeAfter( int v1 )
{
   int tv = findVert( v1 );
   int v2;

   if ( tv == ( int ) vlist.size() - 1 )
      v2 = vlist[ 0 ];
   else
      v2 = vlist[ tv + 1 ];

   //search this list instead of parents to save processing.
   for ( int i = 0; i < ( int ) elist.size(); i++ )
   {
      Edge *e = EDGE( elist[ i ] );

      if ( ( e->getVert( 0 ) == v1 && e->getVert( 1 ) == v2 ) ||
            ( e->getVert( 1 ) == v1 && e->getVert( 0 ) == v2 ) )
      {
         return elist[ i ];
      }
   }

   cerr << "Unable to find edge after " << v1 << " in Face:" << index << endl;
   return -1;
}

int Face::getEdgeBefore( int v2 )
{
   int tv = findVert( v2 );
   int v1;

   if ( tv == 0 )
      v1 = vlist[ vlist.size() - 1 ];
   else
      v1 = vlist[ tv - 1 ];

   //search this list instead of parents to save processing.
   for ( int i = 0; i < ( int ) elist.size(); i++ )
   {
      Edge *e = EDGE( elist[ i ] );

      if ( ( e->getVert( 0 ) == v1 && e->getVert( 1 ) == v2 ) ||
            ( e->getVert( 1 ) == v1 && e->getVert( 0 ) == v2 ) )
      {
         return elist[ i ];
      }
   }

   cerr << "Unable to find edge before " << v1 << " in Face:" << index << endl;
   return -1;
}

void Face::normalize()
{
   getNormal( 0 );

}


Vector4 Face::getUVCenter()
{
   Vector4 c( 0, 0, 0, 0 );
   int n = ( int ) uvlist.size();

   for ( int i = 0; i < n; i++ )
   {
      c += UV( uvlist[ i ] ) -> getPosition();
   }

   c.x /= n;
   c.y /= n;
   c.z /= n;

   return c;

}
