/* avl_functions.c
 * - Functions to compare avl nodes and stuff
 *
 * Copyright (c) 1999 Jack Moffitt, Barath Raghavan, and Alexander Havng
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

/* Last overhauled 990420 <eel@musiknet.se> */

#ifdef HAVE_CONFIG_H
#ifdef _WIN32
#include <win32config.h>
#else
#include <config.h>
#endif
#endif

#include "definitions.h"

#include <stdio.h>
#ifndef __USE_BSD
#define __USE_BSD
#endif
#ifndef __EXTENSIONS__
#define __EXTENSIONS__
#endif

#include <string.h>
#include <sys/types.h>
#include <time.h>

#ifndef _WIN32
#include <sys/socket.h>
#include <netinet/in.h>
#endif

#include "avl.h"
#include "threads.h"
#include "icetypes.h"
#include "icecast.h"
#include "utility.h"
#include "ice_string.h"
#include "connection.h"
#include "log.h"
#include "avl_functions.h"
#include "memory.h"
#include "authenticate/basic.h"
#include "item.h"
#include "sock.h"

int
compare_groups (const void *first, const void *second, void *param)
{
	group_t *v1 = (group_t *) first, *v2 = (group_t *) second;
	
	if (!first || !second || !v1->name || !v2->name)
	{
		xa_debug (2, "WARNING: compare_groups called with NULL pointers!");
		return 0;
	}

	return (ice_strcmp (v1->name, v2->name));
}

int
compare_users (const void *first, const void *second, void *param)
{
	ice_user_t *v1 = (ice_user_t *) first, *v2 = (ice_user_t *) second;
	
	if (!first || !second || !v1->name || !v2->name)
	{
		xa_debug (2, "WARNING: compare_users called with NULL pointers!");
		return 0;
	}

	return (ice_strcmp (v1->name, v2->name));
}

int
compare_mounts (const void *first, const void *second, void *param)
{
	mount_t *v1 = (mount_t *) first, *v2 = (mount_t *) second;
	
	if (!first || !second || !v1->name || !v2->name)
	{
		xa_debug (2, "WARNING: compare_mounts called with NULL pointers!");
		return 0;
	}

	return (ice_strcmp (v1->name, v2->name));
}

int
compare_vars (const void *first, const void *second, void *param)
{
	varpair_t *v1 = (varpair_t *) first, *v2 = (varpair_t *) second;
	
	if (!first || !second || !v1->name || !v2->name)
	{
		xa_debug (2, "WARNING: compare_vars called with NULL pointers!");
		return 0;
	}

	return (ice_strcmp (v1->name, v2->name));
}
	
#ifdef DEBUG_MEMORY
int
compare_mem (const void *firstp, const void *secondp, void *param)
{
	meminfo_t *first = (meminfo_t *) firstp, *second = (meminfo_t *) secondp;

	if (!first || !second)
	{
		fprintf (stderr, "WARNING, compare_mem called with NULL pointers!");
		return 0;
	}

        if ((unsigned long int) (first->ptr) > (unsigned long int) (second->ptr))
		return 1;
	else if ((unsigned long int) (first->ptr) < (unsigned long int) (second->ptr))
		return -1;
	return 0;
}
#endif

int
compare_relay_ids (const void *first, const void *second, void *param)
{
	relay_id_t *r1 = (relay_id_t *) first, *r2 = (relay_id_t *) second;

	return compare_strings (r1->host, r2->host, param);
}

int
compare_relays (const void *first, const void *second, void *param)
{
	relay_t *r1 = (relay_t *) first, *r2 = (relay_t *) second;
	char cfirst[BUFSIZE], csecond[BUFSIZE];

	if (!r1 || !r1->req.host || !r1->req.path || !r2 || !r2->req.host || !r2->req.path) {
		write_log (LOG_DEFAULT, "WARNING: compare_relays() called with NULL pointers!");
	}

	snprintf (cfirst, BUFSIZE, "%s:%d%s", r1->req.host, r1->req.port, r1->req.path);
	snprintf (csecond, BUFSIZE, "%s:%d%s", r2->req.host, r2->req.port, r2->req.path);
	return ice_strcmp (cfirst, csecond);
}

int
compare_strings (const void *first, const void *second, void *param)
{
	char *a1 = (char *)first, *a2 = (char *)second;

	if (!first || !second)
	{
		write_log (LOG_DEFAULT, "WARNING: compare_strings called with null pointers");
		return 0;
	}

	xa_debug (4, "DEBUG: Comparing [%s] with [%s]", a1, a2);
	return (ice_strcasecmp (a1, a2));
}
	
int
compare_aliases (const void *first, const void *second, void *param)
{
	alias_t *a1 = (alias_t *) first, *a2 = (alias_t *) second;
	char full[BUFSIZE], full2[BUFSIZE];

	if (!a1 || !a2 || !a1->name || !a2->name || !a1->name->host || !a1->name->path || !a2->name->host || !a2->name->path)
	{
		write_log (LOG_DEFAULT, "WARNING: NULL pointers in comparison");
		return -1;
	}
	
	snprintf (full, BUFSIZE, "%s:%d%s", a1->name->host, a1->name->port, a1->name->path);
	snprintf (full2, BUFSIZE, "%s:%d%s", a2->name->host, a2->name->port, a2->name->path);

	return ice_strcmp (full, full2);
}

int
compare_restricts (const void *first, const void *second, void *param)
{
	restrict_t *t1 = (restrict_t *)first, *t2 = (restrict_t *)second;
	
	if (!first || !second)
	{
		write_log (LOG_DEFAULT, "WARNING: compare_restricts called with NULL pointers");
		return 0;
	}

	if (t1->id > t2->id)
		return 1;
	if (t1->id < t2->id)
		return -1;
	return 0;
}

int
compare_threads (const void *first, const void *second, void *param)
{
	mythread_t *t1, *t2;
	t1 = (mythread_t *)first;
	t2 = (mythread_t *)second;

	if (!first || !second)
	{
		write_log (LOG_DEFAULT, "WARNING: compare_threads called with NULL pointers");
		return 0;
	}

	if (t1->id > t2->id)
		return 1;
	if (t1->id < t2->id)
		return -1;
	return 0;
}

int
compare_mutexes (const void *first, const void *second, void *param)
{
	mutex_t *t1, *t2;
	t1 = (mutex_t *)first;
	t2 = (mutex_t *)second;

	if (!first || !second)
	{
		write_log (LOG_DEFAULT, "WARNING: compare_mutex called with NULL pointers");
		return 0;
	}

	if (t1->mutexid > t2->mutexid)
		return 1;
	if (t1->mutexid < t2->mutexid)
		return -1;
	return 0;
}

int compare_connection(const void *first, const void *second, void *param)
{
	connection_t *a1, *a2;

	if (!first || !second)
	{
		write_log (LOG_DEFAULT, "WARNING!!! - Null pointer connection!");
		return -1;
	}

	a1 = (connection_t *)first;
	a2 = (connection_t *)second;

	if (a1->type != a2->type)
	{
		write_log (LOG_DEFAULT, "WARNING!!!! - Comparing different type connections");
		return -1;
	}

	if (a1->id > a2->id)
		return 1;
	else if (a1->id < a2->id)
		return -1;
	else
		return 0;
}

void 
zero_trav(avl_traverser *trav)
{
	if (!trav)
	{
		write_log (LOG_DEFAULT, "WARNING: zero_trav called with NULL trav");
		return;
	}
	trav->init = 0;
	trav->nstack = 0;
	trav->p = NULL;
}

int compare_directories(const void *first, const void *second, void *param)
{
	directory_server_t *dir1, *dir2;
	char s1[BUFSIZE], s2[BUFSIZE];

	dir1 = (directory_server_t *)first;
	dir2 = (directory_server_t *)second;

	if (!first || !second)
	{
		write_log (LOG_DEFAULT, "WARNING: compare_directory called with NULL pointers");
		return 0;
	}

	snprintf(s1, BUFSIZE, "%s:%d", dir1->host, dir1->port);
	snprintf(s2, "%s:%d", dir2->host, dir2->port);

	return compare_strings (s1, s2, param);
}

int
compare_sockets (const void *first, const void *second, void *param)
{
	ice_socket_t *is1 = (ice_socket_t *) first, *is2 = (ice_socket_t *) second;
	
	if (is1 == NULL || is2 == NULL) {
		fprintf (stderr, "WARNING: compare_sockets called with NULL values");
		return -1;
	} else if (is1->sock < 0 || is2->sock < 0) {
		fprintf (stderr, "WARNING: compare_sockets called with negative socket number");
		return -1;
	}

	if (is1->sock > is2->sock)
		return 1;
	else if (is1->sock < is2->sock)
		return -1;
	return 0;
}

/*
int
compare_item (const void *first, const void *second, void *param)
{
	item_t *t1, *t2;
	t1 = (item_t *)first;
	t2 = (item_t *)second;

	if (!first || !second)
	{
		write_log (LOG_DEFAULT, "WARNING: compare_item called with NULL pointers");
		return 0;
	}

	if (t1->order > t2->order)
		return 1;
	if (t1->order < t2->order)
		return -1;
	return 0;

}
*/

void *
avl_get_any_node (avl_tree *tree)
{
	avl_traverser trav = {0};
	if (!tree)
	{
		write_log (LOG_DEFAULT, "WARNING: avl_get_any_node called with NULL tree");
		return NULL;
	}

	if (avl_count (tree) <= 0) 
		return NULL;

	return (avl_traverse (tree, &trav));
}







