/*
 * SpanDSP - a series of DSP components for telephony
 *
 * lpc10_tests.c - Test the LPC10 low bit rate speech codec.
 *
 * Written by Steve Underwood <steveu@coppice.org>
 *
 * Copyright (C) 2006 Steve Underwood
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2, as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: lpc10_tests.c,v 1.8 2006/10/24 13:45:29 steveu Exp $
 */

/*! \file */

/*! \page lpc10_tests_page LPC10 codec tests
\section lpc10_tests_page_sec_1 What does it do?

\section lpc10_tests_page_sec_2 How is it used?
To perform a general audio quality test, lpc10 should be run. The file ../localtests/short_nb_voice.wav
will be compressed to LPC10 data, decompressed, and the resulting audio stored in post_lpc10.wav.
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <inttypes.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <assert.h>
#include <fcntl.h>
#include <ctype.h>
#include <stdio.h>
#include <audiofile.h>
#include <tiffio.h>

#include "spandsp.h"

#define BLOCK_LEN       180

#define BLOCKS_PER_READ 1

#define IN_FILE_NAME    "../localtests/dam9.wav"
#define REF_FILE_NAME   "../localtests/dam9_lpc55.wav"
#define OUT_FILE_NAME   "post_lpc10.wav"

static void write_bits(FILE *f, uint8_t bits[], int len)
{
    int i;          /* generic loop variable */
    uint8_t mask;   /* The next bit position within the variable "data" to place the next bit. */
    uint8_t data;   /* The contents of the next byte to place in the output. */

    /* Fill in the array bits.
     * The first compressed output bit will be the most significant
     * bit of the byte, so initialize mask to 0x80.  The next byte of
     * compressed data is initially 0, and the desired bits will be
     * turned on below.
     */
    mask = 0x80;
    data = 0;
    for (i = 0;  i < len;  i++)
    {
        /* Turn on the next bit of output data, if necessary. */
        if (bits[i])
            data |= mask;
        /*
         * If the byte data is full, determined by mask becoming 0,
         * then write the byte to the output file, and reinitialize
         * data and mask for the next output byte.  Also add the byte
         * if (i == len-1), because if len is not a multiple of 8,
         * then mask won't yet be 0.  */
        mask >>= 1;
        if ((mask == 0)  ||  (i == len - 1))
        {
            putc(data, f);
            data = 0;
            mask = 0x80;
        }
    }
}

int main(int argc, char *argv[])
{
    AFfilehandle inhandle;
    AFfilehandle refhandle;
    AFfilehandle outhandle;
    AFfilesetup filesetup;
    int frames;
    int outframes;
    float x;
    double pre_energy;
    double post_energy;
    double ref_energy;
    double diff_energy;
    int16_t pre_amp[BLOCKS_PER_READ*BLOCK_LEN];
    int16_t post_amp[BLOCKS_PER_READ*BLOCK_LEN];
    int16_t ref_amp[BLOCKS_PER_READ*BLOCK_LEN];
    int16_t log_amp[BLOCKS_PER_READ*BLOCK_LEN*3];
    uint8_t lpc10_data[BLOCKS_PER_READ*7];
    double xx;
    lpc10_encode_state_t *lpc10_enc_state;
    lpc10_decode_state_t *lpc10_dec_state;
    int i;
    int block_no;
    int log_error;
    const char *in_file_name;

    log_error = TRUE;
    in_file_name = IN_FILE_NAME;
    for (i = 1;  i < argc;  i++)
    {
        if (strcmp(argv[i], "-l") == 0)
        {
            log_error = FALSE;
            continue;
        }
        if (strcmp(argv[i], "-i") == 0)
        {
            in_file_name = argv[++i];
            continue;
        }
    }

    if ((inhandle = afOpenFile(in_file_name, "r", 0)) == AF_NULL_FILEHANDLE)
    {
        printf("    Cannot open wave file '%s'\n", in_file_name);
        exit(2);
    }
    if ((x = afGetFrameSize(inhandle, AF_DEFAULT_TRACK, 1)) != 2.0)
    {
        printf("    Unexpected frame size in wave file '%s'\n", in_file_name);
        exit(2);
    }
    if ((x = afGetRate(inhandle, AF_DEFAULT_TRACK)) != (float) SAMPLE_RATE)
    {
        printf("    Unexpected sample rate in wave file '%s'\n", in_file_name);
        exit(2);
    }
    if ((x = afGetChannels(inhandle, AF_DEFAULT_TRACK)) != 1.0)
    {
        printf("    Unexpected number of channels in wave file '%s'\n", in_file_name);
        exit(2);
    }
    if ((filesetup = afNewFileSetup()) == AF_NULL_FILESETUP)
    {
        fprintf(stderr, "    Failed to create file setup\n");
        exit(2);
    }

    if ((refhandle = afOpenFile(REF_FILE_NAME, "r", 0)) == AF_NULL_FILEHANDLE)
    {
        printf("    Cannot open wave file '%s'\n", REF_FILE_NAME);
        exit(2);
    }
    if ((x = afGetFrameSize(refhandle, AF_DEFAULT_TRACK, 1)) != 2.0)
    {
        printf("    Unexpected frame size in wave file '%s'\n", REF_FILE_NAME);
        exit(2);
    }
    if ((x = afGetRate(refhandle, AF_DEFAULT_TRACK)) != (float) SAMPLE_RATE)
    {
        printf("    Unexpected sample rate in wave file '%s'\n", REF_FILE_NAME);
        exit(2);
    }
    if ((x = afGetChannels(refhandle, AF_DEFAULT_TRACK)) != 1.0)
    {
        printf("    Unexpected number of channels in wave file '%s'\n", REF_FILE_NAME);
        exit(2);
    }

    if ((filesetup = afNewFileSetup()) == AF_NULL_FILESETUP)
    {
        fprintf(stderr, "    Failed to create file setup\n");
        exit(2);
    }
    afInitSampleFormat(filesetup, AF_DEFAULT_TRACK, AF_SAMPFMT_TWOSCOMP, 16);
    afInitRate(filesetup, AF_DEFAULT_TRACK, (float) SAMPLE_RATE);
    afInitFileFormat(filesetup, AF_FILE_WAVE);
    afInitChannels(filesetup, AF_DEFAULT_TRACK, 1);

    if ((outhandle = afOpenFile(OUT_FILE_NAME, "w", filesetup)) == AF_NULL_FILEHANDLE)
    {
        fprintf(stderr, "    Cannot create wave file '%s'\n", OUT_FILE_NAME);
        exit(2);
    }
    
    if ((lpc10_enc_state = lpc10_encode_init(NULL, TRUE)) == NULL)
    {
        fprintf(stderr, "    Cannot create encoder\n");
        exit(2);
    }
            
    if ((lpc10_dec_state = lpc10_decode_init(NULL, TRUE)) == NULL)
    {
        fprintf(stderr, "    Cannot create decoder\n");
        exit(2);
    }

    pre_energy = 0.0;
    post_energy = 0.0;
    ref_energy = 0.0;
    diff_energy = 0.0;
    block_no = 0;
    while ((frames = afReadFrames(inhandle, AF_DEFAULT_TRACK, pre_amp, BLOCKS_PER_READ*BLOCK_LEN)) == BLOCKS_PER_READ*BLOCK_LEN
            &&
            (frames = afReadFrames(refhandle, AF_DEFAULT_TRACK, ref_amp, BLOCKS_PER_READ*BLOCK_LEN)) == BLOCKS_PER_READ*BLOCK_LEN)
    {
        lpc10_encode(lpc10_enc_state, lpc10_data, pre_amp, BLOCKS_PER_READ);
        //write_bits(stdout, lpc10_data, BLOCKS_PER_READ*7);
        lpc10_decode(lpc10_dec_state, post_amp, lpc10_data, BLOCKS_PER_READ);
        for (i = 0;  i < BLOCK_LEN;  i++)
        {
            pre_energy += (double) pre_amp[i] * (double) pre_amp[i];
            post_energy += (double) post_amp[i] * (double) post_amp[i];
            ref_energy += (double) ref_amp[i] * (double) ref_amp[i];
            /* The reference file has some odd clipping, so eliminate this from the
               energy measurement. */
            if (ref_amp[i] == 32767  ||  ref_amp[i] == -32768)
                xx = 0.0;
            else
                xx = post_amp[i] - ref_amp[i];
            diff_energy += (double) xx * (double) xx;
            log_amp[i] = xx; //post_amp[i] - ref_amp[i];
        }
        block_no++;
        if (log_error)
            outframes = afWriteFrames(outhandle, AF_DEFAULT_TRACK, log_amp, frames);
        else
            outframes = afWriteFrames(outhandle, AF_DEFAULT_TRACK, post_amp, frames);
    }

    if (afCloseFile(inhandle) != 0)
    {
        printf("    Cannot close wave file '%s'\n", in_file_name);
        exit(2);
    }
    if (afCloseFile(refhandle) != 0)
    {
        printf("    Cannot close wave file '%s'\n", REF_FILE_NAME);
        exit(2);
    }
    if (afCloseFile(outhandle) != 0)
    {
        printf("    Cannot close wave file '%s'\n", OUT_FILE_NAME);
        exit(2);
    }
    afFreeFileSetup(filesetup);
    lpc10_encode_release(lpc10_enc_state);
    lpc10_decode_release(lpc10_dec_state);

    printf("Output energy is %f%% of input energy.\n", 100.0*post_energy/pre_energy);
    printf("Difference energy is %f%% of the total.\n", 100.0*diff_energy/ref_energy);
    if (fabs(1.0 - post_energy/pre_energy) > 0.05
        ||
        fabs(diff_energy/post_energy) > 0.03)
    {
        printf("Tests failed.\n");
        exit(2);
    }
    
    printf("Tests passed.\n");
    return 0;
}
/*- End of function --------------------------------------------------------*/
/*- End of file ------------------------------------------------------------*/
