/*
 * SpanDSP - a series of DSP components for telephony
 *
 * lpc10.h - LPC10 low bit rate speech codec.
 *
 * Written by Steve Underwood <steveu@coppice.org>
 *
 * Copyright (C) 2006 Steve Underwood
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2, as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: lpc10.h,v 1.7 2006/10/24 13:45:28 steveu Exp $
 */

#if !defined(_LPC10_H_)
#define _LPC10_H_

/*! \page lpc10_page LPC10 encoding and decoding
\section lpc10_page_sec_1 What does it do?
The LPC10 module implements the US Department of Defense LPC10
codec. This codec produces compressed data at 2400bps. At such
a low rate high fidelity cannot be expected. However, the speech
clarity is quite good, and this codec is unencumbered by patent
or other restrictions.

\section lpc10_page_sec_2 How does it work?
???.
*/

#define LPC10_SAMPLES_PER_FRAME 180
#define LPC10_BITS_IN_COMPRESSED_FRAME 54

/*!
    LPC10 codec unpacked frame.
*/
typedef struct
{
    int32_t ipitch;
    int32_t irms;
    int32_t irc[10];
} lpc10_frame_t;

/*!
    LPC10 codec encoder state descriptor. This defines the state of
    a single working instance of the LPC10 encoder.
*/
typedef struct
{
    int error_correction;

    /* State used only by function high_pass_100hz */
    float z11;
    float z21;
    float z12;
    float z22;
    
    /* State used by function lpc10_analyse */
    float inbuf[LPC10_SAMPLES_PER_FRAME*3];
    float pebuf[LPC10_SAMPLES_PER_FRAME*3];
    float lpbuf[696];
    float ivbuf[312];
    float bias;
    int32_t osbuf[10];  /* no initial value necessary */
    int32_t osptr;      /* initial value 1 */
    int32_t obound[3];
    int32_t vwin[6];    /* was [2][3] */   /* initial value vwin[4] = 307; vwin[5] = 462; */
    int32_t awin[6];    /* was [2][3] */   /* initial value awin[4] = 307; awin[5] = 462; */
    int32_t voibuf[8];  /* was [2][4] */
    float rmsbuf[3];
    float rcbuf[30];    /* was [10][3] */
    float zpre;


    /* State used by function onset */
    float n;
    float d__;          /* initial value 1.0f */
    float fpc;          /* no initial value necessary */
    float l2buf[16];
    float l2sum1;
    int32_t l2ptr1;     /* initial value 1 */
    int32_t l2ptr2;     /* initial value 9 */
    int32_t lasti;      /* no initial value necessary */
    int hyst;           /* initial value FALSE */

    /* State used by function lpc10_voicing */
    float dither;       /* initial value 20.0f */
    float snr;
    float maxmin;
    float voice[6]	     /* was [2][3] */;   /* initial value is probably unnecessary */
    int32_t lbve;
    int32_t lbue;
    int32_t fbve;
    int32_t fbue;
    int32_t ofbue;
    int32_t sfbue;
    int32_t olbue;
    int32_t slbue;

    /* State used by function dynamic_pitch_tracking */
    float s[60];
    int32_t p[120]	     /* was [60][2] */;
    int32_t ipoint;
    float alphax;

    /* State used by function lpc10_pack */
    int32_t isync;
} lpc10_encode_state_t;

/*!
    LPC10 codec decoder state descriptor. This defines the state of
    a single working instance of the LPC10 decoder.
*/
typedef struct
{
    int error_correction;

    /* State used by function decode */
    int32_t iptold;     /* initial value 60 */
    int first;          /* initial value TRUE */
    int32_t ivp2h;
    int32_t iovoic;
    int32_t iavgp;      /* initial value 60 */
    int32_t erate;
    int32_t drc[10][3];
    int32_t dpit[3];
    int32_t drms[3];

    /* State used by function synths */
    float buf[LPC10_SAMPLES_PER_FRAME*2];
    int32_t buflen;     /* initial value LPC10_SAMPLES_PER_FRAME */

    /* State used by function pitsyn */
    int32_t ivoico;     /* no initial value necessary as long as first_pitsyn is initially TRUE_ */
    int32_t ipito;      /* no initial value necessary as long as first_pitsyn is initially TRUE_ */
    float rmso;         /* initial value 1.0f */
    float rco[10];      /* no initial value necessary as long as first_pitsyn is initially TRUE_ */
    int32_t jsamp;      /* no initial value necessary as long as first_pitsyn is initially TRUE_ */
    int first_pitsyn;   /* initial value TRUE_ */

    /* State used by function bsynz */
    int32_t ipo;
    float exc[166];
    float exc2[166];
    float lpi[3];
    float hpi[3];
    float rmso_bsynz;

    /* State used by function random */
    int32_t j;
    int32_t k;
    int16_t y[5];

    /* State used by function deemp */
    float dei[2];
    float deo[3];
} lpc10_decode_state_t;

#ifdef __cplusplus
extern "C" {
#endif

/*! Initialise an LPC10e encode context.
    \param s The LPC10e context
    \param error_correction ???
    \return A pointer to the LPC10e context, or NULL for error. */
lpc10_encode_state_t *lpc10_encode_init(lpc10_encode_state_t *s, int error_correction);

int lpc10_encode_release(lpc10_encode_state_t *s);

/*! Encode a buffer of linear PCM data to LPC10e.
    \param s The LPC10e context.
    \param ima_data The LPC10e data produced.
    \param amp The audio sample buffer.
    \param len The number of samples in the buffer.
    \return The number of bytes of LPC10e data produced. */
int lpc10_encode(lpc10_encode_state_t *s, uint8_t code[], const int16_t amp[], int quant);

/*! Initialise an LPC10e decode context.
    \param s The LPC10e context
    \param error_correction ???
    \return A pointer to the LPC10e context, or NULL for error. */
lpc10_decode_state_t *lpc10_decode_init(lpc10_decode_state_t *st, int error_correction);

int lpc10_decode_release(lpc10_decode_state_t *s);

/*! Decode a buffer of LPC10e data to linear PCM.
    \param s The LPC10e context.
    \param amp The audio sample buffer.
    \param code The LPC10e data.
    \param quant The number of frames of LPC10e data to be decoded.
    \return The number of samples returned. */
int lpc10_decode(lpc10_decode_state_t *s, int16_t amp[], const uint8_t code[], int quant);


#ifdef __cplusplus
}
#endif

#endif
/*- End of include ---------------------------------------------------------*/
