/*BHEADER**********************************************************************
 * Copyright (c) 2008,  Lawrence Livermore National Security, LLC.
 * Produced at the Lawrence Livermore National Laboratory.
 * This file is part of HYPRE.  See file COPYRIGHT for details.
 *
 * HYPRE is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License (as published by the Free
 * Software Foundation) version 2.1 dated February 1999.
 *
 * $Revision: 2.24 $
 ***********************************************************************EHEADER*/




/******************************************************************************
 *
 * HYPRE_SStructMatrix interface
 *
 *****************************************************************************/

#include "headers.h"

/*--------------------------------------------------------------------------
 *--------------------------------------------------------------------------*/

int
HYPRE_SStructMatrixCreate( MPI_Comm              comm,
                           HYPRE_SStructGraph    graph,
                           HYPRE_SStructMatrix  *matrix_ptr )
{
   /* GEC1202 grid not needed  */
   /*  hypre_SStructGrid      *grid     = hypre_SStructGraphGrid(graph); */
   hypre_SStructStencil ***stencils = hypre_SStructGraphStencils(graph);

   hypre_SStructMatrix    *matrix;
   int                  ***splits;
   int                     nparts;
   hypre_SStructPMatrix  **pmatrices;
   int                  ***symmetric;

   hypre_SStructPGrid     *pgrid;
   int                     nvars;

   int                     stencil_size;
   int                    *stencil_vars;
   int                     pstencil_size;

   HYPRE_SStructVariable   vitype, vjtype;
   int                     part, vi, vj, i;
   int                     size;

   matrix = hypre_TAlloc(hypre_SStructMatrix, 1);

   hypre_SStructMatrixComm(matrix)  = comm;
   hypre_SStructMatrixNDim(matrix)  = hypre_SStructGraphNDim(graph);
   hypre_SStructGraphRef(graph, &hypre_SStructMatrixGraph(matrix));

   /* compute S/U-matrix split */
   nparts = hypre_SStructGraphNParts(graph);
   hypre_SStructMatrixNParts(matrix) = nparts;
   splits = hypre_TAlloc(int **, nparts);
   hypre_SStructMatrixSplits(matrix) = splits;
   pmatrices = hypre_TAlloc(hypre_SStructPMatrix *, nparts);
   hypre_SStructMatrixPMatrices(matrix) = pmatrices;
   symmetric = hypre_TAlloc(int **, nparts);
   hypre_SStructMatrixSymmetric(matrix) = symmetric;
   for (part = 0; part < nparts; part++)
   {
      pgrid = hypre_SStructGraphPGrid(graph, part);
      nvars = hypre_SStructPGridNVars(pgrid);
      splits[part] = hypre_TAlloc(int *, nvars);
      symmetric[part] = hypre_TAlloc(int *, nvars);
      for (vi = 0; vi < nvars; vi++)
      {
         stencil_size  = hypre_SStructStencilSize(stencils[part][vi]);
         stencil_vars  = hypre_SStructStencilVars(stencils[part][vi]);
         pstencil_size = 0;
         splits[part][vi] = hypre_TAlloc(int, stencil_size);
         symmetric[part][vi] = hypre_TAlloc(int, nvars);
         for (i = 0; i < stencil_size; i++)
         {
            vj = stencil_vars[i];
            vitype = hypre_SStructPGridVarType(pgrid, vi);
            vjtype = hypre_SStructPGridVarType(pgrid, vj);
            if (vjtype == vitype)
            {
               splits[part][vi][i] = pstencil_size;
               pstencil_size++;
            }
            else
            {
               splits[part][vi][i] = -1;
            }
         }
         for (vj = 0; vj < nvars; vj++)
         {
            symmetric[part][vi][vj] = 0;
         }
      }
   }

   /* GEC0902 move the IJ creation to the initialization phase  
    * ilower = hypre_SStructGridGhstartRank(grid);
    * iupper = ilower + hypre_SStructGridGhlocalSize(grid) - 1; 
    * HYPRE_IJMatrixCreate(comm, ilower, iupper, ilower, iupper,
    *                    &hypre_SStructMatrixIJMatrix(matrix)); */
   
   hypre_SStructMatrixIJMatrix(matrix)     = NULL;
   hypre_SStructMatrixParCSRMatrix(matrix) = NULL;

   size = 0;
   for (part = 0; part < nparts; part++)
   {
      pgrid = hypre_SStructGraphPGrid(graph, part);
      nvars = hypre_SStructPGridNVars(pgrid);
      for (vi = 0; vi < nvars; vi++)
      {
         size = hypre_max(size, hypre_SStructStencilSize(stencils[part][vi]));
      }
   }
   hypre_SStructMatrixEntriesSize(matrix) = size;
   hypre_SStructMatrixSEntries(matrix) = hypre_TAlloc(int, size);
   hypre_SStructMatrixUEntries(matrix) = hypre_TAlloc(int, size);
   hypre_SStructMatrixTmpColCoords(matrix) = NULL;
   hypre_SStructMatrixTmpCoeffs(matrix)    = NULL;

   hypre_SStructMatrixNSSymmetric(matrix) = 0;
   hypre_SStructMatrixGlobalSize(matrix)  = 0;
   hypre_SStructMatrixRefCount(matrix)    = 1;
  
   /* GEC0902 setting the default of the object_type to HYPRE_SSTRUCT */ 

   hypre_SStructMatrixObjectType(matrix) = HYPRE_SSTRUCT;

   *matrix_ptr = matrix;

   return hypre_error_flag;
}

/*--------------------------------------------------------------------------
 *--------------------------------------------------------------------------*/

int 
HYPRE_SStructMatrixDestroy( HYPRE_SStructMatrix matrix )
{
   hypre_SStructGraph     *graph;
   int                  ***splits;
   int                     nparts;
   hypre_SStructPMatrix  **pmatrices;
   int                  ***symmetric;
   hypre_SStructPGrid     *pgrid;
   int                     nvars;
   int                     part, var;

   if (matrix)
   {
      hypre_SStructMatrixRefCount(matrix) --;
      if (hypre_SStructMatrixRefCount(matrix) == 0)
      {
         graph        = hypre_SStructMatrixGraph(matrix);
         splits       = hypre_SStructMatrixSplits(matrix);
         nparts       = hypre_SStructMatrixNParts(matrix);
         pmatrices    = hypre_SStructMatrixPMatrices(matrix);
         symmetric    = hypre_SStructMatrixSymmetric(matrix);
         for (part = 0; part < nparts; part++)
         {
            pgrid = hypre_SStructGraphPGrid(graph, part);
            nvars = hypre_SStructPGridNVars(pgrid);
            for (var = 0; var < nvars; var++)
            {
               hypre_TFree(splits[part][var]);
               hypre_TFree(symmetric[part][var]);
            }
            hypre_TFree(splits[part]);
            hypre_TFree(symmetric[part]);
            hypre_SStructPMatrixDestroy(pmatrices[part]);
         }
         HYPRE_SStructGraphDestroy(graph);
         hypre_TFree(splits);
         hypre_TFree(pmatrices);
         hypre_TFree(symmetric);
         HYPRE_IJMatrixDestroy(hypre_SStructMatrixIJMatrix(matrix));
         hypre_TFree(hypre_SStructMatrixSEntries(matrix));
         hypre_TFree(hypre_SStructMatrixUEntries(matrix));
         hypre_TFree(hypre_SStructMatrixTmpColCoords(matrix));
         hypre_TFree(hypre_SStructMatrixTmpCoeffs(matrix));
         hypre_TFree(matrix);
      }
   }

   return hypre_error_flag;
}

/*--------------------------------------------------------------------------
 *--------------------------------------------------------------------------*/

int
HYPRE_SStructMatrixInitialize( HYPRE_SStructMatrix matrix )
{
   int                     nparts    = hypre_SStructMatrixNParts(matrix);
   hypre_SStructGraph     *graph     = hypre_SStructMatrixGraph(matrix);
   hypre_SStructPMatrix  **pmatrices = hypre_SStructMatrixPMatrices(matrix);
   int                  ***symmetric = hypre_SStructMatrixSymmetric(matrix);
   hypre_SStructStencil ***stencils  = hypre_SStructGraphStencils(graph);
   int                    *split;

   MPI_Comm                pcomm;
   hypre_SStructPGrid     *pgrid;
   hypre_SStructStencil  **pstencils;
   int                     nvars;

   int                     stencil_size;
   hypre_Index            *stencil_shape;
   int                    *stencil_vars;
   int                     pstencil_ndim;
   int                     pstencil_size;

   int                     part, var, i;

   /* GEC0902 addition of variables for ilower and iupper   */
   MPI_Comm                 comm;
   hypre_SStructGrid       *grid;
   int                     ilower, iupper;
   int                   matrix_type = hypre_SStructMatrixObjectType(matrix);

   /* S-matrix */
   for (part = 0; part < nparts; part++)
   {
      pgrid = hypre_SStructGraphPGrid(graph, part);
      nvars = hypre_SStructPGridNVars(pgrid);
      pstencils = hypre_TAlloc(hypre_SStructStencil *, nvars);
      for (var = 0; var < nvars; var++)
      {
         split = hypre_SStructMatrixSplit(matrix, part, var);
         stencil_size  = hypre_SStructStencilSize(stencils[part][var]);
         stencil_shape = hypre_SStructStencilShape(stencils[part][var]);
         stencil_vars  = hypre_SStructStencilVars(stencils[part][var]);
         pstencil_ndim = hypre_SStructStencilNDim(stencils[part][var]);
         pstencil_size = 0;
         for (i = 0; i < stencil_size; i++)
         {
            if (split[i] > -1)
            {
               pstencil_size++;
            }
         }
         HYPRE_SStructStencilCreate(pstencil_ndim, pstencil_size,
                                    &pstencils[var]);
         for (i = 0; i < stencil_size; i++)
         {
            if (split[i] > -1)
            {
               HYPRE_SStructStencilSetEntry(pstencils[var], split[i],
                                            stencil_shape[i],
                                            stencil_vars[i]);
            }
         }
      }
      pcomm = hypre_SStructPGridComm(pgrid);
      hypre_SStructPMatrixCreate(pcomm, pgrid, pstencils, &pmatrices[part]);
      for (var = 0; var < nvars; var++)
      {
         for (i = 0; i < nvars; i++)
         {
            hypre_SStructPMatrixSetSymmetric(pmatrices[part], var, i,
                                             symmetric[part][var][i]);
         }
      }
      hypre_SStructPMatrixInitialize(pmatrices[part]);
   }

      /* U-matrix */

   /* GEC0902  knowing the kind of matrix we can create the IJMATRIX with the 
    *  the right dimension (HYPRE_PARCSR without ghosts) */

   grid  = hypre_SStructGraphGrid(graph); 
   comm  =  hypre_SStructMatrixComm(matrix); 

   if(matrix_type == HYPRE_PARCSR)
   {
     ilower = hypre_SStructGridStartRank(grid);
     iupper = ilower + hypre_SStructGridLocalSize(grid) - 1;
   }
   
   if(matrix_type == HYPRE_SSTRUCT || matrix_type == HYPRE_STRUCT)
   {
     ilower = hypre_SStructGridGhstartRank(grid);
     iupper = ilower + hypre_SStructGridGhlocalSize(grid) - 1;
   }
    
   HYPRE_IJMatrixCreate(comm, ilower, iupper, ilower, iupper,
                        &hypre_SStructMatrixIJMatrix(matrix)); 


   /* U-matrix */
   hypre_SStructUMatrixInitialize(matrix);

   return hypre_error_flag;
}

/*--------------------------------------------------------------------------
 *--------------------------------------------------------------------------*/

int
HYPRE_SStructMatrixSetValues( HYPRE_SStructMatrix  matrix,
                              int                  part,
                              int                 *index,
                              int                  var,
                              int                  nentries,
                              int                 *entries,
                              double              *values )
{
   int                   ndim  = hypre_SStructMatrixNDim(matrix);
   int                  *Sentries;
   int                  *Uentries;
   int                   nSentries;
   int                   nUentries;
   hypre_SStructPMatrix *pmatrix;
   hypre_Index           cindex;

   hypre_SStructMatrixSplitEntries(matrix, part, var, nentries, entries,
                                   &nSentries, &Sentries,
                                   &nUentries, &Uentries);

   hypre_CopyToCleanIndex(index, ndim, cindex);

   /* S-matrix */
   if (nSentries > 0)
   {
      pmatrix = hypre_SStructMatrixPMatrix(matrix, part);
      hypre_SStructPMatrixSetValues(pmatrix, cindex, var,
                                    nSentries, Sentries, values, 0);
   }
   /* U-matrix */
   if (nUentries > 0)
   {
      hypre_SStructUMatrixSetValues(matrix, part, cindex, var,
                                    nUentries, Uentries, values, 0);
   }

   return hypre_error_flag;
}

/*--------------------------------------------------------------------------
 *--------------------------------------------------------------------------*/

int
HYPRE_SStructMatrixSetBoxValues( HYPRE_SStructMatrix  matrix,
                                 int                  part,
                                 int                 *ilower,
                                 int                 *iupper,
                                 int                  var,
                                 int                  nentries,
                                 int                 *entries,
                                 double              *values )
{
   int                   ndim  = hypre_SStructMatrixNDim(matrix);
   int                  *Sentries;
   int                  *Uentries;
   int                   nSentries;
   int                   nUentries;
   hypre_SStructPMatrix *pmatrix;
   hypre_Index           cilower;
   hypre_Index           ciupper;

   hypre_SStructMatrixSplitEntries(matrix, part, var, nentries, entries,
                                   &nSentries, &Sentries,
                                   &nUentries, &Uentries);

   hypre_CopyToCleanIndex(ilower, ndim, cilower);
   hypre_CopyToCleanIndex(iupper, ndim, ciupper);

   /* S-matrix */
   if (nSentries > 0)
   {
      pmatrix = hypre_SStructMatrixPMatrix(matrix, part);
      hypre_SStructPMatrixSetBoxValues(pmatrix, cilower, ciupper, var,
                                       nSentries, Sentries, values, 0);
   }
   /* U-matrix */
   if (nUentries > 0)
   {
      hypre_SStructUMatrixSetBoxValues(matrix, part, cilower, ciupper, var,
                                       nUentries, Uentries, values, 0);
   }

   return hypre_error_flag;
}

/*--------------------------------------------------------------------------
 *--------------------------------------------------------------------------*/

int
HYPRE_SStructMatrixGetValues( HYPRE_SStructMatrix  matrix,
                              int                  part,
                              int                 *index,
                              int                  var,
                              int                  nentries,
                              int                 *entries,
                              double              *values )
{
   int                   ndim  = hypre_SStructMatrixNDim(matrix);
   int                  *Sentries;
   int                  *Uentries;
   int                   nSentries;
   int                   nUentries;
   hypre_SStructPMatrix *pmatrix;
   hypre_Index           cindex;

   hypre_SStructMatrixSplitEntries(matrix, part, var, nentries, entries,
                                   &nSentries, &Sentries,
                                   &nUentries, &Uentries);

   hypre_CopyToCleanIndex(index, ndim, cindex);

   /* S-matrix */
   if (nSentries > 0)
   {
      pmatrix = hypre_SStructMatrixPMatrix(matrix, part);
      hypre_SStructPMatrixSetValues(pmatrix, cindex, var,
                                    nSentries, Sentries, values, -1);
   }
   /* U-matrix */
   if (nUentries > 0)
   {
      hypre_SStructUMatrixSetValues(matrix, part, cindex, var,
                                    nUentries, Uentries, values, -1);
   }

   return hypre_error_flag;
}

/*--------------------------------------------------------------------------
 *--------------------------------------------------------------------------*/

int
HYPRE_SStructMatrixGetBoxValues( HYPRE_SStructMatrix  matrix,
                                 int                  part,
                                 int                 *ilower,
                                 int                 *iupper,
                                 int                  var,
                                 int                  nentries,
                                 int                 *entries,
                                 double              *values )
{
   int                   ndim  = hypre_SStructMatrixNDim(matrix);
   int                  *Sentries;
   int                  *Uentries;
   int                   nSentries;
   int                   nUentries;
   hypre_SStructPMatrix *pmatrix;
   hypre_Index           cilower;
   hypre_Index           ciupper;
   int                   action;

   hypre_SStructMatrixSplitEntries(matrix, part, var, nentries, entries,
                                   &nSentries, &Sentries,
                                   &nUentries, &Uentries);

   hypre_CopyToCleanIndex(ilower, ndim, cilower);
   hypre_CopyToCleanIndex(iupper, ndim, ciupper);

   /* action < 0: get values */
   action= -1;

   /* S-matrix */
   if (nSentries > 0)
   {
      pmatrix = hypre_SStructMatrixPMatrix(matrix, part);
      hypre_SStructPMatrixSetBoxValues(pmatrix, cilower, ciupper, var,
                                       nSentries, Sentries, values, action);
   }
   /* U-matrix */
   if (nUentries > 0)
   {
      hypre_SStructUMatrixSetBoxValues(matrix, part, cilower, ciupper, var,
                                       nUentries, Uentries, values, action);
   }

   return hypre_error_flag;
}
/*--------------------------------------------------------------------------
 *--------------------------------------------------------------------------*/

int 
HYPRE_SStructMatrixAddToValues( HYPRE_SStructMatrix  matrix,
                                int                  part,
                                int                 *index,
                                int                  var,
                                int                  nentries,
                                int                 *entries,
                                double              *values )
{
   int                   ndim  = hypre_SStructMatrixNDim(matrix);
   int                  *Sentries;
   int                  *Uentries;
   int                   nSentries;
   int                   nUentries;
   hypre_SStructPMatrix *pmatrix;
   hypre_Index           cindex;

   hypre_SStructMatrixSplitEntries(matrix, part, var, nentries, entries,
                                   &nSentries, &Sentries,
                                   &nUentries, &Uentries);

   hypre_CopyToCleanIndex(index, ndim, cindex);

   /* S-matrix */
   if (nSentries > 0)
   {
      pmatrix = hypre_SStructMatrixPMatrix(matrix, part);
      hypre_SStructPMatrixSetValues(pmatrix, cindex, var,
                                    nSentries, Sentries, values, 1);
   }
   /* U-matrix */
   if (nUentries > 0)
   {
      hypre_SStructUMatrixSetValues(matrix, part, cindex, var,
                                    nUentries, Uentries, values, 1);
   }

   return hypre_error_flag;
}

/*--------------------------------------------------------------------------
 *--------------------------------------------------------------------------*/

int 
HYPRE_SStructMatrixAddToBoxValues( HYPRE_SStructMatrix  matrix,
                                   int                  part,
                                   int                 *ilower,
                                   int                 *iupper,
                                   int                  var,
                                   int                  nentries,
                                   int                 *entries,
                                   double              *values )
{
   int                   ndim  = hypre_SStructMatrixNDim(matrix);
   int                  *Sentries;
   int                  *Uentries;
   int                   nSentries;
   int                   nUentries;
   hypre_SStructPMatrix *pmatrix;
   hypre_Index           cilower;
   hypre_Index           ciupper;

   hypre_SStructMatrixSplitEntries(matrix, part, var, nentries, entries,
                                   &nSentries, &Sentries,
                                   &nUentries, &Uentries);

   hypre_CopyToCleanIndex(ilower, ndim, cilower);
   hypre_CopyToCleanIndex(iupper, ndim, ciupper);

   /* S-matrix */
   if (nSentries > 0)
   {
      pmatrix = hypre_SStructMatrixPMatrix(matrix, part);
      hypre_SStructPMatrixSetBoxValues(pmatrix, cilower, ciupper, var,
                                       nSentries, Sentries, values, 1);
   }
   /* U-matrix */
   if (nUentries > 0)
   {
      hypre_SStructUMatrixSetBoxValues(matrix, part, cilower, ciupper, var,
                                       nUentries, Uentries, values, 1);
   }

   return hypre_error_flag;
}

/*--------------------------------------------------------------------------
 *--------------------------------------------------------------------------*/

int 
HYPRE_SStructMatrixAssemble( HYPRE_SStructMatrix matrix )
{
   hypre_SStructGraph      *graph          = hypre_SStructMatrixGraph(matrix);
   int                      nparts         = hypre_SStructMatrixNParts(matrix);
   hypre_SStructPMatrix   **pmatrices      = hypre_SStructMatrixPMatrices(matrix);
   hypre_SStructGrid       *grid           = hypre_SStructGraphGrid(graph);
   int                    **nvneighbors    = hypre_SStructGridNVNeighbors(grid);
   hypre_SStructNeighbor ***vneighbors     = hypre_SStructGridVNeighbors(grid);
   hypre_SStructCommInfo **vnbor_comm_info = hypre_SStructGridVNborCommInfo(grid);
   int                     vnbor_ncomms    = hypre_SStructGridVNborNComms(grid);

   hypre_SStructPMatrix   *pmatrix;
   hypre_SStructStencil   *stencil;
   hypre_Index            *shape;
   int                    *smap;
   int                    *vars;
   hypre_StructMatrix     *smatrix;
   hypre_StructGrid       *sgrid;
   hypre_SStructNeighbor  *vneighbor;

   hypre_Box              *box, *sbox, *ibox;
   hypre_IndexRef          offset;

   int                    *entries;
   int                    *Sentries;
   int                    *Uentries;
   int                     nSentries;
   int                     nUentries;

   double                 *values = NULL;

   int                     nvars, nentries;
   int                     part, var, entry, sentry, b, sb;

   hypre_CommInfo         *comm_info;
   int                     send_part,    recv_part;
   int                     send_var,     recv_var;
   hypre_StructMatrix     *send_matrix, *recv_matrix;
   hypre_CommPkg          *comm_pkg;
   hypre_CommHandle       *comm_handle;
   int                     ci;

   /*------------------------------------------------------
    * Communicate and accumulate within parts
    *
    * NOTE: For matrices, it's important to do this first,
    * before moving off-part couplings below.
    *------------------------------------------------------*/

   for (part = 0; part < nparts; part++)
   {
      hypre_SStructPMatrixAccumulate(pmatrices[part]);
   }

   /*------------------------------------------------------
    * Communicate and accumulate between parts
    *------------------------------------------------------*/

   for (ci = 0; ci < vnbor_ncomms; ci++)
   {
      comm_info = hypre_SStructCommInfoCommInfo(vnbor_comm_info[ci]);
      send_part = hypre_SStructCommInfoSendPart(vnbor_comm_info[ci]);
      recv_part = hypre_SStructCommInfoRecvPart(vnbor_comm_info[ci]);
      send_var  = hypre_SStructCommInfoSendVar(vnbor_comm_info[ci]);
      recv_var  = hypre_SStructCommInfoRecvVar(vnbor_comm_info[ci]);

      send_matrix = hypre_SStructPMatrixSMatrix(
         hypre_SStructMatrixPMatrix(matrix, send_part), send_var, send_var);
      recv_matrix = hypre_SStructPMatrixSMatrix(
         hypre_SStructMatrixPMatrix(matrix, recv_part), recv_var, recv_var);

      if ((send_matrix != NULL) && (recv_matrix != NULL))
      {
         hypre_StructStencil *send_stencil = hypre_StructMatrixStencil(send_matrix);
         hypre_StructStencil *recv_stencil = hypre_StructMatrixStencil(recv_matrix);
         int                  num_values, stencil_size, num_transforms;
         int                 *symm;
         int                 *v_to_s, *s_to_v;
         hypre_Index         *coords, *dirs;
         int                **orders, *order;
         hypre_IndexRef       sentry0;
         hypre_Index          sentry1;
         int                  ti, si, i, j;

         /* to compute 'orders', remember that we are doing reverse communication */
         num_values = hypre_StructMatrixNumValues(recv_matrix);
         symm = hypre_StructMatrixSymmElements(recv_matrix);
         stencil_size = hypre_StructStencilSize(recv_stencil);
         v_to_s = hypre_TAlloc(int, num_values);
         s_to_v = hypre_TAlloc(int, stencil_size);
         for (si = 0, i = 0; si < stencil_size; si++)
         {
            s_to_v[si] = -1;
            if (symm[si] < 0)  /* this is a stored coefficient */
            {
               v_to_s[i] = si;
               s_to_v[si] = i;
               i++;
            }
         }
         hypre_CommInfoGetTransforms(comm_info, &num_transforms, &coords, &dirs);
         orders = hypre_TAlloc(int *, num_transforms);
         order = hypre_TAlloc(int, num_values);
         for (ti = 0; ti < num_transforms; ti++)
         {
            for (i = 0; i < num_values; i++)
            {
               si = v_to_s[i];
               sentry0 = hypre_StructStencilElement(recv_stencil, si);
               for (j = 0; j < 3; j++)
               {
                  hypre_IndexD(sentry1, hypre_IndexD(coords[ti], j)) = 
                     hypre_IndexD(sentry0, j) * hypre_IndexD(dirs[ti], j);
               }
               order[i] = hypre_StructStencilElementRank(send_stencil, sentry1);
               /* currently, both send and recv transforms are parsed */
               if (order[i] > -1)
               {
                  order[i] = s_to_v[order[i]];
               }
            }
            /* want order to indicate the natural order on the remote process */
            orders[ti] = hypre_TAlloc(int, num_values);
            for (i = 0; i < num_values; i++)
            {
               orders[ti][i] = -1;
            }
            for (i = 0; i < num_values; i++)
            {
               if (order[i] > -1)
               {
                  orders[ti][order[i]] = i;
               }
            }
         }
         hypre_TFree(v_to_s);
         hypre_TFree(s_to_v);
         hypre_TFree(order);

         /* want to communicate and add ghost data to real data */
         hypre_CommPkgCreate(comm_info,
                             hypre_StructMatrixDataSpace(send_matrix),
                             hypre_StructMatrixDataSpace(recv_matrix),
                             num_values, orders, 1,
                             hypre_StructMatrixComm(send_matrix), &comm_pkg);
         /* note reversal of send/recv data here */
         hypre_InitializeCommunication(comm_pkg,
                                       hypre_StructMatrixData(recv_matrix),
                                       hypre_StructMatrixData(send_matrix),
                                       1, 0, &comm_handle);
         hypre_FinalizeCommunication(comm_handle);
         hypre_CommPkgDestroy(comm_pkg);

         for (ti = 0; ti < num_transforms; ti++)
         {
            hypre_TFree(orders[ti]);
         }
         hypre_TFree(orders);
      }
   }

   /*------------------------------------------------------
    * Move off-part couplings (described by neighbor info)
    * from S-matrix structure into U-matrix structure.
    *------------------------------------------------------*/

   box  = hypre_BoxCreate();
   ibox = hypre_BoxCreate();

   nentries = hypre_SStructMatrixEntriesSize(matrix);
   entries  = hypre_TAlloc(int, nentries);
   for (entry = 0; entry < nentries; entry++)
   {
      entries[entry] = entry;
   }

   for (part = 0; part < nparts; part++)
   {
      pmatrix  = pmatrices[part];

      nvars = hypre_SStructPMatrixNVars(pmatrix);
      for (var = 0; var < nvars; var++)
      {
         stencil  = hypre_SStructPMatrixStencil(pmatrix, var);
         smap     = hypre_SStructPMatrixSMap(pmatrix, var);
         shape    = hypre_SStructStencilShape(stencil);
         vars     = hypre_SStructStencilVars(stencil);
         nentries = hypre_SStructStencilSize(stencil);

         hypre_SStructMatrixSplitEntries(matrix, part, var, nentries, entries,
                                         &nSentries, &Sentries,
                                         &nUentries, &Uentries);

         for (entry = 0; entry < nSentries; entry++)
         {
            smatrix = hypre_SStructPMatrixSMatrix(pmatrix, var,
                                                  vars[entries[entry]]);
            sentry = smap[entries[entry]];

            /* Shift/intersect neighbor box and move values */
            for (b = 0; b < nvneighbors[part][var]; b++)
            {
               vneighbor = &vneighbors[part][var][b];
               hypre_CopyBox(hypre_SStructNeighborBox(vneighbor), box);

               /* shift box by stencil offset */
               offset = shape[entry];
               hypre_BoxIMinX(box) -= hypre_IndexX(offset);
               hypre_BoxIMinY(box) -= hypre_IndexY(offset);
               hypre_BoxIMinZ(box) -= hypre_IndexZ(offset);
               hypre_BoxIMaxX(box) -= hypre_IndexX(offset);
               hypre_BoxIMaxY(box) -= hypre_IndexY(offset);
               hypre_BoxIMaxZ(box) -= hypre_IndexZ(offset);

               sgrid = hypre_StructMatrixGrid(smatrix);
               hypre_ForStructGridBoxI(sb, sgrid)
                  {
                     sbox = hypre_StructGridBox(sgrid, sb);
                     hypre_IntersectBoxes(box, sbox, ibox);

                     if (hypre_BoxVolume(ibox))
                     {
                        values = hypre_TReAlloc(values, double,
                                                hypre_BoxVolume(ibox));

                        /* move matrix values from S-matrix to U-matrix */
                        hypre_StructMatrixSetBoxValues(smatrix, ibox, ibox,
                                                       1, &sentry, values,
                                                       -2, -1, 0);

                        hypre_SStructUMatrixSetBoxValues(matrix, part,
                                                         hypre_BoxIMin(ibox),
                                                         hypre_BoxIMax(ibox),
                                                         var, 1, &entry,
                                                         values, 1);
                     }
                  }
            }
         }
      }
   }

   hypre_TFree(entries);
   hypre_TFree(values);
   hypre_BoxDestroy(box);
   hypre_BoxDestroy(ibox);

   /*------------------------------------------------------
    * Assemble P and U matrices
    *------------------------------------------------------*/

   for (part = 0; part < nparts; part++)
   {
      hypre_SStructPMatrixAssemble(pmatrices[part]);
   }

   /* U-matrix */
   hypre_SStructUMatrixAssemble(matrix);

   return hypre_error_flag;
}

/*--------------------------------------------------------------------------
 * NOTE: Should set things up so that this information can be passed
 * immediately to the PMatrix.  Unfortunately, the PMatrix is
 * currently not created until the SStructMatrix is initialized.
 *--------------------------------------------------------------------------*/
 
int
HYPRE_SStructMatrixSetSymmetric( HYPRE_SStructMatrix matrix,
                                 int                 part,
                                 int                 var,
                                 int                 to_var,
                                 int                 symmetric )
{
   int                ***msymmetric = hypre_SStructMatrixSymmetric(matrix);
   hypre_SStructGraph   *graph      = hypre_SStructMatrixGraph(matrix);
   hypre_SStructPGrid   *pgrid;

   int pstart = part;
   int psize  = 1;
   int vstart = var;
   int vsize  = 1;
   int tstart = to_var;
   int tsize  = 1;
   int p, v, t;

   if (part == -1)
   {
      pstart = 0;
      psize  = hypre_SStructMatrixNParts(matrix);
   }

   for (p = pstart; p < psize; p++)
   {
      pgrid = hypre_SStructGraphPGrid(graph, p);
      if (var == -1)
      {
         vstart = 0;
         vsize  = hypre_SStructPGridNVars(pgrid);
      }
      if (to_var == -1)
      {
         tstart = 0;
         tsize  = hypre_SStructPGridNVars(pgrid);
      }

      for (v = vstart; v < vsize; v++)
      {
         for (t = tstart; t < tsize; t++)
         {
            msymmetric[p][v][t] = symmetric;
         }
      }
   }

   return hypre_error_flag;
}

/*--------------------------------------------------------------------------
 *--------------------------------------------------------------------------*/
 
int
HYPRE_SStructMatrixSetNSSymmetric( HYPRE_SStructMatrix matrix,
                                   int                 symmetric )
{
   hypre_SStructMatrixNSSymmetric(matrix) = symmetric;

   return hypre_error_flag;
}

/*--------------------------------------------------------------------------
 *--------------------------------------------------------------------------*/

int
HYPRE_SStructMatrixSetObjectType( HYPRE_SStructMatrix  matrix,
                                  int                  type )
{
   hypre_SStructGraph     *graph    = hypre_SStructMatrixGraph(matrix);
   int                  ***splits   = hypre_SStructMatrixSplits(matrix);
   int                     nparts   = hypre_SStructMatrixNParts(matrix);
   hypre_SStructStencil ***stencils = hypre_SStructGraphStencils(graph);

   hypre_SStructPGrid     *pgrid;
   int                     nvars;
   int                     stencil_size;
   int                     part, var, i;

   hypre_SStructMatrixObjectType(matrix) = type ;   

   /* RDF: This and all other modifications to 'split' really belong
    * in the Initialize routine */
   if (type != HYPRE_SSTRUCT && type != HYPRE_STRUCT)
   {
      for (part = 0; part < nparts; part++)
      {
         pgrid = hypre_SStructGraphPGrid(graph, part);
         nvars = hypre_SStructPGridNVars(pgrid);
         for (var = 0; var < nvars; var++)
         {
            stencil_size = hypre_SStructStencilSize(stencils[part][var]);
            for (i = 0; i < stencil_size; i++)
            {
               splits[part][var][i] = -1;
            }
         }
      }
   }
   
   return hypre_error_flag;
}

/*--------------------------------------------------------------------------
 *--------------------------------------------------------------------------*/

int
HYPRE_SStructMatrixGetObject( HYPRE_SStructMatrix   matrix,
                              void                **object )
{
   int            type     = hypre_SStructMatrixObjectType(matrix);
   HYPRE_IJMatrix ijmatrix = hypre_SStructMatrixIJMatrix(matrix);
   hypre_SStructPMatrix *pA;
   hypre_StructMatrix   *sA;
   int                   part, var;
 

   if (type == HYPRE_PARCSR)
   {
       HYPRE_IJMatrixGetObject(ijmatrix, object);
   }

   else if (type == HYPRE_SSTRUCT)
   {
      *object= matrix;
   }

   else if (type == HYPRE_STRUCT)
   {
      /* only one part & one variable */
      part= 0;
      pA  = hypre_SStructMatrixPMatrix(matrix, part);
      var = 0;
      sA  = hypre_SStructPMatrixSMatrix(pA, var, var);
     *object= sA;
   }

   return hypre_error_flag;
}

/*--------------------------------------------------------------------------
 *--------------------------------------------------------------------------*/

int
HYPRE_SStructMatrixGetObject2( HYPRE_SStructMatrix   matrix,
                               void                **object )
{
   int            type     = hypre_SStructMatrixObjectType(matrix);
   HYPRE_IJMatrix ijmatrix = hypre_SStructMatrixIJMatrix(matrix);
   hypre_SStructPMatrix *pA;
   hypre_StructMatrix   *sA;
   int                   part, var;
 

   if (type == HYPRE_PARCSR)
   {
      /* only difference from ..GetObject: here returns an IJMatrix, not a ParCSRMatrix */
      *object = ijmatrix;
   }

   else if (type == HYPRE_SSTRUCT)
   {
      *object= matrix;
   }

   else if (type == HYPRE_STRUCT)
   {
      /* only one part & one variable */
      part= 0;
      pA  = hypre_SStructMatrixPMatrix(matrix, part);
      var = 0;
      sA  = hypre_SStructPMatrixSMatrix(pA, var, var);
     *object= sA;
   }

   return hypre_error_flag;
}

/*--------------------------------------------------------------------------
 *--------------------------------------------------------------------------*/

int
HYPRE_SStructMatrixPrint( const char          *filename,
                          HYPRE_SStructMatrix  matrix,
                          int                  all )
{
   int  nparts = hypre_SStructMatrixNParts(matrix);
   int  part;
   char new_filename[255];

   for (part = 0; part < nparts; part++)
   {
      sprintf(new_filename, "%s.%02d", filename, part);
      hypre_SStructPMatrixPrint(new_filename,
                                hypre_SStructMatrixPMatrix(matrix, part),
                                all);
   }

   /* U-matrix */
   sprintf(new_filename, "%s.UMatrix", filename);
   HYPRE_IJMatrixPrint(hypre_SStructMatrixIJMatrix(matrix), new_filename);

   return hypre_error_flag;
}

/*--------------------------------------------------------------------------
 * HYPRE_StructMatrixMatvec
 *--------------------------------------------------------------------------*/

int
HYPRE_SStructMatrixMatvec( double alpha,
                           HYPRE_SStructMatrix A,
                           HYPRE_SStructVector x,
                           double beta,
                           HYPRE_SStructVector y     )
{
   hypre_SStructMatvec(alpha, A, x, beta, y);

   return hypre_error_flag;
}

