/** 
* @file computing.c
* @brief smoothing part
* @author serge guelton
* @date 2008-02-13
*
* This file is part of hyantes.
*
* hyantes is free software; you can redistribute it and/or modify
* it under the terms of the CeCILL-C License
*
* You should have received a copy of the CeCILL-C License
* along with this program.  If not, see <http://www.cecill.info/licences>.
*/

#include "hs_compat.h"
#include "computing.h"
#include "interpolation.h"
#include "module.h"

#include <math.h>
#include <float.h>
#include <assert.h>
#include <error.h>
#include <stdlib.h>
#include <string.h>


static /*@null@ */ extra_t *gen_extra(const planartree * ptrees);



/* exported function */
/** 
* @brief performs the smoothing on data stored in the tree ptrees, using the window visu and resolution resoLat, resoLon
*       note that this is a front end to make_all_pot, which is the function that can be distibuted
* 
* @param ptrees tree representation of the map
* @param visu coordinate of the visualtion window
* @param resoLat number of different pixels written, for latitude coordinate
* @param resoLon number of different pixels written, for longitude coordinate
* 
* @return newly allocated array of potentials, written in following order:
*   (lat, lon, potential)
*   (0,   0,   p00),
*   (0,   1,   p01),
*   ...
*   (0,resoLon-1,p0(resoLon-1))
*   (1,   0,   p10),
*   ...
*   (resoLat-1,resoLon-1,p(resoLat-1,resoLon-1))
*
*   OR
*   NULL if an error occured
*
*/

hs_potential_t *do_smoothing(const planartree * ptrees, hs_coord_t visu,
   size_t resoLat, size_t resoLon)
{
    hs_potential_t *out = NULL, *iter = NULL;
    extra_t *extra = NULL;
    size_t i, j;
    double echLat, echLon;

    echLat = (visu.MLat - visu.mLat) / (resoLat - 1);
    echLon = (visu.MLon - visu.mLon) / (resoLon - 1);

    /* allocate extra buffer */
    extra = gen_extra(ptrees);
    if(extra == NULL)
    {
        fprintf(stderr,
           "[do_smoothing::gen_extra] unable to generate extra data");
        return NULL;
    }

    /* allocate out buffer */
    assert(resoLat * resoLon > 0);
    out = (hs_potential_t *) malloc(resoLat * resoLon * sizeof(*out));
    if(out == NULL)
    {
        perror("[do_smoothing::malloc] ");
        free(extra);
        return NULL;
    }
    memset(out, 0, resoLat * resoLon * sizeof(*out));
    iter = out;

    /* compute lat and lon */
    for(i = 0; i < resoLat; i++)
    {
        for(j = 0; j < resoLon; j++)
        {
            iter->lat = visu.mLat + i * echLat;
            iter->lon = visu.mLon + j * echLon;
            ++iter;
        }
    }

    {
        struct make_pot_param param = { ptrees, extra };
        (void) make_all_pot(out, resoLat * resoLon, &param);
#if 0
        /* compute pot */
        iter = out;
        for(i = 0; i < resoLat; i++)
        {
            for(j = 0; j < resoLon; j++)
            {
                make_pot(iter, &param);
                ++iter;
            }
        }
#endif
    }

    free(extra);
    return out;

}

/** 
* @brief fills extra information in ptrees concerning precomputed values of cosinus and sinus
* 
* @param ptrees tree representation of the data used
* 
* @return newly allocated array containing precomputed values
*   OR
*       NULL if an error occured
*/
static extra_t *gen_extra(const planartree * ptrees)
{
    const planartree *iter = ptrees;
    extra_t *out = NULL;
    extra_t *outiter = NULL;

    out = (extra_t *) malloc(ptrees->size * sizeof(*out));
    if(out == NULL)
    {
        perror("[gen_extra::malloc] ");
        return NULL;
    }

    memset(out, 0, sizeof(*out) * ptrees->size);

    outiter = out;
    for(; iter->size > 0; ++iter, ++outiter)
    {
        outiter->cos[0] = cos(iter->coo.mLat);
        outiter->cos[1] = cos(iter->coo.mLon);
        outiter->cos[2] = cos(iter->coo.MLat);
        outiter->cos[3] = cos(iter->coo.MLon);

        outiter->sin[0] = sin(iter->coo.mLat);
        outiter->sin[1] = sin(iter->coo.mLon);
        outiter->sin[2] = sin(iter->coo.MLat);
        outiter->sin[3] = sin(iter->coo.MLon);

    }
    return out;
}
