-- #hide
--------------------------------------------------------------------------------
-- |
-- Module      :  Sound.OpenAL.AL.QueryUtils
-- Copyright   :  (c) Sven Panne 2003-2004
-- License     :  BSD-style (see the file libraries/OpenGL/LICENSE)
-- 
-- Maintainer  :  sven.panne@aedion.de
-- Stability   :  provisional
-- Portability :  portable
--
-- This is a purely internal module with utilities to query OpenAL state.
--
--------------------------------------------------------------------------------

module Sound.OpenAL.AL.QueryUtils (
   GetPName(..), marshalGetPName,
   getFloat1, getFloat3, getFloat6
) where

import Control.Monad ( liftM )
import Foreign.Marshal.Array ( allocaArray )
import Foreign.Ptr ( Ptr )
import Sound.OpenAL.AL.BasicTypes ( ALenum, ALfloat )
import Sound.OpenAL.AL.PeekPoke ( peek3, peek6 )

--------------------------------------------------------------------------------

data GetPName =
     GetSourceType
   | GetSourceRelative
   | GetConeInnerAngle
   | GetConeOuterAngle
   | GetPitch
   | GetPosition
   | GetDirection
   | GetVelocity
   | GetLooping
   | GetStreaming
   | GetBuffer
   | GetGain
   | GetByte
   | GetMinGain
   | GetMaxGain
   | GetOrientation
   | GetSourceState
   | GetBuffersQueued
   | GetBuffersProcessed
   | GetPending
   | GetProcessed
   | GetReferenceDistance
   | GetRolloffFactor
   | GetConeOuterGain
   | GetMaxDistance
   | GetFrequency
   | GetBits
   | GetChannels
   | GetSize
   | GetVendor
   | GetVersion
   | GetRenderer
   | GetExtensions
   | GetDopplerFactor
   | GetDopplerVelocity
   | GetDistanceScale
   | GetDistanceModel
   | GetInverseDistance
   | GetInverseDistanceClamped

marshalGetPName :: GetPName -> ALenum
marshalGetPName x = case x of
   GetSourceType -> 0x0200
   GetSourceRelative -> 0x0202
   GetConeInnerAngle -> 0x1001
   GetConeOuterAngle -> 0x1002
   GetPitch -> 0x1003
   GetPosition -> 0x1004
   GetDirection -> 0x1005
   GetVelocity -> 0x1006
   GetLooping -> 0x1007
   GetStreaming -> 0x1008
   GetBuffer -> 0x1009
   GetGain -> 0x100A
   GetByte -> 0x100C
   GetMinGain -> 0x100D
   GetMaxGain -> 0x100E
   GetOrientation -> 0x100F
   GetSourceState -> 0x1010
   GetBuffersQueued -> 0x1015
   GetBuffersProcessed -> 0x1016
   GetPending -> 0x1017
   GetProcessed -> 0x1018
   GetReferenceDistance -> 0x1020
   GetRolloffFactor -> 0x1021
   GetConeOuterGain -> 0x1022
   GetMaxDistance -> 0x1023
   GetFrequency -> 0x2001
   GetBits -> 0x2002
   GetChannels -> 0x2003
   GetSize -> 0x2004
   GetVendor -> 0xB001
   GetVersion -> 0xB002
   GetRenderer -> 0xB003
   GetExtensions -> 0xB004
   GetDopplerFactor -> 0xC000
   GetDopplerVelocity -> 0xC001
   GetDistanceScale -> 0xC002
   GetDistanceModel -> 0xD000
   GetInverseDistance -> 0xD001
   GetInverseDistanceClamped -> 0xD002

--------------------------------------------------------------------------------

-- Note: OpenAL is not yet orthogonal enough to use getFloatv here, too.

getFloat1 :: (ALfloat -> a) -> GetPName -> IO a
getFloat1 f = liftM f . alGetFloat . marshalGetPName

foreign import CALLCONV unsafe "alGetFloat"
   alGetFloat :: ALenum -> IO ALfloat

getFloat3 :: (ALfloat -> ALfloat -> ALfloat -> a) -> GetPName -> IO a
getFloat3 f n = allocaArray 3 $ \buf -> do
   getFloatv n buf
   peek3 f buf

getFloat6 ::
      (ALfloat -> ALfloat -> ALfloat -> a)
   -> GetPName -> IO (a, a)
getFloat6 f n = allocaArray 6 $ \buf -> do
   getFloatv n buf
   peek6 f buf

getFloatv :: GetPName -> Ptr ALfloat -> IO ()
getFloatv = alGetFloatv . marshalGetPName

foreign import CALLCONV unsafe "alGetFloatv"
   alGetFloatv :: ALenum -> Ptr ALfloat -> IO ()
