/*
//
// html.c
//
// HTML markup tag functions
//
// Copyright (c) 1995-96 Jim Nelson.  Permission to distribute
// granted by the author.  No warranties are made on the fitness of this
// source code.
//
*/

#include "htp.h"

char *FindWhitespace(char *str)
{
    assert(str != NULL);

    while(*str != NUL)
    {
        if(isspace(*str))
        {
            break;
        }
        str++;
    }

    return str;
}

char *FindNonWhitespace(char *str)
{
    assert(str != NULL);

    while(*str != NUL)
    {
        if(!isspace(*str))
        {
            break;
        }
        str++;
    }

    return str;
}   

/*
// HTML_ATTRIBUTE functions
*/

BOOL MakeAttribute(HTML_ATTRIBUTE *htmlAttribute, const char *name,
    const char *value, BOOL quoted)
{
    assert(htmlAttribute != NULL);
    assert(name != NULL);

    /* create a duplicate of the string for the attribute structure */
    if((htmlAttribute->name = DuplicateString(name)) == NULL)
    {
        return FALSE;
    }

    /* do the same for the value, if it exists */
    if(value != NULL)
    {
        if((htmlAttribute->value = DuplicateString(value)) == NULL)
        {
            FreeMemory(htmlAttribute->name);
            htmlAttribute->name = NULL;
            return FALSE;
        }
    }
    else
    {
        htmlAttribute->value = NULL;
    }

    /* keep track if this was a quoted value */
    htmlAttribute->quoted = quoted;

    return TRUE;
}   

BOOL ChangeAttributeName(HTML_ATTRIBUTE *htmlAttribute, const char *name)
{
    assert(htmlAttribute != NULL);

    /* although name should never be null, let it slide */
    if(htmlAttribute->name != NULL)
    {
        FreeMemory(htmlAttribute->name);
        htmlAttribute->name = NULL;
    }

    if((htmlAttribute->name = DuplicateString(name)) == NULL)
    {
        return FALSE;
    }

    return TRUE;
}

BOOL ChangeAttributeValue(HTML_ATTRIBUTE *htmlAttribute, const char *value,
    BOOL quoted)
{
    assert(htmlAttribute != NULL);

    /* free the value's memory, if previously defined */
    if(htmlAttribute->value != NULL)
    {
        FreeMemory(htmlAttribute->value);
        htmlAttribute->value = NULL;
    }

    /* if the new value is defined, allocate room and copy it in */
    if(value != NULL)
    {
        if((htmlAttribute->value = DuplicateString(value)) == NULL)
        {
            return FALSE;
        }
    }

    htmlAttribute->quoted = quoted;

    return TRUE;
}   

void DestroyAttribute(HTML_ATTRIBUTE *htmlAttribute)
{
    assert(htmlAttribute != NULL);
    /* the attribute should always have a name */
    assert(htmlAttribute->name != NULL);

    FreeMemory(htmlAttribute->name);
    htmlAttribute->name = NULL;

    if(htmlAttribute->value != NULL)
    {
        FreeMemory(htmlAttribute->value);
        htmlAttribute->value = NULL;
    }
}   

/*
// HTML_MARKUP functions
*/

/*
    this is perhaps the ugliest piece of code in the entire program ... it
    is also one of the most critical.  (Surprise.)

    Allan Todd provided a very important fix: htp 1.0 and before would
    crash and burn if it hit a large comment.  This was problematic for
    JavaScript and VBScript, which embed the interpreted code in comments
    to prevent other browsers from gagging.  Allans solution is simply
    to pack the entire comment into one attribute.
*/
BOOL PlaintextToMarkup(const char *plaintext, HTML_MARKUP *htmlMarkup)
{
    char *plainBuffer;
    char *plainPtr;
    BOOL valueFound;
    BOOL quotedValue;
    BOOL endOfMarkup;
    char *token;
    char *value;
    BOOL isComment;

    assert(plaintext != NULL);
    assert(htmlMarkup != NULL);

    /* although this isnt a good thing, its not something to halt execution over */
    if(plaintext[0] == NUL)
    {
        DEBUG_PRINT(("no plaintext to convert"));
        return FALSE;
    }

    /* allocate and copy the markup field into the structure */
    if((plainBuffer = DuplicateString(plaintext)) == NULL)
    {
        DEBUG_PRINT(("unable to duplicate plaintext string"));
        return FALSE;
    }
    plainPtr = plainBuffer;

    /* initialize the markup structure */
    htmlMarkup->tag = NULL;
    htmlMarkup->attribCount = 0;

    /* walk the markup and build tag and attribute list (re-using the markup */
    /* argument to walk the copied string) */

    /* walk past any initial whitespace */
    plainPtr = FindNonWhitespace(plainPtr);
    if(*plainPtr == NUL)
    {
        FreeMemory(plainBuffer);
        return TRUE;
    }

    /* mark first token as the tag */
    token = plainPtr;

    /* check if this is a comment */
    isComment = (strncmp(token, "!--", 3) != 0) ? FALSE : TRUE;

    endOfMarkup = FALSE;

    /* walk to the first whitespace, mark it as NUL, and this is the tag */
    plainPtr = FindWhitespace(plainPtr);
    if(*plainPtr == NUL)
    {
        endOfMarkup = TRUE;
    }

    /* copy the markup tag into the structure */
    *plainPtr = NUL;
    if((htmlMarkup->tag = DuplicateString(token)) == NULL)
    {
        DEBUG_PRINT(("unable to duplicate markup token"));
        FreeMemory(plainBuffer);
        return FALSE;
    }

    if(endOfMarkup)
    {
        FreeMemory(plainBuffer);
        return TRUE;
    }

    /* advance past NUL */
    plainPtr++;

    /* start walking the rest of markup, looking for attributes and their */
    /* values */
    while(*plainPtr != NUL)
    {
        /* walk past whitespace */
        plainPtr = FindNonWhitespace(plainPtr);

        /* if a comment, put the whole she-bang into a single attribute */
        /* and skeedaddle */
        if(isComment == TRUE)
        {
            if(AddAttributeToMarkup(htmlMarkup, plainPtr, FALSE, FALSE) == TRUE)
            {
                FreeMemory(plainBuffer);
                return TRUE;
            }

            /* couldnt add it for some reason */
            DEBUG_PRINT(("unable to add comment to markup"));
            DestroyMarkupStruct(htmlMarkup);
            FreeMemory(plainBuffer);

            return FALSE;
        }

        /* if not NUL, then hit an attribute */
        if(*plainPtr != NUL)
        {
            /* mark the beginning of the attribute */
            token = plainPtr;
            value = NULL;
            quotedValue = FALSE;

            /* walk through the attribute, looking for whitespace or an */
            /* equal sign */
            valueFound = FALSE;
            while(*plainPtr != NUL)
            {
                if(*plainPtr == '=')
                {
                    valueFound = TRUE;
                    break;
                }
                else if(isspace(*plainPtr))
                {
                    /* end of attribute */
                    break;
                }
                plainPtr++;
            }

            if(*plainPtr != NUL)
            {
                /* mark as NUL to delimit the attribute name */
                *plainPtr = NUL;
                plainPtr++;

                /* skip past whitespace (looking for a value) */
                plainPtr = FindNonWhitespace(plainPtr);

                /* is this an attribute or value for previous attribute? */
                if(*plainPtr != NUL)
                {
                    if(valueFound)
                    {
                        if(*plainPtr == '\"')
                        {
                            /* quoted value, search for end quote */
                            quotedValue = TRUE;
                            plainPtr++;
                        }
                        else
                        {
                            quotedValue = FALSE;
                        }

                        /* mark the beginning of the value */
                        if(*plainPtr != NUL)
                        {
                            value = plainPtr;
                        }

                        /* find the end of the value */
                        while(*plainPtr != NUL)
                        {
                            if(isspace(*plainPtr))
                            {
                                if(quotedValue == FALSE)
                                {
                                    break;
                                }
                            }

                            if((*plainPtr == '\"') && (quotedValue))
                            {
                                break;
                            }

                            plainPtr++;
                        }

                        /* mark the end of the value */
                        if(*plainPtr != NUL)
                        {
                            *plainPtr = NUL;
                            plainPtr++;
                        }
                    }
                }
            }

            /* add the new attribute to the markup structure */
            if(AddAttributeToMarkup(htmlMarkup, token, value, quotedValue) == FALSE)
            {
                DEBUG_PRINT(("unable to add attribute to markup"));
                DestroyMarkupStruct(htmlMarkup);
                FreeMemory(plainBuffer);
                return FALSE;
            }
        }
    }

    FreeMemory(plainBuffer);

    return TRUE;
}   

BOOL AddAttributeToMarkup(HTML_MARKUP *htmlMarkup, const char *name,
    const char *value, BOOL quotedValue)
{
    assert(htmlMarkup != NULL);
    assert(name != NULL);

    if(htmlMarkup->attribCount < MAX_ATTRIBUTE_COUNT)
    {
        if(MakeAttribute(&htmlMarkup->attrib[htmlMarkup->attribCount], name,
            value, quotedValue) == TRUE)
        {
            htmlMarkup->attribCount++;
            return TRUE;
        }
        else
        {
            DEBUG_PRINT(("unable to make attribute name=\"%s\" value=\"%s\"",
                name, value));
        }
    }
    else
    {
        DEBUG_PRINT(("markup structure full!  tag=\"%s\" name=\"%s\" value=\"%s\"",
            htmlMarkup->tag, name, value));
    }

    return FALSE;
}   

void DestroyMarkupStruct(HTML_MARKUP *htmlMarkup)
{
    uint ctr;

    assert(htmlMarkup != NULL);

    /* destroy the tag */
    /* do not assert against this, as this function might be used to */
    /* destroy a partially-built structure */
    if(htmlMarkup->tag != NULL)
    {
        FreeMemory(htmlMarkup->tag);
        htmlMarkup->tag = NULL;
    }

    /* destroy all markup attributes */
    for(ctr = 0; ctr < htmlMarkup->attribCount; ctr++)
    {
        DestroyAttribute(&htmlMarkup->attrib[ctr]);
    }
}   

BOOL IsMarkupTag(HTML_MARKUP *htmlMarkup, const char *tag)
{
    assert(htmlMarkup != NULL);
    assert(tag != NULL);

    return (stricmp(htmlMarkup->tag, tag) == 0) ? TRUE : FALSE;
}   

static uint MarkupAttributeIndex(HTML_MARKUP *htmlMarkup, const char *name)
{
    uint ctr;

    assert(htmlMarkup != NULL);
    assert(name != NULL);

    for(ctr = 0; ctr < htmlMarkup->attribCount; ctr++)
    {
        assert(htmlMarkup->attrib[ctr].name != NULL);

        if(stricmp(htmlMarkup->attrib[ctr].name, name) == 0)
        {
            return ctr;
        }
    }

    return ERROR;
}   

BOOL MarkupToPlaintext(HTML_MARKUP *htmlMarkup, char **plaintext)
{
    uint ctr;
    HTML_ATTRIBUTE *htmlAttribute;
    uint size;
    uint attrSize;
    uint maxAttrSize;
    char *buffer;
    char *attrBuffer;

    assert(htmlMarkup != NULL);
    assert(htmlMarkup->tag != NULL);
    assert(plaintext != NULL);

    /* estimate the required size of the plaintext buffer */

    maxAttrSize = 0;
    attrBuffer = NULL;

    /* additional byte is to account for NUL */
    size = strlen(htmlMarkup->tag) + 1;
    for(ctr = 0; ctr < htmlMarkup->attribCount; ctr++)
    {
        htmlAttribute = &htmlMarkup->attrib[ctr];

        assert(htmlAttribute != NULL);
        assert(htmlAttribute->name != NULL);

        /* an additional byte is added to size for preceding space char */
        attrSize = strlen(htmlAttribute->name) + 1;

        if(htmlAttribute->value != NULL)
        {
            /* additional byte added to account for equal sign */
            attrSize += strlen(htmlAttribute->value) + 1;

            if(htmlAttribute->quoted)
            {
                /* account for the quote characters */
                attrSize += 2;
            }
        }

        /* additional byte added for NULL character */
        attrSize++;

        size += attrSize;
        if(maxAttrSize < attrSize)
        {
            maxAttrSize = attrSize;
        }
    }

    if((buffer = AllocMemory(size)) == NULL)
    {
        DEBUG_PRINT(("unable to allocate plaintext buffer (%u bytes)", size));
        return FALSE;
    }

    if(maxAttrSize != 0)
    {
        if((attrBuffer = AllocMemory(maxAttrSize)) == NULL)
        {
            DEBUG_PRINT(("unable to allocate attribute plaintext buffer (%u bytes)",
                maxAttrSize));
            FreeMemory(buffer);
            return FALSE;
        }
    }
    else
    {
        attrBuffer = NULL;
    }

    /* start copying in the markup as plaintext */
    strcpy(buffer, htmlMarkup->tag);

    for(ctr = 0; ctr < htmlMarkup->attribCount; ctr++)
    {
        htmlAttribute = &htmlMarkup->attrib[ctr];

        /* checked previously, but check again */
        assert(htmlAttribute != NULL);
        assert(htmlAttribute->name != NULL);

        /* this had best be true */
        assert(attrBuffer != NULL);

        /* its a little ugly, but is much quicker than prior tactic */
        /* (an equally ugly set of strcat() functions used conditionally) */
        if(htmlAttribute->value == NULL)
        {
            sprintf(attrBuffer, " %s", htmlAttribute->name);
        }
        else
        {
            if(htmlAttribute->quoted == FALSE)
            {
                sprintf(attrBuffer, " %s=%s", htmlAttribute->name,
                    htmlAttribute->value);
            }
            else
            {
                sprintf(attrBuffer, " %s=\"%s\"", htmlAttribute->name,
                    htmlAttribute->value);
            }
        }

        /* copy in the attribute plaintext into the markup plaintext */
        strcat(buffer, attrBuffer);
    }

    /* free the attribute buffer */
    FreeMemory(attrBuffer);

    /* give the buffer to caller */
    *plaintext = buffer;

    return TRUE;
}   

BOOL IsAttributeInMarkup(HTML_MARKUP *htmlMarkup, const char *name)
{
    assert(htmlMarkup != NULL);

    return (MarkupAttributeIndex(htmlMarkup, name) != ERROR) ? TRUE : FALSE;
}   

const char *MarkupAttributeValue(HTML_MARKUP *htmlMarkup, const char *name)
{
    uint index;

    assert(htmlMarkup != NULL);

    if((index = MarkupAttributeIndex(htmlMarkup, name)) == ERROR)
    {
        return NULL;
    }

    /* check validity of attribute */
    assert(htmlMarkup->attrib[index].name != NULL);

    return htmlMarkup->attrib[index].value;
}   

BOOL ChangeMarkupTag(HTML_MARKUP *htmlMarkup, const char *tag)
{
    assert(htmlMarkup != NULL);
    assert(tag != NULL);

    if(htmlMarkup->tag != NULL)
    {
        FreeMemory(htmlMarkup->tag);
    }

    htmlMarkup->tag = DuplicateString(tag);

    return (htmlMarkup->tag != NULL) ? TRUE : FALSE;
}

const HTML_ATTRIBUTE *MarkupAttribute(HTML_MARKUP *htmlMarkup, uint index)
{
    assert(htmlMarkup != NULL);

    if(index >= htmlMarkup->attribCount)
    {
        return NULL;
    }

    assert(htmlMarkup->attribCount < MAX_ATTRIBUTE_COUNT);

    return &htmlMarkup->attrib[index];
}

