// ExMgr.cpp

/* Copyright (C) 2000-2003 Hewlett-Packard Company
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 */

/* Original author: David Paschal */

#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <stdarg.h>
#include <errno.h>
#include <time.h>
#include <syslog.h>
#undef LOG_INFO
#undef LOG_SYSLOG
#include <ExMlcTransport.h>

#define SAFE_STRING(s) ((s)?((char *)(s)):"")

static ExMgr *gpMgr=0;	/* Needed by the ExWatchdogTimer class. */
static char *gArgv0;
static enum {
	DEBUG_OFF=0,
	DEBUG_WARN,
	DEBUG_ON
} gDebugFlag=DEBUG_OFF;

/*****************************************************************************\
| Logging
\*****************************************************************************/

STATUS logHandleCommand(char *cmd) {
	if (!strcmp(cmd,"log")) {
		gDebugFlag=DEBUG_ON;

	} else if (!strcmp(cmd,"logwarn")) {
		gDebugFlag=DEBUG_WARN;

	} else if (!strcmp(cmd,"nolog")) {
		gDebugFlag=DEBUG_OFF;

	} else {
		return ERROR;
	}

	return OK;
}

#define LOG_COMMON_BUFFER_LEN	1024

void logCommon(char *file,int line,LogType type,int dummy,char *format,...) {
	char firstLine[LOG_COMMON_BUFFER_LEN];
	char secondLine[LOG_COMMON_BUFFER_LEN];
	char *stype="???";
	va_list ap;
	int level=LOG_DEBUG;

	switch (type) {
	   case LOG_TYPE_SYSLOG:
		stype="SYSLOG";
		level=LOG_NOTICE;
		break;
	   case LOG_TYPE_WARNING:
		if (gDebugFlag<DEBUG_WARN) return;
		stype="WARNING";
		level=LOG_WARNING;
		break;
	   case LOG_TYPE_ERROR:
		stype="ERROR";
		level=LOG_ERR;
		break;
	   case LOG_TYPE_ERROR_FATAL:
#ifdef JD_DEBUGLITE
	    if (gpMgr && gpMgr->isInitialized()) {
		printf("\n\n\n\n");
		printf("************************************************\n");
		printf("************************************************\n");
		printf("************************************************\n");
		printf("\n\n\n\n");
		gpMgr->dump();
		printf("\n\n\n\n");
		printf("************************************************\n");
		printf("\n\n\n\n");
	    }
#endif
		stype="FATAL ERROR";
		level=LOG_CRIT;
		break;
	   case LOG_TYPE_ENTRY:
		if (gDebugFlag<DEBUG_ON) return;
		stype="ENTRY";
		break;
	   case LOG_TYPE_EXIT:
		if (gDebugFlag<DEBUG_ON) return;
		stype="EXIT";
		break;
	   case LOG_TYPE_INFO:
		if (gDebugFlag<DEBUG_ON) return;
		stype="INFO";
		break;
	}
	char *socketPrefix="mlc:";
	char *socketSuffix;
	if (!gpMgr || !(socketSuffix=gpMgr->getSocketSuffix())) {
		socketPrefix="";
		socketSuffix="???";
	}
	char *devSeparator="@",*llioName;
	if (!gpMgr || !(llioName=gpMgr->llioNameGet())) {
		devSeparator="";
		llioName="";
	}
	int t=time(0);
	snprintf(firstLine,LOG_COMMON_BUFFER_LEN,
		"%s at %s:%d, dev=<%s%s%s%s>, pid=%d, e=%d, t=%u\n",
		stype,file,line,socketPrefix,socketSuffix,devSeparator,
		llioName,getpid(),errno,t);
	va_start(ap,format);
	if (!format) {
		secondLine[0]=0;
	} else {
		vsnprintf(secondLine,LOG_COMMON_BUFFER_LEN,format,ap);
	}
	va_end(ap);

	syslog(LOG_LPR|level,"%s        %s",firstLine,secondLine);
	if (type==LOG_TYPE_SYSLOG && gDebugFlag<DEBUG_WARN) return;
	printf("\nptal-mlcd: %s%s",firstLine,secondLine);
	fflush(stdout);

	if (type==LOG_TYPE_ERROR_FATAL) {
		/* Uncomment the following line to segfault after a
		 * fatal error, so we can get a stack trace in GDB: */
		// *(int *)0=1;
		exit(1);
	}
}

#define DIE \
	do { \
		LOG_ERROR(cEXBP,0,0,"Got to here.\n"); \
		exit(1); \
	} while(0)

/*****************************************************************************\
| Queue class
\*****************************************************************************/

void Queue::base_add(QueueEntry *entry,QueueEntry *before) {
	LOG_ASSERT(!entry->isEnqueued(),cEXBP,0,cCauseBadParm,
		"base_add: already added, this=0x%8.8X, "
		"entry=0x%8.8X, prev=0x%8.8X, next=0x%8.8X!\n",
		this,entry,entry->prev,entry->next);

	if (!before) {
		entry->prev=tail;
		entry->next=0;
		if (!tail) {
			head=entry;
		} else {
			tail->next=entry;
		}
		tail=entry;
	} else {
		entry->prev=before->prev;
		entry->next=before;
		if (!before->prev) {
			head=entry;
		} else {
			before->prev->next=entry;
		}
		before->prev=entry;
	}
	_depth++;
}

QueueEntry *Queue::base_peek(QueueEntry *entry) {
	QueueEntry *current=head;
	if (entry) while (current && current!=entry) current=current->next;
	return current;
}

QueueEntry *Queue::base_pop(QueueEntry *entry) {
	entry=base_peek(entry);
	if (entry) {
		if (!entry->prev) {
			head=entry->next;
		} else {
			entry->prev->next=entry->next;
		}
		if (!entry->next) {
			tail=entry->prev;
		} else {
			entry->next->prev=entry->prev;
		}
		entry->prev=0;
		entry->next=0;
		_depth--;
	}
	return entry;
}

/*****************************************************************************\
| ExMsg class
\*****************************************************************************/

void ExMsg::send(ExMsgHandler *pMsgHandler) {
	SETTHIS(pMsgHandler);
	pMgr->msgEnqueue(this);
}

/*****************************************************************************\
| ExWatchdogTimer class
\*****************************************************************************/

ExWatchdogTimer::ExWatchdogTimer(ExMsgHandler *pMsgHandler,
    ExMsg *pMsg,char *mystring) {
	SETTHIS(pMsgHandler);
	SETTHIS(pMsg);

	delay.tv_sec=delay.tv_usec=0;

	cancelled=0;
	restart=0;
	started=0;
}

ExWatchdogTimer::~ExWatchdogTimer() {
	/* Panic if a message is still set. */
	LOG_ASSERT(!pMsg,cEXBP,0,cCauseBadState,"");
}

#ifdef JD_DEBUGLITE
void ExWatchdogTimer::dump(void) {
	printf("pMsgHandler=0x%8.8X\n",(int)pMsgHandler);
	printf("pMsg=0x%8.8X\n",(int)pMsg);
	printf("delay=%d seconds, %d usec\n",
		(int)delay.tv_sec,(int)delay.tv_usec);
	printf("cancelled=%d\n",cancelled);
	printf("restart=%d\n",restart);
	printf("started=%d\n",started);
}
#endif

void ExWatchdogTimer::setMsg(ExMsg *pMsg) {
	/* Panic if a message is already set. */
	LOG_ASSERT(!this->pMsg,cEXBP,0,cCauseBadState,"");
	SETTHIS(pMsg);
	if (restart) start();
}

#define MYSTRING ""

void ExWatchdogTimer::start(void) {
#ifdef JD_DEBUGLITE
	int log=0;
#endif

	/* Panic if the caller forgot to set the delay. */
	LOG_ASSERT(delay.tv_sec || delay.tv_usec,cEXBP,0,cCauseBadState,"");

	if (!isPeriodic() && !pMsg) {
		restart=1;
	} else {
		cancelled=0;
		restart=0;
#ifdef JD_DEBUGLITE
		log=1;
#endif
		if (delay.tv_sec==WAIT_FOREVER || delay.tv_usec==WAIT_FOREVER) {
			gpMgr->timerCancel(this);
		} else {
			timer.setTimeout(&delay);
			gpMgr->timerStart(this);
		}
	}
	started=1;
#ifdef JD_DEBUGLITE
	if (log) {
		LOG_INFO(cEXBP,0,
			"Starting ExWatchdogTimer=0x%8.8X (%s), "
			"delay={sec=%d,usec=%d}.\n",
			this,MYSTRING,delay.tv_sec,delay.tv_usec);
	}
#endif
}

void ExWatchdogTimer::cancel(void) {
	restart=started=0;
	if (!isPeriodic() && !pMsg) {
		cancelled=1;
	} else {
		gpMgr->timerCancel(this);
		/* Removed: if (restart) start(); */
	}

	LOG_INFO(cEXBP,0,"Cancelling ExWatchdogTimer=0x%8.8X (%s).\n",
		this,MYSTRING);
}

void ExWatchdogTimer::callback(void) {
	ExMsg *pMsg=removeMsg();

	started=0;

	/* Panic if the timer popped but a message is not set. */
	LOG_ASSERT(pMsg,cEXBP,0,cCauseBadState,"");

	pMsg->send(pMsgHandler);
}

/*****************************************************************************\
| ExWatchdogTimerQueue class
\*****************************************************************************/

void ExWatchdogTimerQueue::add(ExWatchdogTimer *pWatchdogTimer) {
	ExWatchdogTimer *pWatchdogTimer2;

	/* Don't add the timer while it's still in the queue. */
	if (pWatchdogTimer->isEnqueued()) {
		if (pWatchdogTimer->isPeriodic()) return;
		pWatchdogTimer2=pop(pWatchdogTimer);
		LOG_ASSERT(pWatchdogTimer2,cEXBP,0,0);
	}

	/* Insert the timer into the queue in order from least to most
	 * remaining time, but start checking from the tail of the queue. */
	ExWatchdogTimer *prev=(ExWatchdogTimer *)tail,*before=0;
	struct timeval thisTime,prevTime;
	pWatchdogTimer->getRemainingTime(&thisTime);
	while (prev) {
		prev->getRemainingTime(&prevTime);
		if (PolledTimer::compareTimes(&prevTime,&thisTime)<0) break;
		before=prev;
		prev=prev->getPrev();
	}

	base_add(pWatchdogTimer,before);
}

/*****************************************************************************\
| ExBdr class
\*****************************************************************************/

ExBdr::ExBdr(ExBufferPool *pBufferPool,int bufferSize) {
	SETTHIS(pBufferPool);
	SETTHIS(bufferSize);
	startAddress=new unsigned char[bufferSize];
	LOG_ASSERT(startAddress,0,0,cCauseNoMem,"");
	reset();
}

ExBdr::~ExBdr() {
	delete[] startAddress;
}

void ExBdr::returnBuffer(void) {
	if (tcd) tcd->returnBufferNotification(this);
	reset();
	pBufferPool->returnBuffer(this);
}

ExBdr *ExBdr::appendFromPool(void) {
	setNext(pBufferPool->getBuffer());
	return getNext();
}

/*****************************************************************************\
| ExBufferPool class
\*****************************************************************************/

ExBufferPool::ExBufferPool(void *owner,int bufferSize,int requestedBufferCount,
    int port,int bufferManager) {
	SETTHIS(bufferSize);
	SETTHIS(requestedBufferCount);
	/* We allocate buffers as needed rather than up-front. */
	bufferCount=0;
}

ExBufferPool::~ExBufferPool() {
	ExBdr *pBdr;

	while ((pBdr=getBuffer(1))!=0) {
		delete pBdr;
		bufferCount--;
	}
}

ExBdr *ExBufferPool::getBuffer(int noCreate) {
	ExBdr *pBdr=bdrQueue.pop();
	if (!pBdr && !noCreate) {
		pBdr=new ExBdr(this,bufferSize);
		if (pBdr) bufferCount++;
	}

	return pBdr;
}

/*****************************************************************************\
| Global functions implemented in ExMgr
\*****************************************************************************/

STATUS tknobGetWorkingValue(int port,int id,int *pValue) {
	return gpMgr->tknobGet(id,pValue);
}

STATUS exTknobGetInit(int id,int *pValue) {
	return gpMgr->tknobGet(id,pValue);
}

/*****************************************************************************\
| ExSessionLookup class
\*****************************************************************************/

class ExSessionLookup: public ExLookup {
    protected:
	SCD scd;
    public:
	ExSessionLookup(ExService *pService,SCD _scd):
	      ExLookup(pService) { scd=_scd; }
	SCD getSCD(void) { return scd; }
};

/*****************************************************************************\
| ExMgr class
| Constructor, destructor, dump:
\*****************************************************************************/

ExMgr::ExMgr(void) {
	gpMgr=this;

	pBufferPool=new ExBufferPool(this,BUFFER_SIZE,MAX_BUFFERS,
		getPortNumber(),getBufferPoolMgr());
	/* The transport is instantiated last. */

	noFork=0;
	noDrain=0;
	forceMlcDot4=0;
	noDot4=0;
	noMlc=0;
	noPmlMultiplexing=0;
	initialized=0;
	FD_ZERO(&rset);
	FD_ZERO(&wset);
	fdCount=0;
	exContext=EX_CONTEXT_NONE;
	exState=EX_STATE_DOWN;
	exActivateCount=0;
	exCloseCount=0;
	exCloseReason=0;
	tryDot4=0;
	tryMlc=0;
	transportMode=bpUsbCurrentModeUnknown;
	miser=0;

	/* arg[cv]{,Original} are initialized in exMain(). */

	nullInit();

	pFreeMsgPool=new ExMsgQueue;
	pPendingMsgQueue=new ExMsgQueue;
	pActiveTimerQueue=new ExWatchdogTimerQueue;
	pPeriodicTimerQueue=new ExWatchdogTimerQueue;

	consoleAllowRemote=0;
	/* Other attributes are initialized with xxxxInit() functions
	 * called from exMain() after the command line has been parsed. */

	socketPreInit();

	sessionChangeState(SESSION_STATE_STARTUP);

	llioPreInit();

	/* Initialize this last. */
	llioInterface[EX_INTERFACE_MLC].pTransport=
		new ExMlcTransport(this,this,MAX_TRANSACTIONS,
		MAX_TRANSPORT_SESSIONS);
	llioInterface[EX_INTERFACE_PRINT].pTransport=
		new ExTransport(this,this,MAX_TRANSACTIONS,1);
}

ExMgr::~ExMgr() {
	socketDone();
	sessionChangeState(SESSION_STATE_SHUTDOWN);

	delete llioInterface[EX_INTERFACE_MLC].pTransport;
	delete llioInterface[EX_INTERFACE_PRINT].pTransport;
	llioDone();

	delete pBufferPool;
	delete pPeriodicTimerQueue;
	delete pActiveTimerQueue;
	delete pPendingMsgQueue;
	delete pFreeMsgPool;

	gpMgr=0;
}

#ifdef JD_DEBUGLITE
void ExMgr::dump(void) {
	int fd,r,w;

	fflush(stdout);

	printf("\nExMgr:\n");
	printf("pid=%d\n",getpid());
	printf("gDebugFlag=%d\n",gDebugFlag);
	printf("pBufferPool: "); pBufferPool->dump();
	/* The transport is dumped below. */

	printf("noFork=%d\n",noFork);
	printf("noDrain=%d\n",noDrain);
	printf("forceMlcDot4=%d\n",forceMlcDot4);
	printf("noDot4=%d\n",noDot4);
	printf("noMlc=%d\n",noMlc);
	printf("noPmlMultiplexing=%d\n",noPmlMultiplexing);
	printf("initialized=%d\n",initialized);

	for (fd=0;fd<fdCount;fd++) {
		r=FD_ISSET(fd,&rset);
		w=FD_ISSET(fd,&wset);
		if (r || w) {
			printf("fd=%d: read=%d, write=%d\n",fd,r,w);
		}
	}
	printf("fdCount=%d\n",fdCount);
	printf("exContext=%d\n",exContext);
	printf("exState=%d\n",exState);
	printf("exActivateCount=%d\n",exActivateCount);
	printf("exCloseCount=%d\n",exCloseCount);
	printf("exCloseReason=0x%4.4X\n",exCloseReason);
	printf("tryDot4=%d\n",tryDot4);
	printf("tryMlc=%d\n",tryMlc);
	printf("transportMode=%d=%s\n",transportMode,getTransportModeString());
	printf("miser=%d\n",miser);

	printf("argcOriginal=%d\n",argcOriginal);
	printf("argvOriginal: "); fflush(stdout); argDump();
	printf("argc=%d\n",argc);		// We don't dump argv.

	printf("fdNull=%d\n",fdNull);

	printf("pFreeMsgPool: depth=%d\n",pFreeMsgPool->depth());
	printf("pPendingMsgQueue: depth=%d\n",pPendingMsgQueue->depth());
	printf("pActiveTimerQueue: depth=%d\n",pActiveTimerQueue->depth());
	printf("pPeriodicTimerQueue: depth=%d\n",pPeriodicTimerQueue->depth());
	struct timeval selectTimeout,*pTimeout;
	timerGetSelectTimeout(&selectTimeout,&pTimeout);
	printf("select timeout: sec=%d, usec=%d, infinite=%d\n",
		(int)selectTimeout.tv_sec,
		(int)selectTimeout.tv_usec,pTimeout==0);

	printf("consoleAllowRemote=%d\n",consoleAllowRemote);
	printf("consoleOldStdin=%d\n",consoleOldStdin);
	printf("consoleOldStdout=%d\n",consoleOldStdout);
	printf("consoleIsRemote=%d\n",consoleIsRemote);

	printf("socketSuffix=<%s>\n",socketSuffix);
	printf("socketName=<%s>\n",socketName);
	printf("socketFd=%d\n",socketFd);

	printf("pmlTransportSession=%d\n",pmlTransportSession);
	printf("pmlCurrentSession=%d\n",pmlCurrentSession);
	printf("pmlLastSession=%d\n",pmlLastSession);
	/* Interesting sessions are dumped at the end. */

	llioDump();		// May be overridden in derived classes.

	printf("\nPassthrough transport:\n");
	llioInterface[EX_INTERFACE_PRINT].pTransport->dump();

	printf("\nMLC/1284.4 transport:\n");
	llioInterface[EX_INTERFACE_MLC].pTransport->dump();

	printf("\nCommand channel:\n");
	llioInterface[EX_INTERFACE_MLC].pTransport->
		locateTransportChannel(0)->dump();

	SCD scd;
	for (scd=0;scd<MAX_SESSIONS;scd++) {
		if (sessionIsInteresting(scd)) sessionDump(scd);
	}

	fflush(stdout);

}
#endif

/*****************************************************************************\
| Runtime flow of control:
\*****************************************************************************/

void ExMgr__signalHandler(int signum) {
	signal(signum,ExMgr__signalHandler);
	gpMgr->signalHandler(signum);
}

void ExMgr::signalHandler(int signum) {
	if (signum==SIGPIPE) {
		if (exContext!=EX_CONTEXT_SESSION_WRITE) consoleOpenQuiet();
		LOG_WARN(cEXBP,0,"Caught signal SIGPIPE!\n");
	} else if (signum==SIGHUP) {
		LOG_WARN(cEXBP,0,"Caught signal SIGHUP!\n");
	} else if (signum==SIGALRM) {
		LOG_WARN(cEXBP,0,"Caught signal SIGALRM!\n");
	} else {
		LOG_WARN(cEXBP,0,"Caught signal %d!\n",signum);
	}
}

int ExMgr::exMain(int argc,char **argv) {
	argInit(argc,argv);
	socketSuffix=argGetString();
	char *busPrefix;
	if (strstr(socketSuffix,busPrefix="mlc:")==socketSuffix) {
		socketSuffix+=strlen(busPrefix);
	}
	argProcess();
	socketInit();
	llioInit();
	consoleInit();
	if (!noFork) {
		pid_t pid=fork();
		if (pid<0) {
			LOG_ERROR_FATAL(cEXBP,0,cCauseNoMem,
				"exMain: fork failed!\n");
		}
		if (pid) {
			return 0;
		}
	}
	signal(SIGHUP,ExMgr__signalHandler);
	signal(SIGALRM,ExMgr__signalHandler);
	signal(SIGPIPE,ExMgr__signalHandler);
	LOG_SYSLOG(cEXBP,0,"ptal-mlcd successfully initialized.\n");
	initialized=1;

	int r;
	fd_set rset,wset;
	struct timeval selectTimeout,*pTimeout;

	while (42) {
		memcpy(&rset,&this->rset,sizeof(fd_set));
		memcpy(&wset,&this->wset,sizeof(fd_set));

		timerGetSelectTimeout(&selectTimeout,&pTimeout);

#define LOG_SELECT 0
#if LOG_SELECT
		LOG_INFO(cEXBP,0,"exMain: before select, "
			"pTimeout=0x%8.8X, sec=%d, usec=%d.\n",
			pTimeout,selectTimeout.tv_sec,selectTimeout.tv_usec);
		// LOG_INFO(cEXBP,0,"exMain: rset=0x%8.8X.\n",*(int *)(&rset));
#endif
		r=select(fdCount,&rset,&wset,0,pTimeout);
#if LOG_SELECT
		LOG_INFO(cEXBP,0,"exMain: select returns %d.\n",r);
		// LOG_INFO(cEXBP,0,"exMain: rset=0x%8.8X.\n",*(int *)(&rset));
#endif

#if 0
		if (r<0) {
			dump();
		} else
#endif
		if (r>0) {
			consolePostSelect(&rset,&wset);
			socketPostSelect(&rset,&wset);
			sessionPostSelect(&rset,&wset);
			llioService();
		}
		timerService();
		msgService();
		llioServicePrintSubprocessWaitTimer();
	}
}

STATUS ExMgr::_fdRegister(int fd,int r,int w,char *file,int line) {
	if (fd<0) {
		LOG_ERROR(cEXBP,0,cCauseBadParm,
			"fdRegister(invalid fd=%d,r=%d,w=%d) "
			"called from %s:%d!\n",fd,r,w,file,line);
		return ERROR;
	}
#ifdef JD_DEBUGLITE
	if ( /* fd!=llioInterface[EX_INTERFACE_PRINT].fdPipe.write && */
	    fd!=llioInterface[EX_INTERFACE_MLC].fdPipe.write) {
		LOG_ENTRY(cEXBP,0,"fdRegister(fd=%d,r=%d,w=%d) "
			"called from %s:%d.\n",fd,r,w,file,line);
	}
#endif

	if (r>0) FD_SET(fd,&rset); else if (r<0) FD_CLR(fd,&rset);
	if (w>0) FD_SET(fd,&wset); else if (w<0) FD_CLR(fd,&wset);

	if (fd>=fdCount) {
		fdCount=fd+1;
	} else for (fd=fdCount-1;fd>=0;fd--,fdCount--) {
		if (FD_ISSET(fd,&rset) || FD_ISSET(fd,&wset)) break;
	}

	return OK;
}

STATUS ExMgr::exActivate(void) {
	switch (exState) {
	   case EX_STATE_DOWN:
		break;
	   case EX_STATE_UP:
		return OK;
	   case EX_STATE_ACTIVATING:
		return IN_PROGRESS;
	   default:
		LOG_ERROR_FATAL(cEXBP,0,0);
	}

	exActivateCount++;
	if (llioOpen()==ERROR) return ERROR;

	exState=EX_STATE_ACTIVATING;
	setTransportMode(bpUsbCurrentModeUnknown);
	activateOneTransport();
	return IN_PROGRESS;
}

STATUS ExMgr::activateOneTransport(ExInterface iface) {
	if (llioInterface[iface].active) {
		if (llioInterface[iface].transportState==EX_STATE_DOWN) {
			llioInterface[iface].transportState=EX_STATE_ACTIVATING;
			int bufferSize=BUFFER_SIZE;
			if (iface==EX_INTERFACE_PRINT) {
				bufferSize=LLIO_PRINT_BUFFER_LEN;
			}
			llioInterface[iface].pTransport->
				activate(MAX_REVERSE_BUFFERS,bufferSize,
				 MAX_BDRS_PER_TRANSACTION);
			return IN_PROGRESS;
		}
		llioInterface[iface].transportState=EX_STATE_UP;
	}

	return OK;
}

STATUS ExMgr::activateOneTransport(void) {
	if (activateOneTransport(EX_INTERFACE_PRINT)==IN_PROGRESS ||
	    activateOneTransport(EX_INTERFACE_MLC)==IN_PROGRESS) {
		return IN_PROGRESS;
	}
	return OK;
}

void ExMgr::parseDeviceID(void) {
	/* It would be better to parse the device ID string by fields
	 * rather than strstr(), but this works OK for now. */

	/* Detect 1284.4 and MLC. */
	tryDot4=getDefaultTryMlcDot4();
	if (strstr(llioDeviceID,"1284.4DL:4")) {
		tryDot4=1;
	}
	if (noDot4) {
		tryDot4=0;
	}
	tryMlc=getDefaultTryMlcDot4();
	if (strstr(llioDeviceID,"MLC")) {
		tryMlc=1;
	}
	if (noMlc) {
		tryMlc=0;
	}

	/* Detect peripherals that are known to be credit misers. */
	/* Technically the T series isn't a miser, but it doesn't
	 * give credit when a channel is first opened, so we'll
	 * assume it is in order to speed things up. */
	miser=0;
	if (
	    strstr(llioDeviceID,"MDL:OfficeJet;") ||
	    strstr(llioDeviceID,"MDL:OfficeJet LX;") ||
	    strstr(llioDeviceID,"MDL:OfficeJet Series 300;") ||
	    strstr(llioDeviceID,"MDL:OfficeJet Series 500;") ||
	    strstr(llioDeviceID,"MFG:Sony;MDL:All-in-One IJP-V100;") ||
	    strstr(llioDeviceID,"MDL:OfficeJet Series 600;") ||
	    strstr(llioDeviceID,"MDL:Printer/Scanner/Copier 300;") ||
	    strstr(llioDeviceID,"MDL:OfficeJet Series 700;") ||
	    strstr(llioDeviceID,"MDL:OfficeJet T Series;") ||
	    0) {
		miser=1;
	}
}

STATUS ExMgr::tknobGet(int id,int *pValue) {
	if (!pValue) return ERROR;

	int value=*pValue;

	switch (id) {
	   case EX_KNOB_TRANSPORT_FORWARD_DATA_TIMEOUT:
	   case EX_KNOB_MLC_FORWARD_DATA_TIMEOUT:
		value=FORWARD_DATA_TIMEOUT;
		break;
	   case EX_KNOB_MLC_COMMAND_REPLY_TIMEOUT:
		value=COMMAND_REPLY_TIMEOUT;
		break;

	   case EX_KNOB_MLC_TRY_REVISION_DOT4:
		value=tryDot4;
		break;
	   case EX_KNOB_MLC_TRY_REVISION_MLC:
		value=tryMlc;
		break;

	   case EX_KNOB_MLC_INIT_MAX_REMOTE_SOCKETS:
		value=MAX_REMOTE_SOCKETS;
		break;

	   case EX_KNOB_MLC_MUSHER_FIRST_CREDIT_REQUEST_DELAY:
	   case EX_KNOB_MLC_GUSHER_FIRST_CREDIT_REQUEST_DELAY:
		if (miser) value=ExMlcTransportChannel::
			DEFAULT_MISER_FIRST_CREDIT_REQUEST_DELAY;
		break;
	   case EX_KNOB_MLC_MUSHER_NEXT_CREDIT_REQUEST_DELAY:
	   case EX_KNOB_MLC_GUSHER_NEXT_CREDIT_REQUEST_DELAY:
		if (miser) value=ExMlcTransportChannel::
			DEFAULT_MISER_NEXT_CREDIT_REQUEST_DELAY;
		break;
	}

	if (value==*pValue) return ERROR;

	*pValue=value;
	return OK;
}

// TODO: Option to deactivate after all application sessions are closed
// and we get the CloseChannelReply (and llioSubprocess(PRINT) has exited).
// Also, don't print an exClose error message in this case, or at least
// LOG_SYSLOG a less severe message.
void ExMgr::exClose(int reason) {
	exCloseReason=reason;

	switch (exState) {
	   case EX_STATE_DOWN:
		return;
	   case EX_STATE_ACTIVATING:
	   case EX_STATE_UP:
		break;
	   default:
		LOG_ERROR_FATAL(cEXBP,0,0);
	}

	LOG_ERROR(cEXBP,0,cCausePeriphError,
		"exClose(reason=0x%4.4X)\n",reason);
	exCloseCount++;
	exContext=EX_CONTEXT_NONE;

	if (llioInterface[EX_INTERFACE_MLC].active) {
		llioInterface[EX_INTERFACE_MLC].pTransport->
			deactivate(reason);
		llioInterface[EX_INTERFACE_MLC].pTransport->
			physicalPortNotActive();
	}
	if (llioInterface[EX_INTERFACE_PRINT].active) {
		llioInterface[EX_INTERFACE_PRINT].pTransport->
			deactivate(reason);
		llioInterface[EX_INTERFACE_PRINT].pTransport->
			physicalPortNotActive();
	}
	sessionDeactivate();
	llioClose();

	/* Helps prevent USB crashes on reactivation. */
	struct timeval delay={0,LLIO_CLOSE_DELAY};
	PolledTimer::delay(&delay);

	exState=EX_STATE_DOWN;
}

/*****************************************************************************\
| Command-line processing:
\*****************************************************************************/

/* syntaxError() is defined at the end of the file. */

void ExMgr::argProcess(char *arg) {
	if (!strcmp(arg,"-device")) {
		while (42) {
			arg=argPeekString();
			if (!arg || *arg=='-') break;
			llioAddPossibleName(argGetString("-device"));
		}
	} else if (!strcmp(arg,"-devidmatch")) {
		llioAddMatchDeviceID(argGetString(arg));
	} else if (!strcmp(arg,"-remconsole")) {
		consoleAllowRemote=1;
	} else if (!strcmp(arg,"-nofork")) {
		noFork=1;
		consoleAllowRemote=1;
	} else if (!strcmp(arg,"-nodrain")) {
		noDrain=1;
	} else if (!strcmp(arg,"-forcemlcdot4")) {
		forceMlcDot4=1;
	} else if (!strcmp(arg,"-nodot4")) {
		noDot4=1;
	} else if (!strcmp(arg,"-nomlc")) {
		noMlc=1;
	} else if (!strcmp(arg,"-nomlcdot4")) {
		noDot4=1;
		noMlc=1;
	} else if (!strcmp(arg,"-nopml")) {
		noPmlMultiplexing=1;
	} else if (logHandleCommand(arg+1)!=ERROR) {
		/* OK */
	} else {
		syntaxError(arg);
	}
}

void ExMgr::printOptions(void) {
	printf(
"    -devidmatch <s> -- Matches portion of device ID string\n"
"    -remconsole     -- Enables remote debug console on socket %d\n"
"    -nofork         -- Stays in the foreground, enables local console\n"
"    -nodrain        -- Disables channel-%d reset and reverse data drain\n"
"    -forcemlcdot4   -- Forces both 1284.4 and MLC modes\n"
"    -nodot4, -nomlc -- Disables 1284.4 or MLC mode\n"
// "    -nomlcdot4      -- Disables both 1284.4 and MLC modes (broken)\n"
"    -nopml          -- Disables PML multiplexing\n"
"    -log, -logwarn  -- Sets logging level (default=-nolog)\n"
		,SOCKET_CONSOLE,LLIO_CHANNEL_MLC_RESET);
}

// Not JD_DEBUGLITE-only.
void ExMgr::argDump(int fd) {
	if (fd<0) fd=CONSOLE_STDOUT;
	int argc=argcOriginal;
	char **argv=argvOriginal;

	while (argc--) {
		write(fd," \"",2);
		write(fd,*argv,strlen(*argv));
		write(fd,"\"",1);
		argv++;
	}
	write(fd,"\n",1);
}

/*****************************************************************************\
| Message and timer management:
\*****************************************************************************/

ExMsg *ExMgr::getFreeMsg(void) {
	ExMsg *pMsg=pFreeMsgPool->pop();
	if (!pMsg) pMsg=new ExMsg(this);
	return pMsg;
}

void ExMgr::msgService(void) {
	while (42) {
		ExMsg *pMsg=pPendingMsgQueue->pop();
		if (!pMsg) break;
		ExMsgHandler *pMsgHandler=pMsg->getMsgHandler();
		/* LOG_INFO(cEXBP,0,"msgService: pMsg=0x%8.8X, "
			"pMsgHandler=0x%8.8X, type=%d.\n",
			pMsg,pMsgHandler,pMsg->getType()); */
		pMsgHandler->handleMsg(pMsg);
	}
}

void ExMgr::handleMsg(ExMsg *pMsg) {
	ExSessionLookup *pLookup=0;

    if (exState!=EX_STATE_DOWN) {
	switch (pMsg->getType()) {
	   case eEXMSG_ACTIVATE_WAIT:
		sessionActivate();
		break;

	   case eEXMSG_ACTIVATE_RESPONSE:
		/* Do nothing. */
		break;

	   case eEXMSG_LOOKUP_RESPONSE:
		pLookup=(ExSessionLookup *)pMsg->getVoidParam();
		sessionLookupResponse(pLookup->getSCD(),pLookup);
		break;

	   case eEXMSG_REMOTE_SOCK_RESPONSE:
		sessionSetRemoteSocketResponse(
			pMsg->getParam(0),
			pMsg->getParam(1));
		break;

	   case eEXMSG_OPEN_CHAN_RESPONSE:
		sessionOpenChannelResponse(
			pMsg->getParam(0),
			pMsg->getParam(1),
			pMsg->getParam(2),
			pMsg->getParam(3));
		break;

	   case eEXMSG_CLOSE_CHAN_RESPONSE:
		sessionCloseChannelResponse(
			pMsg->getParam(0),
			pMsg->getParam(1));
		break;

	   case eEXMSG_DEACTIVATE_RESPONSE:
		/* Do nothing. */
		break;

	   default:
		LOG_ERROR_FATAL(cEXBP,0,cCauseBadParm,"");
		break;
	}
    }

	returnMsg(pMsg);
}

void ExMgr::timerService(void) {
	ExWatchdogTimer *pWatchdogTimer=pActiveTimerQueue->peek();
	ExWatchdogTimer *next;

	while (pWatchdogTimer) {
		next=pWatchdogTimer->getNext();
		if (pWatchdogTimer->isTimedOut()) {
			pActiveTimerQueue->pop(pWatchdogTimer);
			pWatchdogTimer->callback();
		}
		pWatchdogTimer=next;
	}
}

void ExMgr::timerGetSelectTimeout(struct timeval *timeout,
    struct timeval **pTimeout) {
	struct timeval timeout2;
	ExWatchdogTimer *pWatchdogTimer;
	*pTimeout=0;
	timeout->tv_sec=timeout->tv_usec=0;

	pWatchdogTimer=pActiveTimerQueue->peek();
	if (pWatchdogTimer) {
		*pTimeout=timeout;
		pWatchdogTimer->getRemainingTime(timeout);
	}

	pWatchdogTimer=pPeriodicTimerQueue->peek();
	if (pWatchdogTimer) {
		if (!*pTimeout) {
			*pTimeout=timeout;
			pWatchdogTimer->getRemainingTime(timeout);
		} else {
			pWatchdogTimer->getRemainingTime(&timeout2);
			if (PolledTimer::compareTimes(&timeout2,timeout)<0) {
				timeout->tv_sec=timeout2.tv_sec;
				timeout->tv_usec=timeout2.tv_usec;
			}
		}
	}
}

/*****************************************************************************\
| Debug console management:
\*****************************************************************************/

void ExMgr::consoleInit(void) {
	consoleOldStdin=dup(CONSOLE_STDIN);
	consoleOldStdout=dup(CONSOLE_STDOUT);
	consoleIsRemote=0;
	LOG_ASSERT(consoleOldStdin>=0 && consoleOldStdout>=0,
		cEXBP,0,cCauseFuncFailed,
		"consoleOldStdin=%d, consoleOldStdout=%d!\n",
		consoleOldStdin,consoleOldStdout);

	consoleOpenQuiet();
}

STATUS ExMgr::consolePreopen(void) {
	if (!consoleAllowRemote) {
		LOG_ERROR(cEXBP,0,cCauseFuncFailed,"consolePreopen: "
			"remote console not enabled!\n");
		return ERROR;
	}
	return OK;
}

void ExMgr::consoleOpen(int fdStdin,int fdStdout) {
	if (consoleIsRemote || noFork) {
		printf("\nptal-mlcd debug console closed.\n");
		fflush(stdout);
	}
	consoleOpenQuiet(fdStdin,fdStdout);
}

void ExMgr::consoleOpenQuiet(int fdStdin,int fdStdout) {
	/* Close "old" console and open "new" console. */
	fdRegister(CONSOLE_STDIN,-1,-1);
	dup2(fdStdin,CONSOLE_STDIN);
	dup2(fdStdout,CONSOLE_STDOUT);

	if (fdStdin==fdStdout) {
		consoleIsRemote=1;
	} else {
		consoleIsRemote=0;
		if (!noFork) return;
	}

	fdRegister(CONSOLE_STDIN,1,-1);

	printf("\nptal-mlcd debug console initialized.\n"
		"Type 'help' for a list of valid commands.\n");
	consolePrompt();
}

void ExMgr::consolePrompt(void) {
	printf("\nptal-mlcd mlc:%s -> ",socketSuffix);
	fflush(stdout);
}

void ExMgr::consoleService(void) {
	char buffer[CONSOLE_BUFFER_LEN+1];
	char *token,*end;
	int r;

	r=read(CONSOLE_STDIN,buffer,CONSOLE_BUFFER_LEN);
	LOG_ASSERT(r<=CONSOLE_BUFFER_LEN,cEXBP,0,cCauseFuncFailed);
	if (r<=0) {
		consoleOpenQuiet();
		return;
	}
	buffer[r]=0;

	for (token=buffer;*token<=' ' || *token>=127;token++)
		if (!*token) goto done;
	for (end=token;*end>' ' && *end<127;end++);
	*end=0;

#ifdef JD_DEBUGLITE
	if (!strcmp(token,"dump")) {
		dump();
	} else
#endif
	if (!strcmp(token,"pid")) {
		printf("%d\n",getpid());

	} else if (!strcmp(token,"activate")) {
		printf("exActivate returns %d.\n",exActivate());

	} else if (!strcmp(token,"deactivate")) {
		exClose(MLCD_STATUS_CONSOLE_DEACTIVATE);

	} else if (!strcmp(token,"forceclose")) {
		sessionForceClose();

	} else if (logHandleCommand(token)!=ERROR) {
		/* OK */

	} else {
		printf(
"Valid commands: dump, pid, activate, deactivate, forceclose,\n"
"                log, logwarn, nolog.\n"
			);
	}

done:
	consolePrompt();
}

/*****************************************************************************\
| Socket management:
\*****************************************************************************/

void ExMgr::socketPreInit(void) {
	socketSuffix=0;
	socketName=0;
	fdInit(&socketFd);
}

void ExMgr::socketInit(void) {
	struct sockaddr_un remoteAddr,localAddr;
	socklen_t remoteAddrLen,localAddrLen;

	/* Form the full socket path/name. */
	socketName=new char[strlen(MLCD_SOCKET_PREFIX)+strlen(socketSuffix)+1];
	LOG_ASSERT(socketName,cEXBP,0,cCauseNoMem);
	strcpy(socketName,MLCD_SOCKET_PREFIX);
	strcat(socketName,socketSuffix);
	LOG_ASSERT(strlen(socketName)<sizeof(localAddr.sun_path),cEXBP,0,0,"");

	/* See if another process is already using this socket.
	 * Use a delay/retry loop in case the other process was exiting. */
	remoteAddr.sun_family=AF_UNIX;
	strcpy(remoteAddr.sun_path,socketName);
	remoteAddrLen=sizeof(remoteAddr.sun_family)+strlen(socketName)+1;
	PolledTimer timer;
	struct timeval timeout={SOCKET_CLOSE_WAIT_TIMEOUT,0};
	struct timeval delay={0,SOCKET_CLOSE_WAIT_DELAY};
	timer.setTimeout(&timeout);
	while (42) {
		int fd=socket(AF_UNIX,SOCK_STREAM,0);
		if (fd<0) break;
		int r=connect(fd,(struct sockaddr *)&remoteAddr,
			remoteAddrLen);
		fdClose(&fd);
		if (r<0) break;

		if (timer.isTimedOut()) {
			LOG_ERROR_FATAL(cEXBP,0,0,
				"Another instance of ptal-mlcd is "
				"already using this device name!\n");
		}
		PolledTimer::delay(&delay);
	}

	/* Delete socket. */
	unlink(socketName);

	/* Create and set up new socket. */

	socketFd=socket(AF_UNIX,SOCK_STREAM,0);
	if (socketFd<0) {
		LOG_ERROR_FATAL(cEXBP,0,cCauseFuncFailed,"socket() failed!\n");
	}

	localAddr.sun_family=AF_UNIX;
	strcpy(localAddr.sun_path,socketName);
	localAddrLen=sizeof(localAddr.sun_family)+strlen(socketName)+1;
	if (bind(socketFd,(struct sockaddr *)&localAddr,localAddrLen)<0) {
		LOG_ERROR_FATAL(cEXBP,0,cCauseFuncFailed,
			"bind(%s) failed!  Ensure %s exists.\n",
			socketName,MLCD_SOCKET_PREFIX);
	}
	if (chmod(socketName,0777)<0) {
		LOG_ERROR(cEXBP,0,cCauseFuncFailed,"chmod() failed!\n");
	}
	if (listen(socketFd,SOCKET_BACKLOG)<0) {
		LOG_ERROR_FATAL(cEXBP,0,cCauseFuncFailed,"listen() failed!\n");
	}

	/* Prepare socketFd for select(). */
	fdSetNonblocking(socketFd);
	fdRegister(socketFd,1,-1);
}

void ExMgr::socketDone(void) {
	fdClose(&socketFd);
	unlink(socketName);
}

void ExMgr::socketService(void) {
	int fd;
	struct sockaddr_un remoteAddr;
	socklen_t remoteAddrLen;

	while (42) {
		remoteAddrLen=sizeof(remoteAddr);
		fd=accept(socketFd,
			(struct sockaddr *)&remoteAddr,&remoteAddrLen);
		if (fd<0) break;
		sessionStart(fd);
	}
}

/*****************************************************************************\
| Session management:
\*****************************************************************************/

#ifdef JD_DEBUGLITE

int ExMgr::sessionIsInteresting(SCD scd) {
	return (session[scd].state!=SESSION_STATE_AVAILABLE ||
		session[scd].fd!=ERROR ||
		session[scd].scdlink!=ERROR ||
		session[scd].outstandingForwardBdrCount ||
		(session[scd].pReverseBdrQueue &&
		 session[scd].pReverseBdrQueue->depth()) ||
		session[scd].pCommandBdr ||
		session[scd].pmlTrapsRegistered ||
		session[scd].bitbucketSocket);
}

void ExMgr::sessionDump(SCD scd) {
	char *stype="command";
	if (scd>=FIRST_TRANSPORT_SESSION) stype="transport";
	if (scd>=FIRST_PML_SESSION) stype="PML";

	printf("\nSession %d: type=%s\n",scd,stype);
	printf("\tstate=%d\n",session[scd].state);
	printf("\tfd=%d\n",session[scd].fd);
	printf("\tscdlink=%d\n",session[scd].scdlink);
	printf("\tpLookup=0x%8.8X\n",(int)session[scd].pLookup);
	printf("\toutstandingForwardBdrCount=%d\n",
		session[scd].outstandingForwardBdrCount);
	printf("\tpReverseBdrQueue: depth=%d\n",
		session[scd].pReverseBdrQueue->depth());
	printf("\ttcd=0x%8.8X\n",(int)session[scd].tcd);
	printf("\tpCommandBdr=0x%8.8X\n",(int)session[scd].pCommandBdr);
	printf("\tpmlTrapsRegistered=%d\n",session[scd].pmlTrapsRegistered);
	printf("\tbitbucketSocket=%d\n",session[scd].bitbucketSocket);

	if (session[scd].tcd) {
		printf("\nTransport channel for session %d:\n",scd);
		session[scd].tcd->dump();
	}
}

#endif

void ExMgr::sessionChangeState(SessionState state) {
	if (state==SESSION_STATE_STARTUP) {
		sessionPmlEnable();
	}

	for (SCD scd=0;scd<MAX_SESSIONS;scd++) {
		sessionChangeState(scd,state);
	}
}

void ExMgr::sessionChangeState(SCD scd,SessionState state,int param) {
	SessionState oldState=session[scd].state;
	SCD scdlink;

	LOG_ENTRY(cEXBP,0,"sessionChangeState(scd=%d,state=%d,param=%d): "
		"oldState=%d.\n",
		scd,state,param,oldState);

	switch (state) {
	   case SESSION_STATE_STARTUP:
		fdInit(&session[scd].fd);
		session[scd].pLookup=0;
		if ((scd>=FIRST_COMMAND_SESSION && scd<=LAST_COMMAND_SESSION) ||
		    scd==pmlTransportSession) {
			session[scd].pLookup=new ExSessionLookup(this,scd);
			if (scd==pmlTransportSession) {
				session[scd].pLookup->setServiceName(
					MLCD_SERVICE_NAME_PML);
			}
		}
		session[scd].outstandingForwardBdrCount=0;
		session[scd].pReverseBdrQueue=new ExBdrQueue;
		session[scd].tcd=0;
		session[scd].pCommandBdr=0;
		oldState=SESSION_STATE_STARTUP;
		state=SESSION_STATE_AVAILABLE;
	   case SESSION_STATE_AVAILABLE:
		session[scd].scdlink=ERROR;
		session[scd].pmlTrapsRegistered=0;
		session[scd].bitbucketSocket=0;
		if (oldState==SESSION_STATE_STARTUP) break;
		if (session[scd].pCommandBdr) {
			forwardDataResponse(scd,session[scd].pCommandBdr,
				MLCD_STATUS_SUCCESSFUL);
			session[scd].pCommandBdr=0;
		}
		LOG_ASSERT(!session[scd].outstandingForwardBdrCount,
			cEXBP,0,cCauseBadState);
		if (session[scd].fd!=ERROR) {
			fdClose(&session[scd].fd);
			session[scd].pReverseBdrQueue->empty();
		}
	   case SESSION_STATE_CLOSE_PENDING:
		if (scd==pmlTransportSession) {
			for (SCD pmlscd=FIRST_PML_SESSION;
			     pmlscd<=LAST_PML_SESSION;pmlscd++) {
				/* This is recursive. */
				sessionChangeState(pmlscd,
					SESSION_STATE_AVAILABLE);
			}
		}
		if (session[scd].fd!=ERROR) {
			nullDup(session[scd].fd);
		}
		break;

	   case SESSION_STATE_COMMAND:
		if (oldState==SESSION_STATE_AVAILABLE) {
			/* Initialize FD if this is a new command session. */
			LOG_ASSERT(param!=ERROR,cEXBP,0,0);
			session[scd].fd=param;
			fdRegister(session[scd].fd,1,-1);
		}
		if ((scdlink=session[scd].scdlink)!=ERROR) {
			/* scd is the active command session.
			 * scdlink is the failed transport or PML session.
			 * So let's break the link. */
			session[scd].scdlink=ERROR;
			/* This is recursive. */
			sessionChangeState(scdlink,SESSION_STATE_AVAILABLE);
		}
enableRead:
		fdRegister(session[scd].fd,1,0);
		break;

	   case SESSION_STATE_ACTIVATE_PENDING:
disableRead:
		if (session[scd].fd!=ERROR) {
			fdRegister(session[scd].fd,-1,0);
		}
		break;

	   case SESSION_STATE_LOOKUP_PENDING:
		goto disableRead;

	   case SESSION_STATE_SET_REMOTE_SOCKET_PENDING:
		if (scd==pmlTransportSession) break;
		/* scd is the new transport or PML session.
		 * scdlink is the old command session. */
		/* Create the link. */
		LOG_ASSERT(param!=ERROR,cEXBP,0,0);
		session[scd].scdlink=param;
	   case SESSION_STATE_OPEN_PENDING:
		if (scd==pmlTransportSession) break;
		/* Update the link state. */
		scdlink=session[scd].scdlink;
		session[scdlink].scdlink=scd;	/* Still creating the link. */
		session[scdlink].state=state;
		goto disableRead;

	   case SESSION_STATE_OPEN:
		if (scd==pmlTransportSession) {
			session[scd].fd=nullDup();
			break;
		}
		if (/* scd>=FIRST_COMMAND_SESSION &&
		    scd<=LAST_COMMAND_SESSION && */
		    session[scd].bitbucketSocket) {
			goto enableRead;
		}

		/* scd is the new transport or PML session.
		 * scdlink is the old command session. */
		if (session[scd].scdlink==ERROR) {
			/* A PML session hasn't already been linked since
			 * it doesn't go through all the state transitions
			 * that a transport session does. */
			scdlink=param;
		} else {
			scdlink=session[scd].scdlink;
		}
		LOG_ASSERT(scdlink!=ERROR,cEXBP,0,0);
		/* Transfer the FD and command reply BDR from the old command
		 * session to the new session, break the session link,
		 * and set the command session to SESSION_STATE_AVAILABLE. */
		session[scd].fd=session[scdlink].fd;
		fdInit(&session[scdlink].fd);
		session[scd].scdlink=ERROR;
		session[scdlink].scdlink=ERROR;
		session[scdlink].pReverseBdrQueue->empty(
			session[scd].pReverseBdrQueue);
		session[scd].pmlTrapsRegistered=0;
		/* This is recursive. */
		sessionChangeState(scdlink,SESSION_STATE_AVAILABLE);
		goto enableRead;

	   case SESSION_STATE_SHUTDOWN:
		/* This is recursive. */
		sessionChangeState(scd,SESSION_STATE_AVAILABLE);
		if (session[scd].pLookup) {
			delete session[scd].pLookup;
		}
		if (session[scd].pReverseBdrQueue) {
			delete session[scd].pReverseBdrQueue;
		}
		break;

	   default:
		LOG_ERROR_FATAL(cEXBP,0,0);
	}
	session[scd].state=state;
}

SCD ExMgr::sessionFindAvailable(SCD first) {
	SCD scd,last=first;

	if (first==FIRST_COMMAND_SESSION)        last=LAST_COMMAND_SESSION;
	else if (first==FIRST_TRANSPORT_SESSION) last=LAST_TRANSPORT_SESSION;
	else if (first==FIRST_PML_SESSION)       last=LAST_PML_SESSION;
	else                                     LOG_ERROR_FATAL(cEXBP,0,0);

	for (scd=first;scd<=last;scd++) {
		if (session[scd].state==SESSION_STATE_AVAILABLE) {
			LOG_INFO(cEXBP,0,
				"sessionFindAvailable(first=%d): scd=%d.\n",
				first,scd);
			return scd;
		}
	}

	return ERROR;
}

SCD ExMgr::sessionStart(int fd) {
	SCD scd=sessionFindAvailable(FIRST_COMMAND_SESSION);
	if (scd==ERROR) {
		LOG_ERROR(cEXBP,0,0,"sessionStart: no more sessions!\n");
		fdClose(&fd);
		return ERROR;
	}

	sessionChangeState(scd,SESSION_STATE_COMMAND,fd);
	fdSetNonblocking(session[scd].fd);

	return scd;
}

void ExMgr::sessionPostSelect(fd_set *rset,fd_set *wset) {
	SCD scd;

	for (scd=0;scd<MAX_SESSIONS;scd++) {
		if (session[scd].fd==ERROR) continue;
		if (FD_ISSET(session[scd].fd,rset)) {
			if (session[scd].state==SESSION_STATE_OPEN) {
				sessionHandleForwardData(scd);
			} else if (session[scd].state==SESSION_STATE_COMMAND) {
				sessionReadCommand(scd);
			} else {
				LOG_ERROR(cEXBP,0,cCauseBadState,
					"sessionPostSelect: bad state=%d "
					"for scd=%d, fd=%d!\n",
					session[scd].state,scd,
					session[scd].fd);
				fdRegister(session[scd].fd,-1,0);
			}
		}

		if (session[scd].fd==ERROR) continue;
		if (FD_ISSET(session[scd].fd,wset)) {
			sessionServiceOutput(scd);
		}
	}
}

void ExMgr::sessionReadCommand(SCD scd) {
	LOG_ASSERT(!session[scd].pCommandBdr,cEXBP,0,0);
	session[scd].pCommandBdr=
		pullForwardData(scd,sizeof(union MlcdCmdUnion));
	if (session[scd].state!=SESSION_STATE_AVAILABLE) {
		sessionProcessCommand(scd);
	}
}

#define COMMAND_VALIDATE(pkt) if (datalen<(int)sizeof(data->pkt)) goto malformed
#define COMMAND_PREPARE_REPLY memset(data,0,sizeof(*data));
#define COMMAND_SET_STATUS(s) data->reply.status=s
#define COMMAND_ACTIVATE(pkt) \
	do { \
		if (session[scd].state==SESSION_STATE_ACTIVATE_PENDING && \
		    status!=MLCD_STATUS_SUCCESSFUL) { \
pkt##_failed: \
			COMMAND_PREPARE_REPLY; \
			COMMAND_SET_STATUS(status); \
			COMMAND_SEND_REPLY(pkt); \
			sessionChangeState(scd,SESSION_STATE_COMMAND); \
			return; \
		} \
		r=exActivate(); \
		if (r==ERROR) { \
			status=MLCD_STATUS_FAILED_TO_ACTIVATE; \
			goto pkt##_failed; \
		} else if (r==IN_PROGRESS) { \
			sessionChangeState(scd, \
				SESSION_STATE_ACTIVATE_PENDING); \
			return; \
		} \
		if (session[scd].state==SESSION_STATE_ACTIVATE_PENDING) { \
			sessionChangeState(scd,SESSION_STATE_COMMAND); \
		} \
	} while(0)

#define COMMAND_STRNCPY(dest,src) \
	do { \
		int len=strlen((char *)src); \
		int maxlen=sizeof(data->dest); \
		if (len>=maxlen) len=maxlen-1; \
		memcpy(data->dest,src,len); \
		data->dest[len]=0; \
	} while(0)

#define COMMAND_SEND_REPLY(pkt) \
	do { \
		session[scd].pCommandBdr->setDataLength(sizeof(data->pkt)); \
		reverseDataReceived(scd,session[scd].pCommandBdr, \
			MLCD_STATUS_SUCCESSFUL); \
		session[scd].pCommandBdr=0; \
		forwardDataResponse(scd,session[scd].pCommandBdr, \
			MLCD_STATUS_SUCCESSFUL); \
	} while(0)

#define COMMAND_SUCCESSFUL_OPEN \
	do { \
		COMMAND_PREPARE_REPLY; \
		COMMAND_SET_STATUS(MLCD_STATUS_SUCCESSFUL); \
		data->openReply.maxForwardDatalen= \
			tcd->getMaxForwardDatalen(); \
		data->openReply.maxReverseDatalen= \
			tcd->getMaxReverseDatalen(); \
		COMMAND_SEND_REPLY(openReply); \
		sessionChangeState(scdlink,SESSION_STATE_OPEN,scd); \
	} while(0)

#define COMMAND_FAKESOCK_REPLY(setBlocking) \
	do { \
		if (setBlocking) { \
			consoleFdSetBlocking(session[scd].fd); \
		} \
		COMMAND_PREPARE_REPLY; \
		COMMAND_SET_STATUS(MLCD_STATUS_SUCCESSFUL); \
		data->openReply.maxForwardDatalen=DEFAULT_MAX_DATALEN; \
		data->openReply.maxReverseDatalen=DEFAULT_MAX_DATALEN; \
		datalen=sizeof(data->openReply); \
		r=sessionWrite(scd,(unsigned char *)data,datalen); \
		if (r!=datalen) LOG_WARN(cEXBP,0, \
			"sessionProcessCommand(scd=%d): " \
			"error opening fake socket=%d, " \
			"reply write returns %d, expected=%d!\n", \
			scd,socketID,r,datalen); \
	} while(0)

#define COMMAND_FAKESOCK_DONE \
	do { \
		forwardDataResponse(scd,session[scd].pCommandBdr, \
			MLCD_STATUS_SUCCESSFUL); \
		session[scd].pCommandBdr=0; \
		if (session[scd].state!=SESSION_STATE_OPEN) { \
			sessionChangeState(scd,SESSION_STATE_AVAILABLE); \
		} \
		return; \
	} while(0)

#define COMMAND_FAKESOCK_BEGIN_WRITE \
	ExContext oldContext=exContext; \
	exContext=EX_CONTEXT_SESSION_WRITE

#define COMMAND_FAKESOCK_END_WRITE exContext=oldContext

#define COMMAND_FAKESOCK_BEGIN_CONSOLE \
	int oldStdout=dup(CONSOLE_STDOUT); \
	dup2(session[scd].fd,CONSOLE_STDOUT)

#define COMMAND_FAKESOCK_END_CONSOLE \
	dup2(oldStdout,CONSOLE_STDOUT); \
	close(oldStdout)

void ExMgr::sessionProcessCommand(SCD scd,int status) {
	if (!session[scd].pCommandBdr) {
		LOG_ERROR(cEXBP,0,cCauseBadParm,
			"sessionProcessCommand: pCommandBdr==0!\n");
		return;
	}

	STATUS r;
	union MlcdCmdUnion *data=(union MlcdCmdUnion *)
		session[scd].pCommandBdr->getStartAddress();
	int datalen=session[scd].pCommandBdr->getDataLength();
	int socketID;
	char *devID;

	LOG_ENTRY(cEXBP,0,"sessionProcessCommand(scd=%d,status=%d): "
		"datalen=%d, command=%d, state=%d.\n",
		scd,status,datalen,data->request.command,session[scd].state);
	COMMAND_VALIDATE(request);

	switch (data->request.command) {
	   case MLCD_CMD_GET_STATUS:
		COMMAND_VALIDATE(getStatus);
		COMMAND_PREPARE_REPLY;
		data->getStatusReply.exState=exState;
		COMMAND_SET_STATUS(MLCD_STATUS_SUCCESSFUL);
		COMMAND_SEND_REPLY(getStatusReply);
		break;

	   case MLCD_CMD_GET_DEVICE_ID:
	   case MLCD_CMD_GET_PREVIOUS_DEVICE_ID:
		COMMAND_VALIDATE(getDeviceID);
		if (data->request.command==MLCD_CMD_GET_DEVICE_ID) {
			COMMAND_ACTIVATE(getDeviceIDReply);
			devID=llioDeviceID;
		} else {
			devID=llioSavedDeviceID;
		}
		COMMAND_PREPARE_REPLY;
		if (!devID) {
			COMMAND_SET_STATUS(MLCD_STATUS_NO_DEVICE_ID_STRING);
		} else {
			COMMAND_STRNCPY(getDeviceIDReply.deviceID,devID);
			COMMAND_SET_STATUS(MLCD_STATUS_SUCCESSFUL);
		}
		COMMAND_SEND_REPLY(getDeviceIDReply);
		break;

	   case MLCD_CMD_LOOKUP:
		COMMAND_VALIDATE(lookup);
		if (session[scd].state==SESSION_STATE_COMMAND) {
			data->lookup.serviceName[MLCD_MAX_SERVICE_NAME_LEN]=0;
			session[scd].pLookup->setServiceName(
				data->lookup.serviceName);
			if (sessionTryLocalLookup(scd)!=ERROR) {
				goto lookupDone;
			}
		}
		COMMAND_ACTIVATE(lookupReply);
		if (session[scd].state==SESSION_STATE_COMMAND) {
			if (!strcmp(session[scd].pLookup->getServiceName(),
			     "PRINT") && llioEnablePrintInterface &&
			     llioInterface[EX_INTERFACE_PRINT].active) {
				session[scd].pLookup->setSocketID(
					SOCKET_COMPOSITE_PRINT,1);
				session[scd].pLookup->setStatus(OK);
				goto lookupDone;
			}
			if (!llioInterface[EX_INTERFACE_MLC].active) {
				session[scd].pLookup->setStatus(
					MLCD_STATUS_LOOKUP_NOT_SUPPORTED);
				goto lookupDone;
			}
			llioInterface[EX_INTERFACE_MLC].pTransport->
				lookupRemoteSocket(session[scd].pLookup);
			sessionChangeState(scd,SESSION_STATE_LOOKUP_PENDING);

		} else if (session[scd].state==SESSION_STATE_LOOKUP_PENDING) {
lookupDone:
			COMMAND_PREPARE_REPLY;
			COMMAND_SET_STATUS(session[scd].pLookup->getStatus());
			if (data->reply.status==MLCD_STATUS_SUCCESSFUL) {
				data->lookupReply.socketID=
					session[scd].pLookup->getSocketID();
			}
			LOG_INFO(cEXBP,0,"lookupReply(scd=%d): "
				"status=%d, socketID=%d.\n",
				scd,data->lookupReply.status,
				data->lookupReply.socketID);
			COMMAND_SEND_REPLY(lookupReply);
			sessionChangeState(scd,SESSION_STATE_COMMAND);

		} else {
			LOG_ERROR_FATAL(cEXBP,0,0);
		}
		break;

	   case MLCD_CMD_REVERSE_LOOKUP:
		COMMAND_VALIDATE(reverseLookup);
		if (session[scd].state==SESSION_STATE_COMMAND) {
			session[scd].pLookup->setSocketID(
				data->reverseLookup.socketID);
			if (sessionTryLocalLookup(scd)!=ERROR) {
				goto reverseLookupDone;
			}
		}
		COMMAND_ACTIVATE(reverseLookupReply);
		if (session[scd].state==SESSION_STATE_COMMAND) {
			if (!llioInterface[EX_INTERFACE_MLC].active) {
				session[scd].pLookup->setStatus(
					MLCD_STATUS_LOOKUP_NOT_SUPPORTED);
				goto reverseLookupDone;
			}
			llioInterface[EX_INTERFACE_MLC].pTransport->
				lookupRemoteSocket(session[scd].pLookup);
			sessionChangeState(scd,SESSION_STATE_LOOKUP_PENDING);

		} else if (session[scd].state==SESSION_STATE_LOOKUP_PENDING) {
reverseLookupDone:
			COMMAND_PREPARE_REPLY;
			COMMAND_SET_STATUS(session[scd].pLookup->getStatus());
			if (data->reply.status==MLCD_STATUS_SUCCESSFUL) {
				COMMAND_STRNCPY(reverseLookupReply.serviceName,
					session[scd].pLookup->getServiceName());
			}
			COMMAND_SEND_REPLY(reverseLookupReply);
			sessionChangeState(scd,SESSION_STATE_COMMAND);

		} else {
			LOG_ERROR_FATAL(cEXBP,0,0);
		}
		break;

	   case MLCD_CMD_OPEN:
		COMMAND_VALIDATE(open);
		socketID=data->open.socketID;
		if (socketID==SOCKET_CONSOLE) {
			if (consolePreopen()==ERROR) {
				status=MLCD_STATUS_NO_CONSOLE;
				goto openFailure;
			}
			COMMAND_FAKESOCK_REPLY(1);
			consoleOpen(session[scd].fd);
			COMMAND_FAKESOCK_DONE;
		}
		if (socketID==SOCKET_PID) {
			COMMAND_FAKESOCK_REPLY(1);
			pid_t pid=getpid();
			snprintf(data->getDeviceIDReply.deviceID,
				MLCD_MAX_DEVICE_ID_LEN,"%d\n",pid);
			sessionWrite(scd,(unsigned char *)
				data->getDeviceIDReply.deviceID,
				strlen(data->getDeviceIDReply.deviceID));
			COMMAND_FAKESOCK_DONE;
		}
		if (socketID==SOCKET_CMDLINE) {
			COMMAND_FAKESOCK_REPLY(1);
			COMMAND_FAKESOCK_BEGIN_WRITE;
			argDump(session[scd].fd);
			COMMAND_FAKESOCK_END_WRITE;
			COMMAND_FAKESOCK_DONE;
		}
		if (socketID==SOCKET_DEVNODE) {
			COMMAND_FAKESOCK_REPLY(1);
			COMMAND_FAKESOCK_BEGIN_WRITE;
			char *s=llioNameGet();
			if (s) write(session[scd].fd,s,strlen(s));
			write(session[scd].fd,"\n",1);
			COMMAND_FAKESOCK_END_WRITE;
			COMMAND_FAKESOCK_DONE;
		}
		if (socketID==SOCKET_DUMP) {
			COMMAND_FAKESOCK_REPLY(1);
#ifdef JD_DEBUGLITE
			COMMAND_FAKESOCK_BEGIN_WRITE;
			COMMAND_FAKESOCK_BEGIN_CONSOLE;
			dump();
			COMMAND_FAKESOCK_END_CONSOLE;
			COMMAND_FAKESOCK_END_WRITE;
#endif
			COMMAND_FAKESOCK_DONE;
		}
		if (socketID==SOCKET_RESERVE_SCANNER ||
		    socketID==SOCKET_RESERVE_FAX_SEND ||
		    socketID==SOCKET_RESERVE_FAX_RECEIVE ||
		    0) {
			SCD i;
			for (i=FIRST_COMMAND_SESSION;
			     i<=LAST_COMMAND_SESSION;i++) {
				if (session[i].bitbucketSocket==socketID) {
					status=MLCD_STATUS_CONNECTION_REFUSED;
					goto openFailure;
				}
			}
			session[scd].bitbucketSocket=socketID;
			COMMAND_FAKESOCK_REPLY(0);
			sessionChangeState(scd,SESSION_STATE_OPEN);
			COMMAND_FAKESOCK_DONE;
		}
		if (socketID==SOCKET_GLOB_DEVNODES &&
		    consoleAllowRemote && exState==EX_STATE_DOWN) {
			COMMAND_FAKESOCK_REPLY(1);
			COMMAND_FAKESOCK_BEGIN_WRITE;
			llioGlobInit();
			llioGlob();
			for (int i=0;i<((int)llioGlobBuffer.gl_pathc);i++) {
				write(session[scd].fd,
					llioGlobBuffer.gl_pathv[i],
					strlen(llioGlobBuffer.gl_pathv[i]));
				write(session[scd].fd,"\n",1);
			}
			llioGlobDone();
			COMMAND_FAKESOCK_END_WRITE;
			COMMAND_FAKESOCK_DONE;
		}
		COMMAND_ACTIVATE(openReply);
		if (session[scd].state==SESSION_STATE_COMMAND) {
			SCD scdlink;
			TCD tcd;
			if (pmlTransportSession!=ERROR &&
			    (tcd=session[pmlTransportSession].tcd)!=0 &&
			    socketID==tcd->getRemoteSocket()) {
				scdlink=sessionFindAvailable(FIRST_PML_SESSION);
				if (scdlink==ERROR) goto tooManySessions;
				COMMAND_SUCCESSFUL_OPEN;
				return;
			}
			if (socketID==SOCKET_COMPOSITE_PRINT &&
			    llioPrintInterfaceIsAvailable()) {
				COMMAND_FAKESOCK_REPLY(1);
				llioSubprocess(EX_INTERFACE_PRINT,
					session[scd].fd);
				COMMAND_FAKESOCK_DONE;
			}
			if (!llioInterface[EX_INTERFACE_MLC].active ||
			    socketID<FIRST_TRANSPORT_SOCKET ||
			    socketID>LAST_TRANSPORT_SOCKET) {
				status=MLCD_STATUS_UNSUPPORTED_SOCKET;
				goto openFailure;
			}
			scdlink=sessionFindAvailable(FIRST_TRANSPORT_SESSION);
			if (scdlink==ERROR) {
tooManySessions:
				status=MLCD_STATUS_TOO_MANY_SESSIONS;
openFailure:
				COMMAND_PREPARE_REPLY;
				COMMAND_SET_STATUS(status);
				COMMAND_SEND_REPLY(openReply);
				sessionChangeState(scd,SESSION_STATE_COMMAND);
				return;
			}
			/* It might be better to consider 0 a valid
			 * requested size and consider <0 to cause the
			 * default size to be set.  Of course, we'd have
			 * to change libptal as well. */
			if (!data->open.maxForwardDatalen) {
				data->open.maxForwardDatalen=
					DEFAULT_MAX_DATALEN;
			}
			if (!data->open.maxReverseDatalen) {
				data->open.maxReverseDatalen=
					DEFAULT_MAX_DATALEN;
			}
			session[scdlink].tcd->setRemoteSocket(socketID,
				data->open.maxForwardDatalen,
				data->open.maxReverseDatalen);
			sessionChangeState(scdlink,
				SESSION_STATE_SET_REMOTE_SOCKET_PENDING,scd);

		} else if (session[scd].state==
		    SESSION_STATE_SET_REMOTE_SOCKET_PENDING) {
			if (status!=MLCD_STATUS_SUCCESSFUL) {
				goto openFailure;
			}
			session[session[scd].scdlink].tcd->open(
				data->open.maxForwardDatalen,
				data->open.maxReverseDatalen);
			sessionChangeState(session[scd].scdlink,
				SESSION_STATE_OPEN_PENDING);

		} else if (session[scd].state==SESSION_STATE_OPEN_PENDING) {
			if (status!=MLCD_STATUS_SUCCESSFUL) {
				goto openFailure;
			}
			SCD scdlink=session[scd].scdlink;
			TCD tcd=session[scdlink].tcd;
			COMMAND_SUCCESSFUL_OPEN;

		} else {
			LOG_ERROR_FATAL(cEXBP,0,0);
		}
		break;

	   default:
malformed:
		LOG_ERROR(cEXBP,0,cCauseBadParm,
			"sessionProcessCommand(scd=%d): "
			"malformed request, cmd=%d, len=%d!\n",
			scd,data->request.command,datalen);
		sessionChangeState(scd,SESSION_STATE_AVAILABLE);
	}
}

STATUS ExMgr::sessionTryLocalLookup(int scd) {
	static struct {
		char socketID;
		char *serviceName;
	} localServiceLookupTable[]={
		{SOCKET_CONSOLE,"PTAL-MLCD-CONSOLE"},
		{SOCKET_PID    ,"PTAL-MLCD-PID"},
		{SOCKET_CMDLINE,"PTAL-MLCD-CMDLINE"},
		{SOCKET_DEVNODE,"PTAL-MLCD-DEVNODE"},
		{SOCKET_DUMP   ,"PTAL-MLCD-DUMP"},
		{SOCKET_RESERVE_SCANNER,"PTAL-MLCD-RESERVE-SCANNER"},
		{SOCKET_RESERVE_FAX_SEND,"PTAL-MLCD-RESERVE-FAX-SEND"},
		{SOCKET_RESERVE_FAX_RECEIVE,"PTAL-MLCD-RESERVE-FAX-RECEIVE"},
		{SOCKET_COMPOSITE_PRINT,"PRINT"},
		{SOCKET_GLOB_DEVNODES,"PTAL-MLCD-GLOB-DEVNODES"},
		{0,0}
	};
	int i;
	ExLookup *pLookup=session[scd].pLookup;
	int lastSet=pLookup->getLastSet();
	int socketID=pLookup->getSocketID();
	char *serviceName=pLookup->getServiceName();

	pLookup->setStatus(OK);
	for (i=0;localServiceLookupTable[i].serviceName;i++) {
		if (lastSet==ExLookup::LAST_SET_SERVICE_NAME) {
			if (localServiceLookupTable[i].socketID==
			     SOCKET_COMPOSITE_PRINT) {
				continue;
			}
			if (!strcmp(localServiceLookupTable[i].serviceName,
			     serviceName)) {
				pLookup->setSocketID(
					localServiceLookupTable[i].socketID,
					1);
				return OK;
			}
		} else if (lastSet==ExLookup::LAST_SET_SOCKET_ID) {
			if (localServiceLookupTable[i].socketID==socketID) {
				pLookup->setServiceName(
					localServiceLookupTable[i].serviceName,
					0,1);
				return OK;
			}
		} else break;
	}

	return ERROR;
}

void ExMgr::sessionActivate(void) {
	if (activateOneTransport()!=OK) return;

	SCD scd;

	if (llioInterface[EX_INTERFACE_MLC].active && !noPmlMultiplexing) {
		sessionPmlEnable();
	} else {
		sessionPmlDisable();
	}

	for (scd=FIRST_TRANSPORT_SESSION;scd<=LAST_TRANSPORT_SESSION;scd++) {
	    if (llioInterface[EX_INTERFACE_MLC].active) {
		session[scd].tcd=
			llioInterface[EX_INTERFACE_MLC].pTransport->
			allocateChannel(this,scd,0,SESSION_MIN_BUFFERS,
			  SESSION_BENEFIT_OF_MORE_BUFFERS);

		if (scd==pmlTransportSession) {
			llioInterface[EX_INTERFACE_MLC].pTransport->
				lookupRemoteSocket(session[scd].pLookup);
			sessionChangeState(scd,SESSION_STATE_LOOKUP_PENDING);
		}
	    }
	}

	if (pmlTransportSession==ERROR) {
		sessionInitialRequestsComplete();
	}
}

void ExMgr::sessionInitialRequestsComplete(void) {
	if (llioInterface[EX_INTERFACE_MLC].active) {
		llioInterface[EX_INTERFACE_MLC].pTransport->
			initialRequestsComplete();
	}
	if (llioInterface[EX_INTERFACE_PRINT].active) {
		llioInterface[EX_INTERFACE_PRINT].pTransport->
			initialRequestsComplete();
	}

	if (pmlTransportSession!=ERROR) {
		session[pmlTransportSession].tcd->open(
			PML_MAX_FORWARD_DATALEN,
			PML_MAX_REVERSE_DATALEN);
		sessionChangeState(pmlTransportSession,
			SESSION_STATE_OPEN_PENDING);
	} else {
		sessionActivateResponse();
	}
}

void ExMgr::sessionActivateResponse(void) {
	exActivateComplete();

	for (SCD scd=0;scd<MAX_SESSIONS;scd++) {
		if (session[scd].state==SESSION_STATE_ACTIVATE_PENDING) {
			sessionProcessCommand(scd);
		}
	}
}

void ExMgr::sessionLookupResponse(SCD scd,ExSessionLookup *pLookup) {
	LOG_ASSERT(pLookup==session[scd].pLookup,cEXBP,0,0);

	LOG_INFO(cEXBP,0,"sessionLookupResponse(scd=%d): "
		"status=%d, socketID=%d.\n",
		scd,pLookup->getStatus(),pLookup->getSocketID());

	if (scd==pmlTransportSession) {
		int status=session[scd].pLookup->getStatus();
		if (status!=MLCD_STATUS_SUCCESSFUL) {
			sessionPmlDisable();
			sessionInitialRequestsComplete();
		} else {
			session[scd].tcd->setRemoteSocket(
				session[scd].pLookup->getSocketID(),
				PML_MAX_FORWARD_DATALEN,
				PML_MAX_REVERSE_DATALEN);
			sessionChangeState(scd,
				SESSION_STATE_SET_REMOTE_SOCKET_PENDING);
		}

	} else if (scd>=FIRST_COMMAND_SESSION && scd<=LAST_COMMAND_SESSION) {
		sessionProcessCommand(scd);

	} else {
		LOG_ERROR_FATAL(cEXBP,0,0);
	}
}

void ExMgr::sessionSetRemoteSocketResponse(SCD scd,int status) {
	if (scd==pmlTransportSession) {
		if (status!=MLCD_STATUS_SUCCESSFUL) {
			sessionPmlDisable();
		} else {
			sessionChangeState(scd,SESSION_STATE_ACTIVATE_PENDING);
		}
		sessionInitialRequestsComplete();

	} else if (scd>=FIRST_TRANSPORT_SESSION &&
	    scd<=LAST_TRANSPORT_SESSION &&
	    session[scd].scdlink!=ERROR) {
		sessionProcessCommand(session[scd].scdlink,status);

	} else {
		LOG_ERROR_FATAL(cEXBP,0,0);
	}
}

void ExMgr::sessionOpenChannelResponse(SCD scd,
    int maxForwardDatalen,int maxReverseDatalen,int status) {
	if (scd==pmlTransportSession) {
		if (status!=MLCD_STATUS_SUCCESSFUL) {
			sessionPmlDisable();
		} else {
			sessionChangeState(scd,SESSION_STATE_OPEN);
		}
		sessionActivateResponse();

	} else if (scd>=FIRST_TRANSPORT_SESSION &&
	    scd<=LAST_TRANSPORT_SESSION &&
	    session[scd].scdlink!=ERROR) {
		sessionProcessCommand(session[scd].scdlink,status);

	} else {
		LOG_ERROR_FATAL(cEXBP,0,0);
	}
}

void ExMgr::sessionForceClose(void) {
	SCD scd;

	for (scd=FIRST_TRANSPORT_SESSION;scd<=LAST_TRANSPORT_SESSION;scd++) {
		if (scd!=pmlTransportSession &&
		    session[scd].outstandingForwardBdrCount) {
			sessionStartClose(scd,1);
		}
	}
}

void ExMgr::sessionStartClose(SCD scd,int flushForwardData) {
	if (scd>=FIRST_TRANSPORT_SESSION && scd<=LAST_TRANSPORT_SESSION) {
		sessionChangeState(scd,SESSION_STATE_CLOSE_PENDING);
		if (sessionFlushClose(scd,flushForwardData)==ERROR) {
			LOG_INFO(cEXBP,0,"sessionStartClose(scd=%d,"
				"flushForwardData=%d): delaying close.\n",
				scd,flushForwardData);
		}
	} else if (scd>=FIRST_PML_SESSION && scd<=LAST_PML_SESSION) {
		sessionPmlDisengage(scd);
		sessionCloseChannelResponse(scd,MLCD_STATUS_SUCCESSFUL);
	} else {
		sessionChangeState(scd,SESSION_STATE_AVAILABLE);
	}
}

STATUS ExMgr::sessionFlushClose(SCD scd,int flushForwardData) {
	if (scd>=FIRST_TRANSPORT_SESSION &&
	    scd<=LAST_TRANSPORT_SESSION &&
	    session[scd].state==SESSION_STATE_CLOSE_PENDING &&
	    (!session[scd].outstandingForwardBdrCount || flushForwardData)) {
		if (flushForwardData) {
			LOG_WARN(cEXBP,0,"sessionFlushClose(scd=%d,"
				"flushForwardData=%d)\n",
				scd,flushForwardData);
		}
		session[scd].tcd->close(flushForwardData);
		return OK;
	}
	return ERROR;
}

void ExMgr::sessionCloseChannelResponse(SCD scd,int status) {
	sessionChangeState(scd,SESSION_STATE_AVAILABLE);
}

void ExMgr::sessionHandleForwardData(SCD scd) {
	ExContext oldContext=exContext;
	exContext=EX_CONTEXT_SESSION_HANDLE_FORWARD_DATA;
	fdRegister(session[scd].fd,-1,0);
	if (/* scd>=FIRST_COMMAND_SESSION && scd<=LAST_COMMAND_SESSION && */
	    session[scd].bitbucketSocket) {
		ExBdr *pBdr=pullForwardData(scd,BUFFER_SIZE);
		if (pBdr) {
			forwardDataResponse(scd,pBdr,MLCD_STATUS_SUCCESSFUL);
		}
	} else if (scd<FIRST_PML_SESSION) {
		session[scd].tcd->forwardDataAvailable();
	} else /* if (scd<=LAST_PML_SESSION) */ {
		if (!sessionPmlIsEngaged()) {
			session[pmlTransportSession].tcd->
				forwardDataAvailable();
		}
	}
	exContext=oldContext;
}

ExBdr *ExMgr::pullForwardData(SCD scd,int maxForwardDatalen) {
	SCD oldscd=scd;
	int reregister=1;

	if (exContext>EX_CONTEXT_SESSION_HANDLE_FORWARD_DATA) {
		if (session[scd].state==SESSION_STATE_CLOSE_PENDING) {
			reregister=-1;
		}
		if (session[scd].state!=SESSION_STATE_OPEN_PENDING) {
			fdRegister(session[scd].fd,reregister,0);
		}
		return 0;
	}

	if (oldscd==pmlTransportSession) {
searchAgain:
		if (sessionPmlIsEngaged()) return 0;
		scd=pmlLastSession;
		while (42) {
			scd++;
			if (scd>LAST_PML_SESSION) scd=FIRST_PML_SESSION;
			if (session[scd].state==SESSION_STATE_OPEN &&
			    session[scd].fd!=ERROR &&
			    !fdIsRegisteredForRead(session[scd].fd)) {
				sessionPmlEngage(scd);
				break;
			}
			if (scd==pmlLastSession) return 0;
		}

	} else if (session[scd].fd==ERROR) {
		return 0;
	}

	if (maxForwardDatalen>BUFFER_SIZE) maxForwardDatalen=BUFFER_SIZE;
	ExBdr *pBdr=pBufferPool->getBuffer();

	unsigned char *buffer=pBdr->getStartAddress();
	int r=read(session[scd].fd,(char *)buffer,maxForwardDatalen);
	int e=errno;
	if (r<=0) {
		pBdr->returnBuffer();
		pBdr=0;
		if (r<0 && e==EAGAIN) {
			if (oldscd==pmlTransportSession) {
				sessionPmlDisengage(scd);
			}
		} else {
			LOG_INFO(cEXBP,0,"pullForwardData(scd=%d,max=%d): "
				"read(fd=%d) returns %d.\n",
				scd,maxForwardDatalen,session[scd].fd,r);
			sessionStartClose(scd);
			if (session[scd].fd==ERROR) return pBdr;
			reregister=-1;
		}
	} else {
		pBdr->setDataLength(r);
		session[scd].outstandingForwardBdrCount++;
		LOG_ASSERT(session[scd].outstandingForwardBdrCount>0,
			cEXBP,0,cCauseBadState);
	}

	fdRegister(session[scd].fd,reregister,0);
	if (!pBdr && oldscd==pmlTransportSession) {
		goto searchAgain;
	}
	return pBdr;
}

void ExMgr::forwardDataResponse(SCD scd,ExBdr *pBdr,int status) {
	if (scd==pmlTransportSession) {
		scd=pmlCurrentSession;
	}

	if (pBdr) {
		pBdr->setTCD(0);
		pBdr->returnBuffer();
	}
	session[scd].outstandingForwardBdrCount--;
	LOG_ASSERT(session[scd].outstandingForwardBdrCount>=0,
		cEXBP,0,cCauseBadState);
	sessionFlushClose(scd);
}

void ExMgr::reverseDataReceived(SCD scd,ExBdr *pBdr,int status) {
	SCD oldscd=scd;

	if (scd==pmlTransportSession && sessionPmlIsEngaged()) {
		scd=pmlCurrentSession;
	}

	int doRegister=1;
	ExBdr *next,*before=0;
	ExBdrQueue *pReverseBdrQueue=session[scd].pReverseBdrQueue;

	LOG_ASSERT(pBdr,cEXBP,0,0);
	LOG_ASSERT(status==MLCD_STATUS_SUCCESSFUL,cEXBP,0,0);

	if (session[scd].state==SESSION_STATE_OPEN_PENDING) {
		if (scd>=FIRST_TRANSPORT_SESSION &&
		    scd<=LAST_TRANSPORT_SESSION) {
			pReverseBdrQueue=
				session[session[scd].scdlink].pReverseBdrQueue;
			doRegister=0;
		} else {
			before=pReverseBdrQueue->peek();
		}
	}

	while (pBdr) {
		next=pBdr->getNext();
		pBdr->setNext(0);
		pReverseBdrQueue->add(pBdr,before);
		pBdr=next;
	}

	if (doRegister) {
		fdRegister(session[scd].fd,0,1);
	}

	if (oldscd==pmlTransportSession) {
		sessionPmlDisengage(scd);
		session[pmlTransportSession].tcd->forwardDataAvailable();
	}
}

void ExMgr::sessionServiceOutput(SCD scd) {
	ExBdr *pBdr=0;
	unsigned char *data;
	int datalen,r;

	if (session[scd].state==SESSION_STATE_OPEN_PENDING) {
		return;
	}

	while (42) {
		pBdr=session[scd].pReverseBdrQueue->peek();
		if (!pBdr) {
			fdRegister(session[scd].fd,0,-1);
			break;
		}

		datalen=pBdr->getDataLength();
		if (datalen) {
			data=pBdr->getStartAddress()+pBdr->getDataOffset();
			r=sessionWrite(scd,data,datalen);
			if (r<0) {
				nullDup(session[scd].fd);
			} else if (r!=datalen) {
				pBdr->unprependData(r);
				break;
			}
		}

		pBdr=session[scd].pReverseBdrQueue->pop();
		pBdr->returnBuffer();
	}
}

int ExMgr::sessionWrite(SCD scd,const unsigned char *data,int datalen) {
	ExContext oldContext=exContext;
	exContext=EX_CONTEXT_SESSION_WRITE;
	int r=write(session[scd].fd,(char *)data,datalen);
	exContext=oldContext;
	if (r<0 && (errno==EAGAIN /* || errno==EINTR */ )) r=0;
	LOG_INFO(cEXBP,0,
		"sessionWrite(scd=%d,datalen=%d) returns %d for fd=%d.\n",
		scd,datalen,r,session[scd].fd);
	return r;
}

void ExMgr::sessionDeactivate(void) {
	for (SCD scd=0;scd<MAX_SESSIONS;scd++) {
		if (session[scd].state==SESSION_STATE_ACTIVATE_PENDING) {
			sessionProcessCommand(scd,
				MLCD_STATUS_FAILED_TO_ACTIVATE);
		} else {
			sessionChangeState(scd,SESSION_STATE_AVAILABLE);
		}
		session[scd].tcd=0;
	}
}

/*****************************************************************************\
| Low-level I/O (LLIO):
\*****************************************************************************/

#ifdef JD_DEBUGLITE
void ExMgr::llioDump(void) {
	int i;

	for (i=0;i<llioPossibleNameCount;i++) {
		printf("llioPossibleName[%d]=<%s>\n",i,
			SAFE_STRING(llioPossibleName[i]));
	}
	printf("llioPossibleNameCount=%d\n",llioPossibleNameCount);
	printf("llioName=<%s>\n",SAFE_STRING(llioName));
	printf("llioGlobBuffer.gl_pathc=%d\n",llioGlobBuffer.gl_pathc);
	printf("llioGlobBuffer.gl_offs=%d\n",llioGlobBuffer.gl_offs);
	printf("llioGlobFlags=%d\n",llioGlobFlags);
	for (i=0;i<llioMatchDeviceIDCount;i++) {
		printf("llioMatchDeviceID[%d]=<%s>\n",i,
			SAFE_STRING(llioMatchDeviceID[i]));
	}
	printf("llioMatchDeviceIDCount=%d\n",llioMatchDeviceIDCount);
	printf("llioInterface[EX_INTERFACE_MLC].pForwardBdrQueue: depth=%d\n",
		llioInterface[EX_INTERFACE_MLC].pForwardBdrQueue->depth());
	printf("llioInterface[EX_INTERFACE_MLC].active=%d\n",
		llioInterface[EX_INTERFACE_MLC].active);
	printf("llioInterface[EX_INTERFACE_MLC].transportState=%d\n",
		llioInterface[EX_INTERFACE_MLC].transportState);
	printf("llioInterface[EX_INTERFACE_MLC].fdPipe.read=%d\n",
		llioInterface[EX_INTERFACE_MLC].fdPipe.read);
	printf("llioInterface[EX_INTERFACE_MLC].fdPipe.write=%d\n",
		llioInterface[EX_INTERFACE_MLC].fdPipe.write);
	printf("llioInterface[EX_INTERFACE_MLC].fdDevice=%d\n",
		llioInterface[EX_INTERFACE_MLC].fdDevice);
	printf("llioInterface[EX_INTERFACE_MLC]."
		"overrideMaxForwardPacketSize=%d\n",
		llioInterface[EX_INTERFACE_MLC].overrideMaxForwardPacketSize);
	printf("llioInterface[EX_INTERFACE_MLC].pid=%d\n",
		llioInterface[EX_INTERFACE_MLC].pid);
	printf("llioInterface[EX_INTERFACE_PRINT].pForwardBdrQueue: depth=%d\n",
		llioInterface[EX_INTERFACE_PRINT].pForwardBdrQueue->depth());
	printf("llioInterface[EX_INTERFACE_PRINT].active=%d\n",
		llioInterface[EX_INTERFACE_PRINT].active);
	printf("llioInterface[EX_INTERFACE_PRINT].transportState=%d\n",
		llioInterface[EX_INTERFACE_PRINT].transportState);
	printf("llioInterface[EX_INTERFACE_PRINT].fdPipe.read=%d\n",
		llioInterface[EX_INTERFACE_PRINT].fdPipe.read);
	printf("llioInterface[EX_INTERFACE_PRINT].fdPipe.write=%d\n",
		llioInterface[EX_INTERFACE_PRINT].fdPipe.write);
	printf("llioInterface[EX_INTERFACE_PRINT].fdDevice=%d\n",
		llioInterface[EX_INTERFACE_PRINT].fdDevice);
	printf("llioInterface[EX_INTERFACE_PRINT]."
		"overrideMaxForwardPacketSize=%d\n",
		llioInterface[EX_INTERFACE_PRINT].overrideMaxForwardPacketSize);
	printf("llioInterface[EX_INTERFACE_PRINT].pid=%d\n",
		llioInterface[EX_INTERFACE_PRINT].pid);
	printf("llioEnablePrintInterface=%d\n",llioEnablePrintInterface);
	printf("llioDeviceID=<%s>\n",SAFE_STRING(llioDeviceID));
	printf("llioSavedDeviceID=<%s>\n",SAFE_STRING(llioSavedDeviceID));
}
#endif

void ExMgr::llioPreInit(void) {
	llioPossibleNameCount=0;
	llioName=0;
	llioMatchDeviceIDCount=0;
	llioInterface[EX_INTERFACE_MLC].pForwardBdrQueue=new ExBdrQueue;
	llioInterface[EX_INTERFACE_MLC].active=0;
	llioInterface[EX_INTERFACE_MLC].transportState=EX_STATE_DOWN;
	fdInit(&llioInterface[EX_INTERFACE_MLC].fdPipe.read);
	fdInit(&llioInterface[EX_INTERFACE_MLC].fdPipe.write);
	fdInit(&llioInterface[EX_INTERFACE_MLC].fdDevice);
	pidInit(&llioInterface[EX_INTERFACE_MLC].pid);
	llioInterface[EX_INTERFACE_PRINT].pForwardBdrQueue=new ExBdrQueue;
	llioInterface[EX_INTERFACE_PRINT].active=0;
	llioInterface[EX_INTERFACE_PRINT].transportState=EX_STATE_DOWN;
	fdInit(&llioInterface[EX_INTERFACE_PRINT].fdPipe.read);
	fdInit(&llioInterface[EX_INTERFACE_PRINT].fdPipe.write);
	fdInit(&llioInterface[EX_INTERFACE_PRINT].fdDevice);
	pidInit(&llioInterface[EX_INTERFACE_PRINT].pid);
	llioEnablePrintInterface=1;
	llioPrintSubprocessWaitTimer=new ExWatchdogTimer;
	llioPrintSubprocessWaitTimer->setDelay(LLIO_PRINT_SUBPROCESS_POLL_RATE);
	llioDeviceID=0;
	llioSavedDeviceID=0;
}

void ExMgr::llioAddPossibleName(char *name) {
	LOG_ASSERT(llioPossibleNameCount<LLIO_MAX_POSSIBLE_NAMES,
		cEXBP,0,cCauseBadState,
		"More than %d -device options given!\n",
		LLIO_MAX_POSSIBLE_NAMES);
	llioPossibleName[llioPossibleNameCount++]=name;
}

void ExMgr::llioAddMatchDeviceID(char *s) {
	LOG_ASSERT(llioMatchDeviceIDCount<LLIO_MAX_DEVICE_ID_MATCHES,
		cEXBP,0,cCauseBadState,
		"More than %d -devidmatch options given!\n",
		LLIO_MAX_DEVICE_ID_MATCHES);
	llioMatchDeviceID[llioMatchDeviceIDCount++]=s;
}

void ExMgr::llioDone(void) {
	llioClose();
	delete llioInterface[EX_INTERFACE_MLC].pForwardBdrQueue;
	delete llioInterface[EX_INTERFACE_PRINT].pForwardBdrQueue;
}

STATUS ExMgr::llioOpen(void) {
	int r=ERROR,i,j=0;
	int *opened=0;

	llioInterface[EX_INTERFACE_MLC].transportState=EX_STATE_DOWN;
	llioInterface[EX_INTERFACE_PRINT].transportState=EX_STATE_DOWN;
	llioGlobInit();
	llioGlob();
	opened=new int[llioGlobBuffer.gl_pathc+1];
	if (!opened) {
		LOG_ERROR(cEXBP,0,cCauseNoMem,
			"llioOpen: 'opened=new char[%d]' failed!\n",
			llioGlobBuffer.gl_pathc+1);
		goto abort;
	}
	for (i=0;i<((int)llioGlobBuffer.gl_pathc);i++) opened[i]=0;
	for (i=0;;i++) {
		if (i>=((int)llioGlobBuffer.gl_pathc)) {
			if (llioGlobBuffer.gl_pathc<=1 || j++>=10) {
				LOG_ERROR(cEXBP,0,cCausePeriphError,
					"Couldn't find device!\n");
				goto abort;
			}
			// TODO: Delay a short random amount of time.
			i=0;
		}
		if (!opened[i]) {
			llioNameSet(llioGlobBuffer.gl_pathv[i]);
			if (llioOpenOne()==ERROR) {
				LOG_WARN(cEXBP,0,"llioOpenOne failed!\n");
			} else {
				opened[i]=1;
				if (llioMatchDevice(opened+i)!=ERROR) break;
				llioClose();
			}
		}
	}
	LOG_INFO(cEXBP,0,"llioOpen: llioName=<%s>, llioDeviceID=<%s>.\n",
		SAFE_STRING(llioName),llioDeviceID);

	parseDeviceID();
	llioInterface[EX_INTERFACE_MLC].overrideMaxForwardPacketSize=0;
	llioInterface[EX_INTERFACE_PRINT].overrideMaxForwardPacketSize=0;
	if (llioPickInterfaces()==ERROR) goto abort;
	if (!llioInterface[EX_INTERFACE_MLC].active &&
	    !llioInterface[EX_INTERFACE_PRINT].active) {
		LOG_ERROR(cEXBP,0,cCausePeriphError,
			"llioOpen: No supported interface on this device!\n");
		goto abort;
	}

	if (llioInterface[EX_INTERFACE_MLC].active) {
		if (llioSetup(EX_INTERFACE_MLC)==ERROR) {
			LOG_ERROR(cEXBP,0,cCausePeriphError,
				"Couldn't set up MLC interface!\n");
			goto abort;
		}
		if (llioDrainReverseData()==ERROR) goto abort;
	} else if (!noDot4 || !noMlc) {
		LOG_ERROR(cEXBP,0,cCausePeriphError,
"Raw (non-MLC/1284.4) mode is not currently supported.  Use \"-nomlcdot4\"\n"
"if you really want this, but it's not completely robust or bug-free.\n");
		goto abort;
	}
	if (llioInterface[EX_INTERFACE_PRINT].active) {
		if (llioSetup(EX_INTERFACE_PRINT)==ERROR) {
			LOG_ERROR(cEXBP,0,cCausePeriphError,
				"Couldn't set up print interface!\n");
			goto abort;
		}
	}
	if (llioSubprocess(EX_INTERFACE_MLC)==ERROR) goto abort;

	r=OK;
abort:
	if (r==ERROR) llioClose();
	llioGlobDone();
	if (opened) delete[] opened;
	return r;
}

STATUS ExMgr::llioOpenOne(void) {
	alarm(LLIO_OPEN_TIMEOUT);
	int fd=open(llioName,O_RDWR|O_NONBLOCK);
	alarm(0);
	if (fd<0) return ERROR;
	llioInterface[EX_INTERFACE_MLC].fdDevice=fd;
	return OK;
}

STATUS ExMgr::llioMatchDevice(int *pOpened) {
	llioDeviceID=llioGetDeviceID(pOpened);
	if (!llioDeviceID) {
		LOG_WARN(cEXBP,0,"llioMatchDevice: llioGetDeviceID failed!\n");
		return ERROR;
	}
	if (llioSavedDeviceID) {
		delete[] llioSavedDeviceID;
	}
	llioSavedDeviceID=llioDeviceID;

	return llioMatchDevice();
}

STATUS ExMgr::llioMatchDevice(void) {
	for (int i=0;i<llioMatchDeviceIDCount;i++) {
		if (!strstr(llioDeviceID,llioMatchDeviceID[i])) {
			LOG_INFO(cEXBP,0,"llioMatchDevice: "
				"didn't match <%s> in device ID string!\n",
				llioMatchDeviceID[i]);
			return ERROR;
		}
	}

	return OK;
}

void ExMgr::resetTransport(ExTransport *pTransport) {
	pTransport->transportResetComplete();
}

void ExMgr::llioClose(void) {
	ExBdr *pBdr;

	while (42) {
		pBdr=llioInterface[EX_INTERFACE_MLC].pForwardBdrQueue->pop();
		if (!pBdr) {
			pBdr=llioInterface[EX_INTERFACE_PRINT].
				pForwardBdrQueue->pop();
			if (!pBdr) break;
		}
		pBdr->getTCD()->forwardDataResponse(pBdr,
			MLCD_STATUS_WRITE_ERROR);
	}

	llioReset();

	fdClose(&llioInterface[EX_INTERFACE_MLC].fdPipe.read);
	fdClose(&llioInterface[EX_INTERFACE_MLC].fdPipe.write);
	fdClose(&llioInterface[EX_INTERFACE_MLC].fdDevice);
	fdClose(&llioInterface[EX_INTERFACE_PRINT].fdPipe.read);
	fdClose(&llioInterface[EX_INTERFACE_PRINT].fdPipe.write);
	fdClose(&llioInterface[EX_INTERFACE_PRINT].fdDevice);

	pidKill(&llioInterface[EX_INTERFACE_MLC].pid);
	pidKill(&llioInterface[EX_INTERFACE_PRINT].pid);

	llioNameSet(0);
	/* We don't delete it here anymore because of llioSavedDeviceID. */
	llioDeviceID=0;

	llioCancelPrintSubprocessWaitTimer();
}

STATUS ExMgr::llioDrainReverseData(void) {
	if (noDrain) return OK;
	ExInterface iface=EX_INTERFACE_MLC;

	/* Send a fake Init packet, and drain the InitReply and any other
	 * stale reverse data.  That seems to be necessary in order to
	 * jump-start the zero-byte packets on USB. */
	static const unsigned char mlcInitPacket[]= {0,0,0,8,0,0,0,0x03};
	static const unsigned char dot4InitPacket[]={0,0,0,8,1,0,0,0x20};
	static const int lenInitHeader=6,lenInitData=2;
	const unsigned char *initPacket=0;
	if (iface!=EX_INTERFACE_MLC) {

	} else if (tryDot4) {
		initPacket=dot4InitPacket;
	} else if (tryMlc) {
		initPacket=mlcInitPacket;
	}
	if (initPacket) {
		int r=llioWrite(iface,initPacket,lenInitHeader,1);
		if (r!=lenInitHeader) {
			LOG_ERROR(cEXBP,0,cCausePeriphError,
				"llioDrainReverseData: llioWrite(header) "
				"returns %d, expected=%d!\n",r,lenInitHeader);
			return ERROR;
		}
		r=llioWrite(iface,initPacket+lenInitHeader,lenInitData,1);
		if (r!=lenInitData) {
			LOG_ERROR(cEXBP,0,cCausePeriphError,
				"llioDrainReverseData: llioWrite(data) "
				"returns %d, expected=%d!\n",r,lenInitData);
			return ERROR;
		}
	}

	unsigned char drainBuffer[LLIO_MLC_BUFFER_LEN];
	int lenDrainBuffer=llioGetDrainBufferSize(LLIO_MLC_BUFFER_LEN);
	if (lenDrainBuffer>LLIO_MLC_BUFFER_LEN) {
		lenDrainBuffer=LLIO_MLC_BUFFER_LEN;
	}
	int reverse=0,pleaseStop=0,r,avail;
	PolledTimer timer;
	struct timeval timeout={0,LLIO_DRAIN_TIMEOUT};
	goto prime;
    while (42) {
	avail=llioReverseDataIsAvailable(iface);
	if (timer.isTimedOut()) {
		avail=0;
		pleaseStop=1;
	}

	if (!avail) {
		if (reverse) {
			if (llioReverseToForward()==ERROR) {
				LOG_ERROR(cEXBP,0,cCausePeriphError,
					"llioDrainReverseData: "
					"llioReverseToForward failed!\n");
				return ERROR;
			}
			reverse=0;
		}
		if (pleaseStop) break;
		continue;
	}

	if (!reverse) {
		if (llioForwardToReverse()==ERROR) {
			LOG_ERROR(cEXBP,0,cCausePeriphError,
				"llioDrainReverseData: "
				"llioForwardToReverse failed!\n");
			return ERROR;
		}
		reverse=1;
	}

	r=llioRead(iface,drainBuffer,lenDrainBuffer,0);
	if (r<0) {
		LOG_ERROR(cEXBP,0,cCausePeriphError,
			"llioDrainReverseData: llioRead returns %d!\n",r);
		return ERROR;
	}
	if (r>0) {
#ifdef JD_DEBUGLITE
		for (int i=0;i<r;i++) {
			LOG_INFO(cEXBP,0,"llioDrainReverseData: "
				"drained reverse byte=0x%2.2X.\n",
				drainBuffer[i]);
		}
#endif
prime:
		timer.setTimeout(&timeout);
	}
    }

	return OK;
}

#define DUMP_BUFFER(data,datalen) \
	if (gDebugFlag>=DEBUG_ON) { \
		int j; \
		for (i=0;i<datalen;i+=16) { \
			printf("%4.4X  ",i); \
			for (j=0;j<16;j++) { \
				if (i+j>=datalen) { \
					printf("   "); \
				} else { \
					printf("%2.2X ",data[i+j]); \
				} \
			} \
			printf(" "); \
			for (j=0;j<16 && i+j<datalen;j++) { \
				if (data[i+j]<' ' || data[i+j]>=127) { \
					printf("."); \
				} else { \
					printf("%c",data[i+j]); \
				} \
			} \
			printf("\n"); \
		} \
	}

void ExMgr::llioService(void) {
	if (exState==EX_STATE_DOWN) return;

	int iface=EX_INTERFACE_MLC;
	if (!llioInterface[iface].active) return;

	ExContext oldContext=exContext;
	int r=0,requestedDatalen,datalen=0,grcFlags,i;
	ExBdr *pBdr,*pFirstBdr;
	unsigned char *data;
	char *msg=0,*msg2=0;
#ifdef JD_DEBUGLITE
	int readTotalCount=0,writeTotalCount=0;
#endif
	int readCount=0,writeCount=0;

    while (42) {
	if (exState==EX_STATE_DOWN) return;
	exContext=EX_CONTEXT_LLIO_SERVICE_INPUT;

	/* Check to see if reverse data is available. */
	pFirstBdr=0;
	if (fdIsReadable(llioInterface[iface].fdPipe.read)) {
		/* Read the header and data in separate chunks as requested
		 * by the transport. */
		pBdr=0;
		data=0;
		datalen=0;
	    do {
		/* Find out how much data to read in this chunk. */
		r=llioInterface[iface].pTransport->
			getReverseCount((char *)data,datalen,
			 &requestedDatalen,&grcFlags);
		if (r==ERROR) {
			msg="getReverseCount";
abort:
			if (msg) LOG_ERROR(cEXBP,0,0,
				"llioService: %s failed!\n",msg);
			if (msg2) LOG_ERROR(cEXBP,0,0,
				"llioService: %s returns %d, "
				"expected=%d!\n",msg2,r,datalen);
			while (pFirstBdr) {
				pBdr=pFirstBdr;
				pFirstBdr=pBdr->getNext();
				pBdr->setNext(0);
				pBdr->returnBuffer();
			}
			exClose(MLCD_STATUS_READ_ERROR);
			return;
		}

		while (requestedDatalen) {
			/* Get the first or next buffer. */
			if (!pBdr) {
				pBdr=pFirstBdr=pBufferPool->getBuffer();
			} else if (pBdr->getDataLength()>=pBdr->getSize()) {
				pBdr=pBdr->appendFromPool();
			}
			if (!pBdr) {
				msg="getBuffer";
				goto abort;
			}

			/* Read the data. */
			data=pBdr->getStartAddress()+pBdr->getDataLength();
			datalen=pBdr->getSize()-pBdr->getDataLength();
			if (datalen>requestedDatalen) datalen=requestedDatalen;
			r=fdRead(llioInterface[iface].fdPipe.read,
				data,datalen,!(grcFlags&ExTransport::
				 GRC_FLAG_PARTIAL_DATA_OK));
			pBdr->setDataLength(pBdr->getDataLength()+datalen);
			LOG_INFO(cEXBP,0,"llioService: "
				"Received %d bytes on %s interface:\n",
				r,getInterfaceString(iface));
			DUMP_BUFFER(data,r);
			if (r!=datalen) {
				if (r>0 && grcFlags&
				     ExTransport::GRC_FLAG_PARTIAL_DATA_OK) {
					break;
				}
				msg2="fdRead";
				goto abort;
			}

			requestedDatalen-=datalen;
		}
	    } while (!(grcFlags&ExTransport::GRC_FLAG_END_OF_TRANSACTION));

		/* Send reverse data chain (if any) to transport. */
		if (pFirstBdr) {
			llioInterface[iface].pTransport->
				reverseDataReceived(pFirstBdr,
				 MLCD_STATUS_SUCCESSFUL);
			readCount++;
		}
	}

	if (exState==EX_STATE_DOWN) return;
	exContext=EX_CONTEXT_LLIO_SERVICE_OUTPUT;

	/* Send queued forward data. */
	fdRegister(llioInterface[iface].fdPipe.write,0,-1);
	pBdr=pFirstBdr=llioInterface[iface].pForwardBdrQueue->pop();
	while (pFirstBdr) {
		data=pBdr->getStartAddress();
		datalen=pBdr->getDataLength();
		r=fdWrite(llioInterface[iface].fdPipe.write,data,datalen,1);
		if (r!=datalen) {
			LOG_ERROR(cEXBP,0,0,
				"llioService: fdWrite returns %d, "
				"expected=%d!\n",r,datalen);
			pFirstBdr->getTCD()->forwardDataResponse(
				pFirstBdr,MLCD_STATUS_WRITE_ERROR);
			exClose(MLCD_STATUS_WRITE_ERROR);
			return;
		}
		LOG_INFO(cEXBP,0,"llioService: "
			"Sent %d bytes on %s interface:\n",
			r,getInterfaceString(iface));
		DUMP_BUFFER(data,r);

		pBdr=pBdr->getNext();
		if (!pBdr) {
			pFirstBdr->getTCD()->forwardDataResponse(
				pFirstBdr,MLCD_STATUS_SUCCESSFUL);
			writeCount++;
			break;
		}
	}

	if (!readCount && !writeCount) break;
#ifdef JD_DEBUGLITE
	readTotalCount+=readCount;
	writeTotalCount+=writeCount;
#endif
	readCount=writeCount=0;
    }

#ifdef JD_DEBUGLITE
	if (readTotalCount || writeTotalCount) {
		LOG_EXIT(cEXBP,0,"llioService: "
			"readTotalCount=%d, writeTotalCount=%d.\n",
			readTotalCount,writeTotalCount);
	}
#endif

	if (exState==EX_STATE_DOWN) return;
	exContext=oldContext;
}

#define LLIO_SUBPROCESS_WRITE_TO_DEVICE \
	do { \
		r=llioWrite(iface, \
			(unsigned char *)forward.buffer+forward.offset, \
			forward.datalen,exact); \
		if (r<0 || (exact && r!=forward.datalen)) { \
			LOG_ERROR(cEXBP,0,cCauseFuncFailed, \
				"llioSubprocess: llioWrite returns %d, " \
				"expected=%d!\n",r,forward.datalen); \
			goto abort; \
		} \
		forward.offset+=r; \
		forward.datalen-=r; \
		pollHit=1; \
	} while(0)

STATUS ExMgr::llioSubprocess(ExInterface iface,int fdSession) {
	int fdForward[2]={ERROR,ERROR};		// [0]=reading, [1]=writing.
	int fdReverse[2]={ERROR,ERROR};
	// int fdConsole[2]={ERROR,ERROR};
	fdInit(&llioInterface[iface].fdPipe.read);
	fdInit(&llioInterface[iface].fdPipe.write);
	// fdInit(&llioInterface[iface].fdConsole);

	if (!llioInterface[iface].active) {
		return OK;
	}

	/* TODO: A graceful way to pass child error messages back to
	 * the parent process, in case the console gets redirected.
	 * Be careful not to deadlock! */
	// pipe(fdConsole);

	if (fdSession==ERROR) {
		if (pipe(fdForward)<0 || pipe(fdReverse)<0) {
			LOG_ERROR(cEXBP,0,cCauseNoMem,
				"llioSubprocess: pipe() failed!\n");
			fdClose(&fdForward[0]);
			fdClose(&fdForward[1]);
			fdClose(&fdReverse[0]);
			fdClose(&fdReverse[1]);
			// fdClose(&fdConsole[0]);
			// fdClose(&fdConsole[1]);
			return ERROR;
		}
	}

	llioInterface[iface].pid=fork();
	if (llioInterface[iface].pid) {
		/* Parent process:
		 * Write forward data, read reverse data. */
		if (fdSession==ERROR) {
			fdClose(&fdForward[0]);
			fdClose(&fdReverse[1]);
		}

		if (llioInterface[iface].pid<0) {
			LOG_ERROR(cEXBP,0,cCauseNoMem,
				"llioSubprocess: fork() failed!\n");
			if (fdSession==ERROR) {
				fdClose(&fdForward[1]);
				fdClose(&fdReverse[0]);
			}
			return ERROR;
		}

		if (fdSession==ERROR) {
			llioInterface[iface].fdPipe.write=fdForward[1];
			llioInterface[iface].fdPipe.read=fdReverse[0];
			fdRegister(llioInterface[iface].fdPipe.read,1,-1);
		}
		if (iface==EX_INTERFACE_PRINT) {
			llioStartPrintSubprocessWaitTimer();
		}
		return OK;
	}

	LOG_ENTRY(cEXBP,0,"llioSubprocess(iface=%d)\n",iface);

	/* TODO: Keep track of and close all other FDs that aren't
	 * needed by the child process? */

	/* Child process:
	 * Read forward data from the pipe, write it to the device.
	 * Read reverse data from the device, write it to the pipe. */
	if (fdSession!=ERROR) {
		llioInterface[iface].fdPipe.read=fdSession;
		llioInterface[iface].fdPipe.write=fdSession;
	} else {
		llioInterface[iface].fdPipe.read=fdForward[0];
		fdClose(&fdForward[1]);
		fdClose(&fdReverse[0]);
		llioInterface[iface].fdPipe.write=fdReverse[1];
	}

	int lenBuffer=LLIO_MLC_BUFFER_LEN;
	if (iface==EX_INTERFACE_PRINT) lenBuffer=LLIO_PRINT_BUFFER_LEN;
	// int shouldTryToReadExtra=llioShouldTryToReadExtra(iface);
	struct {
		char *buffer;
		int offset;
		int datalen;
	} forward={0,0,0}, reverse={0,0,0};
	forward.buffer=new char[lenBuffer];
	reverse.buffer=new char[lenBuffer];
	if (!forward.buffer || !reverse.buffer) {
		LOG_ERROR(cEXBP,0,cCauseNoMem,
			"llioSubprocess: can't allocate buffers!\n");
		goto abort;
	}

  {
	time_t pollLastHitTime=0;
	enum PollState {
		POLL_STATE_HIGH,
		POLL_STATE_MEDIUM,
		POLL_STATE_LOW
	} pollState=POLL_STATE_HIGH;
	int pollHit=1,pollNow=0;

	int grcCount,grcFlags,grcOldFlags,grcLastLength;
	char *grcLastData;
	llioInterface[iface].pTransport->resetReverseCount();
	llioInterface[iface].pTransport->
		getReverseCount(0,0,&grcCount,&grcFlags);

	struct timeval timeout,*pTimeout;
	fd_set rset,wset;
	int r,fdn=llioInterface[iface].fdDevice;
	if (fdn<llioInterface[iface].fdPipe.read) fdn=llioInterface[iface].fdPipe.read;
	if (fdn<llioInterface[iface].fdPipe.write) fdn=llioInterface[iface].fdPipe.write;
	fdn++;

    while (42) {
	if (pollHit) {
		pollLastHitTime=time(0);
		if (pollState!=POLL_STATE_HIGH) {
			pollState=POLL_STATE_HIGH;
			LOG_INFO(cEXBP,0,"llioSubprocess: pollState=%d.\n",
				pollState);
		}
		pollHit=0;
	} else {
		int threshold=LLIO_POLL_THRESHOLD_MEDIUM;
		PollState nextState=POLL_STATE_MEDIUM;
		if (pollState>=POLL_STATE_MEDIUM) {
			threshold=LLIO_POLL_THRESHOLD_LOW;
			nextState=POLL_STATE_LOW;
		}
		time_t t=time(0);
		if (pollLastHitTime<(t-threshold) &&
		    nextState!=pollState) {
			pollState=nextState;
			LOG_INFO(cEXBP,0,"llioSubprocess: pollState=%d.\n",
				pollState);
		}
	}

	int exact=0;
	FD_ZERO(&rset);
	FD_ZERO(&wset);
	pTimeout=0;
	if (!forward.datalen) {
		FD_SET(llioInterface[iface].fdPipe.read,&rset);
	} else if (llioInterface[iface].fdDevice>=0) {
		FD_SET(llioInterface[iface].fdDevice,&wset);
	}
	if (reverse.datalen) {
		FD_SET(llioInterface[iface].fdPipe.write,&wset);
	} else if (llioInterface[iface].fdDevice>=0 &&
	    (pollState==POLL_STATE_HIGH || pollNow)) {
		FD_SET(llioInterface[iface].fdDevice,&rset);
		pollNow=0;
	} else {
		timeout.tv_sec=0;
		if (pollState==POLL_STATE_HIGH) {
			timeout.tv_usec=LLIO_POLL_RATE_HIGH;
		} else if (pollState==POLL_STATE_MEDIUM) {
			timeout.tv_usec=LLIO_POLL_RATE_MEDIUM;
		} else /* if (pollState==POLL_STATE_LOW) */ {
			timeout.tv_usec=LLIO_POLL_RATE_LOW;
		}
		pTimeout=&timeout;
		pollNow=1;
	}

	r=select(fdn,&rset,&wset,0,pTimeout);
	if (r<0) {
		LOG_ERROR(cEXBP,0,cCauseFuncFailed,
			"llioSubprocess: select() failed!\n");
		goto abort;
	}

	/* Service forward (host->device) direction. */
	exact=0;
	if (forward.datalen) {
		LLIO_SUBPROCESS_WRITE_TO_DEVICE;
	} else if (FD_ISSET(llioInterface[iface].fdPipe.read,&rset)) {
		forward.offset=0;
	    do {
		grcLastData=forward.buffer+forward.offset+forward.datalen;
		if (!(grcFlags&ExTransport::GRC_FLAG_PARTIAL_DATA_OK)) {
			exact=1;
		} else if (llioInterface[iface].overrideMaxForwardPacketSize) {
			grcCount=llioInterface[iface].
				overrideMaxForwardPacketSize;
		}
		if (grcCount<=0) {
			r=0;
		} else {
			if (forward.offset+forward.datalen+grcCount>lenBuffer) {
				LOG_ERROR(cEXBP,0,0,
					"llioSubprocess: forward.offset=%d + "
					"forward.datalen=%d + grcCount=%d "
					"> lenBuffer=%d!\n",forward.offset,
					forward.datalen,grcCount,lenBuffer);
				goto abort;
			}
			r=fdRead(llioInterface[iface].fdPipe.read,
				(unsigned char *)grcLastData,grcCount,exact);
			if (r<=0 || (exact && r!=grcCount)) {
				if ((exact && r) || (!exact && r<0)) {
				    LOG_ERROR(cEXBP,0,cCauseFuncFailed,
					"llioSubprocess: fdRead returns %d, "
					"expected=%d!\n",r,grcCount);
				}
				goto abort;
			}
		}
		grcLastLength=r;
		forward.datalen+=r;
		grcOldFlags=grcFlags;
		llioInterface[iface].pTransport->
			getReverseCount(grcLastData,grcLastLength,
			&grcCount,&grcFlags);
		LLIO_SUBPROCESS_WRITE_TO_DEVICE;
	    } while (!(grcOldFlags&ExTransport::GRC_FLAG_END_OF_TRANSACTION));
	}

	/* Service reverse (device->host) direction. */
	// TODO: Double the reverse buffer size, buffer multiple reverse
	//   packets until less than one buffer-size of free space remains.
	//   For libusb, try to read maximum buffer size.
	// TODO: For USB, don't read data if there's no IN endpoint.
	exact=0;
	if (reverse.datalen) goto writeToPipe;
	if (llioInterface[iface].fdDevice<0) {
		if (llioReverseDataIsAvailable(iface)) goto readFromDevice;
	} else if (FD_ISSET(llioInterface[iface].fdDevice,&rset)) {
readFromDevice:
		reverse.offset=0;
		if (llioForwardToReverse()==ERROR) {
			LOG_ERROR(cEXBP,0,cCausePeriphError,"llioSubprocess: "
				"llioForwardToReverse() failed!\n");
			goto abort;
		}

	    do {
		grcLastData=reverse.buffer+reverse.datalen;
		if (!(grcFlags&ExTransport::GRC_FLAG_PARTIAL_DATA_OK)) {
			exact=1;
			if (!reverse.datalen) exact=-1;
		}
		if (grcCount<=0) {
			r=0;
		} else {
			if (reverse.datalen+grcCount>lenBuffer) {
				LOG_ERROR(cEXBP,0,0,"llioSubprocess: "
					"reverse.datalen=%d + grcCount=%d "
					"> lenBuffer=%d!\n",
					reverse.datalen,grcCount,lenBuffer);
				goto abort;
			}
			r=llioRead(iface,(unsigned char *)grcLastData,
				grcCount,exact);
			if (!r && exact<0) break;
			if (r<0 || (exact && r!=grcCount)) {
				LOG_ERROR(cEXBP,0,cCauseFuncFailed,
					"llioSubprocess: llioRead returns %d, "
					"expected=%d!\n",r,grcCount);
				goto abort;
			}
		}
		grcLastLength=r;
		reverse.datalen+=r;
		grcOldFlags=grcFlags;
		llioInterface[iface].pTransport->
			getReverseCount(grcLastData,grcLastLength,
			&grcCount,&grcFlags);
		pollHit=1;
	    } while (!(grcOldFlags&ExTransport::GRC_FLAG_END_OF_TRANSACTION));

		if (llioReverseToForward()==ERROR) {
			LOG_ERROR(cEXBP,0,cCausePeriphError,"llioSubprocess: "
				"llioReverseToForward() failed!\n");
			goto abort;
		}

		if (reverse.datalen) {
writeToPipe:
			/* Never "exact", to prevent deadlocks. */
			r=fdWrite(llioInterface[iface].fdPipe.write,
				(unsigned char *)reverse.buffer+reverse.offset,
				reverse.datalen,0);
			if (r<0) {
				if (errno==EPIPE &&
				    iface==EX_INTERFACE_PRINT) {
					LOG_WARN(cEXBP,0,
						"llioSubprocess: fdWrite "
						"returns %d, discarding "
						"reverse data!\n",r);
					r=reverse.datalen;
				} else {
					LOG_ERROR(cEXBP,0,cCauseFuncFailed,
						"llioSubprocess: fdWrite "
						"returns %d!\n",r);
					goto abort;
				}
			}
			reverse.offset+=r;
			reverse.datalen-=r;
		}
	}
    }
  }
abort:
	exit(0);
	return ERROR;
}

int ExMgr::llioGenericRead(ExInterface iface,int fd,
    unsigned char *data,int datalen,int exact) {
	int datalen0=datalen,r,countup=0,retryCount=0;
	int isDevice=(iface!=EX_INTERFACE_NONE);
	int returnOnZeroByteRead=(isDevice && exact<0);    // ==firstChunk
	PolledTimer timer;
	struct timeval timeout={LLIO_READ_LOOP_TIMEOUT,0};
	timer.setTimeout(&timeout);

	if (fd==ERROR) {
retryNonFd:
		r=_llioRead(iface,data,datalen,exact);
	} else {
		if (returnOnZeroByteRead || !exact) {
			fdSetNonblocking(fd);
		} else {
retryFd:
			fdSetBlocking(fd);
			alarm(LLIO_READ_TIMEOUT);
		}
		r=read(fd,(char *)data,datalen);
	}

	if (r<=0) {
		if (r<0) {
			if (errno!=EAGAIN && errno!=EINTR &&
			    errno!=EIO && errno!=ETIMEDOUT) {
				if (!countup) countup=r;
				goto abort;
			}
			r=0;
		}
		if (!r && (returnOnZeroByteRead || !isDevice)) {
			goto abort;
		}
	}

	countup+=r;
	if (exact && r<datalen && !timer.isTimedOut()) {
		retryCount++;
		if (r && iface!=EX_INTERFACE_NONE) {
		    LOG_WARN(cEXBP,0,
			"llioGenericRead(datalen=%d): "
			"retrying (r=%d,countup=%d,retryCount=%d).\n",
			datalen0,r,countup,retryCount);
		}
		data+=r;
		datalen-=r;
		returnOnZeroByteRead=0;
		if (fd!=ERROR) goto retryFd;
		goto retryNonFd;
	}
abort:
	alarm(0);
	return countup;
}

int ExMgr::llioGenericWrite(ExInterface iface,int fd,
    const unsigned char *data,int datalen,int exact) {
	int r;
	if (fd==ERROR) {
		r=_llioWrite(iface,data,datalen,exact);
	} else {
		if (!exact) {
			fdSetNonblocking(fd);
		} else {
			fdSetBlocking(fd);
			alarm(LLIO_WRITE_TIMEOUT);
		}
		r=write(fd,(char *)data,datalen);
		if (r<0 && errno==EAGAIN) r=0;
	}
	alarm(0);
	return r;
}

/*****************************************************************************\
| ParMgr class
\*****************************************************************************/

#ifndef PAR_PLATFORM_NONE

#define ffs ffs__ExMgr		// Workaround link failure on FreeBSD.
#include <ParPort.h>

#define DEFAULT_BASEHIGH(baseLow) ((baseLow)+0x400)

class ParMgr: public ExMgr {
    protected:
	int baseLow;
	int baseHigh;
	int portType;
	int forwardHwAssist;
	int reverseHwAssist;
	int allowBoundedEcp;
	ParPort *pParPort;

    public:
	ParMgr(void) {
		baseLow=0x378;
		baseHigh=DEFAULT_BASEHIGH(baseLow);
		portType=ParPort::PORTTYPE_UNKNOWN;
		forwardHwAssist=reverseHwAssist=1;
		allowBoundedEcp=1;
		pParPort=0;
	}
#ifdef JD_DEBUGLITE
	virtual void llioDump(void);
#endif
    protected:
	virtual void argProcess(char *arg);
    public:
	static void printOptions(void);
    protected:
	virtual void llioInit(void);
	virtual char *llioGetDeviceID(int *pOpened) {
		char *s=(char *)pParPort->getDeviceID();
		if (!s) s=(char *)pParPort->getDeviceID();
		return s;
	}
	virtual STATUS llioSetup(ExInterface iface);
	virtual void llioReset(void) { pParPort->terminate(); }
	virtual int llioGetDrainBufferSize(int proposed) {
		return 1;
	}
	virtual int _llioReverseDataIsAvailable(ExInterface iface) {
		return pParPort->statusReverseDataIsAvailable();
	}
	virtual STATUS llioForwardToReverse(void) {
		return pParPort->startReverse();
	}
	virtual STATUS llioReverseToForward(void) {
		return pParPort->finishReverse();
	}
	virtual int llioRead(ExInterface iface,
	    unsigned char *data,int datalen,int extra) {
		return pParPort->read(data,datalen);
	}
	virtual int llioWrite(ExInterface iface,
	    const unsigned char *data,int datalen,int extra) {
		return pParPort->write(data,datalen);
	}
};

#ifdef JD_DEBUGLITE
void ParMgr::llioDump(void) {
	ExMgr::llioDump();
	printf("baseLow=0x%3.3X\n",baseLow);
	printf("baseHigh=0x%3.3X\n",baseHigh);
	printf("portType=0x%3.3X\n",portType);
	printf("forwardHwAssist=%d\n",forwardHwAssist);
	printf("reverseHwAssist=%d\n",reverseHwAssist);
	printf("allowBoundedEcp=%d\n",allowBoundedEcp);
	if (pParPort) {
		printf("\nParPort:\n");
		pParPort->dump();
	}
}
#endif

void ParMgr::argProcess(char *arg) {
	if (!strcmp(arg,"-base")) {
		baseLow=argGetInt(arg);
		baseHigh=DEFAULT_BASEHIGH(baseLow);

	} else if (!strcmp(arg,"-basehigh")) {
		baseHigh=argGetInt(arg);

	} else if (!strcmp(arg,"-porttype")) {
		forwardHwAssist=reverseHwAssist=1;
		arg=argGetString(arg);
		if (!strcmp(arg,"spp")) {
			portType=ParPort::PORTTYPE_SPP;
		} else if (!strcmp(arg,"bpp")) {
			portType=ParPort::PORTTYPE_BPP;
		} else if (!strcmp(arg,"ecp")) {
			portType=ParPort::PORTTYPE_ECP;
		} else if (!strcmp(arg,"ecphwfwd")) {
			portType=ParPort::PORTTYPE_ECPHW;
			reverseHwAssist=0;
		} else if (!strcmp(arg,"ecphwrev")) {
			portType=ParPort::PORTTYPE_ECPHW;
			forwardHwAssist=0;
		} else if (!strcmp(arg,"ecphw")) {
			portType=ParPort::PORTTYPE_ECPHW;
		} else {
			syntaxError(arg);
		}

	} else if (!strcmp(arg,"-nobecp")) {
		allowBoundedEcp=0;

	} else if (!strcmp(arg,"-forcebecp")) {
		allowBoundedEcp=-1;

	} else {
		ExMgr::argProcess(arg);
	}
}

void ParMgr::printOptions(void) {
	printf(
"    -base 0x<addr>      -- I/O port base address (default=0x378)\n"
"    -basehigh 0x<addr>  -- ECP high base address (default=base+0x400)\n"
"    -device <dev>       -- Path and filename of device node (optional)\n"
"    -porttype {spp,bpp,ecp,ecphwfwd,ecphwrev,ecphw}\n"
"                        -- Overrides port-type detection (default=ecphw)\n"
"    -nobecp, -forcebecp -- Disables or forces use of bounded-ECP mode\n"
		);
}

void ParMgr::llioInit(void) {
	ExMgr::llioInit();

	pParPort=new ParPort(baseLow,baseHigh,portType);
	if (!pParPort) {
		LOG_ERROR_FATAL(cEXTP,0,cCauseNoMem);
	}
	if (pParPort->getPortType()<=ParPort::PORTTYPE_UNKNOWN) {
		LOG_ERROR_FATAL(cEXTP,0,cCauseBadParm,
			"llioInit: No parallel port detected at "
			"baseLow=0x%X, baseHigh=0x%X!\n",
			baseLow,baseHigh);
	}
	if (portType!=ParPort::PORTTYPE_SPP &&
	    pParPort->getPortType()==ParPort::PORTTYPE_SPP) {
		LOG_ERROR(cEXBP,0,0,
"Your parallel port should be set to ECP or bidirectional (BPP or PS/2)\n"
"for proper/optimal operation.  Check your BIOS settings.\n"
			);
	}

	if (!llioPossibleNameCount) {
		llioAddPossibleName("/dev/null");
	}

	int pt=pParPort->getPortType();
	char *pts="???";
	if (pt==ParPort::PORTTYPE_ECPHW) {
		if (forwardHwAssist && reverseHwAssist) {
			pts="ecphw";
		} else if (forwardHwAssist) {
			pts="ecphwfwd";
		} else if (reverseHwAssist) {
			pts="ecphwrev";
		}
		goto logEcp;
	} else if (pt==ParPort::PORTTYPE_ECP) {
		pts="ecp";
logEcp:
		LOG_SYSLOG(cEXBP,0,"Using parallel port "
			"'-porttype %s -base 0x%3.3X -basehigh 0x%3.3X'.\n",
			pts,baseLow,baseHigh);

	} else if (pt==ParPort::PORTTYPE_BPP) {
		pts="bpp";
		goto logNonEcp;
	} else if (pt==ParPort::PORTTYPE_SPP) {
		pts="spp";
		goto logNonEcp;
	} else {
logNonEcp:
		LOG_SYSLOG(cEXBP,0,"Using parallel port "
			"'-porttype %s -base 0x%3.3X'.\n",
			pts,baseLow);
	}
}

STATUS ParMgr::llioSetup(ExInterface iface) {
	/* We don't currently support raw printing over parallel. */
	LOG_ASSERT(iface==EX_INTERFACE_MLC,cEXBP,0,0);

	/* That said, we do use the raw-print interface to hide the
	 * device node FD. */
	llioInterface[EX_INTERFACE_PRINT].fdDevice=
		llioInterface[iface].fdDevice;
	fdInit(&llioInterface[iface].fdDevice);

	/* Pick forward and reverse modes.  Default to bounded ECP mode. */
	int forwardMode=ParPort::MODE_BOUNDED_ECP;
	int reverseMode=ParPort::MODE_BOUNDED_ECP;
	int disableHwAssistForNow=0;

	/* Allow user to disable bounded ECP mode. */
	if (!allowBoundedEcp) {
		forwardMode=reverseMode=ParPort::MODE_ECP;
	}

	/* For these products, use regular ECP mode (unless the user
	 * really wants bounded ECP mode) for forward data and nibble
	 * mode for reverse data. */
	if (strstr(llioDeviceID,"MDL:OfficeJet;") ||
	    strstr(llioDeviceID,"MDL:OfficeJet LX;") ||
	    strstr(llioDeviceID,"MDL:OfficeJet Series 300;") ||
	    0) {
		if (allowBoundedEcp>=0) {
			forwardMode=ParPort::MODE_ECP;
		}
		reverseMode=ParPort::MODE_NIBBLE;
		// Prevents MLC command reply timeouts when printing:
		disableHwAssistForNow=1;
	}

	/* ECP reverse isn't possible for unidirectional parallel ports,
	 * so use nibble mode instead. */
	if (!pParPort->portTypeIsBidirectional()) {
		reverseMode=ParPort::MODE_NIBBLE;
	}

	/* Enable hardware-assisted ECP transfers if possible and desired. */
	if (pParPort->portTypeIsEcpWithHwAssist() && !disableHwAssistForNow) {
		if (forwardHwAssist && ParPort::modeIsEcp(forwardMode)) {
			forwardMode|=ParPort::MODE_HW_ASSIST;
		}
		if (reverseHwAssist && ParPort::modeIsEcp(reverseMode)) {
			reverseMode|=ParPort::MODE_HW_ASSIST;
		}
	}

	/* Try to negotiate into the new forward mode. */
	if (pParPort->setModes(forwardMode,reverseMode)==ERROR) {
		// If bounded ECP fails, then try to fall back to regular ECP.
		forwardMode&=~ParPort::MODE_BOUNDED_ECP;
		reverseMode&=~ParPort::MODE_BOUNDED_ECP;
		if (pParPort->setModes(forwardMode,reverseMode)==ERROR) {
			LOG_ERROR(cEXBP,0,0,"llioSetup: "
				"setModes(fwd=0x%3.3X,rev=0x%3.3X) failed!\n",
				forwardMode,reverseMode);
			return ERROR;
		}
	}

	/* The delays are needed for the LaserJet 1100. */
	struct timeval delay={0,LLIO_CH78_RESET_DELAY};

    if (!noDrain) {
	/* Do MLC reset. */
	PolledTimer::delay(&delay);
	if (pParPort->setChannel(LLIO_CHANNEL_MLC_RESET)==ERROR) {
		LOG_ERROR(cEXBP,0,0,"llioSetup: setChannel(%d) failed!\n",
			LLIO_CHANNEL_MLC_RESET);
		return ERROR;
	}
	PolledTimer::delay(&delay);
	unsigned char c=0;
	if (pParPort->write(&c,1)==ERROR) {
		LOG_ERROR(cEXBP,0,0,"llioSetup: write(reset byte) failed!\n");
		return ERROR;
	}
    }

	/* Switch to MLC packet channel. */
	PolledTimer::delay(&delay);
	if (pParPort->setChannel(LLIO_CHANNEL_MLC_PACKET)==ERROR) {
		LOG_ERROR(cEXBP,0,0,"llioSetup: setChannel(%d) failed!\n",
			LLIO_CHANNEL_MLC_PACKET);
		return ERROR;
	}
	PolledTimer::delay(&delay);

	return OK;
}

#endif

/*****************************************************************************\
| UsbMgr class
\*****************************************************************************/

#if defined(USB_PLATFORM_LINUX)

#include <sys/ioctl.h>

/* Copied somewhat from /usr/src/linux/drivers/usb/printer.c: */
#define LPIOC_GET_DEVICE_ID	_IOC(_IOC_READ ,'P',1,MAX_DEVICE_ID_SIZE)
#define LPIOC_GET_PROTOCOLS	_IOC(_IOC_READ ,'P',2,sizeof(int[2]))
#define LPIOC_SET_PROTOCOL	_IOC(_IOC_WRITE,'P',3,0)
#define LPIOC_HP_SET_CHANNEL	_IOC(_IOC_WRITE,'P',4,0)
#define LPIOC_GET_BUS_ADDRESS	_IOC(_IOC_READ ,'P',5,sizeof(int[2]))
#define LPIOC_GET_VID_PID	_IOC(_IOC_READ ,'P',6,sizeof(int[2]))

#elif defined(USB_PLATFORM_LIBUSB)

#ifndef HAVE_LIBUSB
#error HAVE_LIBUSB not defined!
#endif

#elif !defined(USB_PLATFORM_NONE)
#error USB_PLATFORM_* not defined!
#endif

#ifndef USB_PLATFORM_NONE

#define MAX_DEVICE_ID_SIZE	1024

#ifdef HAVE_LIBUSB
#include <usb.h>

#define INTERFACE_ID(altsetting) \
	(((altsetting)->bInterfaceClass<<16)| \
	 ((altsetting)->bInterfaceSubClass<<8)| \
	 ((altsetting)->bInterfaceProtocol))
#endif

// TODO: -nomlcdot4 mode fails to deactivate when the device goes away.

class UsbMgr: public ExMgr {
    protected:
	static const int VENDOR_ID_HP=0x03F0;
	static const int USB_PROTOCOL_711=1;
	static const int USB_PROTOCOL_712=2;
	static const int USB_PROTOCOL_713=3;
	static const int USB_DEVICE_ID_REQUEST=0;
	static const int USB_HP_CHANNEL_CHANGE_REQUEST=0;
	static const int INTERFACE_ID_711=0x070101;
	static const int INTERFACE_ID_712=0x070102;
	static const int INTERFACE_ID_713=0x070103;
	static const int INTERFACE_ID_OLD_MLC=0xFFFFFF;
	static const int INTERFACE_ID_MLC=0xFFD400;
	static const char LIBUSB_DIRNAME_PREFIX='%';
	static const char LIBUSB_FILENAME_PREFIX='%';

    protected:
	int llioMatchVendorID;
	int llioMatchProductID;
	int llioAllowComposite;
	int llioGlobUsb;
	int llioAllowChannelChangeRequest;
	int llioBusAddress;
	int llioDeviceAddress;
	int llioVendorID;
	int llioProductID;
	int llioSupportedProtocols;
	int llioCurrentProtocol;
	int llioDesiredProtocol;
#ifdef HAVE_LIBUSB
	struct usb_device *libusbCurrentDevice;
	struct libusbAltSetting_s {
		struct usb_interface_descriptor *desc;
		int epRead;
		int epWrite;
		int maxForwardPacketSize;
	} libusbAltSetting711,libusbAltSetting712,libusbAltSetting713,
	  libusbAltSettingOldMlc,libusbAltSettingNewMlc;
	struct {
		struct libusbAltSetting_s *pAltSetting;
		struct usb_dev_handle *handle;
	} libusbInterface[EX_INTERFACE_COUNT];
#endif
    public:
	UsbMgr(void) {
		llioMatchVendorID=ERROR;
		llioMatchProductID=ERROR;
		llioAllowComposite=1;
		llioGlobUsb=1;
		llioAllowChannelChangeRequest=1;
	}
    protected:
#ifdef JD_DEBUGLITE
	virtual void llioDump(void);
#endif
	virtual void argProcess(char *arg);
    public:
	static void printOptions(void);
    protected:
	virtual int getDefaultTryMlcDot4(void) {
#ifdef HAVE_LIBUSB
		if (libusbAltSetting713.desc ||
		    libusbAltSettingOldMlc.desc ||
		    libusbAltSettingNewMlc.desc) {
			return 1;
		}
#endif
		return (ExMgr::getDefaultTryMlcDot4() ||
			(llioSupportedProtocols&(1<<USB_PROTOCOL_713))!=0);
	}
	virtual void llioInit(void);
#ifdef HAVE_LIBUSB
	void libusbHandleInit(struct usb_dev_handle **handle) {
		*handle=0;
	}
	void libusbHandleClose(struct usb_dev_handle **handle) {
		if (*handle) {
			usb_close(*handle);
			libusbHandleInit(handle);
		}
	}
	struct usb_dev_handle *libusbHandleOpen(int bInterfaceNumber,
	    int *pOpened=0) {
		usb_dev_handle *handle=usb_open(libusbCurrentDevice);
		if (!handle) {
			LOG_ERROR(cEXBP,0,cCausePeriphError,
				"usb_open failed!\n");
		} else if (usb_claim_interface(handle,bInterfaceNumber)<0) {
			if (pOpened) {
				*pOpened=0;
				LOG_WARN(cEXBP,0,
					"Couldn't claim interface=%d!\n",
					bInterfaceNumber);
			} else {
				LOG_ERROR(cEXBP,0,cCauseFuncFailed,
					"Couldn't claim interface=%d!\n",
					bInterfaceNumber);
			}
			libusbHandleClose(&handle);
		}
		return handle;
	}
	virtual int llioGlobOne(char *filename,int flags=0) {
		if (filename[0]==LIBUSB_DIRNAME_PREFIX) {
			flags|=GLOB_NOCHECK;
		}
		return ExMgr::llioGlobOne(filename,flags);
	}
	virtual void llioGlob(void);
	virtual STATUS llioOpenOne(void);
	STATUS libusbFindDevice(char *strDirname,unsigned int lenDirname,
	    char *strFilename,unsigned int lenFilename,
	    int busAddress,int deviceAddress);
	STATUS libusbParseDescriptors(void);
#endif
	virtual STATUS llioMatchDevice(void) {
		if ((llioMatchVendorID!=ERROR &&
		     llioMatchVendorID!=llioVendorID) ||
		    (llioMatchProductID!=ERROR &&
		     llioMatchProductID!=llioProductID)) {
			LOG_INFO(cEXBP,0,"llioMatchDevice: "
				"found vendor/product ID 0x%4.4X/0x%4.4X, "
				"expected 0x%4.4X/0x%4.4X!\n",
				llioVendorID,llioProductID,
				llioMatchVendorID,llioMatchProductID);
			return ERROR;
		}
		if (llioMatchVendorID!=ERROR && llioMatchProductID!=ERROR) {
			return OK;
		}
		return ExMgr::llioMatchDevice();
	}
	virtual char *llioGetDeviceID(int *pOpened);
	virtual STATUS llioPickInterfaces(void);
	virtual STATUS llioSetup(ExInterface iface);
	STATUS llioSetProtocol(ExInterface iface);
	STATUS llioSetChannel(ExInterface iface,int channel);
#ifdef HAVE_LIBUSB
	virtual void llioReset(void) {
		libusbHandleClose(&libusbInterface[EX_INTERFACE_PRINT].handle);
		libusbHandleClose(&libusbInterface[EX_INTERFACE_MLC].handle);
	}

	// virtual int llioShouldTryToReadExtra(ExInterface iface) {
	//	return (libusbInterface[iface].handle!=0);
	// }
	virtual int _llioRead(ExInterface iface,
	    unsigned char *data,int datalen,int exact) {
		return usb_bulk_read(libusbInterface[iface].handle,
			libusbInterface[iface].pAltSetting->epRead,
			(char *)data,datalen,LLIO_READ_TIMEOUT*1000);
	}

	virtual int _llioWrite(ExInterface iface,
	    const unsigned char *data,int datalen,int exact) {
		int r=usb_bulk_write(libusbInterface[iface].handle,
			libusbInterface[iface].pAltSetting->epWrite,
			(char *)data,datalen,LLIO_WRITE_TIMEOUT*1000);

		if (r>0) {
			/* If this transfer was a multiple of the USB
			 * packet size, then add an empty runt packet
			 * to make the peripheral happy. */
			if (!(datalen%libusbInterface[iface].pAltSetting->
			      maxForwardPacketSize)) {
				_llioWrite(iface,data,0,exact);
			}

		/* Work around Linux usbdevfs bug where successful retry
		 * of timed-out usb_bulk_write has the wrong data-toggle
		 * bit, causing the device to ignore it.  For the print
		 * interface (which can NAK indefinitely on bulk-OUT
		 * transfers), keep stuffing dummy empty bulk-OUT packets
		 * until one is successfully transferred (which the device
		 * will ignre), to guarantee that the next (retry) data
		 * packet will have the correct data toggle bit. */
		} else if (r<0 && errno==ETIMEDOUT &&
		    iface==EX_INTERFACE_PRINT) {
			while (usb_bulk_write(libusbInterface[iface].handle,
				libusbInterface[iface].pAltSetting->epWrite,
				(char *)data,0,LLIO_WRITE_TIMEOUT*1000));
			r=0;
		}
		return r;
	}
#endif
};

#ifdef JD_DEBUGLITE
void UsbMgr::llioDump(void) {
	ExMgr::llioDump();
	printf("llioMatchVendorID=0x%4.4X\n",llioMatchVendorID);
	printf("llioMatchProductID=0x%4.4X\n",llioMatchProductID);
	printf("llioAllowComposite=%d\n",llioAllowComposite);
	printf("llioGlobUsb=%d\n",llioGlobUsb);
	printf("llioAllowChannelChangeRequest=%d\n",
		llioAllowChannelChangeRequest);
	printf("llioBusAddress=%d\n",llioBusAddress);
	printf("llioDeviceAddress=%d\n",llioDeviceAddress);
	printf("llioVendorID=0x%4.4X\n",llioVendorID);
	printf("llioProductID=0x%4.4X\n",llioProductID);
	printf("llioSupportedProtocols=0x%2.2X\n",llioSupportedProtocols);
	printf("llioCurrentProtocol=%d\n",llioCurrentProtocol);
	printf("llioDesiredProtocol=%d\n",llioDesiredProtocol);
#ifdef HAVE_LIBUSB
    if (libusbInterface[EX_INTERFACE_MLC].pAltSetting) {
	printf("libusbInterface[EX_INTERFACE_MLC]=0x%6.6X,\n"
		"    epRead=0x%2.2X, epWrite=0x%2.2X,\n"
		"    maxForwardPacketSize=%d\n",
		INTERFACE_ID(libusbInterface[EX_INTERFACE_MLC].pAltSetting->desc),
		libusbInterface[EX_INTERFACE_MLC].pAltSetting->epRead,
		libusbInterface[EX_INTERFACE_MLC].pAltSetting->epWrite,
		libusbInterface[EX_INTERFACE_MLC].pAltSetting->maxForwardPacketSize);
    }
    if (libusbInterface[EX_INTERFACE_PRINT].pAltSetting) {
	printf("libusbInterface[EX_INTERFACE_PRINT]=0x%6.6X,\n"
		"    epRead=0x%2.2X, epWrite=0x%2.2X,\n"
		"    maxForwardPacketSize=%d\n",
		INTERFACE_ID(libusbInterface[EX_INTERFACE_PRINT].pAltSetting->desc),
		libusbInterface[EX_INTERFACE_PRINT].pAltSetting->epRead,
		libusbInterface[EX_INTERFACE_PRINT].pAltSetting->epWrite,
		libusbInterface[EX_INTERFACE_PRINT].pAltSetting->maxForwardPacketSize);
    }
#endif
}
#endif

void UsbMgr::argProcess(char *arg) {
	if (!strcmp(arg,"-vpidmatch")) {
		int vpid=argGetInt(arg);
		llioMatchVendorID=(vpid>>16)&0xFFFF;
		llioMatchProductID=vpid&0xFFFF;
	} else if (!strcmp(arg,"-nocomp")) {
		llioAllowComposite=0;
	} else if (!strcmp(arg,"-nocompprint")) {
		llioEnablePrintInterface=0;
	} else if (!strcmp(arg,"-forcecomp")) {
		llioAllowComposite=-1;
	} else if (!strcmp(arg,"-noglobusb")) {
		llioGlobUsb=0;
	} else if (!strcmp(arg,"-nochannelchange")) {
		llioAllowChannelChangeRequest=0;
	} else {
		ExMgr::argProcess(arg);
	}
}

void UsbMgr::printOptions(void) {
#ifdef USB_PLATFORM_LINUX
	printf(
"    -device \"<dev>\"       -- Linux device node path/filename (wildcards OK)\n"
		);
#endif
#ifdef HAVE_LIBUSB
	// IMPORTANT: ptal-init greps for "libusb" to detect libusb support.
	printf(
"    -device %c<bus>%c<dev>  -- libusb bus and device numbers (optional)\n"
		,LIBUSB_DIRNAME_PREFIX,LIBUSB_FILENAME_PREFIX);
#endif
	printf(
"    -vpidmatch 0xVVVVPPPP -- Matches USB vendor and product ID\n"
		);
#ifdef HAVE_LIBUSB
	// IMPORTANT: ptal-init greps for "libusb" to detect libusb support.
	printf(
"    -nocomp, -forcecomp   -- Disables or forces composite-USB support\n"
"    -nocompprint          -- Disables composite-USB print interface\n"
"    -noglobusb            -- Disables implicit libusb device globbing\n"
		);
#endif
	printf(
"    -nochannelchange      -- Disables HP USB \"channel change\" request\n"
		);
}

void UsbMgr::llioInit(void) {
#ifndef HAVE_LIBUSB
	if (!llioPossibleNameCount) syntaxError("-device");
#else
	usb_init();
	libusbHandleInit(&libusbInterface[EX_INTERFACE_MLC].handle);
	libusbHandleInit(&libusbInterface[EX_INTERFACE_PRINT].handle);
	libusbInterface[EX_INTERFACE_MLC].pAltSetting=0;
	libusbInterface[EX_INTERFACE_PRINT].pAltSetting=0;
#endif
	ExMgr::llioInit();
}

#ifdef HAVE_LIBUSB

void UsbMgr::llioGlob(void) {
	ExMgr::llioGlob();

	usb_find_busses();
	usb_find_devices();

	if (!llioGlobUsb) return;

	struct usb_bus *currentBus=usb_busses;
	for (;currentBus;currentBus=currentBus->next) {
		int lenDirname=strlen(currentBus->dirname);

		struct usb_device *currentDevice=currentBus->devices;
		for (;currentDevice;currentDevice=currentDevice->next) {
			int lenFilename=strlen(currentDevice->filename);
			char *devnode=new char[1+lenDirname+1+lenFilename+1];
			if (!devnode) {
				LOG_ERROR(cEXBP,0,cCauseNoMem,
					"llioGlob: 'new char[%d]' failed!\n",
					1+lenDirname+1+lenFilename+1);
				return;
			}
			devnode[0]=LIBUSB_DIRNAME_PREFIX;
			memcpy(devnode+1,currentBus->dirname,lenDirname);
			devnode[1+lenDirname]=LIBUSB_FILENAME_PREFIX;
			memcpy(devnode+1+lenDirname+1,currentDevice->filename,
				lenFilename);
			devnode[1+lenDirname+1+lenFilename]=0;
			llioGlobOne(devnode);
			delete[] devnode;
		}
	}
}

STATUS UsbMgr::llioOpenOne(void) {
	if (llioName[0]!=LIBUSB_DIRNAME_PREFIX) {
#ifdef USB_PLATFORM_LINUX
		return ExMgr::llioOpenOne();
#else
		LOG_WARN(cEXBP,0,"llioOpenOne: "
			"bad libusb device specifier <%s>!\n",llioName);
		return ERROR;
#endif
	}

	char *dirname=llioName,*filename;
	int lenDirname=0,lenFilename=0;

#if 1
	if (*dirname==LIBUSB_DIRNAME_PREFIX) dirname++;
#else
	while (*dirname==LIBUSB_DIRNAME_PREFIX) dirname++;
#endif

	while (dirname[lenDirname]!=LIBUSB_FILENAME_PREFIX) {
		if (!dirname[lenDirname]) {
			LOG_ERROR(cEXBP,0,cCauseBadParm,
				"llioOpenOne: bad or missing libusb "
				"suffix in <%s>!\n",llioName);
			return ERROR;
		}
		lenDirname++;
	}

	filename=dirname+lenDirname;
	while (*filename==LIBUSB_FILENAME_PREFIX) filename++;

	while (filename[lenFilename]) lenFilename++;

	return libusbFindDevice(dirname,lenDirname,filename,lenFilename,0,0);
}

STATUS UsbMgr::libusbFindDevice(char *strDirname,unsigned int lenDirname,
    char *strFilename,unsigned int lenFilename,
    int busAddress,int deviceAddress) {
	struct usb_bus *currentBus=usb_busses;
    for (;currentBus;currentBus=currentBus->next) {
	if (strDirname) {
		if (strlen(currentBus->dirname)!=lenDirname ||
		    memcmp(currentBus->dirname,strDirname,lenDirname)) {
			continue;
		}
	} else {
		if (atoi(currentBus->dirname)!=busAddress) {
			continue;
		}
	}

	struct usb_device *currentDevice=currentBus->devices;
      for (;currentDevice;currentDevice=currentDevice->next) {
	if (strFilename) {
		if (strlen(currentDevice->filename)!=lenFilename ||
		    memcmp(currentDevice->filename,strFilename,lenFilename)) {
			continue;
		}
	} else {
		if (atoi(currentDevice->filename)!=deviceAddress) {
			continue;
		}
	}

	libusbCurrentDevice=currentDevice;
	return OK;
      }
    }

	LOG_ERROR(cEXBP,0,cCauseFuncFailed,
		"libusbFindDevice: Couldn't find device!\n");
	return ERROR;
}

STATUS UsbMgr::libusbParseDescriptors(void) {
	libusbAltSetting711.desc=0;
	libusbAltSetting712.desc=0;
	libusbAltSetting713.desc=0;
	libusbAltSettingOldMlc.desc=0;
	libusbAltSettingNewMlc.desc=0;

	if (!libusbCurrentDevice) return ERROR;

	if (llioInterface[EX_INTERFACE_MLC].fdDevice==ERROR) {
		llioBusAddress=atoi(libusbCurrentDevice->bus->dirname);
		llioDeviceAddress=atoi(libusbCurrentDevice->filename);
		llioVendorID=libusbCurrentDevice->descriptor.idVendor;
		llioProductID=libusbCurrentDevice->descriptor.idProduct;
		llioSupportedProtocols=0;
		llioDesiredProtocol=llioCurrentProtocol=0;
	}

	/* Note that we only search the first configuration descriptor. */
	for (int iindex=0;
	     iindex<libusbCurrentDevice->config->bNumInterfaces;
	     iindex++) {

	for (int asindex=0;
	     asindex<libusbCurrentDevice->config->
	      interface[iindex].num_altsetting;
	     asindex++) {
		struct usb_interface_descriptor *asdesc=
			&libusbCurrentDevice->config->
			 interface[iindex].altsetting[asindex];
		int interfaceID=INTERFACE_ID(asdesc);
		LOG_INFO(cEXBP,0,"Found interface=0x%6.6X.\n",interfaceID);
		struct libusbAltSetting_s *pAltSetting;
		if (interfaceID==INTERFACE_ID_711) {
			pAltSetting=&libusbAltSetting711;
		} else if (interfaceID==INTERFACE_ID_712) {
			pAltSetting=&libusbAltSetting712;
		} else if (interfaceID==INTERFACE_ID_713) {
			pAltSetting=&libusbAltSetting713;
		} else if (interfaceID==INTERFACE_ID_OLD_MLC &&
			   llioVendorID==VENDOR_ID_HP &&
		           llioAllowComposite) {
			pAltSetting=&libusbAltSettingOldMlc;
		} else if (interfaceID==INTERFACE_ID_MLC &&
			   llioVendorID==VENDOR_ID_HP &&
		           llioAllowComposite) {
			pAltSetting=&libusbAltSettingNewMlc;
		} else {
			continue;
		}
		pAltSetting->desc=asdesc;
		pAltSetting->epWrite=ERROR;
		pAltSetting->epRead=ERROR;

	for (int epindex=0;
	     epindex<pAltSetting->desc->bNumEndpoints;
	     epindex++) {
		struct usb_endpoint_descriptor *ep=
			&pAltSetting->desc->endpoint[epindex];
		if ((ep->bmAttributes&USB_ENDPOINT_TYPE_MASK)!=
		     USB_ENDPOINT_TYPE_BULK) continue;
		if ((ep->bEndpointAddress&USB_ENDPOINT_DIR_MASK)==
		     USB_ENDPOINT_OUT) {
			if (pAltSetting->epWrite!=ERROR) {
				LOG_WARN(cEXBP,0,"Ignoring duplicate bulk-OUT "
					"endpoint=0x%2.2X in favor of %d!\n",
					ep->bEndpointAddress,
					pAltSetting->epWrite);
			} else {
				pAltSetting->epWrite=USB_ENDPOINT_OUT|
					(ep->bEndpointAddress&
					 USB_ENDPOINT_ADDRESS_MASK);
				pAltSetting->maxForwardPacketSize=
					ep->wMaxPacketSize;
			}
		} else if ((ep->bEndpointAddress&USB_ENDPOINT_DIR_MASK)==
		     USB_ENDPOINT_IN) {
			if (pAltSetting->epRead!=ERROR) {
				LOG_WARN(cEXBP,0,"Ignoring duplicate bulk-IN "
					"endpoint=0x%2.2X in favor of %d!\n",
					ep->bEndpointAddress,
					pAltSetting->epRead);
			} else {
				pAltSetting->epRead=USB_ENDPOINT_IN|
					(ep->bEndpointAddress&
					 USB_ENDPOINT_ADDRESS_MASK);
			}
		}
	}

		if (pAltSetting->epWrite==ERROR ||
		    (pAltSetting!=&libusbAltSetting711 &&
		     pAltSetting->epRead==ERROR)) {
			LOG_WARN(cEXBP,0,"Ignoring interface=0x%6.6X "
				"because epWrite=0x%2.2X, epRead=0x%2.2X!\n",
				interfaceID,pAltSetting->epWrite,
				pAltSetting->epRead);
			pAltSetting->desc=0;
			pAltSetting->epWrite=ERROR;
			pAltSetting->epRead=ERROR;
#ifdef JD_DEBUGLITE
		} else {
			LOG_INFO(cEXBP,0,"Parsed interface=0x%6.6X, "
				"epWrite=0x%2.2X, epRead=0x%2.2X.\n",
				interfaceID,pAltSetting->epWrite,
				pAltSetting->epRead);
#endif
		}

	} }

	if (libusbAltSettingOldMlc.desc &&
	    !libusbAltSetting712.desc && !libusbAltSetting711.desc) {
		LOG_WARN(cEXBP,0,"Ignoring old-1284.4-only interface.\n");
		libusbAltSettingOldMlc.desc=0;
		libusbAltSettingOldMlc.epWrite=ERROR;
		libusbAltSettingOldMlc.epRead=ERROR;
	}

	if (!libusbAltSetting711.desc &&
	    !libusbAltSetting712.desc &&
	    !libusbAltSetting713.desc &&
	    !libusbAltSettingOldMlc.desc &&
	    !libusbAltSettingNewMlc.desc) {
		LOG_WARN(cEXBP,0,"Ignoring non-printer-class device.\n");
		return ERROR;
	}

	return OK;
}

#endif

char *UsbMgr::llioGetDeviceID(int *pOpened) {
	/* First, get some general information about the device. */

#ifdef USB_PLATFORM_LINUX
	/* Find out what we can about the device using Linux printer.c
	 * ioctl()s. */
    if (llioInterface[EX_INTERFACE_MLC].fdDevice!=ERROR) {
	int twoints[2];

	llioBusAddress=llioDeviceAddress=ERROR;
	llioVendorID=llioProductID=ERROR;
	llioSupportedProtocols=ERROR;
	llioDesiredProtocol=llioCurrentProtocol=USB_PROTOCOL_713;

	if (ioctl(llioInterface[EX_INTERFACE_MLC].fdDevice,LPIOC_GET_BUS_ADDRESS,&twoints)>=0) {
		llioBusAddress=twoints[0];
		llioDeviceAddress=twoints[1];
	}
	if (ioctl(llioInterface[EX_INTERFACE_MLC].fdDevice,LPIOC_GET_VID_PID,&twoints)>=0) {
		llioVendorID=twoints[0];
		llioProductID=twoints[1];
	}
	if (ioctl(llioInterface[EX_INTERFACE_MLC].fdDevice,LPIOC_GET_PROTOCOLS,&twoints)>=0) {
		llioSupportedProtocols=twoints[1];
		llioDesiredProtocol=llioCurrentProtocol=twoints[0];
	}
    }
#endif

#ifdef HAVE_LIBUSB
	/* If we started out using Linux printer.c, try to establish
	 * a libusb correspondence. */
	if (llioInterface[EX_INTERFACE_MLC].fdDevice!=ERROR) {
		libusbFindDevice(0,0,0,0,llioBusAddress,llioDeviceAddress);
	}
	/* Now read the descriptors via libusb. */
	if (libusbParseDescriptors()==ERROR) {
		if (llioInterface[EX_INTERFACE_MLC].fdDevice==ERROR) {
			return 0;
		}
		libusbCurrentDevice=0;
	}
#endif

	/* Now get the device ID string. */
	unsigned char deviceID[MAX_DEVICE_ID_SIZE+1];
	int len;
	int r=ERROR;

#ifdef USB_PLATFORM_LINUX
	if (llioInterface[EX_INTERFACE_MLC].fdDevice!=ERROR &&
	    ioctl(llioInterface[EX_INTERFACE_MLC].fdDevice,LPIOC_GET_DEVICE_ID,deviceID)>=0) {
		r=OK;
	}
#endif

#ifdef HAVE_LIBUSB
    if (r==ERROR && libusbCurrentDevice) {
	int cfgnum=libusbCurrentDevice->config->bConfigurationValue;
	int ifnum=0,asnum=0;
	if (libusbAltSetting713.desc) {
		ifnum=libusbAltSetting713.desc->bInterfaceNumber;
		asnum=libusbAltSetting713.desc->bAlternateSetting;
	} else if (libusbAltSetting712.desc) {
		ifnum=libusbAltSetting712.desc->bInterfaceNumber;
		asnum=libusbAltSetting712.desc->bAlternateSetting;
	} else if (libusbAltSetting711.desc) {
		ifnum=libusbAltSetting711.desc->bInterfaceNumber;
		asnum=libusbAltSetting711.desc->bAlternateSetting;
	} else if (libusbAltSettingNewMlc.desc) {
		ifnum=libusbAltSettingNewMlc.desc->bInterfaceNumber;
		asnum=libusbAltSettingNewMlc.desc->bAlternateSetting;
	} else if (libusbAltSettingOldMlc.desc) {
		ifnum=libusbAltSettingOldMlc.desc->bInterfaceNumber;
		asnum=libusbAltSettingOldMlc.desc->bAlternateSetting;
	}
	usb_dev_handle *handle=libusbHandleOpen(ifnum,pOpened);
	if (!handle) return 0;

	int wValue=cfgnum;
	int wIndex=(ifnum<<8)|(asnum);
#if 1
	// TODO: This is technically wrong, but it works better
	// with Linux:
	wIndex=ifnum;
#endif
	int maxDeviceIDSize=MAX_DEVICE_ID_SIZE;
#ifndef USB_PLATFORM_LINUX
	// This shouldn't be necessary, but FreeBSD seems to require it:
	if (usb_control_msg(handle,
	     USB_TYPE_CLASS|USB_ENDPOINT_IN|USB_RECIP_INTERFACE,
	     USB_DEVICE_ID_REQUEST,
	     wValue,wIndex,
	     (char *)deviceID,2,
	     LLIO_WRITE_TIMEOUT*1000)>=0) {
		maxDeviceIDSize=BEND_GET_WORD(deviceID);
		if (maxDeviceIDSize>MAX_DEVICE_ID_SIZE) {
			maxDeviceIDSize=MAX_DEVICE_ID_SIZE;
		}
	}
#endif
	if (usb_control_msg(handle,
	     USB_TYPE_CLASS|USB_ENDPOINT_IN|USB_RECIP_INTERFACE,
	     USB_DEVICE_ID_REQUEST,
	     wValue,wIndex,
	     (char *)deviceID,maxDeviceIDSize,
	     LLIO_WRITE_TIMEOUT*1000)>=0) {
		r=OK;
	}
	libusbHandleClose(&handle);
    }
#endif

	if (r==ERROR) {
		LOG_ERROR(cEXBP,0,cCausePeriphError,
			"llioGetDeviceID failed!\n");
		return 0;
	}

	len=BEND_GET_WORD(deviceID);
	if (len>MAX_DEVICE_ID_SIZE) len=MAX_DEVICE_ID_SIZE;
	len-=2;
	if (len<=0) {
		LOG_ERROR(cEXBP,0,cCausePeriphError,
			"llioGetDeviceID: empty string!\n");
		return 0;
	}
	deviceID[2+len]=0;
	char *copy=new char[len+1];
	if (!copy) {
		LOG_ERROR(cEXBP,0,cCauseNoMem,
			"llioGetDeviceID: 'new char[%d]' failed!\n",len+1);
	} else {
		strcpy(copy,(char *)deviceID+2);
	}
	return copy;
}

// TODO: For USB on "MDL:DESKJET 8", no MLC, no 7/1/1, no IN on 7/1/2.
/* Platform/device/interface possibilities:
- Old Linux
  - 7/1/[21] (print)
  - 7/1/3 (MLC)
- New Linux
  - 7/1/[21] (print)
  - 7/1/[32] (MLC)
- New Linux + libusb
  - 7/1/[21] (print, Linux)
  - 7/1/[32] (MLC, Linux)
  - FF/D4/00 (MLC, libusb) + 7/1/[21] (print, Linux)
- libusb (Linux support either not compiled in or not accessible)
  - 7/1/[21] (print)
  - 7/1/[32], FF/D4/00 (MLC)
  - FF/D4/00 (MLC) + 7/1/[21] (print)
*/

STATUS UsbMgr::llioPickInterfaces(void) {
	if (ExMgr::llioPickInterfaces()==ERROR) return ERROR;

#ifdef HAVE_LIBUSB
	libusbInterface[EX_INTERFACE_MLC].pAltSetting=0;
	libusbInterface[EX_INTERFACE_PRINT].pAltSetting=0;

    if (libusbCurrentDevice) {
	/* Pick interfaces(s) and alternate setting(s) we want to use. */
	if (libusbAltSettingNewMlc.desc) {
		libusbInterface[EX_INTERFACE_MLC].pAltSetting=
			&libusbAltSettingNewMlc;
		goto printComposite;

	} else if (libusbAltSettingOldMlc.desc && (llioAllowComposite<0 ||
	     (strstr(llioDeviceID,"MDL:photosmart 140 series;") ||
	      strstr(llioDeviceID,"MDL:photosmart 240 series;")))) {
		libusbInterface[EX_INTERFACE_MLC].pAltSetting=
			&libusbAltSettingOldMlc;
printComposite:
		if (libusbAltSetting712.desc) goto print712;
		if (libusbAltSetting711.desc) goto print711;

	} else if (libusbAltSetting713.desc &&
	    llioInterface[EX_INTERFACE_MLC].active) {
		libusbInterface[EX_INTERFACE_MLC].pAltSetting=
			&libusbAltSetting713;
		llioDesiredProtocol=USB_PROTOCOL_713;

	} else if (libusbAltSetting712.desc) {
		if (llioInterface[EX_INTERFACE_MLC].active &&
		    llioVendorID==VENDOR_ID_HP) {
			libusbInterface[EX_INTERFACE_MLC].pAltSetting=
				&libusbAltSetting712;
		} else {
print712:
			libusbInterface[EX_INTERFACE_PRINT].pAltSetting=
				&libusbAltSetting712;
		}
		llioDesiredProtocol=USB_PROTOCOL_712;

	} else if (libusbAltSetting711.desc) {
print711:
		libusbInterface[EX_INTERFACE_PRINT].pAltSetting=
			&libusbAltSetting711;
		llioDesiredProtocol=USB_PROTOCOL_711;
	} else {
		LOG_ERROR(cEXBP,0,cCausePeriphError,
			"llioPickInterfaces: no supported USB interface "
			"(vendor ID=0x%4.4X)!\n",llioVendorID);
		return ERROR;
	}

	/* If we found a print-only interface, then enable it. */
	if (libusbInterface[EX_INTERFACE_PRINT].pAltSetting) {
		llioInterface[EX_INTERFACE_PRINT].active=1;
	}

	/* Disable MLC interface if already disabled or not available. */
	if (!libusbInterface[EX_INTERFACE_MLC].pAltSetting ||
	    !llioInterface[EX_INTERFACE_MLC].active) {
		libusbInterface[EX_INTERFACE_MLC].pAltSetting=0;
		llioInterface[EX_INTERFACE_MLC].active=0;
	}

	/* If we have an fdDevice and only one interface, then revert
	 * to the non-libusb method. */
	if (llioInterface[EX_INTERFACE_MLC].fdDevice!=ERROR &&
	    ((!llioInterface[EX_INTERFACE_MLC].active)!=
	     (!llioInterface[EX_INTERFACE_PRINT].active))) {
		libusbInterface[EX_INTERFACE_MLC].pAltSetting=0;
		libusbInterface[EX_INTERFACE_PRINT].pAltSetting=0;
		goto pickLinux;
	}
    } else {
pickLinux:
#endif

	if (llioSupportedProtocols&(1<<USB_PROTOCOL_713) &&
	    llioInterface[EX_INTERFACE_MLC].active) {
		llioDesiredProtocol=USB_PROTOCOL_713;

	} else if (llioSupportedProtocols&(1<<USB_PROTOCOL_712)) {
		llioDesiredProtocol=USB_PROTOCOL_712;
		if (!llioInterface[EX_INTERFACE_MLC].active ||
		    llioVendorID!=VENDOR_ID_HP) {
			goto printLinux;
		}

	} else if (llioSupportedProtocols&(1<<USB_PROTOCOL_711)) {
		llioDesiredProtocol=USB_PROTOCOL_711;
printLinux:
		llioInterface[EX_INTERFACE_MLC].active=0;
		llioInterface[EX_INTERFACE_PRINT].active=1;

	} else {
		LOG_ERROR(cEXBP,0,cCausePeriphError,
			"llioPickInterfaces: no supported USB interface "
			"(supported=0x%2.2X, current=%d, "
			"vendor ID=0x%4.4X)!\n",llioSupportedProtocols,
			llioCurrentProtocol,llioVendorID);
		return ERROR;
	}
#ifdef HAVE_LIBUSB
    }
#endif

	/* The print interface has priority for the fdDevice. */
	if (llioInterface[EX_INTERFACE_MLC].fdDevice!=ERROR &&
	    llioInterface[EX_INTERFACE_PRINT].active) {
		llioInterface[EX_INTERFACE_PRINT].fdDevice=
			llioInterface[EX_INTERFACE_MLC].fdDevice;
		fdInit(&llioInterface[EX_INTERFACE_MLC].fdDevice);
	}

#ifdef HAVE_LIBUSB
	/* Work around Linux usbdevfs bug where timed-out usb_bulk_write
	 * doesn't report number of bytes successfully transferred.
	 * For the print interface (which can NAK indefinitely on
	 * bulk-OUT transfers), always send less than the maximum USB
	 * packet size at a time, to guarantee an all-or-nothing result. */
	if (llioInterface[EX_INTERFACE_PRINT].active &&
	    llioInterface[EX_INTERFACE_PRINT].fdDevice==ERROR) {
		llioInterface[EX_INTERFACE_PRINT].
			overrideMaxForwardPacketSize=
			libusbInterface[EX_INTERFACE_PRINT].pAltSetting->
			maxForwardPacketSize-1;
	}
#endif

	return OK;
}

STATUS UsbMgr::llioSetup(ExInterface iface) {
	if (llioSetProtocol(iface)==ERROR) return ERROR;

	if (llioAllowChannelChangeRequest &&
	    !strstr(llioDeviceID,"MDL:photosmart 140 series;") &&
	    !strstr(llioDeviceID,"MDL:photosmart 240 series;")) {
		int channel=LLIO_CHANNEL_MLC_PACKET;
		if (iface==EX_INTERFACE_PRINT) {
			channel=LLIO_CHANNEL_PRINT;
		} else if (llioVendorID==VENDOR_ID_HP &&
		    !strstr(llioDeviceID,"MDL:OfficeJet G") &&
		    !strstr(llioDeviceID,"MDL:OfficeJet K") &&
		    !strstr(llioDeviceID,"MDL:OfficeJet  K") &&
		    !noDrain) {
			if (llioSetChannel(iface,
			     LLIO_CHANNEL_MLC_RESET)!=ERROR) {
				/* I don't think a delay is necessary here. */
				unsigned char nullByte=0;
				llioWrite(iface,&nullByte,1,1);
			}
		}
		if (llioSetChannel(iface,channel)==ERROR) {
			LOG_ERROR(cEXBP,0,cCausePeriphError,
				"llioSetChannel(%d) failed!\n",
				LLIO_CHANNEL_MLC_PACKET);
			return ERROR;
		}
	}

	return OK;
}

STATUS UsbMgr::llioSetProtocol(ExInterface iface) {
#ifdef USB_PLATFORM_LINUX
    if (llioInterface[iface].fdDevice!=ERROR) {
	if (ioctl(llioInterface[iface].fdDevice,LPIOC_SET_PROTOCOL,
	     llioDesiredProtocol)<0) {
		if (llioSupportedProtocols>0 ||
		    (iface==EX_INTERFACE_MLC &&
		     llioDesiredProtocol!=USB_PROTOCOL_713) ||
		    (iface==EX_INTERFACE_PRINT &&
		     llioDesiredProtocol!=USB_PROTOCOL_712 &&
		     llioDesiredProtocol!=USB_PROTOCOL_711)) {
			LOG_ERROR(cEXBP,0,cCauseFuncFailed,
				"llioSetProtocol failed: "
				"llioDesiredProtocol=%d, "
				"llioSupportedProtocols=0x%2.2X!\n",
				llioDesiredProtocol,llioSupportedProtocols);
			return ERROR;
		}
	}
	llioCurrentProtocol=llioDesiredProtocol;
	return OK;
    }
#endif

#ifdef HAVE_LIBUSB
    if (libusbCurrentDevice) {
	libusbInterface[iface].handle=libusbHandleOpen(
		libusbInterface[iface].pAltSetting->desc->bInterfaceNumber);
	if (!libusbInterface[iface].handle) return ERROR;

	if (usb_set_altinterface(libusbInterface[iface].handle,
	     libusbInterface[iface].pAltSetting->desc->bAlternateSetting)<0) {
		LOG_ERROR(cEXBP,0,cCausePeriphError,
			"Couldn't set alternate setting=%d!\n",
			libusbInterface[iface].pAltSetting->desc->
			 bAlternateSetting);
		return ERROR;
	}

	return OK;
    }
#endif

	LOG_ERROR(cEXBP,0,cCauseBadState,"llioSetProtocol: no method!\n");
	return ERROR;
}

STATUS UsbMgr::llioSetChannel(ExInterface iface,int channel) {
    if (llioVendorID!=VENDOR_ID_HP) {
	if ((iface==EX_INTERFACE_MLC && channel==LLIO_CHANNEL_MLC_PACKET) ||
	    (iface==EX_INTERFACE_PRINT && channel==LLIO_CHANNEL_PRINT)) {
		return OK;
	}
	return ERROR;
    }

#ifdef USB_PLATFORM_LINUX
    if (llioInterface[iface].fdDevice!=ERROR) {
	if (ioctl(llioInterface[iface].fdDevice,LPIOC_HP_SET_CHANNEL,
	     channel)<0) {
		if (llioSupportedProtocols>0 ||
		    (iface==EX_INTERFACE_MLC &&
		     channel!=LLIO_CHANNEL_MLC_PACKET) ||
		    (iface==EX_INTERFACE_PRINT &&
		     channel!=LLIO_CHANNEL_PRINT)) {
			return ERROR;
		}
	}
	return OK;
    }
#endif

#ifdef HAVE_LIBUSB
	char newChannel;
    if (libusbInterface[iface].handle) {
	if (usb_control_msg(libusbInterface[iface].handle,
	     USB_ENDPOINT_IN|USB_TYPE_VENDOR|USB_RECIP_INTERFACE,
	     USB_HP_CHANNEL_CHANGE_REQUEST,channel,
	     libusbInterface[iface].pAltSetting->desc->bInterfaceNumber,
	     &newChannel,1,LLIO_WRITE_TIMEOUT*1000)<0) {
		return ERROR;
	}
	return OK;
    }
#endif

	LOG_ERROR(cEXBP,0,cCauseBadState,"llioSetChannel: no method!\n");
	return ERROR;
}

#endif

/*****************************************************************************\
| Main:
\*****************************************************************************/

void ExMgr::syntaxError(char *arg) {
	if (arg) {
		LOG_ERROR(cEXBP,0,cCauseBadParm,
			"problem with argument <%s>!\n",arg);
	}
	printf(
"\nSyntax: %s [mlc:]<bus>:<name> [<options>...]\n"
/* IMPORTANT: ptal-init greps for the following line: */
"<bus> is the connection type, one of: { "	/* ptal-init dependency! */
		,gArgv0);
#ifndef PAR_PLATFORM_NONE
	printf("par ");			/* ptal-init dependency! */
#endif
#ifndef USB_PLATFORM_NONE
	printf("usb ");			/* ptal-init dependency! */
#endif
	printf(
"}\n"
"<name> is the desired name or number suffix for this device\n"
"Common options:\n"
		);
	/*ExMgr::*/printOptions();

#ifndef PAR_PLATFORM_NONE
	printf(
"Valid options for 'par' (parallel-port connection):\n"
		);
	ParMgr::printOptions();
#endif
#ifndef USB_PLATFORM_NONE
	printf(
"Valid options for 'usb' (USB connection):\n"
		);
	UsbMgr::printOptions();
#endif

#if 0
	printf("\n");
#endif

	exit(1);
}

int main(int argc,char **argv) {
	char *socketSuffix;
	char *busPrefix;
	char *busSuffix;
	enum { BUS_UNKNOWN, BUS_PARALLEL, BUS_USB } bus=BUS_UNKNOWN;
	ExMgr *pMgr=0;

	/* Standard I/O file descriptors may be missing when invoked from
	 * a Linux USB hotplug script.  Let /dev/null take their place. */
	while (42) {
		int fd=open("/dev/null",O_RDWR);
		if (fd<0) break;
		if (fd>ExMgr::CONSOLE_STDERR) {
			close(fd);
			break;
		}
	}
	/* Set up syslog. */
	openlog("ptal-mlcd",LOG_NDELAY,LOG_LPR);

	gArgv0=*argv;
	argc--; argv++;
	if (argc<=0 || **argv=='-') ExMgr::syntaxError(*argv);
	socketSuffix=*argv;
	if (strstr(socketSuffix,busPrefix="mlc:")==socketSuffix) {
		socketSuffix+=strlen(busPrefix);
	}
	if (strstr(socketSuffix,busPrefix="par:")==socketSuffix) {
		bus=BUS_PARALLEL;
	} else if (strstr(socketSuffix,busPrefix="usb:")==socketSuffix) {
		bus=BUS_USB;
	} else {
		ExMgr::syntaxError(socketSuffix);
	}
	busSuffix=socketSuffix+strlen(busPrefix);
	if (!*busSuffix || strpbrk(busSuffix,":/")) {
		ExMgr::syntaxError(socketSuffix);
	}

	if (bus==BUS_PARALLEL) {
#ifndef PAR_PLATFORM_NONE
		pMgr=new ParMgr;
#endif
	} else if (bus==BUS_USB) {
#ifndef USB_PLATFORM_NONE
		pMgr=new UsbMgr;
#endif
	}
	if (!pMgr) ExMgr::syntaxError(socketSuffix);

	return pMgr->exMain(argc,argv);
}
