/* libhpojip -- HP OfficeJet image-processing library. */

/* Copyright (C) 1995-2002 Hewlett-Packard Company
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 *
 * In addition, as a special exception, Hewlett-Packard Company
 * gives permission to link the code of this program with any
 * version of the OpenSSL library which is distributed under a
 * license identical to that listed in the included LICENSE.OpenSSL
 * file, and distribute linked combinations including the two.
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * this file, you may extend this exception to your version of the
 * file, but you are not obligated to do so.  If you do not wish to
 * do so, delete this exception statement from your version.
 */

/* Original author: Mark Overton and others.
 *
 * Ported to Linux by David Paschal.
 * This file is not currently functional due to its reliance on
 * MS-Windows functionality.
 */

/******************************************************************************\
 *
 * xheader.c - Adds header to top of page data
 *
 ******************************************************************************
 *
 * Name of Global Jump-Table:
 *
 *    headerTbl
 *
 * Items in aXformInfo array passed into setXformSpec:
 *
 *    aXformInfo[IP_HEADER_SPEC] contains a pointer to a XHEADER_SPEC struct
 *    containing the specification of the header.  This type is defined in xheader.h.
 *
 * Capabilities and Limitations:
 *
 *    Puts a header on top of the page consisting of one line of black text
 *    on a white background.  This text can be appended to or replace the
 *    top of the page.  The text consists of three strings that are
 *    left-justified, centered and right-justified.
 *    After this header has been output, xheader.c merely passes the data
 *    through unchanged.
 *    iPixelsPerRow must be a multiple of 16 due to Windows bitmap functions.
 *    The page data can be bilevel, 8-bit grey or 24-bit color.  The header
 *    is output in the same format as the page data, so the rows that are
 *    output by this xform appear to be one seamless page.
 *
 * Default Input Traits, and Output Traits:
 *
 *          trait             default input             output
 *    -------------------  ---------------------  ------------------------
 *    iPixelsPerRow         * passed into output   same as default input
 *    iBitsPerPixel         * passed into output   same as default input
 *    iComponentsPerPixel   * passed into output   same as default input
 *    lHorizDPI             * passed into output   same as default input
 *    lVertDPI              * passed into output   same as default input
 *    lNumRows                used if valid        boosted by header size
 *    iNumPages               passed into output   same as default input
 *    iPageNum                passed into output   same as default input
 *
 *    Above, a "*" by an item indicates it must be valid (not negative).
 *
 * Mar 1998 Mark Overton -- wrote original code
 *
\******************************************************************************/

#include <tchar.h>
#include "hpojip.h"
#include "ipdefs.h"
#include "string.h"    /* for memset and memcpy */


#if 0
    #include "stdio.h"
    #include <tchar.h>

    #define PRINT(msg,arg1,arg2) \
        _ftprintf(stderr, msg, (int)arg1, (int)arg2)
#else
    #define PRINT(msg,arg1,arg2)
#endif

#define CHECK_VALUE 0x4ba1dace


typedef struct {
    IP_IMAGE_TRAITS traits;   /* traits of the input image */
    XHEADER_SPEC    spec;     /* specification of the header */
    PSTR      pHeader;        /* bilevel bitmap containing header */
    WORD      wHeaderRows;    /* # rows in the header */
    DWORD     dwRowsDone;     /* number of rows output so far */
    DWORD     dwBytesInRow;   /* # bytes in each raster row */
    DWORD     dwBiBytesInRow; /* # bytes in each row of bilevel */
    DWORD     dwInNextPos;    /* file pos for subsequent input */
    DWORD     dwOutNextPos;   /* file pos for subsequent output */
    DWORD     dwValidChk;     /* struct validity check value */
} HED_INST, *PHED_INST;



/*****************************************************************************\
 *
 * header_openXform - Creates a new instance of the transformer
 *
 *****************************************************************************
 *
 * This returns a handle for the new instance to be passed into
 * all subsequent calls.
 *
 * Return value: IP_DONE=success; IP_FATAL_ERROR=misc error.
 *
\*****************************************************************************/

static WORD header_openXform (
    IP_XFORM_HANDLE *pXform)   /* out: returned handle */
{
    PHED_INST g;

    INSURE (pXform != NULL);
    IP_MEM_ALLOC (sizeof(HED_INST), g);
    *pXform = g;
    memset (g, 0, sizeof(HED_INST));
    g->dwValidChk = CHECK_VALUE;
    return IP_DONE;

    fatal_error:
    return IP_FATAL_ERROR;
}



/*****************************************************************************\
 *
 * header_setDefaultInputTraits - Specifies default input image traits
 *
 *****************************************************************************
 *
 * The header of the file-type handled by the transform probably does
 * not include *all* the image traits we'd like to know.  Those not
 * specified in the file-header are filled in from info provided by
 * this routine.
 *
 * Return value: IP_DONE=success; IP_FATAL_ERROR=misc error.
 *
\*****************************************************************************/

static WORD header_setDefaultInputTraits (
    IP_XFORM_HANDLE  hXform,     /* in: handle for xform */
    PIP_IMAGE_TRAITS pTraits)    /* in: default image traits */
{
    PHED_INST g;

    HANDLE_TO_PTR (hXform, g);

    /* Insure that values we care about are correct */
    INSURE (pTraits->iBitsPerPixel > 0);
    INSURE (pTraits->iComponentsPerPixel > 0);
    INSURE (pTraits->iPixelsPerRow>0 && (pTraits->iPixelsPerRow&0x0f)==0);
    INSURE (pTraits->lHorizDPI > 0);
    INSURE (pTraits->lVertDPI > 0);

    INSURE ( (pTraits->iBitsPerPixel==1  && pTraits->iComponentsPerPixel==1)
          || (pTraits->iBitsPerPixel==8  && pTraits->iComponentsPerPixel==1)
          || (pTraits->iBitsPerPixel==24 && pTraits->iComponentsPerPixel==3) );

    g->dwBytesInRow   = (pTraits->iBitsPerPixel*pTraits->iPixelsPerRow+7) / 8;
    g->dwBiBytesInRow = (pTraits->iPixelsPerRow+7) / 8;
    g->traits = *pTraits;   /* a structure copy */
    return IP_DONE;

    fatal_error:
    return IP_FATAL_ERROR;
}



/*****************************************************************************\
 *
 * header_setXformSpec - Provides xform-specific information
 * 
 *
\*****************************************************************************/

static WORD header_setXformSpec (
    IP_XFORM_HANDLE hXform,         /* in: handle for xform */
    DWORD_OR_PVOID  aXformInfo[])   /* in: xform information */
{
    PHED_INST g;
    XHEADER_SPEC *pSpec;

    HANDLE_TO_PTR (hXform, g);
    pSpec = (XHEADER_SPEC*)aXformInfo[IP_HEADER_SPEC].pvoid;
    INSURE (pSpec->pszTypeFace != NULL);
    INSURE (pSpec->fHeightPoints > 0.0);
    g->spec = *pSpec;   /* a structure copy */

    /* allocate and copy the strings */
    if (pSpec->pszLeftStr != NULL) {
        IP_MEM_ALLOC (strlen(pSpec->pszLeftStr), g->spec.pszLeftStr);
        strcpy (g->spec.pszLeftStr, pSpec->pszLeftStr); 
    }
    if (pSpec->pszCenterStr != NULL) {
        IP_MEM_ALLOC (strlen(pSpec->pszCenterStr), g->spec.pszCenterStr);
        strcpy (g->spec.pszCenterStr, pSpec->pszCenterStr); 
    }
    if (pSpec->pszRightStr != NULL) {
        IP_MEM_ALLOC (strlen(pSpec->pszRightStr), g->spec.pszRightStr);
        strcpy (g->spec.pszRightStr, pSpec->pszRightStr); 
    }
    IP_MEM_ALLOC (strlen(pSpec->pszTypeFace), g->spec.pszTypeFace);
    strcpy (g->spec.pszTypeFace, pSpec->pszTypeFace);  

    return IP_DONE;

    fatal_error:
    return IP_FATAL_ERROR;
}



/*****************************************************************************\
 *
 * header_getHeaderBufSize- Returns size of input buf needed to hold header
 *
\*****************************************************************************/

static WORD header_getHeaderBufSize (
    IP_XFORM_HANDLE  hXform,         /* in:  handle for xform */
    DWORD           *pdwInBufLen)    /* out: buf size for parsing header */
{
    /* since input is raw pixels, there is no header, so set it to zero */
    *pdwInBufLen = 0;
    return IP_DONE;
}



/****************************************************************************\
 *
 * makeBilevelBitmap - create a bilevel bitmap of the header
 *
 * After this returns, g->pHeader points to the bitmap.
 *
\****************************************************************************/

static BOOL makeBilevelBitmap (PHED_INST g)
{
    HDC        hDC;
    HDC        hDCMem;
    HFONT      hFont;
    UINT       uHeight;
    UINT       uMargin;
    HBITMAP    hBitmap;
    TEXTMETRIC tm;
    UINT       uBMBytes;

    hDC = CreateIC (_T("DISPLAY"), NULL, NULL, NULL);

    if (hDC != NULL) {
        hDCMem = CreateCompatibleDC (hDC);

        if (hDCMem != NULL) {
            uHeight = (UINT)(g->traits.lVertDPI    *
                             g->spec.fHeightPoints *
                             (1.0/(65536.0*72.0))  + 0.5);
            hFont = CreateFont (
                    -(int)uHeight, 0,0,0,
                    FW_NORMAL, 0,0,0,
                    g->spec.wCharSet,
                    OUT_STROKE_PRECIS, CLIP_TT_ALWAYS, PROOF_QUALITY,
                    0x04 | FF_DONTCARE | VARIABLE_PITCH,
                    g->spec.pszTypeFace);

            if (hFont != NULL) {
                SelectObject (hDCMem, hFont);
                SetTextColor (hDCMem, RGB(255,255,255));
                SetBkColor (hDCMem, RGB(0,0,0));
                GetTextMetrics (hDCMem, &tm);
                uHeight = tm.tmHeight;
                uBMBytes = g->dwBiBytesInRow * uHeight;
                g->pHeader = malloc (uBMBytes);

                if (g->pHeader != NULL) {
                    memset (g->pHeader, 0, uBMBytes);
                    hBitmap = CreateBitmap
                        (g->traits.iPixelsPerRow, uHeight, 1, 1, g->pHeader);

                    if (hBitmap != NULL) {
                        SelectObject (hDCMem, hBitmap);
                        uMargin = (UINT)(g->traits.lHorizDPI   *
                                         g->spec.fMarginPoints *
                                         (1.0/(65536.0*72.0))  + 0.5);

                        /**** draw the left string ****/

                        SetTextAlign (hDCMem, TA_LEFT | TA_TOP);
                        TextOut (hDCMem, uMargin, 0, 
                                 g->spec.pszLeftStr,
                                 strlen(g->spec.pszLeftStr));

                        /**** draw the centered string ****/

                        SetTextAlign (hDCMem, TA_CENTER | TA_TOP);
                        TextOut (hDCMem,
                                 g->traits.iPixelsPerRow/2, 0, 
                                 g->spec.pszCenterStr,
                                 strlen(g->spec.pszCenterStr));

                        /**** draw the right string ****/

                        SetTextAlign (hDCMem, TA_RIGHT | TA_TOP);
                        TextOut (hDCMem,
                                 g->traits.iPixelsPerRow - uMargin, 0,
                                 g->spec.pszRightStr,
                                 strlen(g->spec.pszRightStr));

                        if (GetBitmapBits (hBitmap, uBMBytes, g->pHeader))
                            g->wHeaderRows = uHeight;  /* success! */
                        DeleteObject (hBitmap);
                    }
                }
                SelectObject (hDCMem, GetStockObject(SYSTEM_FONT));
                DeleteObject (hFont);
            }
            DeleteDC (hDCMem);
        }
        DeleteDC (hDC);
    }

    return g->wHeaderRows > 0;
}



/*****************************************************************************\
 *
 * header_getActualTraits - Parses header, and returns input & output traits
 *
\*****************************************************************************/

static WORD header_getActualTraits (
    IP_XFORM_HANDLE  hXform,         /* in:  handle for xform */
    DWORD            dwInputAvail,   /* in:  # avail bytes in input buf */
    PBYTE            pbInputBuf,     /* in:  ptr to input buffer */
    PDWORD           pdwInputUsed,   /* out: # bytes used from input buf */
    PDWORD           pdwInputNextPos,/* out: file-pos to read from next */
    PIP_IMAGE_TRAITS pIntraits,      /* out: input image traits */
    PIP_IMAGE_TRAITS pOutTraits)     /* out: output image traits */
{
    PHED_INST g;

    HANDLE_TO_PTR (hXform, g);

    /* Since there is no header, we'll report no usage of input */
    *pdwInputUsed    = 0;
    *pdwInputNextPos = 0;

    if (! makeBilevelBitmap(g)) /* generate the bitmap */
        goto fatal_error;

    *pIntraits  = g->traits;   /* structure copies */
    *pOutTraits = g->traits;

    if (pOutTraits->lNumRows>=0 && g->wHeaderRows>0) {
        pOutTraits->lNumRows += g->wHeaderRows - 1;
        /* The -1 above is because we discard the first input row.
         * See comments in the 'header_convert' function.  */
    }

    return IP_DONE | IP_READY_FOR_DATA;

    fatal_error:
    return IP_FATAL_ERROR;
}



/****************************************************************************\
 *
 * header_getActualBufSizes - Returns buf sizes needed for remainder of job
 *
\****************************************************************************/

static WORD header_getActualBufSizes (
    IP_XFORM_HANDLE hXform,          /* in:  handle for xform */
    PDWORD          pdwMinInBufLen,  /* out: min input buf size */
    PDWORD          pdwMinOutBufLen) /* out: min output buf size */
{
    PHED_INST g;

    HANDLE_TO_PTR (hXform, g);
    *pdwMinInBufLen = *pdwMinOutBufLen = g->dwBytesInRow;
    return IP_DONE;

    fatal_error:
    return IP_FATAL_ERROR;
}



/*****************************************************************************\
 *
 * transferHeaderRow - transfers one row from header to output buffer
 *
\*****************************************************************************/

static void transferHeaderRow (
    PHED_INST g,
    PBYTE     pOutputBuf)
{
    PBYTE pRow;
    PBYTE pBi;
    BYTE  bi;
    BYTE  bMask;
    int   iMore;

    pRow = g->pHeader + g->dwBiBytesInRow*g->dwRowsDone;
    iMore = g->dwBiBytesInRow;

    switch (g->traits.iBitsPerPixel) {
        case 1: {   /* bilevel */

            if (g->spec.rgbWhite.rgbRed & 1) {
                /* need to flip all the pixels */
                ULONG *pSource, *pDest;
                pSource = (ULONG*)pRow;
                pDest = (ULONG*)pOutputBuf;
                for ( ; iMore>0; iMore-=4)
                    *pDest++ = ~*pSource++;
            } else {
                /* just a straight copy */
                memcpy (pOutputBuf, pRow, g->dwBytesInRow);
            }
            break;
        }

        case 8: {   /* 8-bit gray */
            PBYTE pDest;
            memset (pOutputBuf, g->spec.rgbWhite.rgbRed,
                    g->traits.iPixelsPerRow);
            pBi = pRow;
            pDest = pOutputBuf;
            for ( ; iMore>0; iMore-=1) {
                bi = *pBi;
                for (bMask=0x80u; bMask!=0; bMask>>=1) {
                    if (bi & bMask)
                        *pDest = g->spec.rgbBlack.rgbRed;
                    pDest += 1;
                }
                pBi += 1;
            }
            break;
        }

        case 24: {   /* 24-bit color (3 component) */
            PBYTE pDest;
            int nwhite=0, nblack=0;
            pBi = pRow;
            pDest = pOutputBuf;
            for ( ; iMore>0; iMore-=1) {
                bi = *pBi;
                for (bMask=0x80u; bMask!=0; bMask>>=1) {
                    *(DWORD*)pDest = (bi & bMask)
                        ? (nblack++,*(DWORD*)&(g->spec.rgbBlack))
                        : (nwhite++,*(DWORD*)&(g->spec.rgbWhite));
                    pDest += 3;
                }
                pBi += 1;
            }
            break;
        }
    }  /* end of switch */
}



/*****************************************************************************\
 *
 * header_convert - Converts one row
 *
\*****************************************************************************/

static WORD header_convert (
    IP_XFORM_HANDLE hXform,
    DWORD           dwInputAvail,     /* in:  # avail bytes in in-buf */
    PBYTE           pbInputBuf,       /* in:  ptr to in-buffer */
    PDWORD          pdwInputUsed,     /* out: # bytes used from in-buf */
    PDWORD          pdwInputNextPos,  /* out: file-pos to read from next */
    DWORD           dwOutputAvail,    /* in:  # avail bytes in out-buf */
    PBYTE           pbOutputBuf,      /* in:  ptr to out-buffer */
    PDWORD          pdwOutputUsed,    /* out: # bytes written in out-buf */
    PDWORD          pdwOutputThisPos) /* out: file-pos to write the data */
{
    PHED_INST g;
    BOOL      bAteRow, bSentRow;
    WORD      wRetVal;

    HANDLE_TO_PTR (hXform, g);

    *pdwInputUsed = *pdwOutputUsed = 0;
    *pdwInputNextPos  = g->dwInNextPos;
    *pdwOutputThisPos = g->dwOutNextPos;
    bAteRow = bSentRow = FALSE;

    /**** Check if we should copy input row to output row ****/

    if (pbInputBuf!=NULL &&
        (g->spec.bOverlay || g->dwRowsDone>=g->wHeaderRows)) {
        /* we're not flushing, and not in an appended header */
        memcpy (pbOutputBuf, pbInputBuf, g->dwBytesInRow);
        bAteRow  = TRUE;
        bSentRow = TRUE;
    } else if (g->dwRowsDone==0 && g->wHeaderRows>0) {
        /* If this is the first call, then an input row is available which
         * we did not consume above . This could crash the main routine
         * because it assumes that the first row is always consumed.
         * (Availability of following rows is controlled by IP_READY_FOR_DATA,
         * set below.)  So discard the first input row to "consume" it.
         */
        bAteRow = TRUE;
    }

    /**** Check if we should output next row of header ****/

    if (g->dwRowsDone < g->wHeaderRows) {
        transferHeaderRow (g, pbOutputBuf);
        bSentRow = TRUE;
    }

    /**** Update buffer variables ****/

    if (bAteRow) {
        INSURE (dwInputAvail >= g->dwBytesInRow);
        *pdwInputUsed     = g->dwBytesInRow;
        g->dwInNextPos   += g->dwBytesInRow;
        *pdwInputNextPos  = g->dwInNextPos;
    }

    if (bSentRow) {
        INSURE (dwOutputAvail >= g->dwBytesInRow);
        *pdwOutputUsed   = g->dwBytesInRow;
        g->dwOutNextPos += g->dwBytesInRow;

        g->dwRowsDone += 1;
    }

    /**** Compute return bits, and return them ****/

    wRetVal = (bSentRow ? IP_PRODUCED_ROW : IP_DONE)  |
              (bAteRow  ? IP_CONSUMED_ROW : 0)  |
              (g->spec.bOverlay || g->dwRowsDone>=g->wHeaderRows
                 ? IP_READY_FOR_DATA : 0);

    PRINT (_T("header_convert: Return value = %04x\n"), wRetVal, 0);
    return wRetVal;

    fatal_error:
    return IP_FATAL_ERROR;
}



/*****************************************************************************\
 *
 * header_insertedData - client inserted into our output stream
 *
\*****************************************************************************/

static WORD header_insertedData (
    IP_XFORM_HANDLE hXform,
    DWORD           dwNumBytes)
{
    fatalBreakPoint ();
    return IP_FATAL_ERROR;   /* must never be called (can't insert data) */
}



/*****************************************************************************\
 *
 * header_newPage - Tells us to flush this page, and start a new page
 *
\*****************************************************************************/

static WORD header_newPage (
    IP_XFORM_HANDLE hXform)
{
    PHED_INST g;

    HANDLE_TO_PTR (hXform, g);
    return IP_DONE;   /* can't insert page-breaks, so ignore this call */

    fatal_error:
    return IP_FATAL_ERROR;

}



/*****************************************************************************\
 *
 * header_closeXform - Destroys this instance
 *
\*****************************************************************************/

static WORD header_closeXform (IP_XFORM_HANDLE hXform)
{
    PHED_INST     g;
    XHEADER_SPEC *pSpec;

    HANDLE_TO_PTR (hXform, g);
    pSpec = &(g->spec);

    if (pSpec->pszLeftStr   != NULL) IP_MEM_FREE (pSpec->pszLeftStr);
    if (pSpec->pszCenterStr != NULL) IP_MEM_FREE (pSpec->pszCenterStr);
    if (pSpec->pszRightStr  != NULL) IP_MEM_FREE (pSpec->pszRightStr);
    if (pSpec->pszTypeFace  != NULL) IP_MEM_FREE (pSpec->pszTypeFace);
    if (g->pHeader          != NULL) IP_MEM_FREE (g->pHeader);

    g->dwValidChk = 0;
    IP_MEM_FREE (g);    /* free memory for the instance */

    return IP_DONE;

    fatal_error:
    return IP_FATAL_ERROR;
}



/*****************************************************************************\
 *
 * headerTbl - Jump-table for transform driver
 *
\*****************************************************************************/

IP_XFORM_TBL headerTbl = {
    header_openXform,
    header_setDefaultInputTraits,
    header_setXformSpec,
    header_getHeaderBufSize,
    header_getActualTraits,
    header_getActualBufSizes,
    header_convert,
    header_newPage,
    header_insertedData,
    header_closeXform
};

/* End of File */
